<?php

/**
 * Handles Purchased Order Fund Handler.
 * 
 * @since 3.1.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

if (!class_exists('WAL_Purchased_Order_Fund_Handler')) {

	/**
	 * Class.
	 */
	class WAL_Purchased_Order_Fund_Handler {

		/**
		 * Order total.
		 *
		 * @var float
		 * @since 3.1.0
		 * */
		private $order_total;

		/**
		 * Purchased order fund.
		 *
		 * @var float
		 * @since 3.1.0
		 * */
		private $purchased_order_fund = 0;

		/**
		 * Class initialization.
		 *        
		 * @param float $order_total
		 * @since 3.1.0
		 * 
		 * */
		public function __construct( $order_total ) {
			$this->order_total = floatval($order_total);

			$this->prepare_purchased_order_fund();
		}

		/**
		 * Get fund.
		 *
		 * @param int $order_id       
		 * @since 3.1.0
		 * 
		 * @return float
		 */
		public static function get_fund( $order_id ) {
			$purchased_order = new self($order_id);

			return $purchased_order->get_purchased_order_amount();
		}

		/**
		 * Get purchased order fund.
		 *
		 * @since 3.1.0
		 * @return string
		 */
		private function get_purchased_order_amount() {
			return $this->purchased_order_fund;
		}

		/**
		 * Prepare purchased order fund.
		 *
		 * @since 3.1.0
		 * @return float
		 * */
		private function prepare_purchased_order_fund() {
			if ('2' !== get_option('wal_module_product_purchase_mode', '1') || !$this->is_valid()) {
				return $this->purchased_order_fund;
			}

			$this->purchased_order_fund = $this->get_purchased_order_fund();
		}

		/*
		 * Get purchased order fund
		 *
		 * @since 3.1.0
		 * @return float.
		 */

		private function get_purchased_order_fund() {
			if ('2' === get_option('wal_module_product_purchase_order_fund_type')) {
				$this->purchased_order_fund = floatval(get_option('wal_module_product_purchase_amount'));
			} else {
				$this->purchased_order_fund = ( $this->order_total / 100 ) * floatval(get_option('wal_module_product_purchase_amount'));
			}

			return floatval($this->purchased_order_fund);
		}

		/**
		 * Is valid restriction
		 *        
		 * @since 3.1.0      
		 * @return bool
		 */
		private function is_valid() {
			$minimum_order_total = floatval(get_option('wal_module_product_purchase_order_total_minimum_amount'));
			$maximum_order_total = floatval(get_option('wal_module_product_purchase_order_total_maximum_amount'));

			if (!empty($minimum_order_total) && $minimum_order_total >= $this->order_total) {
				return false;
			}

			if (!empty($maximum_order_total) && $maximum_order_total <= $this->order_total) {
				return false;
			}

			return true;
		}
	}

}
