<?php

/**
 * Handles order total range fund handler.
 * 
 * @since 3.4.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

if (!class_exists('WAL_Order_Total_Range_Fund_Handler')) {

	/**
	 * Class.
	 * 
	 * @since 3.4.0
	 */
	class WAL_Order_Total_Range_Fund_Handler {

		/**
		 * Order total.
		 *
		 * @since 3.4.0 
		 * @var float
		 * */
		private $order_total;

		/**
		 * Fund.
		 *
		 * @since 3.4.0 
		 * @var float
		 * */
		private $fund = 0;

		/**
		 * Valid rules.
		 * 
		 * @since 3.4.0
		 * @var array
		 * */
		private $valid_rules = array();

		/**
		 * Rules.
		 * 
		 * @since 3.4.0
		 * @var array
		 * */
		private $rule = array();

		/**
		 * Class initialization.
		 *       
		 * @since 3.4.0  
		 * @var float $order_total         
		 * */
		public function __construct( $order_total ) {
			$this->order_total = floatval($order_total);

			$this->prepare_fund();
		}

		/**
		 * Get fund.     
		 *  
		 * @since 3.4.0
		 * @param float $order_total 
		 * @return float
		 */
		public static function get_fund( $order_total ) {
			$purchased_order = new self($order_total);

			return $purchased_order->get_amount();
		}

		/**
		 * Get amount.
		 *
		 * @since 3.4.0
		 * @return float
		 */
		private function get_amount() {
			return (float) $this->fund;
		}

		/**
		 * Prepare fund.
		 *
		 * @since 3.4.0
		 * @return float
		 * */
		private function prepare_fund() {
			// Return if not product purchase range mode
			if ('3' !== get_option('wal_module_product_purchase_mode', '1')) {
				return $this->fund;
			}

			$order_total_range_rules = get_option('wal_module_product_purchase_order_total_range_rules', array());
			//Return if purchase order range rules empty
			if (!wal_check_is_array($order_total_range_rules)) {
				return $this->fund;
			}

			foreach ($order_total_range_rules as $key => $order_range_rules) {
				$this->rule = $order_range_rules;
				if (!$this->is_valid_rule()) {
					continue;
				}

				$this->valid_rules[$key] = '1' == $this->rule['fund_type'] ? $this->rule['fund_value'] : $this->rule['fund_value'] / 100 * $this->order_total;
			}

			$this->prepare_amount_by_priority();
		}

		/**
		 * Prepare amount by priority
		 * 
		 * @since 3.4.0
		 * @return float
		 * */
		private function prepare_amount_by_priority() {
			// Return if not valid rules
			if (!wal_check_is_array($this->valid_rules)) {
				return $this->fund;
			}

			switch (get_option('wal_module_product_purchase_order_total_range_priority', '1')) {
				case '2':
					$this->fund = end($this->valid_rules);
					break;
				case '3':
					$this->fund = min($this->valid_rules);
					break;
				case '4':
					$this->fund = max($this->valid_rules);
					break;
				default:
					$this->fund = reset($this->valid_rules);
					break;
			}
		}

		/**
		 * Is valid rule?
		 *        
		 * @since 3.4.0      
		 * @return bool
		 */
		private function is_valid_rule() {
			$min_amount = isset($this->rule['min_order_total']) ? $this->rule['min_order_total'] : '';
			$max_amount = isset($this->rule['max_order_total']) ? $this->rule['max_order_total'] : '';
			// Return if empty minumum amount and maximum amount
			if (!$min_amount && !$max_amount) {
				return true;
			}

			if ($min_amount <= $this->order_total && $max_amount >= $this->order_total) {
				return true;
			}

			if (( !$min_amount && $max_amount ) && $max_amount >= $this->order_total) {
				return true;
			}

			if (( !$max_amount && $min_amount ) && $min_amount <= $this->order_total) {
				return true;
			}

			return false;
		}
	}

}
