<?php

/**
 * Handles the Order.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

if (!class_exists('WAL_Order_Handler')) {

	/**
	 * Class.
	 */
	class WAL_Order_Handler {

		/**
		 * Class Initialization.
		 */
		public static function init() {
			// Maybe set partial fund when the placing order is not completed.
			add_action('wp_loaded', array( __CLASS__, 'set_partial_fund_session' ));

			// May be debit partial fund from the order via shortcode checkout.
			add_action('woocommerce_checkout_update_order_meta', array( __CLASS__, 'maybe_debit_partial_fund' ), 10);
			// May be debit partial fund from the order via block checkout.
			add_action('woocommerce_store_api_checkout_order_processed', array( __CLASS__, 'maybe_debit_partial_fund' ));

			// May be add the top-up data in the order via shortcode checkout.
			add_action('woocommerce_checkout_update_order_meta', array( __CLASS__, 'maybe_add_topup_data' ), 10);
			// May be add the top-up data in the order via block checkout.
			add_action('woocommerce_store_api_checkout_order_processed', array( __CLASS__, 'maybe_add_topup_data' ), 10);

			// May be debit the amount from the user wallet in the order.
			add_action('woocommerce_process_shop_order_meta', array( __CLASS__, 'process_shop_order_meta' ), 60, 1);
			// Credit the wallet amount when the refund the order.
			add_action('woocommerce_create_refund', array( __CLASS__, 'maybe_credit_fund_manual_order_refund' ), 10, 2);
			// Redirect the cart page after Top-up placed.
			add_filter('woocommerce_get_checkout_order_received_url', array( __CLASS__, 'maybe_retain_topup_cart' ), 99999, 2);
			// Remove partial fund from woocommerce session
			add_action('woocommerce_thankyou', array( __CLASS__, 'remove_partial_fund_session' ), 10, 1);

			/**
			 * This hook is used to alter the partial fund refund hooks.
			 * 
			 * @since 1.0
			 */
			$partial_fund_order_statuses = apply_filters('wal_partial_fund_refund_hooks', array(
				'woocommerce_order_status_refunded',
				'woocommerce_order_status_cancelled',
				'woocommerce_order_status_failed',
					)
			);

			if (wal_check_is_array($partial_fund_order_statuses)) {
				foreach ($partial_fund_order_statuses as $order_status) {
					// Add the hooks for failed partial fund.
					add_action($order_status, array( __CLASS__, 'failed_partial_fund' ), 1);
				}
			}

			// May be credit the topup fund in wallet based on order status.
			$topup_order_statuses = get_option('wal_general_topup_order_status');
			if (wal_check_is_array($topup_order_statuses)) {
				foreach ($topup_order_statuses as $order_status) {
					add_action("woocommerce_order_status_{$order_status}", array( __CLASS__, 'maybe_credit_topup_fund' ), 1);
				}
			}

			/**
			 * This hook is used to alter the top-up product refund hooks.
			 * 
			 * @since 1.0
			 */
			$topup_refund_hooks = apply_filters('wal_topup_product_refund_hooks', array(
				'woocommerce_order_status_refunded',
				'woocommerce_order_status_cancelled',
				'woocommerce_order_status_failed',
					)
			);

			if (wal_check_is_array($topup_refund_hooks)) {
				foreach ($topup_refund_hooks as $hook) {
					// Add the hooks for failed topup product.
					add_action($hook, array( __CLASS__, 'failed_topup_product' ), 1);
				}
			}
		}

		/**
		 * Maybe set partial fund when the placing order is not completed.
		 * 
		 * @since 2.3.0
		 */
		public static function set_partial_fund_session() {
			// Return if the current order ID session is not exists.
			// If the partial fund session is exists.
			if (!is_object(WC()->session) || !WC()->session->get('order_awaiting_payment') || WC()->session->get('wal_partial_fund')) {
				return;
			}

			// Get the partial fund from the order.
			$order_id = WC()->session->get('order_awaiting_payment');
			$order = wc_get_order($order_id);
			if (!is_object($order)) {
				return;
			}
			//Return if current order status reached follwing order statuses
			if (!in_array($order->get_status(), array( 'completed', 'processing', 'on-hold' ))) {
				return;
			}

			$partial_fund = $order->get_meta('wal_partial_fund', true);
			if (!$partial_fund) {
				return;
			}

			WC()->session->set('wal_partial_fund', $partial_fund);
		}

		/**
		 * May be debit partial fund from the order when placing the order.
		 * 
		 * @since 1.0.0
		 * @param int/object $order_id
		 * @return void
		 * */
		public static function maybe_debit_partial_fund( $order_id ) {
			$order_id = is_object($order_id) ? $order_id->get_id() : $order_id;
			$order = wc_get_order($order_id);
			if (!is_object($order)) {
				return;
			}

			// Return if the partial fund already applied.
			if (!WC()->session->get('wal_partial_fund')) {
				return;
			}

			$fees = WC()->cart->get_fees();
			// Return if the fees is not exists in the cart.
			if (!wal_check_is_array($fees)) {
				return '';
			}

			if (!array_key_exists(WAL()->wallet_fee_name(), $fees)) {
				return '';
			}

			$amount = abs($fees[WAL()->wallet_fee_name()]->total);
			$partial_fund_log = get_option('wal_localization_wallet_partial_fund_debit_log');
			$event_message = str_replace('{order_id}', '#' . $order->get_id(), $partial_fund_log);

			/**
			 * This hook is used to alter the wallet partial debit amount.
			 *
			 * @param float $amount
			 * @since 1.0.0
			 */
			$amount = apply_filters('wal_partial_debit_amount', $amount);

			$args = array(
				'user_id' => $order->get_customer_id(),
				'order_id' => $order->get_id(),
				'amount' => $amount,
				'event_id' => 6,
				'event_message' => $event_message,
				'currency' => $order->get_currency(),
				'update_usage_total' => true,
			);

			$transaction_log_id = wal_debit_wallet_fund($args);

			// Add the note.
			/* translators: %1s- Amount, %2s - User name */
			$note = sprintf(__('The amount of %1$s debited from user %2$s', 'wallet-for-woocommerce'), wal_price($amount), $order->get_user()->display_name);
			$order->add_order_note($note);

			// Update the partial fund details.
			$order->update_meta_data('wal_wallet_fund', $amount);
			$order->update_meta_data('wal_partial_fund', $amount);

			// Update the meta.
			$order->update_meta_data('wal_wallet_fund_debited', 'yes');
			$order->update_meta_data('wal_partial_fund_debited', 'yes');
			$order->save();

			// Unset the session.
			WC()->session->set('wal_partial_fund', null);

			/**
			 * This hook is used to do extra action after order partial fund debited.
			 * 
			 * @since 1.0.0
			 * @param int $transaction_log_id
			 * @param int $order_id
			 * @param object $order
			 */
			do_action('wal_order_partial_fund_debited', $transaction_log_id, $order_id, $order);
		}

		/**
		 * May be failed the partial fund based on order status.
		 * 
		 * @since 1.0.0
		 * @param int $order_id
		 * @return void
		 * */
		public static function failed_partial_fund( $order_id ) {
			$order = wc_get_order($order_id);
			if (!is_object($order)) {
				return;
			}

			if ('yes' !== $order->get_meta('wal_partial_fund_debited')) {
				return;
			}

			$partial_fund = $order->get_meta('wal_partial_fund', true);
			if (!$partial_fund) {
				return;
			}

			$partial_fund_log = get_option('wal_localization_wallet_partial_fund_credit_log');
			$event_message = str_replace('{order_id}', '#' . $order->get_id(), $partial_fund_log);

			$args = array(
				'user_id' => $order->get_customer_id(),
				'order_id' => $order->get_id(),
				'amount' => $partial_fund,
				'event_id' => 5,
				'event_message' => $event_message,
				'currency' => $order->get_currency(),
				'update_usage_total' => true,
			);

			$transaction_log_id = wal_credit_wallet_fund($args);

			// Add the note.
			/* translators: %1s- Amount, %2s - User name */
			$note = sprintf(__('The amount of %1$s credited to user %2$s', 'wallet-for-woocommerce'), wal_price($partial_fund), $order->get_user()->display_name);
			$order->add_order_note($note);

			// Delete the meta.
			$order->delete_meta_data('wal_wallet_fund_debited');
			$order->delete_meta_data('wal_partial_fund_debited');
			$order->save();

			/**
			 * This hook is used to do extra action after order partial fund credited.
			 * 
			 * @since 1.0.0
			 * @param int $transaction_log_id
			 * @param int $order_id
			 * @param object $order
			 */
			do_action('wal_order_partial_fund_credited', $transaction_log_id, $order_id, $order);
		}

		/**
		 * May be add Top-up data in the order when placing the order.
		 * 
		 * @since 1.0.0
		 * @param int $order_id
		 * @return void
		 * */
		public static function maybe_add_topup_data( $order_id ) {
			$order_id = is_object($order_id) ? $order_id->get_id() : $order_id;
			$order = wc_get_order($order_id);
			if (!is_object($order)) {
				return;
			}

			// Return if a cart object is not initialized.
			if (!is_object(WC()->cart)) {
				return;
			}

			foreach (WC()->cart->get_cart() as $key => $value) {

				if (!isset($value['wal_topup'])) {
					continue;
				}

				// Prepare the topup data.
				$topup_data = array(
					'wal_topup_price' => floatval($value['wal_topup']['price']),
					'wal_topup_product' => $value['wal_topup']['product_id'],
					'wal_topup_mode' => $value['wal_topup']['mode'],
				);

				foreach ($topup_data as $key => $value) {
					$order->add_meta_data($key, $value);
				}
			}

			$order->save();

			WC()->session->__unset('wal_fund_amount');

			/**
			 * This hook is used to do extra action after top-up order meta updated.
			 * 
			 * @since 1.0.0
			 * @param int $order_id
			 * @param object $order
			 */
			do_action('wal_updated_topup_order_meta', $order_id, $order);
		}

		/**
		 * May be credit the top up fund in the wallet.
		 * 
		 * @since 1.0.0
		 * @param int $order_id
		 * @return void
		 * */
		public static function maybe_credit_topup_fund( $order_id ) {
			$order = wc_get_order($order_id);
			if (!is_object($order)) {
				return;
			}

			if ('manual' !== $order->get_meta('wal_topup_mode')) {
				return;
			}
			/**
			 * This hook is used to do extra action before order top-up fund credit.
			 * 
			 * @since 1.0.0
			 * @param int $order_id
			 * @param object $order
			 */
			do_action('wal_before_credit_order_topup_fund', $order_id, $order);

			// Return if the topup fund is already credited.
			if ('yes' === $order->get_meta('wal_topup_fund_credited')) {
				return;
			}

			$topup_fund = $order->get_meta('wal_topup_price');
			// Return if the topup fund is not applied.
			if (!$topup_fund) {
				return;
			}

			$partial_fund_log = get_option('wal_localization_wallet_topup_fund_credit_log');
			$event_message = str_replace('{order_id}', '#' . $order->get_id(), $partial_fund_log);

			$args = array(
				'user_id' => $order->get_customer_id(),
				'order_id' => $order->get_id(),
				'amount' => floatval($topup_fund),
				'event_id' => 3,
				'event_message' => $event_message,
				'currency' => $order->get_currency(),
				'update_topup_total' => true,
			);

			$transaction_log_id = wal_credit_wallet_fund($args);

			// Add the note.
			/* translators: %1s- Amount, %2s - User name */
			$note = sprintf(__('The topup amount of %1$s credited to user %2$s', 'wallet-for-woocommerce'), $topup_fund, $order->get_user()->display_name);
			$order->add_order_note($note);

			// Update the meta.
			$order->update_meta_data('wal_topup_fund_credited', 'yes');
			$order->save();

			self::maybe_award_topup_bonus($order, $topup_fund);
			/**
			 * This hook is used to do extra action after order top-up fund credited.
			 * 
			 * @since 1.0.0
			 * @param int $transaction_log_id
			 * @param int $order_id
			 * @param object $order
			 */
			do_action('wal_order_topup_fund_credited', $transaction_log_id, $order_id, $order);
		}

		/**
		 * May be award top-up bonus.
		 *
		 * @since 2.5.0
		 * @param Object $order 
		 * @param float $topup_fund 
		 * 
		 * @return void
		 * */
		public static function maybe_award_topup_bonus( $order, $topup_fund ) {
			if (!is_object($order)) {
				return;
			}
			// Return if the topup fund is already credited.
			if ('yes' === $order->get_meta('wal_topup_bonus_fund_credited')) {
				return;
			}

			$topup_bonus_amount = WAL_Topup_Bonus_Handler::get_amount($topup_fund);
			if (empty($topup_bonus_amount)) {
				return;
			}

			$args = array(
				'user_id' => $order->get_customer_id(),
				'order_id' => $order->get_id(),
				'amount' => $topup_bonus_amount,
				'event_id' => 21,
				'event_message' => str_replace('{order_id}', '#' . $order->get_id(), wal_topup_bonus_fund_credit_log_label()),
				'currency' => $order->get_currency(),
				'update_topup_total' => true,
			);

			$transaction_log_id = wal_credit_wallet_fund($args);

			// Add the note.
			/* translators: %1s- Amount, %2s - User name */
			$note = sprintf(__('The topup bonus amount of %1$s credited to user %2$s', 'wallet-for-woocommerce'), wal_price($topup_bonus_amount), $order->get_user()->display_name);
			$order->add_order_note($note);

			// Update the meta.
			$order->update_meta_data('wal_topup_bonus_fund_credited', 'yes');
			$order->save();

			/**
			 * This hook is used to do extra action after order top-up bonus fund credited.
			 * 
			 * @since 2.5.0
			 * @param int $transaction_log_id
			 * @param int $order_id
			 * @param object $order
			 */
			do_action('wal_order_topup_bonus_fund_credited', $transaction_log_id, $order->get_id(), $order);
		}

		/**
		 * May be failed the partial fund based on order status.
		 * 
		 * @since 1.0.0
		 * @param int $order_id
		 * @return void
		 * */
		public static function failed_topup_product( $order_id ) {
			$order = wc_get_order($order_id);
			if (!is_object($order)) {
				return;
			}

			if ('manual' !== $order->get_meta('wal_topup_mode')) {
				return;
			}
			/**
			 * This hook is used to do extra action before order top-up fund debit.
			 * 
			 * @since 1.0.0
			 * @param int $order_id
			 * @param object $order
			 */
			do_action('wal_before_debit_order_topup_fund', $order_id, $order);

			$topup_fund = $order->get_meta('wal_topup_price');
			// Return if the topup fund is not applied.
			if (!$topup_fund) {
				return;
			}

			// Return if the topup fund is not already credited.
			if ('yes' !== $order->get_meta('wal_topup_fund_credited')) {
				return;
			}

			$partial_fund_log = get_option('wal_localization_wallet_topup_fund_debit_log');
			$find = array( '{order_id}', '{order_status}' );
			$replace = array( '#' . $order->get_id(), wal_display_post_status(get_post_status($order->get_id()), false) );
			$event_message = str_replace($find, $replace, $partial_fund_log);

			$args = array(
				'user_id' => $order->get_customer_id(),
				'order_id' => $order->get_id(),
				'amount' => $topup_fund,
				'event_id' => 4,
				'event_message' => $event_message,
				'currency' => $order->get_currency(),
				'update_topup_total' => true,
			);

			$transaction_log_id = wal_debit_wallet_fund($args);

			// Add the note.
			/* translators: %1s- Amount, %2s - User name */
			$note = sprintf(__('The topup amount of %1$s debited from user %2$s', 'wallet-for-woocommerce'), wal_price($topup_fund), $order->get_user()->display_name);
			$order->add_order_note($note);

			$order->delete_meta_data('wal_topup_fund_credited');
			$order->save();

			self::maybe_debit_topup_bonus($order, $topup_fund);

			/**
			 * This hook is used to do extra action after order top-up fund debited.
			 * 
			 * @since 1.0.0
			 * @param int $transaction_log_id
			 * @param int $order_id
			 * @param object $order
			 */
			do_action('wal_order_topup_fund_debited', $transaction_log_id, $order_id, $order);
		}

		/**
		 * May be debit top-up bonus.
		 *
		 * @since 2.5.0
		 * @param Object $order 
		 * @param float $topup_fund 
		 * 
		 * @return void
		 * */
		public static function maybe_debit_topup_bonus( $order, $topup_fund ) {
			if (!is_object($order)) {
				return;
			}

			// Return if the topup fund is already not credited.
			if ('yes' !== $order->get_meta('wal_topup_bonus_fund_credited')) {
				return;
			}

			$topup_bonus_amount = WAL_Topup_Bonus_Handler::get_amount($topup_fund);
			if (empty($topup_bonus_amount)) {
				return;
			}

			$find = array( '{order_id}', '{order_status}' );
			$replace = array( '#' . $order->get_id(), wal_display_post_status(get_post_status($order->get_id()), false) );
			$event_message = str_replace($find, $replace, wal_topup_bonus_fund_debit_log());

			$args = array(
				'user_id' => $order->get_customer_id(),
				'order_id' => $order->get_id(),
				'amount' => $topup_bonus_amount,
				'event_id' => 22,
				'event_message' => $event_message,
				'currency' => $order->get_currency(),
				'update_topup_total' => true,
			);

			$transaction_log_id = wal_debit_wallet_fund($args);

			/**
			 * This hook is used to do extra action after order top-up bonus fund debited.
			 * 
			 * @since 2.5.0
			 * @param int $transaction_log_id
			 * @param object $order
			 */
			do_action('wal_order_topup_bonus_fund_debited', $transaction_log_id, $order);
		}

		/**
		 * Process shop order meta
		 * 
		 * @since 3.4.0
		 * @param int $order_id
		 */
		public static function process_shop_order_meta( $order_id ) {
			self::maybe_debit_manual_order_user_fund($order_id);
			self::maybe_debit_manual_order_user_apply_fund($order_id);
		}

		/**
		 * Maybe debit manual order user apply fund
		 * 
		 * @since 3.4.0
		 * @param int $order_id
		 */
		public static function maybe_debit_manual_order_user_apply_fund( $order_id ) {
			// Get the order object.
			$order = wc_get_order($order_id);
			if (!is_object($order)) {
				return;
			}

			// Return if the wallet amount is already debited for the order.
			if ($order->get_meta('wal_partial_fund_debited_manually', true) || !$order->get_meta('wal_partial_fund_applied', true)) {
				return;
			}

			$fund = $order->get_meta('wal_partial_fund');
			// Return if fund is empty
			if (!$fund) {
				return;
			}

			try {
				// Throw error if the current user is guest.
				if (!$order->get_customer_id()) {
					throw new Exception(__('Guest Users are not allowed to earn Funds. Hence, it is not possible to deduct funds for this order.', 'wallet-for-woocommerce'));
				}

				$wallet_id = wal_get_wallet_id_by_user_id($order->get_customer_id());
				if ($wallet_id) {
					$wallet = wal_get_wallet($wallet_id);
					if (!is_object($wallet)) {
						return;
					}
					// Throw the error if the wallet is in active. 
					if ('wal_inactive' == $wallet->get_status()) {
						throw new Exception(__('Unable to deduct the funds from this User Account due to the account being Inactive.', 'wallet-for-woocommerce'));
					}
				}

				$payment_gateway_debit_log = get_option('wal_localization_wallet_partial_fund_debit_log');
				$event_message = str_replace('{order_id}', '#' . $order->get_id(), $payment_gateway_debit_log);

				$args = array(
					'user_id' => $order->get_customer_id(),
					'order_id' => $order->get_id(),
					'amount' => $fund,
					'event_id' => 8,
					'event_message' => $event_message,
					'currency' => $order->get_currency(),
					'update_usage_total' => true,
				);

				$transaction_log_id = wal_debit_wallet_fund($args);
				if (!$transaction_log_id) {
					throw new exception(__('Amount not debited.', 'wallet-for-woocommerce'));
				}

				// Add the note.
				/* translators: %1s- Amount, %2s - User name */
				$note = sprintf(__('The amount of %1$s debited from user %2$s', 'wallet-for-woocommerce'), wal_price($order->get_meta('wal_partial_fund')), $order->get_user()->display_name);
				$order->add_order_note($note);

				// Update the meta.
				$order->update_meta_data('wal_wallet_fund_debited', 'yes');
				$order->update_meta_data('wal_partial_fund_debited', 'yes');
				$order->update_meta_data('wal_partial_fund_debited_manually', 'yes');
				$order->save();

				/**
				 * This hook is used to do extra action after gateway order fund debited.
				 * 
				 * @since 3.4.0
				 * @param int $transaction_log_id
				 * @param int $order_id
				 * @param object $order
				 */
				do_action('wal_order_fund_debited', $transaction_log_id, $order->get_id(), $order);
			} catch (Exception $ex) {
				// Add the note.
				$order->add_order_note($ex->getMessage());
			}
		}

		/**
		 * May be debit the amount from the user for manual order.
		 * 
		 * @since 2.1.0
		 * @param int $order_id
		 * */
		public static function maybe_debit_manual_order_user_fund( $order_id ) {
			// Get the order object.
			$order = wc_get_order($order_id);
			if (!is_object($order)) {
				return;
			}

			// Prevent deducting amounts another order type.
			if ('shop_order' !== $order->get_type()) {
				return;
			}

			// Return if the wallet amount is already debited for the order.
			if ($order->get_meta('wal_wallet_fund_debited', true) || $order->get_meta('wal_partial_fund_debited', true)) {
				return;
			}

			// Return if the order payment method is not a wallet.
			if ('wal_wallet' !== $order->get_payment_method()) {
				return;
			}

			try {
				// Throw error if the current user is guest.
				if (!$order->get_customer_id()) {
					throw new Exception(__('Guest Users are not allowed to earn Funds. Hence, it is not possible to deduct funds for this order.', 'wallet-for-woocommerce'));
				}

				$wallet_id = wal_get_wallet_id_by_user_id($order->get_customer_id());
				if ($wallet_id) {
					$wallet = wal_get_wallet($wallet_id);

					// Throw the error if the wallet is in active. 
					if ('wal_inactive' == $wallet->get_status()) {
						throw new Exception(__('Unable to deduct the funds from this User Account due to the account being Inactive.', 'wallet-for-woocommerce'));
					}

					// Throw the error if the wallet amount less than order total. 
					if (wal_convert_price($wallet->get_balance()) < $order->get_total()) {
						throw new Exception(__('Unable to deduct the funds from this User Account due to Insufficient Balance.', 'wallet-for-woocommerce'));
					}
				}

				$payment_gateway_debit_log = get_option('wal_localization_wallet_payment_gateway_debit_log');
				$event_message = str_replace('{order_id}', '#' . $order->get_id(), $payment_gateway_debit_log);

				$args = array(
					'user_id' => $order->get_customer_id(),
					'order_id' => $order->get_id(),
					'amount' => $order->get_total(),
					'event_id' => 8,
					'event_message' => $event_message,
					'currency' => $order->get_currency(),
					'update_usage_total' => true,
				);

				$transaction_log_id = wal_debit_wallet_fund($args);
				if (!$transaction_log_id) {
					throw new exception(__('Amount not debited.', 'wallet-for-woocommerce'));
				}

				// Add the note.
				/* translators: %1s- Amount, %2s - User name */
				$note = sprintf(__('The amount of %1$s debited from user %2$s', 'wallet-for-woocommerce'), wal_price($order->get_total()), $order->get_user()->display_name);
				$order->add_order_note($note);

				// Update the fund details.
				$order->update_meta_data('wal_gateway_total_fund', $order->get_total());
				$order->update_meta_data('wal_gateway_remaining_fund', $order->get_total());
				$order->update_meta_data('wal_wallet_fund', $order->get_total());

				// Update the meta.
				$order->update_meta_data('wal_wallet_fund_debited', 'yes');
				$order->update_meta_data('wal_gateway_fund_debited', 'yes');
				$order->save();

				/**
				 * This hook is used to do extra action after gateway order fund debited.
				 * 
				 * @since 2.1.0
				 * @param int $transaction_log_id
				 * @param int $order_id
				 * @param object $order
				 */
				do_action('wal_order_gateway_fund_debited', $transaction_log_id, $order->get_id(), $order);
			} catch (Exception $ex) {
				// Add the note.
				$order->add_order_note($ex->getMessage());
			}
		}

		/**
		 * May be credit the amount from the user for manual order refund.
		 * 
		 * @since 2.1.0
		 * @param string/int $refund
		 * @param array $args
		 * */
		public static function maybe_credit_fund_manual_order_refund( $refund, $args ) {
			if (!isset($args['order_id']) || !isset($_REQUEST['wal_is_refund'])) {
				return;
			}

			$order = wc_get_order($args['order_id']);
			if (!is_object($order) || !$order->get_customer_id()) {
				return;
			}

			$refund_credit_log = get_option('wal_localization_wallet_refund_log');
			$find = array( '{order_id}' );
			$replace = array( '#' . $order->get_id() );
			$event_message = str_replace($find, $replace, $refund_credit_log);

			$args = array(
				'user_id' => $order->get_customer_id(),
				'order_id' => $order->get_id(),
				'amount' => $args['amount'],
				'event_id' => 20,
				'event_message' => $event_message,
				'currency' => $order->get_currency(),
				'update_usage_total' => true,
			);

			$transaction_log_id = wal_credit_wallet_fund($args);
			if (!$transaction_log_id) {
				throw new exception(esc_html__('Amount not credited.', 'wallet-for-woocommerce'));
			}

			// Add the note.
			/* translators: %1s- Amount, %2s - User name */
			$note = sprintf(__('The amount of %1$s credited to user %2$s', 'wallet-for-woocommerce'), wal_price($args['amount']), $order->get_user()->display_name);
			$order->add_order_note($note);

			/**
			 * This hook is used to do extra action after gateway order fund credited.
			 * 
			 * @since 1.0.0
			 * @param int $transaction_log_id
			 * @param int $order_id
			 * @param object $order
			 */
			do_action('wal_order_gateway_fund_credited', $transaction_log_id, $args['order_id'], $order);
		}

		/**
		 * May be retain the cart that before topup and redirect to the cart page.
		 * 
		 * @since 2.6.0
		 * @param string $url
		 * @param WC_Order $order
		 * @return string
		 */
		public static function maybe_retain_topup_cart( $url, $order ) {
			// Avoid it if the option is disabled.
			if ('yes' !== get_option('wal_general_retain_cart_after_topup')) {
				return $url;
			}

			// Avoid it if the order is valid,
			// Session is not exists.
			// Guest is placed the order.
			if (!is_object($order) || !is_object(WC()->session) || !$order->get_customer_id()) {
				return $url;
			}

			$cart = maybe_unserialize(base64_decode(get_user_meta($order->get_customer_id(), 'wal_topup_retain_cart', true)));
			if (!$cart) {
				return $url;
			}

			// Set the cart contents in the cart.
			WC()->session->cart = $cart;

			// Delete the cart contents after retianed from the user meta.
			delete_user_meta($order->get_customer_id(), 'wal_topup_retain_cart');

			return wc_get_cart_url();
		}

		/**
		 * Remove partial fund from woocommerce session
		 * 
		 * @since 3.2.0
		 * @param int $order_id     
		 * 
		 * @return void
		 */
		public static function remove_partial_fund_session( $order_id ) {
			$order = wc_get_order($order_id);
			//Return if not order object
			if (!is_object($order)) {
				return;
			}
			//Return it is not object
			if (!is_object(WC()->session)) {
				return;
			}

			// Unset the session.
			WC()->session->__unset('wal_partial_fund');
			WC()->session->__unset('wal_auto_deduct_partial_fund');
		}
	}

	WAL_Order_Handler::init();
}
