<?php

/**
 * Initialize the Plugin.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Install')) {

	/**
	 * Class.
	 */
	class WAL_Install {

		/**
		 * Class initialization.
		 */
		public static function init() {
			add_action('admin_init', array( __CLASS__, 'check_version' ));
			add_filter('plugin_action_links_' . WAL_PLUGIN_SLUG, array( __CLASS__, 'settings_link' ));
		}

		/**
		 * Check current version of the plugin is updated when activating plugin, if not run updater.
		 */
		public static function check_version() {
			if (version_compare(get_option('wal_version'), WAL_VERSION, '>=')) {
				return;
			}

			self::install();
		}

		/**
		 * Install.
		 */
		public static function install() {

			WAL_Pages::create_pages(); // Create pages.
			self::set_default_values(); // Default values.
			self::maybe_create_topup_product();
			self::update_version();
		}

		/**
		 * Update current version.
		 */
		private static function update_version() {
			update_option('wal_version', WAL_VERSION);
		}

		/**
		 * Add the plugin settings link.
		 * 
		 * @return array
		 */
		public static function settings_link( $links ) {
			$setting_page_link = '<a href="' . wal_get_settings_page_url() . '">' . __('Settings', 'wallet-for-woocommerce') . '</a>';

			array_unshift($links, $setting_page_link);

			return $links;
		}

		/**
		 * Set the settings default values.
		 */
		public static function set_default_values() {

			if (!class_exists('WAL_Settings')) {
				include_once WAL_PLUGIN_PATH . '/inc/admin/menu/class-wal-settings.php';
			}

			// Get the settings.
			$settings = WAL_Settings::get_settings_pages();
			foreach ($settings as $setting) {
				$sections = $setting->get_sections();

				// Update the section settings.
				if (wal_check_is_array($sections)) {
					foreach ($sections as $section_key => $section) {
						$settings_array = $setting->get_settings($section_key);
						foreach ($settings_array as $value) {
							//Check if the default and id key is exists.
							if (isset($value['default']) && isset($value['id'])) {
								//Check if option are saved or not.
								if (get_option($value['id']) === false) {
									add_option($value['id'], $value['default']);
								}
							}
						}
					}
				} else {
					$settings_fields = $setting->get_settings($setting->get_id());
					foreach ($settings_fields as $value) {
						//Check if default and id key is exists.
						if (isset($value['default']) && isset($value['id'])) {
							//Check if option are saved or not.
							if (get_option($value['id']) === false) {
								add_option($value['id'], $value['default']);
							}
						}
					}
				}
			}

			//Default for the notifications.
			$notifications = WAL_Notification_Instances::get_notifications();
			foreach ($notifications as $notification) {
				$settings = $notification->get_settings_array();
				if (!wal_check_is_array($settings)) {
					continue;
				}

				foreach ($settings as $setting) {
					//Check if the default and id key is exists.
					if (isset($setting['default']) && isset($setting['id'])) {
						//Check if option are saved or not.
						if (get_option($setting['id']) === false) {
							add_option($setting['id'], $setting['default']);
						}
					}
				}
			}

			//Default for the modules.
			$modules = WAL_Module_Instances::get_modules();
			foreach ($modules as $module) {
				$tabs = $module->setting_tabs();

				// Update the tab settings.
				if (wal_check_is_array($tabs)) {
					foreach ($tabs as $tab_key => $section) {
						$settings_array = $module->setting_options($tab_key);
						foreach ($settings_array as $value) {
							//Check if the default and id key is exists.
							if (isset($value['default']) && isset($value['id'])) {
								//Check if option are saved or not.
								if (get_option($value['id']) === false) {
									add_option($value['id'], $value['default']);
								}
							}
						}
					}
				} else {
					$settings_fields = $module->setting_options($module->get_id());
					foreach ($settings_fields as $value) {
						//Check if default and id key is exists.
						if (isset($value['default']) && isset($value['id'])) {
							//Check if option are saved or not.
							if (get_option($value['id']) === false) {
								add_option($value['id'], $value['default']);
							}
						}
					}
				}
			}

			//Default for the wallet withdrawal gateways.
			$gateways = WAL_Module_Instances::get_module_by_id('wallet_withdrawal')->payment_gateways();
			foreach ($gateways as $gateway) {
				$settings = $gateway->get_settings_array();
				if (!wal_check_is_array($settings)) {
					continue;
				}

				foreach ($settings as $setting) {
					//Check if the default and id key is exists.
					if (isset($setting['default']) && isset($setting['id'])) {
						//Check if option are saved or not.
						if (get_option($setting['id']) === false) {
							add_option($setting['id'], $setting['default']);
						}
					}
				}
			}
		}

		/**
		 * May be create the top-up product.
		 */
		public static function maybe_create_topup_product() {
			$topup_product = array_filter((array) get_option('wal_general_topup_product_id', array()), 'wal_array_filter');

			// Return if the top-up product already exists.
			if (wal_check_is_array($topup_product)) {
				$topup_product_id = reset($topup_product);
				$product = wc_get_product($topup_product_id);
				if ($product) {
					return true;
				}
			}

			// Create a topup product.
			$product_id = wal_create_new_product('wallet');

			update_option('wal_general_topup_product_id', array( $product_id ));
			update_post_meta($product_id, 'wal_topup_product', 'yes');
		}
	}

	WAL_Install::init();
}
