<?php

/**
 * Current User Wallet.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}
if ( ! class_exists( 'WAL_Current_User_Wallet' ) ) {

	/**
	 * Class.
	 */
	class WAL_Current_User_Wallet {

		/**
		 * User ID.
		 * 
		 * @var int
		 */
		protected static $user_id ;

		/**
		 * Wallet ID.
		 * 
		 * @var int
		 */
		protected static $wallet_id ;

		/**
		 * User name.
		 * 
		 * @var string
		 */
		protected static $user_name ;

		/**
		 * User email.
		 * 
		 * @var string
		 */
		protected static $user_email ;

		/**
		 * User.
		 * 
		 * @var object
		 */
		protected static $user ;

		/**
		 * Wallet.
		 * 
		 * @var object
		 */
		protected static $wallet ;

		/**
		 * Initialize the current user wallet details.
		 * 
		 * @return void
		 */
		public static function init() {

			self::populate_data() ;
		}

		/**
		 * Populate Data.
		 * 
		 * @return void
		 */
		public static function populate_data() {
			
			/**
			 * This hook is used to alter the user id.
			 *
			 * @param int $user_id
			 * @since 4.9.0
			 */
			self::$user_id = apply_filters('wal_get_current_user_id', get_current_user_id());

			//Return if the user is guest.
			if ( ! self::$user_id ) {
				return ;
			}

			self::$wallet_id = wal_get_wallet_id_by_user_id( self::$user_id ) ;

			//Return if the user not having wallet.
			if ( ! self::$wallet_id ) {
				return ;
			}

			self::$wallet = wal_get_wallet( self::$wallet_id ) ;
		}

		/**
		 * Has user wallet?.
		 * 
		 * @return bool
		 */
		public static function has_user_wallet() {
			if ( ! self::get_wallet_id() ) {
				return false ;
			}

			if ( ! is_object( self::get_wallet() ) ) {
				return false ;
			}

			return true ;
		}

		/**
		 * Get the user.
		 * 
		 * @return object/bool.
		 */
		public static function get_user() {
			if ( self::$user ) {
				return self::$user ;
			}

			self::$user = false ;
			if ( self::get_user_id() ) {
				self::$user = get_userdata( self::get_user_id() ) ;
			}

			return self::$user ;
		}

		/**
		 * Get the user ID.
		 * 
		 * @retrun int/NULL
		 */
		public static function get_user_id() {
			return self::$user_id ;
		}

		/**
		 * Get the user name.
		 * 
		 * @return string/bool
		 */
		public static function get_user_name() {
			if ( self::$user_name ) {
				return self::$user_name ;
			}

			self::$user_name = false ;
			if ( ! is_object( self::get_user() ) ) {
				return self::$user_name ;
			}

			self::$user_name = self::get_user()->display_name ;

			return self::$user_name ;
		}

		/**
		 * Get the user email.
		 * 
		 * @return string/bool
		 */
		public static function get_user_email() {
			if ( self::$user_email ) {
				return self::$user_email ;
			}

			self::$user_email = false ;
			if ( ! is_object( self::get_user() ) ) {
				return self::$user_email ;
			}

			self::$user_email = self::get_user()->user_email ;

			return self::$user_email ;
		}

		/**
		 * Get the wallet ID.
		 * 
		 * @return int/NULL
		 */
		public static function get_wallet_id() {
			return self::$wallet_id ;
		}

		/**
		 * Get the Wallet Object.
		 * 
		 * @return object/NULL
		 */
		public static function get_wallet() {
			return self::$wallet ;
		}

		/**
		 * Get the wallet status.
		 * 
		 * @return string
		 */
		public static function get_status() {
			if ( ! self::has_user_wallet() ) {
				return 'wal_active' ;
			}

			return self::get_wallet()->get_status() ;
		}

		/**
		 * Get the wallet balance.
		 * 
		 * @return float
		 */
		public static function get_balance() {
			if ( ! self::has_user_wallet() ) {
				return 0 ;
			}
					   
			return self::get_wallet()->get_balance() ;
		}

		/**
		 * Get the wallet usage total.
		 * 
		 * @return float
		 */
		public static function get_usage_total() {
			if ( ! self::has_user_wallet() ) {
				return 0 ;
			}

			return self::get_wallet()->get_usage_total() ;
		}

		/**
		 * Get the wallet top up total.
		 * 
		 * @return float
		 */
		public static function get_topup_total() {
			if ( ! self::has_user_wallet() ) {
				return 0 ;
			}

			return self::get_wallet()->get_topup_total() ;
		}

		/**
		 * Get the wallet expired total.
		 * 
		 * @return float
		 */
		public static function get_expired_total() {
			if ( ! self::has_user_wallet() ) {
				return 0 ;
			}

			return self::get_wallet()->get_expired_total() ;
		}

		/**
		 * Get the wallet transferred total.
		 * 
		 * @return float
		 */
		public static function get_transferred_total() {
			if ( ! self::has_user_wallet() ) {
				return 0 ;
			}

			return self::get_wallet()->get_transferred_total() ;
		}

		/**
		 * Get the wallet received total.
		 * 
		 * @return float
		 */
		public static function get_received_total() {
			if ( ! self::has_user_wallet() ) {
				return 0 ;
			}

			return self::get_wallet()->get_received_total() ;
		}

		/**
		 * Get the wallet withdrew total.
		 * 
		 * @since 1.8
		 * @return float
		 */
		public static function get_withdrew_total() {
			if ( ! self::has_user_wallet() ) {
				return 0 ;
			}

			return self::get_wallet()->get_withdrew_total() ;
		}

		/**
		 * Get the wallet withdraw payment gateway.
		 * 
		 * @since 1.8
		 * @return string
		 */
		public static function get_withdrawal_payment_gateway() {
			if ( ! self::has_user_wallet() ) {
				return '' ;
			}

			return self::get_wallet()->get_withdrawal_payment_gateway() ;
		}

		/**
		 * Get the expiry date.
		 * 
		 * @return string
		 */
		public static function get_expiry_date( $format = false ) {
			if ( ! self::has_user_wallet() ) {
				return '-' ;
			}

			if ( $format ) {
				$expiry_date = self::get_wallet()->get_formatted_current_expiry_date() ;
			} else {
				$expiry_date = self::get_wallet()->get_current_expiry_date() ;
			}

			return $expiry_date ;
		}

		/**
		 * Is valid fund usage?.
		 * 
		 * @return bool
		 */
		public static function is_valid_fund_usage() {
			if ( ! self::has_user_wallet() ) {
				return false ;
			}

			// Return false, if the wallet status is not active.
			if ( 'wal_active' != self::get_status() ) {
				return false ;
			}

			$wallet = self::get_wallet() ;

			return WAL_Fund_Usage_Validation::is_valid( $wallet ) ;
		}

		/**
		 * Get Fund Usage Amount.
		 * 
		 * @return float
		 */
		public static function get_fund_to_apply( $fund = 0 ) {
			// Return false if the page is pay order.
			if ( function_exists( 'is_wc_endpoint_url' ) && is_wc_endpoint_url( 'order-pay' ) ) {
				return 0 ;
			}

			// Return true if the cart object is not initialized.
			if ( ! is_object( WC()->cart ) ) {
				return 0 ;
			}
			
			$cart_contents = WC()->cart->get_cart() ;
			if ( ! wal_check_is_array( $cart_contents ) ) {
				return 0 ;
			}

			$total_value = array();
			$product_type = get_option( 'wal_general_wallet_usage_product_restriction_type' ) ;
			foreach ( $cart_contents as $cart_content ) {

				$variation = wc_get_product( $cart_content[ 'variation_id' ] );
				if ( $variation && $variation->is_type( 'variation' ) ) {
					$product_id = $variation->get_parent_id();
				} else {
					$product_id = $cart_content[ 'product_id' ];
				}
				$line_total  = ( 'yes' == get_option( 'woocommerce_prices_include_tax' ) ) ? ( $cart_content['line_subtotal'] + $cart_content['line_tax'] ): $cart_content['line_subtotal'];
				switch ( $product_type ) {
					case '2':
						$include_products = array_filter( ( array ) get_option( 'wal_general_wallet_usage_include_product' ), 'wal_array_filter' ) ;
						// Return true if any selected products not in the cart.
						if ( in_array( $product_id, $include_products ) ) {
							$total_value[] = $line_total ;
						}

						break ;
					case '3':
						$exclude_products = array_filter( ( array ) get_option( 'wal_general_wallet_usage_exclude_product' ), 'wal_array_filter' ) ;
						// Return true if any selected products in the cart.
						if ( ! in_array( $product_id, $exclude_products ) ) {
							$total_value[] = $line_total ;
						}
						break ;
					case '4':
						$include_categories = array_filter( ( array ) get_option( 'wal_general_wallet_usage_include_categories' ), 'wal_array_filter' ) ;
						// Included categories.
						$product_categories = get_the_terms( $product_id, 'product_cat' ) ;

						if ( wal_check_is_array( $product_categories ) ) {
							foreach ( $product_categories as $product_category ) {
								$category_ids[] = $product_category->term_id ;
								// Return false if any selected products of category in the cart.
								if ( in_array( $product_category->term_id, $include_categories ) ) {
									$total_value[] = $line_total ;
								}
							}
						}
						break ;
					case '5':
						$exclude_categories = array_filter( ( array ) get_option( 'wal_general_wallet_usage_exclude_categories' ), 'wal_array_filter' ) ;
						// Excluded categories.
						$product_categories = get_the_terms( $product_id, 'product_cat' ) ;
						if ( wal_check_is_array( $product_categories ) ) {
							foreach ( $product_categories as $product_category ) {
								// Return true if any selected products of category in the cart.
								if ( ! in_array( $product_category->term_id, $exclude_categories ) ) {
									$total_value[] = $line_total ;
									break;
								}
							}
						}
						break ;
				}
			}

			$total = ( '1' == $product_type ) ? wal_get_wc_cart_total(true, true) : array_sum( $total_value ) ;
			$fund = ( $fund > $total ) ? $total : $fund;

			return $fund ;
		}

		/**
		 * Get Product Name.
		 * 
		 * @return string
		 */
		public static function get_product_names( $product_ids ) {
			$product_name = array();
			if ( ! wal_check_is_array($product_ids) ) {
				return $product_name;
			}

			foreach ($product_ids as $product_id) {
				$product_name[] = get_the_title($product_id);
			}

			return implode(',', $product_name);
		}

		/**
		 * Get Category Name.
		 * 
		 * @return string
		 */
		public static function get_category_names( $category_ids ) {
			$category_name = array();
			if ( ! wal_check_is_array($category_ids) ) {
				return $category_name;
			}

			foreach ($category_ids as $category_id) {
				$category_name[] = wal_get_category_name($category_id);
			}

			return implode(',', $category_name);
		}

		/**
		 * Reset the properties.
		 * 
		 * @return void
		 */
		public static function reset() {
			self::$user_id    = null ;
			self::$wallet_id  = null ;
			self::$user_email = null ;
			self::$user       = null ;
			self::$wallet     = null ;

			self::populate_data() ;
		}
	}

}
