<?php

/**
 * Handles the Cron.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly
}

if ( ! class_exists( 'WAL_Cron_Handler' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Cron_Handler {

		/**
		 *  Class initialization.
		 * */
		public static function init() {

			// Maybe set the WP schedule event.
			add_action( 'init', array( __CLASS__, 'maybe_set_wp_schedule_event' ), 10 ) ;
			// Handle the wallet expired.
			add_action( 'wal_wallet_expired', array( __CLASS__, 'maybe_handle_wallet_expired' ) ) ;
			// Handle the wallet expiry reminder emails.
			add_action( 'wal_wallet_expiry_reminder_emails', array( __CLASS__, 'maybe_handle_wallet_expiry_reminder_emails' ) ) ;
		}

		/**
		 * Maybe set the WP schedule event.
		 * 
		 * @return void.
		 * */
		public static function maybe_set_wp_schedule_event() {
			// Check if the wallet expired schedule event is already exists.
			if ( ! wp_next_scheduled( 'wal_wallet_expired' ) ) {
				// Add the wallet expired schedule event.
				wp_schedule_event( time(), 'hourly', 'wal_wallet_expired' ) ;
			}

			// Check if the wallet expiry reminder emails schedule event is already exists.
			if ( ! wp_next_scheduled( 'wal_wallet_expiry_reminder_emails' ) ) {
				// Add the wallet expiry reminder emails schedule event.
				wp_schedule_event( time(), 'twicedaily', 'wal_wallet_expiry_reminder_emails' ) ;
			}
		}

		/**
		 * Handles the WP cron.
		 * 
		 * @return void.
		 * */
		public static function maybe_handle_wallet_expired() {
			// Update the WP cron current date. 
			update_option( 'wal_wallet_expired_wp_cron_last_updated_date', WAL_Date_Time::get_mysql_date_time_format( 'now', true ) ) ;
			/**
			 * This hook is used to handle the expired cron.
			 * 
			 * @since 1.0
			 */
			do_action( 'wal_handle_expired_cron' ) ;

			$current_date = WAL_Date_Time::get_mysql_date_time_format( 'now', true ) ;
			$args         = array(
				'post_type'   => WAL_Register_Post_Types::WALLET_POSTTYPE,
				'post_status' => 'wal_active',
				'fields'      => 'ids',
				'numberposts' => '-1',
				'meta_query'  => array(
					array(
						'key'     => 'wal_current_expiry_date',
						'value'   => $current_date,
						'compare' => '<=',
						'type'    => 'DATETIME',
					),
					array(
						'key'     => 'wal_current_expiry_date',
						'value'   => '',
						'compare' => '!=',
					),
				),
					) ;

			$post_ids = get_posts( $args ) ;

			// Return if the post ids is not exists.
			if ( ! wal_check_is_array( $post_ids ) ) {
				return ;
			}

			foreach ( $post_ids as $post_id ) {
				$wallet = wal_get_wallet( $post_id ) ;
				if ( ! $wallet->exists() ) {
					continue ;
				}

				$meta_args = array(
					'wal_balance'       => 0,
					'wal_expired_total' => floatval( $wallet->get_balance() ) + floatval( $wallet->get_expired_total() ),
						) ;

				wal_update_wallet( $post_id, $meta_args, array( 'post_status' => 'wal_expired' ) ) ;

				// Insert transaction log.
				$args = array(
					'user_id'       => $wallet->get_user_id(),
					'amount'        => $wallet->get_balance(),
					'event_id'      => 9,
					'event_message' => get_option( 'wal_localization_wallet_expired_log' ),
					'update_wallet' => false,
						) ;

				$transaction_log_id = wal_debit_wallet_fund( $args ) ;
				if ( $transaction_log_id ) {
					/**
					 * This hook is used to send the email when wallet fund expired.
					 * 
					 * @since 1.0
					 */
					do_action( 'wal_wallet_fund_expired', $wallet ) ;
				}
			}
		}

		/**
		 * Handles the Wallet expiry reminder emails.
		 * 
		 * @return void.
		 * */
		public static function maybe_handle_wallet_expiry_reminder_emails() {
			// Update the WP cron current date. 
			update_option( 'wal_wallet_expiry_reminder_email_last_updated_date', WAL_Date_Time::get_mysql_date_time_format( 'now', true ) ) ;

			/**
			 * This hook is used to send the wallet fund expiry reminder emails.
			 * 
			 * @since 1.0
			 */
			do_action( 'wal_wallet_fund_expiry_reminder_emails' ) ;
		}
	}

	WAL_Cron_Handler::init() ;
}
