<?php

/**
 * Handles category total range fund handler.
 * 
 * @since 5.1.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

if (!class_exists('WAL_Category_Total_Range_Fund_Handler')) {

	/**
	 * Class.
	 * 
	 * @since 5.1.0
	 */
	class WAL_Category_Total_Range_Fund_Handler {

		/**
		 * Order Object.
		 *
		 * @since 5.1.0 
		 * @var array
		 * */
		private $order;

		/**
		 * Category Total.
		 *
		 * @since 5.1.0 
		 * @var array
		 * */
		private $category_total = array();

		/**
		 * Fund.
		 *
		 * @since 5.1.0 
		 * @var float
		 * */
		private $fund = 0;

		/**
		 * Valid rules.
		 * 
		 * @since 5.1.0
		 * @var array
		 * */
		private $valid_rules = array();

		/**
		 * Selected Category Name.
		 * 
		 * @since 5.1.0
		 * @var array
		 * */
		protected static $selected_category_name = array();

		/**
		 * Matched Rule Key.
		 * 
		 * @since 5.1.0
		 * @var int
		 * */
		protected static $matched_rule_key;

		/**
		 * Rules.
		 * 
		 * @since 5.1.0
		 * @var array
		 * */
		private $rule = array();

		/**
		 * Class initialization.
		 *       
		 * @since 5.1.0  
		 * @var float $order         
		 * */
		public function __construct( $order ) {
			$this->order = $order;

			$this->prepare_fund();
		}

		/**
		 * Get fund.     
		 *  
		 * @since 5.1.0
		 * @param float $order 
		 * @return float
		 */
		public static function get_fund( $order ) {
			$purchased_order = new self($order);

			return $purchased_order->get_amount();
		}

		/**
		 * Get fund.     
		 *  
		 * @since 5.1.0
		 * @param float $order 
		 * @return float
		 */
		public static function get_category_name() {
			return self::$selected_category_name;
		}

		/**
		 * Get amount.
		 *
		 * @since 5.1.0
		 * @return float
		 */
		private function get_amount() {
			return (float) $this->fund;
		}

		/**
		 * Get Matched Rule Key.
		 *
		 * @since 5.1.0
		 * @return int
		 */
		public static function get_matched_rule_key() {
			return self::$matched_rule_key;
		}

		/**
		 * Prepare fund.
		 *
		 * @since 5.1.0
		 * @return float
		 * */
		private function prepare_fund() {
			// Return if not category total range mode
			if ('4' !== get_option('wal_module_product_purchase_mode', '1')) {
				return $this->fund;
			}

			$category_total_range_rules = get_option('wal_module_category_total_range_rules', array());
			//Return if purchase category range rules empty
			if (!wal_check_is_array($category_total_range_rules)) {
				return $this->fund;
			}

			$this->prepare_category_total();

			$awarded_rule_ids = get_user_meta(get_current_user_id(), 'wal_awarded_rule_ids', true);
			foreach ($category_total_range_rules as $key => $category_total_range_rule) {
				$this->rule = $category_total_range_rule;

				if (!$this->is_fund_awarded_rule( $key, $awarded_rule_ids )) {
					continue;
				}

				if (!$this->is_valid_rule()) {
					continue;
				}
				
				$this->valid_rules[$key] = $this->rule['fund_value'];
			}

			$this->prepare_amount_by_priority();
		}

		/**
		 * Prepare amount by priority
		 * 
		 * @since 5.1.0
		 * @return float
		 * */
		private function prepare_amount_by_priority() {
			// Return if not valid rules
			if (!wal_check_is_array($this->valid_rules)) {
				return $this->fund;
			}

			switch (get_option('wal_module_category_total_range_priority', '1')) {
				case '2':
					$this->fund = end($this->valid_rules);
					break;
				case '3':
					$this->fund = min($this->valid_rules);
					break;
				case '4':
					$this->fund = max($this->valid_rules);
					break;
				default:
					$this->fund = reset($this->valid_rules);
					break;
			}

			self::$matched_rule_key = array_search($this->fund, $this->valid_rules);
		}

		/**
		 * Is valid rule?
		 *        
		 * @since 5.1.0      
		 * @return bool
		 */
		private function is_fund_awarded_rule( $key, $awarded_rule_ids ) {
			if ( ! isset($this->rule['user_restriction']) ) {
				return true;
			}

			if ( wal_check_is_array($awarded_rule_ids) && in_array($key, $awarded_rule_ids)) {
				return false;
			}
			
			return true;
		}

		/**
		 * Is valid rule?
		 *        
		 * @since 5.1.0      
		 * @return bool
		 */
		private function is_valid_rule() {
			$included_category = isset($this->rule['included_category']) ? $this->rule['included_category'] : array();
			if (!wal_check_is_array($included_category)) {
				return true;
			}

			if ( !wal_check_is_array($this->category_total) ) {
				return false;
			}

			$min_amount = isset($this->rule['min_category_total']) ? $this->rule['min_category_total'] : '';
			$max_amount = isset($this->rule['max_category_total']) ? $this->rule['max_category_total'] : '';
			// Return if empty minimum amount and maximum amount
			if (!$min_amount && !$max_amount) {
				return true;
			}

			$selected_category_total = 0;
			foreach ($included_category as $category_id) {
				if ( ! isset($this->category_total[$category_id]) ) {
					continue;
				}

				$selected_category_total += $this->category_total[$category_id];
				self::$selected_category_name[] = wal_get_category_name($category_id);
				if (!empty($min_amount) && !empty($max_amount)) {
					if ( ( $min_amount <= $selected_category_total ) && ( $max_amount >= $selected_category_total )) {
						return true;
					}
				} else if (!empty($min_amount) && empty($max_amount)) {
					if ($min_amount <= $selected_category_total) {
						return true;
					}
				} else if (empty($min_amount) && !empty($max_amount)) {
					if ($max_amount >= $selected_category_total) {
						return true;
					}
				}
			}

			return false;
		}

		/**
		 * Prepare Category total from cart/order object
		 *        
		 * @since 5.1.0      
		 * @return array
		 */
		private function prepare_category_total() {
			foreach ($this->order as $item) {
				$product = wc_get_product($item['product_id']);
				if (!is_object($product)) {
					continue;
				}

				if ('incl' === get_option('woocommerce_tax_display_cart')) {
					$regular_price = floatval($item['line_subtotal_tax']) + floatval($item['line_subtotal']);
				} else {
					$regular_price = floatval($item['line_subtotal']);
				}

				$category_ids = wp_get_post_terms( $item['product_id'], 'product_cat' );
				
				foreach ($category_ids as $category) {
					if ( isset($this->category_total[$category->term_id]) ) {
						$this->category_total[$category->term_id] += $regular_price;
					} else {
						$this->category_total[$category->term_id] = $regular_price;
					}
				}
			}

			return $this->category_total;
		}
	}

}
