<?php
/**
 * Handles the action scheduler.
 *
 * @since 3.5.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Action_Scheduler_Handler' ) ) {

	/**
	 * Class.
	 *
	 * @since 3.5.0
	 */
	class WAL_Action_Scheduler_Handler {

		/**
		 * Events.
		 *
		 * @since 3.5.0
		 * @var array
		 */
		private static $events = array(
			'wal_reset_user_balance'         => 2,
			'wal_reset_user_transaction_log' => 2,
		);

		/**
		 * Class Initialization.
		 *
		 * @since 3.5.0
		 */
		public static function init() {
			foreach ( self::$events as $event => $argument_count ) {
				$method = str_replace( 'wal_', '', $event );
				if ( ! method_exists( __CLASS__, $method ) ) {
					continue;
				}

				add_action( $event, array( __CLASS__, $method ), 10, $argument_count );
			}
		}

		/**
		 * Delete the user balance.
		 *
		 * @since 3.5.0
		 * @param array $user_ids User IDs.
		 * @param bool  $completed Whether the scheduling is completed or not.
		 * @return void
		 */
		public static function reset_user_balance( $user_ids, $completed ) {
			// Return if the user IDs is empty.
			if ( ! wal_check_is_array( $user_ids ) ) {
				delete_option( 'wal_processing_user_wallet_balance_reset' );
				if ( 'yes' !== get_option( 'wal_processing_user_wallet_transaction_log_reset' ) ) {
					update_option( 'wal_user_wallet_reseted', 'no_data' );
				}
				return;
			}

			// Reset wallet balance.
			foreach ( $user_ids as $user_id ) {
				$wallet_id = wal_get_wallet_id_by_user_id( $user_id );
				if ( $wallet_id ) {
					wal_update_wallet( $wallet_id, array( 'wal_balance' => 0 ), array() );
				}
			}

			if ( $completed ) {
				delete_option( 'wal_processing_user_wallet_balance_reset' );
				if ( 'yes' !== get_option( 'wal_processing_user_wallet_transaction_log_reset' ) ) {
					update_option( 'wal_user_wallet_reseted', 'yes' );
				}
			}
		}

		/**
		 * Delete the user transaction log.
		 *
		 * @since 3.5.0
		 * @param array $transaction_log_ids Transaction log IDs.
		 * @param bool  $completed Whether the scheduling is completed or not.
		 */
		public static function reset_user_transaction_log( $transaction_log_ids, $completed ) {
			// Return if the transaction log IDs is empty.
			if ( ! wal_check_is_array( $transaction_log_ids ) ) {
				delete_option( 'wal_processing_user_wallet_transaction_log_reset' );
				if ( 'yes' !== get_option( 'wal_processing_user_wallet_balance_reset' ) ) {
					update_option( 'wal_user_wallet_reseted', 'no_data' );
				}
				return;
			}

			// Reset wallet transaction log.
			foreach ( $transaction_log_ids as $transaction_log_id ) {
				wal_delete_transaction_log( $transaction_log_id, true );
			}

			if ( $completed ) {
				delete_option( 'wal_processing_user_wallet_transaction_log_reset' );
				if ( 'yes' !== get_option( 'wal_processing_user_wallet_balance_reset' ) ) {
					update_option( 'wal_user_wallet_reseted', 'yes' );
				}
			}
		}
	}

	WAL_Action_Scheduler_Handler::init();
}
