<?php

/**
 * Common functions.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!function_exists('wal_page_screen_ids')) {

	/**
	 * Get the page screen IDs.
	 *
	 * @return array
	 */
	function wal_page_screen_ids() {
		$screen_id = sanitize_title(WAL()->menu_name());
		/**
		 * This hook is used to alter the page screen IDs.
		 * 
		 * @since 1.0
		 */
		return apply_filters(
				'wal_page_screen_ids', array(
			'wal_wallet',
			'wal_transaction_log',
			'wal_wallet_withdraw',
			$screen_id . '_page_wal_funds',
			$screen_id . '_page_wal_settings',
			$screen_id . '_page_wal_modules',
			$screen_id . '_page_wal_transaction_logs',
			'shop_order',           
			'woocommerce_page_wc-orders',
			'woocommerce_page_wal_funds',
			'woocommerce_page_wal_settings',
			'woocommerce_page_wal_modules',
			'woocommerce_page_wal_transaction_logs',
				)
		);
	}

}

if (!function_exists('wal_get_wc_categories')) {

	/**
	 * Get the WC Categories.
	 *
	 * @return array
	 */
	function wal_get_wc_categories() {
		$categories = array();
		$wc_categories = get_terms('product_cat');

		if (!wal_check_is_array($wc_categories)) {
			return $categories;
		}

		foreach ($wc_categories as $category) {
			$categories[$category->term_id] = $category->name;
		}

		return $categories;
	}

}

if (!function_exists('wal_get_wp_user_roles')) {

	/**
	 * Get the WordPress User Roles.
	 *
	 * @return array
	 */
	function wal_get_wp_user_roles() {
		static $wal_roles;
		if ($wal_roles) {
			return $wal_roles;
		}

		global $wp_roles;
		$wal_roles = array();

		if (!isset($wp_roles->roles) || !wal_check_is_array($wp_roles->roles)) {
			return $wal_roles;
		}

		foreach ($wp_roles->roles as $slug => $role) {
			$wal_roles[$slug] = $role['name'];
		}

		return $wal_roles;
	}

}

if (!function_exists('wal_get_wc_order_statuses')) {

	/**
	 * Get the WC order statuses.
	 *
	 * @return array
	 */
	function wal_get_wc_order_statuses( $negative_status = true ) {
		$order_statuses = wc_get_order_statuses();
				
		if (!$negative_status) {
			unset($order_statuses['wc-failed']);
			unset($order_statuses['wc-cancelled']);
			unset($order_statuses['wc-refunded']);
						unset($order_statuses['wc-pending']);
						unset($order_statuses['wc-checkout-draft']);
		}

		$order_statuses_keys = array_keys($order_statuses);
		$order_statuses_keys = str_replace('wc-', '', $order_statuses_keys);
		$order_statuses_values = array_values($order_statuses);

		return array_combine($order_statuses_keys, $order_statuses_values);
	}

}

if (!function_exists('wal_get_wc_available_gateways')) {

	/**
	 * Get the WC available payment gateways.
	 * 
	 * @return array
	 */
	function wal_get_wc_available_gateways() {
		$available_gateways = array();
		$wc_gateways = WC()->payment_gateways->get_available_payment_gateways();

		if (!wal_check_is_array($wc_gateways)) {
			return $available_gateways;
		}

		foreach ($wc_gateways as $gateway) {
			if ('wal_wallet' == $gateway->id) {
				continue;
			}

			$available_gateways[$gateway->id] = $gateway->title;
		}

		return $available_gateways;
	}

}

if (!function_exists('wal_get_allowed_setting_tabs')) {

	/**
	 * Get setting tabs.
	 *
	 * @return array
	 */
	function wal_get_allowed_setting_tabs() {
		/**
		 * This hook is used to alter the settings tabs array.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_settings_tabs_array', array());
	}

}

if (!function_exists('wal_get_settings_page_url')) {

	/**
	 * Get the settings page URL.
	 *
	 * @return URL
	 */
	function wal_get_settings_page_url( $args = array() ) {
		$url = add_query_arg(array( 'page' => 'wal_settings' ), admin_url('admin.php'));

		if (wal_check_is_array($args)) {
			$url = add_query_arg($args, $url);
		}

		return $url;
	}

}

if (!function_exists('wal_get_module_page_url')) {

	/**
	 * Get the module page URL.
	 *
	 * @return URL
	 */
	function wal_get_module_page_url( $args = array() ) {
		$url = add_query_arg(array( 'page' => 'wal_modules' ), admin_url('admin.php'));

		if (wal_check_is_array($args)) {
			$url = add_query_arg($args, $url);
		}

		return $url;
	}

}

if (!function_exists('wal_get_wallet_page_url')) {

	/**
	 * Get the wallet page URL.
	 *
	 * @return URL
	 */
	function wal_get_wallet_page_url( $args = array() ) {
		$url = add_query_arg(array( 'post_type' => WAL_Register_Post_Types::WALLET_POSTTYPE ), admin_url('edit.php'));

		if (wal_check_is_array($args)) {
			$url = add_query_arg($args, $url);
		}

		return $url;
	}

}

if (!function_exists('wal_get_user_transaction_logs_page_url')) {

	/**
	 * Get the user transaction logs page URL.
	 *
	 * @return URL
	 */
	function wal_get_user_transaction_logs_page_url( $args = array() ) {
		$url = add_query_arg(array( 'page' => 'wal_transaction_logs' ), admin_url('admin.php'));

		if (wal_check_is_array($args)) {
			$url = add_query_arg($args, $url);
		}

		return $url;
	}

}

if (!function_exists('wal_wallet_details_data')) {

	/**
	 * Get the wallet details data.
	 *
	 * @return array
	 */
	function wal_wallet_details_data( &$wallet ) {

		$data = array(
			'customer_name' => array(
				'label' => __('Username', 'wallet-for-woocommerce'),
				'value' => $wallet->get_user_name(),
			),
			'customer_email' => array(
				'label' => __('Email ID', 'wallet-for-woocommerce'),
				'value' => $wallet->get_user_email(),
			),
			'balance' => array(
				'label' => __('Available Balance', 'wallet-for-woocommerce'),
				'value' => wal_convert_price_by_currency($wallet->get_balance()),
			),
			'usage_total' => array(
				'label' => __('Total Used Amount', 'wallet-for-woocommerce'),
				'value' => wal_convert_price_by_currency($wallet->get_usage_total()),
			),
			'expired_total' => array(
				'label' => __('Total Expired Amount', 'wallet-for-woocommerce'),
				'value' => wal_convert_price_by_currency($wallet->get_expired_total()),
			),
			'topup_total' => array(
				'label' => __('Total Top-up Amount', 'wallet-for-woocommerce'),
				'value' => wal_convert_price_by_currency($wallet->get_topup_total()),
			),
			'last_activity_date' => array(
				'label' => __('Last Activity Date', 'wallet-for-woocommerce'),
				'value' => $wallet->get_formatted_last_activity_date(),
			),
		);
				
		return $data;
	}

}

if (!function_exists('wal_create_new_product')) {

	/**
	 * Create a new product based on title.
	 * 
	 * @return Integer
	 * */
	function wal_create_new_product( $product_title ) {
		$args = array(
			'post_author' => get_current_user_id(),
			'post_status' => 'publish',
			'post_title' => $product_title,
			'post_type' => 'product',
		);

		$product_id = wp_insert_post($args);

		$terms = array( 'exclude-from-search', 'exclude-from-catalog' ); // For hidden..
		wp_set_post_terms($product_id, $terms, 'product_visibility', false);

		$meta_keys = array(
			'_stock_status' => 'instock',
			'total_sales' => '0',
			'_downloadable' => 'no',
			'_virtual' => 'yes',
			'_regular_price' => '0',
			'_price' => '0',
			'_sale_price' => '',
			'_featured' => '',
			'_sold_individually' => 'yes',
			'_manage_stock' => 'no',
			'_backorders' => 'no',
			'_stock' => '',
			'_tax_status' => 'none',
		);

		foreach ($meta_keys as $key => $value) {
			update_post_meta($product_id, sanitize_key($key), $value);
		}

		return $product_id;
	}

}

if (!function_exists('wal_display_action')) {

	/**
	 * Display the post action.
	 *
	 * @since 4.1.0
	 * @param string $status
	 * @param int $id
	 * @param string $current_url
	 * @param bool $action
	 * @return string
	 */
	function wal_display_action( $status, $id, $current_url, $action = false ) {
		switch ($status) {
			case 'edit':
				$status_name = '<span class="dashicons dashicons-edit"></span>';
				$title = __('Edit', 'wallet-for-woocommerce');
				return '<a class="wal-wallet-actions" href="' . esc_url(add_query_arg(array( 'post' => $id, 'action' => $status ), $current_url)) . '" title="' . $title . '">' . $status_name . '</a>';
			case 'deactivate':
				$status_name = '<img src="' . esc_url(WAL_PLUGIN_URL . '/assets/images/button-off.png') . '"/>';
				$title = __('Deactivate', 'wallet-for-woocommerce');
				return '<a class="wal-wallet-actions wal-deactivate-post" href="' . wp_nonce_url($current_url, 'deactivate-post_' . $id) . '" data-id="' . $id . '" title="' . $title . '">' . $status_name . '</a>';
			case 'activate':
				$status_name = '<img src="' . esc_url(WAL_PLUGIN_URL . '/assets/images/button-on.png') . '"/>';
				$title = __('Activate', 'wallet-for-woocommerce');
				return '<a class="wal-wallet-actions wal-activate-post" href="' . wp_nonce_url($current_url, 'activate-post_' . $id) . '" title="' . $title . '">' . $status_name . '</a>';
			case 'view_transactions':
				$status_name = '<img src="' . esc_url(WAL_PLUGIN_URL . '/assets/images/view.png') . '"/>';
				$title = __('View Transactions', 'wallet-for-woocommerce');
				return '<a class="wal-wallet-actions" href="' . esc_url( $current_url ) . '" title="' . $title . '">' . $status_name . '</a>';
		}
		
		return '';
	}

}

if ( ! function_exists( 'wal_get_shipping_methods' ) ) {

	/**
	 * Get shipping methods.
	 *
	 * @since 4.3.0
	 * @return array
	 */
	function wal_get_shipping_methods() {
		static $active_shipping_methods;
		if ($active_shipping_methods) {
			return $active_shipping_methods;
		}
		$shipping_zones          = WC_Shipping_Zones::get_zones();
		$active_shipping_methods = array();

		if ( wal_check_is_array( $shipping_zones ) ) {
			foreach ( $shipping_zones as $zone ) {
				$shipping_zone = new WC_Shipping_Zone( $zone['zone_id'] );
				if ( ! is_object( $shipping_zone ) ) {
					continue;
				}

				$zone_shipping_methods = $shipping_zone->get_shipping_methods();
				if ( ! wal_check_is_array( $zone_shipping_methods ) ) {
					continue;
				}

				foreach ( $zone_shipping_methods as $shipping_method_id => $shipping_method ) {
					if ( $shipping_method->is_enabled() ) {
						$active_shipping_methods[ $shipping_method_id ] = $shipping_zone->get_zone_name() . ' - ' . $shipping_method->get_title();
					}
				}
			}
		}

		/**
		* This hook is used to alter the shipping methods.
		*
		* @since 4.3.0
		*/
		return apply_filters( 'wal_shipping_methods', $active_shipping_methods );
	}
}

if (!function_exists('wal_get_shipping_method_instance_ids_from_order')) {
	
	/**
	 * Get a shipping method instance ids from order.
	 *
	 * @since 4.3.0
	 * @param object $order
	 * @return string
	 */
	function wal_get_shipping_method_instance_ids_from_order( $order ) {
		if (!is_object($order)) {
			return ;
		}

		foreach ( $order->get_items( 'shipping' ) as $item_id => $item ) {
			$shipping_method_instance_ids[] = $item->get_instance_id();
		}

		return $shipping_method_instance_ids;
	}
}

if (!function_exists('wal_get_valid_shipping_methods')) {
	
	/**
	 * Get a valid shipping method.
	 *
	 * @since 4.3.0
	 * @return array
	 */
	function wal_get_valid_shipping_methods() {
		return array_filter(get_option('wal_module_product_purchase_allowed_shipping_method'), 'wal_array_filter');
	}
}

if (!function_exists('wal_get_valid_shipping_titles')) {
	
	/**
	 * Get a valid shipping titles.
	 *
	 * @since 4.3.0
	 * @return string
	 */
	function wal_get_valid_shipping_titles() {

		return array_intersect_key(wal_get_shipping_methods(), array_combine(wal_get_valid_shipping_methods(), wal_get_valid_shipping_methods()));
	}
}
