<?php
/**
 * Gift Vouchers List Table.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php'  ;
}

if ( ! class_exists( 'WAL_Gift_Vouchers_List_Table' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Gift_Vouchers_List_Table extends WP_List_Table {

		/**
		 * Per page count.
		 * 
		 * @var int
		 * */
		private $perpage = 20 ;

		/**
		 * Database.
		 * 
		 * @var object
		 * */
		private $database ;

		/**
		 * Offset.
		 * 
		 * @var int
		 * */
		private $offset ;

		/**
		 * Order BY.
		 * 
		 * @var string
		 * */
		private $orderby = 'ORDER BY ID DESC' ;

		/**
		 * Post type.
		 * 
		 * @var string
		 * */
		private $post_type = WAL_Register_Post_Types::GIFT_VOUCHER_POSTTYPE ;

		/**
		 * List Slug.
		 * 
		 * @var string
		 * */
		private $list_slug = 'wal_gift_vouchers' ;

		/**
		 * Base URL.
		 * 
		 * @var string
		 * */
		private $base_url ;

		/**
		 * Current URL.
		 * 
		 * @var string
		 * */
		private $current_url ;

		/**
		 * Constructor.
		 */
		public function __construct() {
			global $wpdb ;
			$this->database = &$wpdb ;

			// Prepare the required data.
			$this->base_url = wal_get_gift_vouchers_page_url() ;

			parent::__construct(
					array(
						'singular' => 'gift_voucher',
						'plural'   => 'gift_vouchers',
						'ajax'     => false,
					)
			) ;
		}

		/**
		 * Prepares the list of items for displaying.
		 * */
		public function prepare_items() {
			//Prepare the current url.
			$this->current_url = add_query_arg( array( 'paged' => absint( $this->get_pagenum() ) ), $this->base_url ) ;

			// Prepare the bulk actions.
			$this->perform_bulk_action() ;

			// Prepare the offset.
			$this->offset = $this->perpage * ( absint( $this->get_pagenum() ) - 1 ) ;

			// Prepare the header columns.
			$this->_column_headers = array( $this->get_columns(), $this->get_hidden_columns(), $this->get_sortable_columns() ) ;

			// Prepare the query clauses.
			$join    = $this->get_query_join() ;
			$where   = $this->get_query_where() ;
			$limit   = $this->get_query_limit() ;
			$offset  = $this->get_query_offset() ;
			$orderby = $this->get_query_orderby() ;

			// Prepare the all items.
			$count_items = $this->database->get_var( 'SELECT COUNT(DISTINCT ID) FROM ' . $this->database->posts . " AS p $where" ) ;

			// Prepare the current page items.
			$prepare_query = $this->database->prepare( 'SELECT DISTINCT ID FROM ' . $this->database->posts . " AS p $join $where $orderby LIMIT %d,%d", $offset, $limit ) ;

			$items = $this->database->get_results( $prepare_query, ARRAY_A ) ;

			// Prepare the item object.
			$this->prepare_item_object( $items ) ;

			// Prepare the pagination arguments.
			$this->set_pagination_args(
					array(
						'total_items' => $count_items,
						'per_page'    => $this->perpage,
					)
			) ;
		}

		/**
		 * Render the table.
		 * */
		public function render() {
			if ( isset( $_REQUEST[ 's' ] ) && strlen( wc_clean( wp_unslash( $_REQUEST[ 's' ] ) ) ) ) { // @codingStandardsIgnoreLine.
				/* translators: %s: search keywords */
				$searched_words = sprintf( __( 'Search results for &#8220;%s&#8221;', 'wallet-for-woocommerce' ) , wc_clean( wp_unslash( $_REQUEST[ 's' ] ) ) );
				echo '<span class="subtitle">' . esc_html($searched_words) . '</span>';
			}

			// Output the table.
			$this->prepare_items() ;
			$this->views() ;
			$this->search_box( __( 'Search Gift Voucher', 'wallet-for-woocommerce' ), 'wal-gift-voucher' ) ;
			$this->display() ;
		}

		/**
		 * Get a list of columns.
		 * 
		 * @return array
		 * */
		public function get_columns() {
			$columns = array(
				'cb'            => '<input type="checkbox" />', // Render a checkbox instead of text
				'code'          => __( 'Voucher Code', 'wallet-for-woocommerce' ),
				'balance'       => __( 'Wallet Balance', 'wallet-for-woocommerce' ),
				'created_date'  => __( 'Created on', 'wallet-for-woocommerce' ),
				'expired_date'  => __( 'Expiry Date', 'wallet-for-woocommerce' ),
				'redeemed_by'   => __( 'Redeemed by', 'wallet-for-woocommerce' ),
				'redeemed_date' => __( 'Redeemed on', 'wallet-for-woocommerce' ),
					) ;

			return $columns ;
		}

		/**
		 * Get a list of hidden columns.
		 * 
		 * @return array
		 * */
		public function get_hidden_columns() {
			return array() ;
		}

		/**
		 * Get a list of sortable columns.
		 * 
		 * @return void
		 * */
		public function get_sortable_columns() {
			return array(
				'balance'       => array( 'balance', false ),
				'code'          => array( 'code', false ),
				'created_date'  => array( 'created_date', false ),
				'redeemed_date' => array( 'redeemed_date', false ),
				'expired_date'  => array( 'expired_date', false ),
					) ;
		}

		/**
		 * Message to be displayed when there are no items.
		 */
		public function no_items() {
			esc_html_e( 'No gift vouchers to show.', 'wallet-for-woocommerce' ) ;
		}

		/**
		 * Get a list of bulk actions.
		 * 
		 * @return array
		 * */
		protected function get_bulk_actions() {
			$action             = array() ;
			$action[ 'delete' ] = __( 'Delete', 'wallet-for-woocommerce' ) ;
			/**
			 * This hook is used to alter the gift voucher bulk actions.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->list_slug . '_bulk_actions', $action ) ;
		}

		/**
		 * Get the name of the primary column.
		 * 
		 * @rerurn string
		 * */
		protected function get_primary_column_name() {
			return 'code' ;
		}

		/**
		 * Prepare the list of row action links.
		 * 
		 * @rerurn array
		 * */
		protected function prepare_row_actions( $item ) {
			$actions = array() ;

			$actions[ 'delete' ] = wal_display_action( 'delete', $item->get_id(), $this->current_url ) ;

			return $actions ;
		}

		/**
		 * Display the list of views available on this table.
		 * 
		 * @return array
		 * */
		public function get_views() {
			$args        = array() ;
			$status_link = array() ;

			$status_link_array = array(
				'all'         => __( 'All', 'wallet-for-woocommerce' ),
				'wal_unused'  => __( 'Unused', 'wallet-for-woocommerce' ),
				'wal_used'    => __( 'Used', 'wallet-for-woocommerce' ),
				'wal_expired' => __( 'Expired', 'wallet-for-woocommerce' ),
					) ;

			foreach ( $status_link_array as $status_name => $status_label ) {
				$status_count = $this->get_total_item_for_status( $status_name ) ;

				if ( ! $status_count ) {
					continue ;
				}

				$args[ 'status' ] = $status_name ;

				$label = $status_label . ' (' . $status_count . ')' ;

				$class = array( strtolower( $status_name ) ) ;
				if ( isset( $_GET[ 'status' ] ) && ( sanitize_title( $_GET[ 'status' ] ) == $status_name ) ) { // @codingStandardsIgnoreLine.
					$class[] = 'current' ;
				}

				if ( ! isset( $_GET[ 'status' ] ) && 'all' == $status_name ) { // @codingStandardsIgnoreLine.
					$class[] = 'current' ;
				}

				$status_link[ $status_name ] = $this->get_edit_link( $args, $label, implode( ' ', $class ) ) ;
			}

			return $status_link ;
		}

		/**
		 * Get a edit link.
		 * 
		 * @rerurn string
		 * */
		private function get_edit_link( $args, $label, $class = '' ) {
			$url        = add_query_arg( $args, $this->base_url ) ;
			$class_html = '' ;
			if ( ! empty( $class ) ) {
				$class_html = sprintf(
						' class="%s"', esc_attr( $class )
						) ;
			}

			return sprintf(
					'<a href="%s"%s>%s</a>', esc_url( $url ), $class_html, $label
					) ;
		}

		/**
		 * Get the total item by status.
		 * 
		 * @return int
		 * */
		private function get_total_item_for_status( $status = '' ) {
			// Get the current status item ids.
			$prepare_query = $this->database->prepare( 'SELECT COUNT(DISTINCT ID) FROM ' . $this->database->posts . " WHERE post_type=%s and post_status IN('" . $this->format_status( $status ) . "')", $this->post_type ) ;

			return $this->database->get_var( $prepare_query ) ;
		}

		/**
		 * Format the status.
		 * 
		 * @return string
		 * */
		private function format_status( $status ) {

			if ( 'all' == $status ) {
				$statuses = wal_get_gift_voucher_statuses() ;
				$status   = implode( "', '", $statuses ) ;
			}

			return $status ;
		}

		/**
		 * Prepare a each column data.
		 * */
		protected function extra_tablenav( $which ) {
			if ( 'top' != $which ) {
				return ;
			}

			$export_url = add_query_arg( array( 'post_type' => $this->post_type, 'wal_export_csv' => 'gift_voucher' ), admin_url( 'edit.php' ) ) ;
			?>
			<a href="<?php echo esc_url( $export_url ) ; ?>" class="wal-export-csv button button-primary"><?php esc_html_e( 'Export Gift Voucher as CSV', 'wallet-for-woocommerce' ) ; ?></a>
			<?php
		}

		/**
		 * Bulk action functionality.
		 * */
		public function perform_bulk_action() {

			$ids = isset( $_REQUEST[ 'id' ] ) ? wc_clean( wp_unslash( ( $_REQUEST[ 'id' ] ) ) ) : array() ; // @codingStandardsIgnoreLine.
			$ids = ! is_array( $ids ) ? explode( ',', $ids ) : $ids ;

			if ( ! wal_check_is_array( $ids ) ) {
				return ;
			}

			if ( ! current_user_can( 'edit_posts' ) ) {
				wp_die( '<p class="error">' . esc_html__( 'Sorry, you are not allowed to edit this item.', 'wallet-for-woocommerce' ) . '</p>' ) ;
			}

			$action = $this->current_action() ;

			foreach ( $ids as $id ) {
				if ( 'delete' === $action ) {
					wp_delete_post( $id, true ) ;
				}
			}

			wp_safe_redirect( $this->current_url ) ;
			exit() ;
		}

		/**
		 * Prepare the CB column data.
		 * 
		 * @return string
		 * */
		protected function column_cb( $item ) {
			return sprintf(
					'<input type="checkbox" name="id[]" value="%s" />', $item->get_id()
					) ;
		}

		/**
		 * Prepare a each column data.
		 * */
		protected function column_default( $item, $column_name ) {

			switch ( $column_name ) {

				case 'code':
					return $item->get_code() ;
					break ;

				case 'balance':
					return wal_price( $item->get_amount() ) ;
					break ;

				case 'redeemed_by':
					$user_name = $item->get_user_name() ;
					if ( $item->get_user_email() ) {
						$user_name .= '<br />(' . $item->get_user_email() . ')' ;
					}

					return $user_name ;
					break ;

				case 'created_date':
					return $item->get_formatted_created_date() ;
					break ;

				case 'redeemed_date':
					return $item->get_formatted_redeemed_date() ;
					break ;

				case 'expired_date':
					return $item->get_formatted_expiry_date() ;
					break ;
			}
		}

		/**
		 * Prepare the item Object.
		 * 
		 * @return void
		 * */
		private function prepare_item_object( $items ) {
			$prepare_items = array() ;
			if ( wal_check_is_array( $items ) ) {
				foreach ( $items as $item ) {
					$prepare_items[] = wal_get_gift_voucher( $item[ 'ID' ] ) ;
				}
			}

			$this->items = $prepare_items ;
		}

		/**
		 * Get the query join clauses.
		 * 
		 * @return string
		 * */
		private function get_query_join() {
			$join = '' ;
			if ( empty( $_REQUEST[ 'orderby' ] ) ) { // @codingStandardsIgnoreLine.
				return $join ;
			}

			$join = ' INNER JOIN ' . $this->database->postmeta . ' AS pm ON ( pm.post_id = p.ID )' ;
			/**
			 * This hook is used to alter the query join fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->list_slug . '_query_join', $join ) ;
		}

		/**
		 * Get the query where clauses.
		 * 
		 * @return string
		 * */
		private function get_query_where() {
			$current_status = 'all' ;
			if ( isset( $_GET[ 'status' ] ) && ( sanitize_title( $_GET[ 'status' ] ) != 'all' ) ) {
				$current_status = sanitize_title( $_GET[ 'status' ] ) ;
			}

			$where = " where post_type='" . $this->post_type . "' and post_status IN('" . $this->format_status( $current_status ) . "')" ;

			// Search.
			$where = $this->custom_search( $where ) ;
			/**
			 * This hook is used to alter the query where fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->list_slug . '_query_where', $where ) ;
		}

		/**
		 * Get the query limit clauses.
		 * 
		 * @return string
		 * */
		private function get_query_limit() {
			/**
			 * This hook is used to alter the query limit fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->list_slug . '_query_limit', $this->perpage ) ;
		}

		/**
		 * Get the query offset clauses.
		 * 
		 * @return string
		 * */
		private function get_query_offset() {
			/**
			 * This hook is used to alter the query offset fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->list_slug . '_query_offset', $this->offset ) ;
		}

		/**
		 * Get the query order by clauses.
		 * 
		 * @return string
		 * */
		private function get_query_orderby() {
			$order = 'DESC' ;
			if ( ! empty( $_REQUEST[ 'order' ] ) && is_string( $_REQUEST[ 'order' ] ) ) { // @codingStandardsIgnoreLine.
				if ( 'ASC' === strtoupper( wc_clean( wp_unslash( $_REQUEST[ 'order' ] ) ) ) ) { // @codingStandardsIgnoreLine.
					$order = 'ASC' ;
				}
			}

			// Order By.
			if ( isset( $_REQUEST[ 'orderby' ] ) ) {
				switch ( wc_clean( wp_unslash( $_REQUEST[ 'orderby' ] ) ) ) { // @codingStandardsIgnoreLine.
					case 'balance':
						$this->orderby = " AND pm.meta_key='wal_amount' ORDER BY CAST(pm.meta_value AS DECIMAL) " . $order ;
						break ;
					case 'code':
						$this->orderby = " AND pm.meta_key='wal_code' ORDER BY pm.meta_value " . $order ;
						break ;
					case 'expired_date':
						$this->orderby = " AND pm.meta_key='wal_expiry_date' ORDER BY CAST(pm.meta_value AS DATETIME) " . $order ;
						break ;
					case 'redeemed_date':
						$this->orderby = " AND pm.meta_key='wal_redeemed_date' ORDER BY CAST(pm.meta_value AS DATETIME) " . $order ;
						break ;
					case 'created_date':
						$this->orderby = ' ORDER BY p.post_date ' . $order ;
						break ;
				}
			}
			/**
			 * This hook is used to alter the query order by fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->list_slug . '_query_orderby', $this->orderby ) ;
		}

		/**
		 * Custom Search.
		 * 
		 * @retrun string
		 * */
		public function custom_search( $where ) {
			if ( ! isset( $_REQUEST[ 's' ] ) ) { // @codingStandardsIgnoreLine.
				return $where ;
			}

			$post_ids = array() ;
			$terms    = explode( ' , ', wc_clean( wp_unslash( $_REQUEST[ 's' ] ) ) ) ; // @codingStandardsIgnoreLine.

			foreach ( $terms as $term ) {
				$term       = $this->database->esc_like( ( $term ) ) ;
				$post_query = new WAL_Query( $this->database->prefix . 'posts', 'p' ) ;
				$post_query->select( 'DISTINCT `p`.ID' )
						->leftJoin( $this->database->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`' )
						->where( '`p`.post_type', $this->post_type )
						->whereIn( '`p`.post_status', wal_get_gift_voucher_statuses() )
						->whereIn( '`pm`.meta_key', array( 'wal_code' ) )
						->whereLike( '`pm`.meta_value', '%' . $term . '%' ) ;

				$post_ids = $post_query->fetchCol( 'ID' ) ;
			}

			$post_ids = wal_check_is_array( $post_ids ) ? $post_ids : array( 0 ) ;
			$where    .= ' AND (id IN (' . implode( ' , ', $post_ids ) . '))' ;

			return $where ;
		}
	}

}
