<?php

/**
 * Auto Topup Users List Table.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php'  ;
}

if ( ! class_exists( 'WAL_Auto_Topup_Users_List_Table' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Auto_Topup_Users_List_Table extends WP_List_Table {

		/**
		 * Per page count.
		 * 
		 * @var int
		 * */
		private $perpage = 20 ;

		/**
		 * Database.
		 * 
		 * @var object
		 * */
		private $database ;

		/**
		 * Offset.
		 * 
		 * @var int
		 * */
		private $offset ;

		/**
		 * Order BY.
		 * 
		 * @var string
		 * */
		private $orderby = 'ORDER BY ID DESC' ;

		/**
		 * Post type.
		 * 
		 * @var string
		 * */
		private $post_type = WAL_Register_Post_Types::AUTO_TOPUP_USERS_POSTTYPE ;

		/**
		 * List Slug.
		 * 
		 * @var string
		 * */
		private $list_slug = 'wal_autotopup_users' ;

		/**
		 * Base URL.
		 * 
		 * @var string
		 * */
		private $base_url ;

		/**
		 * Current URL.
		 * 
		 * @var string
		 * */
		private $current_url ;

		/**
		 * Constructor.
		 */
		public function __construct() {
			global $wpdb ;
			$this->database = &$wpdb ;
			$this->base_url = admin_url( 'admin.php?page=wal_modules&module=auto_topup&tab=auto_topup' ) ;

			parent::__construct(
					array(
						'singular' => 'autotopup_user',
						'plural'   => 'autotopup_users',
						'ajax'     => false,
					)
			) ;
		}

		/**
		 * Prepares the list of items for displaying.
		 * */
		public function prepare_items() {
			//Prepare the current url.
			$this->current_url = add_query_arg( array( 'paged' => absint( $this->get_pagenum() ) ), $this->base_url ) ;

			// Prepare the offset.
			$this->offset = $this->perpage * ( absint( $this->get_pagenum() ) - 1 ) ;

			// Prepare the header columns.
			$this->_column_headers = array( $this->get_columns(), $this->get_hidden_columns(), $this->get_sortable_columns() ) ;

			// Prepare the query clauses.
			$join    = $this->get_query_join() ;
			$where   = $this->get_query_where() ;
			$limit   = $this->get_query_limit() ;
			$offset  = $this->get_query_offset() ;
			$orderby = $this->get_query_orderby() ;

			// Prepare the all items.
			$count_items = $this->database->get_var( 'SELECT COUNT(DISTINCT ID) FROM ' . $this->database->posts . " AS p $where" ) ;

			// Prepare the current page items.
			$prepare_query = $this->database->prepare( 'SELECT DISTINCT ID FROM ' . $this->database->posts . " AS p $join $where $orderby LIMIT %d,%d", $offset, $limit ) ;

			$items = $this->database->get_results( $prepare_query, ARRAY_A ) ;

			// Prepare the item object.
			$this->prepare_item_object( $items ) ;

			// Prepare the pagination arguments.
			$this->set_pagination_args(
					array(
						'total_items' => $count_items,
						'per_page'    => $this->perpage,
					)
			) ;
		}

		/**
		 * Render the table.
		 * */
		public function render() {
			if ( isset( $_REQUEST[ 's' ] ) && strlen( wc_clean( wp_unslash( $_REQUEST[ 's' ] ) ) ) ) { // @codingStandardsIgnoreLine.
				/* translators: %s: search keywords */
				$searched_words = sprintf( __( 'Search results for &#8220;%s&#8221;', 'wallet-for-woocommerce' ) , wc_clean( wp_unslash( $_REQUEST[ 's' ] ) ) );
				echo '<span class="subtitle">' . esc_html($searched_words) . '</span>';
			}

			// Output the table.
			$this->prepare_items() ;
			$this->views() ;
			$this->display() ;
		}

		/**
		 * Get a list of columns.
		 * 
		 * @return array
		 * */
		public function get_columns() {
			$columns = array(
				'username' => __('Username and Email', 'wallet-for-woocommerce'),
				'status' => __('Status', 'wallet-for-woocommerce'),
				'topup_amt' => __('Auto Top-up Amount', 'wallet-for-woocommerce'),
				'threshold_amt' => __('Threshold Value', 'wallet-for-woocommerce'),
				'authorized_order' => __('Authorized Order Number', 'wallet-for-woocommerce'),              
				'last_charge_date' => __('Last Auto Top-up Date', 'wallet-for-woocommerce'),
				'last_activity' => __('Last Activity', 'wallet-for-woocommerce'),
			);

			return $columns ;
		}

		/**
		 * Get a list of hidden columns.
		 * 
		 * @return array
		 * */
		public function get_hidden_columns() {
			return array() ;
		}

		/**
		 * Get a list of sortable columns.
		 * 
		 * @return void
		 * */
		public function get_sortable_columns() {
			return array(
				'topup_amt'        => array( 'topup_amt', false ),
				'threshold_amt'    => array( 'threshold_amt', false ),
				'authorized_order' => array( 'authorized_order', false ),
				'last_charge_date' => array( 'last_charge_date', false ),
					) ;
		}

		/**
		 * Message to be displayed when there are no items.
		 */
		public function no_items() {
			esc_html_e( 'No auto topup users to show.', 'wallet-for-woocommerce' ) ;
		}

		/**
		 * Get a list of bulk actions.
		 * 
		 * @return array
		 * */
		protected function get_bulk_actions() {
			$action = array() ;
			/**
			 * This hook is used to alter the Auto Top-up bulk actions.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->list_slug . '_bulk_actions', $action ) ;
		}

		/**
		 * Display the list of views available on this table.
		 * 
		 * @return array
		 * */
		public function get_views() {
			$args        = array() ;
			$status_link = array() ;

			$status_link_array = array(
				'all'           => __( 'All', 'wallet-for-woocommerce' ),
				'wal_active'    => __( 'Active', 'wallet-for-woocommerce' ),
				'wal_cancelled' => __( 'Cancelled', 'wallet-for-woocommerce' ),
					) ;

			foreach ( $status_link_array as $status_name => $status_label ) {
				$status_count = $this->get_total_item_for_status( $status_name ) ;

				if ( ! $status_count ) {
					continue ;
				}

				$args[ 'status' ] = $status_name ;

				$label = $status_label . ' (' . $status_count . ')' ;

				$class = array( strtolower( $status_name ) ) ;
				if ( isset( $_GET[ 'status' ] ) && ( sanitize_title( $_GET[ 'status' ] ) == $status_name ) ) { // @codingStandardsIgnoreLine.
					$class[] = 'current' ;
				}

				if ( ! isset( $_GET[ 'status' ] ) && 'all' == $status_name ) { // @codingStandardsIgnoreLine.
					$class[] = 'current' ;
				}

				$status_link[ $status_name ] = $this->get_edit_link( $args, $label, implode( ' ', $class ) ) ;
			}

			return $status_link ;
		}

		/**
		 * Get a edit link.
		 * 
		 * @rerurn string
		 * */
		private function get_edit_link( $args, $label, $class = '' ) {
			$url        = add_query_arg( $args, $this->base_url ) ;
			$class_html = '' ;
			if ( ! empty( $class ) ) {
				$class_html = sprintf(
						' class="%s"', esc_attr( $class )
						) ;
			}

			return sprintf(
					'<a href="%s"%s>%s</a>', esc_url( $url ), $class_html, $label
					) ;
		}

		/**
		 * Get the total item by status.
		 * 
		 * @return int
		 * */
		private function get_total_item_for_status( $status = '' ) {
			// Get the current status item ids.
			$prepare_query = $this->database->prepare( 'SELECT COUNT(DISTINCT ID) FROM ' . $this->database->posts . " WHERE post_type=%s and post_status IN('" . $this->format_status( $status ) . "')", $this->post_type ) ;

			return $this->database->get_var( $prepare_query ) ;
		}

		/**
		 * Format the status.
		 * 
		 * @return string
		 * */
		private function format_status( $status ) {
			if ( 'all' == $status ) {
				$statuses = wal_get_auto_topup_statuses() ;
				$status   = implode( "', '", $statuses ) ;
			}

			return $status ;
		}

		/**
		 * Prepare a each column data.
		 * */
		protected function column_default( $item, $column_name ) {
			switch ( $column_name ) {
				case 'username':
					$username = '' ;
					$user     = $item->get_user() ;

					if ( $user ) {
						$username = '<a href="' . esc_url( add_query_arg( 'user_id', absint( $user->ID ), admin_url( 'user-edit.php' ) ) ) . '" target="_blank">' . esc_html( $user->display_name ) . '</br>' . esc_html( $user->user_email ) . '</a>' ;
					}

					return $username ;
				case 'status':
					return wal_display_post_status( $item->get_status() ) ;
				case 'topup_amt':
					return wc_price( $item->get_topup_amount(), array( 'currency' => $item->get_currency() ) ) ;
				case 'threshold_amt':
					return wc_price( $item->get_threshold_amount(), array( 'currency' => $item->get_currency() ) ) ;
				case 'authorized_order':
					/* translators: %s - URL,%1s - Order Number, %2s - View Orders */
					return wp_kses_post(sprintf(__('%1$s<br><a href="#" data-id="%2$1s" class="wal-auto-topup-renewal-order-details">%3$2s</a>'), $item->get_post_name_with_edit_link($item->get_authorized_order()), $item->get_authorized_order(), esc_html__('View Authorized Orders', 'wallet-for-woocommerce')));
				case 'last_charge_date':
					return '' === $item->get_last_charge_date() ? '--' : $item->get_formatted_last_charge_date() ;
				case 'last_activity':
					return '' === $item->get_last_activity() ? '--' : $item->get_last_activity() ;
			}
		}

		/**
		 * Prepare the item Object.
		 * 
		 * @return void
		 * */
		private function prepare_item_object( $items ) {
			$prepare_items = array() ;
			if ( wal_check_is_array( $items ) ) {
				foreach ( $items as $item ) {
					$prepare_items[] = wal_get_auto_topup( $item[ 'ID' ] ) ;
				}
			}

			$this->items = $prepare_items ;
		}

		/**
		 * Get the query join clauses.
		 * 
		 * @return string
		 * */
		private function get_query_join() {
			$join = '' ;
			if ( empty( $_REQUEST[ 'orderby' ] ) ) { // @codingStandardsIgnoreLine.
				return $join ;
			}

			$join = ' INNER JOIN ' . $this->database->postmeta . ' AS pm ON ( pm.post_id = p.ID )' ;
			/**
			 * This hook is used to alter the query join fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->list_slug . '_query_join', $join ) ;
		}

		/**
		 * Get the query where clauses.
		 * 
		 * @return string
		 * */
		private function get_query_where() {
			$current_status = 'all' ;
			if ( isset( $_GET[ 'status' ] ) && ( sanitize_title( $_GET[ 'status' ] ) != 'all' ) ) {
				$current_status = sanitize_title( $_GET[ 'status' ] ) ;
			}

			$where = " where post_type='" . $this->post_type . "' and post_status IN('" . $this->format_status( $current_status ) . "') " ;
			/**
			 * This hook is used to alter the query where fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->list_slug . '_query_where', $where ) ;
		}

		/**
		 * Get the query limit clauses.
		 * 
		 * @return string
		 * */
		private function get_query_limit() {
			/**
			 * This hook is used to alter the query limit fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->list_slug . '_query_limit', $this->perpage ) ;
		}

		/**
		 * Get the query offset clauses.
		 * 
		 * @return string
		 * */
		private function get_query_offset() {
			/**
			 * This hook is used to alter the query offset fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->list_slug . '_query_offset', $this->offset);
		}

		/**
		 * Render extra tablenav.
		 *
		 * @since 2.8.0
		 *
		 */
		public function extra_tablenav( $which ) {
			if ('top' != $which) {
				return;
			}

			echo '<div class="wal-hide wal-popup" data-popup="#wal-auto-topup-modal">';
			echo '<div id="wal-auto-topup-modal">';
			echo '</div></div>';
		}

		/**
		 * Get the query order by clauses.
		 * 
		 * @return string
		 * */
		private function get_query_orderby() {
			$order = 'DESC' ;
			if ( ! empty( $_REQUEST[ 'order' ] ) && is_string( $_REQUEST[ 'order' ] ) ) { // @codingStandardsIgnoreLine.
				if ( 'ASC' === strtoupper( wc_clean( wp_unslash( $_REQUEST[ 'order' ] ) ) ) ) { // @codingStandardsIgnoreLine.
					$order = 'ASC' ;
				}
			}

			// Order By.
			if ( isset( $_REQUEST[ 'orderby' ] ) ) {
				switch ( wc_clean( wp_unslash( $_REQUEST[ 'orderby' ] ) ) ) { // @codingStandardsIgnoreLine.
					case 'ID':
						$this->orderby = ' ORDER BY p.ID ' . $order ;
						break ;
					case 'authorized_order':
						$this->orderby = " AND pm.meta_key='wal_authorized_order' ORDER BY pm.meta_value " . $order ;
						break ;
					case 'topup_amt':
						$this->orderby = " AND pm.meta_key='wal_topup_amount' ORDER BY pm.meta_value " . $order ;
						break ;
					case 'threshold_amt':
						$this->orderby = " AND pm.meta_key='wal_threshold_amount' ORDER BY pm.meta_value " . $order ;
						break ;
					case 'last_charge_date':
						$this->orderby = " AND pm.meta_key='wal_last_charge_date' ORDER BY pm.meta_value " . $order ;
						break ;
					case 'date':
						$this->orderby = ' ORDER BY p.post_date ' . $order ;
						break ;
				}
			}
			/**
			 * This hook is used to alter the query order by fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->list_slug . '_query_orderby', $this->orderby ) ;
		}
	}

}
