<?php

/**
 * Wallet Withdrawal List Table.
 * 
 * @since 1.8
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Post_List_Table')) {
	require_once WAL_PLUGIN_PATH . '/inc/abstracts/abstract-wal-admin-post-list-table.php';
}

if (!class_exists('WAL_Wallet_Withdrawal_List_Table')) {

	/**
	 * Class.
	 * */
	class WAL_Wallet_Withdrawal_List_Table extends WAL_Post_List_Table {

		/**
		 * Post Type.
		 * 
		 * @var String
		 */
		protected $post_type = WAL_Register_Post_Types::WALLET_WITHDRAWAL_POSTTYPE;

		/**
		 * Plugin Slug.
		 * 
		 * @var String
		 */
		protected $plugin_slug = 'wal';

		/**
		 * Class initialization.
		 */
		public function __construct() {
			parent::__construct();
		}

		/**
		 * Initialize the extra hooks.
		 */
		protected function init_extra_hooks() {
			// Approve the wallet withdrawal.
			add_action('post_action_approve', array( $this, 'approve_wallet_withdrawal' ));
			// Cancel the wallet withdrawal.
			add_action('post_action_cancel', array( $this, 'cancel_wallet_withdrawal' ));
		}

		/**
		 * Add the custom meta boxes for this post type.
		 * 
		 * @return void
		 */
		public function add_meta_boxes() {
			global $post;
			$wal_withdrawal = wal_get_wallet_withdrawal($post->ID);
			// Remove the publish meta box.
			remove_meta_box('submitdiv', $this->post_type, 'side');
			if ('wal_pending' === $wal_withdrawal->get_status()) {
				// Wallet withdrawal status meta box.
				add_meta_box('wal-status-data', __('Status', 'wallet-for-woocommerce'), array( $this, 'render_status_meta_box' ), $this->post_type, 'side', 'high');
			}
			// Wallet withdrawal data meta box.
			add_meta_box('wal-wallet-withdrawal-data', __('Withdrawal', 'wallet-for-woocommerce'), array( $this, 'render_withdrawal_data_meta_box' ), $this->post_type, 'normal', 'high');
		}

		/**
		 * Define the which columns to show on this screen.
		 *
		 * @return array
		 */
		public function define_columns( $columns ) {
			if (empty($columns) && !is_array($columns)) {
				$columns = array();
			}

			unset($columns['comments'], $columns['date'], $columns['title']);

			$columns['user_name'] = __('Username', 'wallet-for-woocommerce');
			$columns['amount'] = __('Amount', 'wallet-for-woocommerce');
			$columns['fee'] = __('Fee', 'wallet-for-woocommerce');
			$columns['payment_method'] = __('Payment Method', 'wallet-for-woocommerce');
			$columns['created_date'] = __('Created Date', 'wallet-for-woocommerce');
			$columns['status'] = __('Status', 'wallet-for-woocommerce');
			$columns['actions'] = __('Actions', 'wallet-for-woocommerce');

			return $columns;
		}

		/**
		 * Define which columns are sortable.
		 *
		 * @return array
		 */
		public function define_sortable_columns( $columns ) {
			$custom_columns = array(
				'amount' => array( 'amount', true ),
				'fee' => array( 'fee', true ),
				'created_date' => array( 'date', true ),
			);

			return wp_parse_args($custom_columns, $columns);
		}

		/**
		 * Define bulk actions.
		 * 
		 * @return array
		 */
		public function define_bulk_actions( $actions ) {
			unset($actions['edit']);
			unset($actions['trash']);

			$actions['approve'] = __('Approve', 'wallet-for-woocommerce');
			$actions['cancel'] = __('Cancel', 'wallet-for-woocommerce');

			return $actions;
		}

		/**
		 * Disable Month Dropdown.
		 * 
		 * @return bool
		 */
		public function disable_months_dropdown( $bool, $post_type ) {
			return true;
		}

		/**
		 * Render extra tablenav.
		 *
		 * @since 2.8.0
		 * @param $which string.
		 *
		 */
		public function extra_tablenav( $which ) {
			if ('top' != $which) {
				return;
			}

			echo '<div class="wal-hide wal-popup" data-popup="#wal-wallet-withdrawal-modal">';
			echo '<div id="wal-wallet-withdrawal-modal">';
			echo '</div></div>';
		}

		/**
		 * Change messages when a post type is updated.
		 *
		 * @return array
		 */
		public function post_updated_messages( $messages ) {
			global $post;

			$messages[$this->post_type] = array(
				0 => '', // Unused. Messages start at index 1.
				1 => __('Wallet withdrawal Updated Successfully.', 'wallet-for-woocommerce'),
				4 => __('Wallet withdrawal Updated Successfully.', 'wallet-for-woocommerce'),
				6 => __('Wallet withdrawal Updated Successfully.', 'wallet-for-woocommerce'),
				7 => __('Wallet withdrawal Saved.', 'wallet-for-woocommerce'),
				10 => __('Wallet withdrawal Draft Updated.', 'wallet-for-woocommerce'),
			);

			return $messages;
		}

		/**
		 * Approve the wallet withdrawal.
		 *
		 * @return void
		 */
		public function approve_wallet_withdrawal( $post_id ) {
			check_admin_referer('approve-post_' . $post_id);

			if (!current_user_can('edit_post', $post_id)) {
				wp_die(esc_html__('Sorry, you are not allowed to edit this item.'));
			}

			// Approve the wallet withdrawal.
			wal_approve_wallet_withdrawal($post_id);
			wp_set_post_lock($post_id);

			$sendback = add_query_arg('post_type', $this->post_type, admin_url('edit.php'));
			wp_safe_redirect(add_query_arg('approved', 1, $sendback));
			exit();
		}

		/**
		 * Cancel the wallet withdrawal.
		 *
		 * @return void
		 */
		public function cancel_wallet_withdrawal( $post_id ) {
			check_admin_referer('cancel-post_' . $post_id);

			if (!current_user_can('edit_post', $post_id)) {
				wp_die(esc_html__('Sorry, you are not allowed to edit this item.'));
			}

			// Cancel the wallet withdrawal.
			wal_cancel_wallet_withdrawal($post_id);
			wp_set_post_lock($post_id);

			$sendback = add_query_arg('post_type', $this->post_type, admin_url('edit.php'));
			wp_safe_redirect(add_query_arg('cancelled', 1, $sendback));
			exit();
		}

		/**
		 * Handle the bulk actions.
		 *
		 * @return string
		 */
		public function handle_bulk_actions( $redirect_to, $action, $post_ids ) {

			switch ($action) {
				case 'approve':
					$approved = 0;
					$locked = 0;

					foreach ((array) $post_ids as $post_id) {
						if (!current_user_can('delete_post', $post_id)) {
							wp_die(esc_html__('Sorry, you are not allowed to move this item to the Trash.'));
						}

						if (wp_check_post_lock($post_id)) {
							$locked++;
							continue;
						}

						if (!wal_approve_wallet_withdrawal($post_id)) {
							continue;
						}

						$approved++;
					}

					$redirect_to = add_query_arg(
							array(
								'approved' => $approved,
								'ids' => join(',', $post_ids),
								'locked' => $locked,
							), $redirect_to
					);
					break;
				case 'cancel':
					$cancelled = 0;
					$locked = 0;

					foreach ((array) $post_ids as $post_id) {
						if (!current_user_can('delete_post', $post_id)) {
							wp_die(esc_html__('Sorry, you are not allowed to move this item to the Trash.'));
						}

						if (wp_check_post_lock($post_id)) {
							$locked++;
							continue;
						}

						if (!wal_cancel_wallet_withdrawal($post_id)) {
							continue;
						}

						$cancelled++;
					}

					$redirect_to = add_query_arg(
							array(
								'cancelled' => $cancelled,
								'ids' => join(',', $post_ids),
								'locked' => $locked,
							), $redirect_to
					);
					break;
			}

			return esc_url_raw($redirect_to);
		}

		/**
		 * Get row actions to show in the list table.
		 *
		 * @return array
		 */
		protected function get_row_actions( $actions, $post ) {
			//Unset the Quick edit.
			unset($actions['inline hide-if-no-js']);
			//Unset the trash.
			unset($actions['trash']);
			//Unset the edit.
			unset($actions['edit']);

			return $actions;
		}

		/**
		 * Pre-fetch any data for the row each column has access to it.
		 */
		protected function prepare_row_data( $post_id ) {
			if (empty($this->object) || $this->object->get_id() !== $post_id) {
				$this->object = wal_get_wallet_withdrawal($post_id);
			}
		}

		/**
		 * Render the user name column.
		 *
		 * @return void
		 */
		public function render_user_name_column() {
			echo wp_kses_post($this->object->get_user_name());
		}

		/**
		 * Render the amount column.
		 *
		 * @return void
		 */
		public function render_amount_column() {
			echo wp_kses_post(wal_price($this->object->get_amount()));
		}

		/**
		 * Render the fee column.
		 *
		 * @return void
		 */
		public function render_fee_column() {
			echo wp_kses_post(wal_price($this->object->get_fee()));
		}

		/**
		 * Render the payment method column.
		 *
		 * @return void
		 */
		public function render_payment_method_column() {
			echo wp_kses_post($this->object->get_payment_gateway_title());
		}

		/**
		 * Render the created date column.
		 *
		 * @return void
		 */
		public function render_created_date_column() {
			echo wp_kses_post($this->object->get_formatted_created_date());
		}

		/**
		 * Render the status column.
		 *
		 * @return void
		 */
		public function render_status_column() {
			echo wp_kses_post(wal_display_post_status($this->object->get_status()));

			if ('wal_pending' === $this->object->get_status() && !get_post_meta($this->object->get_id(), '_edit_lock', true)) {
				echo '<br/><span class="wal-new-status-label">' . esc_html__('New', 'wallet-for-woocommerce') . '</span>';
			}
		}

		/**
		 * Render the actions column.
		 *
		 * @return void
		 */
		public function render_actions_column() {
			global $wal_withdrawal;
			$wal_withdrawal = wal_get_wallet_withdrawal($this->object->get_id());
			$actions = array();

			$actions[] = sprintf('<a class="wal-wallet-actions wal-edit-post button" data-id=%s href=%s>%s</a>', $this->object->get_id(), get_edit_post_link($this->object->get_id()), __('Edit', 'wallet-for-woocommerce'));
			if ($this->object->has_status('wal_pending')) {
				$link = add_query_arg(array( 'action' => 'approve', 'post' => $this->object->get_id() ), admin_url('post.php'));
				$actions[] = sprintf('<a class="wal-wallet-actions wal-approve-post button" href=%s>%s</a>', wp_nonce_url($link, 'approve-post_' . $this->object->get_id()), __('Approve', 'wallet-for-woocommerce'));

				$link = add_query_arg(array( 'action' => 'cancel', 'post' => $this->object->get_id() ), admin_url('post.php'));
				$actions[] = sprintf('<a class="wal-wallet-actions wal-cancel-post button" data-id=%s href=%s>%s</a>', $this->object->get_id(), wp_nonce_url($link, 'cancel-post_' . $this->object->get_id()), __('Cancel', 'wallet-for-woocommerce'));
			}

			if ($this->object->has_status('wal_cancelled') && !empty($wal_withdrawal->get_cancelled_reason())) {
				$actions[] = sprintf('<a class="wal-wallet-actions wal-wallet-withdrawal-cancelled-reason button" data-id=%s href=#>%s</a>', $this->object->get_id(), __('Reason', 'wallet-for-woocommerce'));
			}

			echo wp_kses_post(implode(' ', $actions));
		}

		/**
		 * Handle order by filters.
		 *
		 * @return array
		 */
		protected function orderby_filters( $query_vars ) {
			// return if the order by key is not exists.
			if (!isset($query_vars['orderby'])) {
				return $query_vars;
			}

			switch (strtolower($query_vars['orderby'])) {
				case 'amount':
					$query_vars['meta_key'] = 'wal_amount';
					$query_vars['orderby'] = 'meta_value_num';
					break;

				case 'fee':
					$query_vars['meta_key'] = 'wal_fee';
					$query_vars['orderby'] = 'meta_value_num';
					break;
			}

			return $query_vars;
		}

		/**
		 * Get the search post IDs.
		 *
		 * @return array
		 */
		protected function get_search_post_ids( $terms ) {

			foreach ($terms as $term) {

				$term = $this->database->esc_like(( $term ));

				// Get the user IDs.
				$user_query = new WAL_Query($this->database->base_prefix . 'users', 'u');
				$user_query->select('DISTINCT `u`.ID')
						->whereLike('`u`.display_name', '%' . $term . '%');
				$user_ids = $user_query->fetchCol('ID');

				// Get the withdrawal IDs.
				$post_query = new WAL_Query($this->database->prefix . 'posts', 'p');
				$post_query->select('DISTINCT `p`.ID')
						->leftJoin($this->database->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`')
						->where('`p`.post_type', $this->post_type)
						->whereIn('`p`.post_status', wal_get_wallet_withdrawal_statuses())
						->where('`pm`.meta_key', 'wal_user_id')
						->whereIn('`pm`.meta_value', $user_ids);

				$withdrawal_ids = $post_query->fetchCol('ID');
			}

			return array_merge($withdrawal_ids, array( 0 ));
		}

		/**
		 * Render the wallet withdrawal status meta box.
		 *
		 * @return void
		 */
		public function render_status_meta_box( $post ) {
			global $wal_withdrawal;

			$withdrawal_id = ( isset($post->ID) ) ? $post->ID : '';
			$wal_withdrawal = wal_get_wallet_withdrawal($withdrawal_id);

			include_once WAL_PLUGIN_PATH . '/inc/admin/menu/views/meta-boxes/html-withdrawal-status.php';
		}

		/**
		 * Render the wallet withdrawal data meta box.
		 *
		 * @return void
		 */
		public function render_withdrawal_data_meta_box( $post ) {
			global $wal_withdrawal;

			$withdrawal_id = ( isset($post->ID) ) ? $post->ID : '';
			$wal_withdrawal = wal_get_wallet_withdrawal($withdrawal_id);

			include_once WAL_PLUGIN_PATH . '/inc/admin/menu/views/meta-boxes/html-withdrawal-data.php';
		}

		/**
		 * Save the meta box data.
		 *
		 * @return void
		 */
		public function save_current_meta_boxes( $post_id, $post ) {
			try {

				$status = isset($_REQUEST['post_status']) ? wc_clean(wp_unslash($_REQUEST['post_status'])) : 'wal_active';
				switch ($status) {
					case 'wal_approved':
						wal_approve_wallet_withdrawal($post_id, true);
						break;

					case 'wal_cancelled':
						$reason = isset($_REQUEST['wal_withdrawal_cancelled_reason']) ? sanitize_textarea_field($_REQUEST['wal_withdrawal_cancelled_reason']) : '';
						wal_cancel_wallet_withdrawal($post_id, $reason, true);
						break;

					default:
						$wal_withdrawal = wal_get_wallet_withdrawal($post_id);
						// Update the wallet data.
						wal_update_wallet_withdrawal($post_id, array(), array( 'post_status' => $status ));
						break;
				}
			} catch (Exception $ex) {
				WC_Admin_Meta_Boxes::add_error($ex->getMessage());
			}
		}
	}

}
