<?php
/**
 * Wallet List Table.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Post_List_Table' ) ) {
	require_once WAL_PLUGIN_PATH . '/inc/abstracts/abstract-wal-admin-post-list-table.php' ;
}

if ( ! class_exists( 'WAL_Wallet_List_Table' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Wallet_List_Table extends WAL_Post_List_Table {

		/**
		 * Post Type.
		 * 
		 * @var String
		 */
		protected $post_type = WAL_Register_Post_Types::WALLET_POSTTYPE ;

		/**
		 * Plugin Slug.
		 * 
		 * @var String
		 */
		protected $plugin_slug = 'wal' ;

		/**
		 * Class initialization.
		 */
		public function __construct() {
			parent::__construct() ;
		}

		/**
		 * Initialize the extra hooks.
		 */
		protected function init_extra_hooks() {
			// Activate the wallet.
			add_action( 'post_action_activate', array( $this, 'activate_wallet' ) ) ;
			// Deactivate the wallet.
			add_action( 'post_action_deactivate', array( $this, 'deactivate_wallet' ) ) ;
		}

		/**
		 * Add the custom meta boxes for this post type.
		 * 
		 * @return void
		 */
		public function add_meta_boxes() {
			// Remove the publish meta box.
			remove_meta_box( 'submitdiv', $this->post_type, 'side' ) ;
			// Wallet status meta box.
			add_meta_box( 'wal-status-data', __( 'Status', 'wallet-for-woocommerce' ), array( $this, 'render_status_meta_box' ), $this->post_type, 'side', 'high' ) ;
			// Wallet data meta box.
			add_meta_box( 'wal-wallet-data', __( 'Settings', 'wallet-for-woocommerce' ), array( $this, 'render_wallet_meta_box' ), $this->post_type, 'normal', 'high' ) ;
		}

		/**
		 * Define the which columns to show on this screen.
		 *
		 * @return array
		 */
		public function define_columns( $columns ) {
			if ( empty( $columns ) && ! is_array( $columns ) ) {
				$columns = array() ;
			}

			unset( $columns[ 'comments' ], $columns[ 'date' ], $columns[ 'title' ] ) ;

			$columns[ 'user_name' ]          = __( 'Username', 'wallet-for-woocommerce' ) ;
			$columns[ 'balance' ]            = __( 'Available Balance', 'wallet-for-woocommerce' ) ;
			$columns[ 'expiry_date' ]        = __( 'Expiry Date', 'wallet-for-woocommerce' ) ;
			$columns[ 'created_date' ]       = __( 'Created Date', 'wallet-for-woocommerce' ) ;
			$columns[ 'last_activity_date' ] = __( 'Last Activity Date', 'wallet-for-woocommerce' ) ;
			$columns[ 'status' ]             = __( 'Status', 'wallet-for-woocommerce' ) ;
			$columns[ 'actions' ]            = __( 'Actions', 'wallet-for-woocommerce' ) ;

			return $columns ;
		}

		/**
		 * Define which columns are sortable.
		 *
		 * @return array
		 */
		public function define_sortable_columns( $columns ) {
			$custom_columns = array(
				'balance'            => array( 'balance', true ),
				'expiry_date'        => array( 'expiry_date', true ),
				'created_date'       => array( 'date', true ),
				'last_activity_date' => array( 'last_activity_date', true ),
					) ;

			return wp_parse_args( $custom_columns, $columns ) ;
		}

		/**
		 * Define bulk actions.
		 * 
		 * @return array
		 */
		public function define_bulk_actions( $actions ) {
			unset( $actions[ 'edit' ] ) ;
			unset( $actions[ 'trash' ] ) ;

			$actions[ 'activate' ]   = __( 'Activate', 'wallet-for-woocommerce' ) ;
			$actions[ 'deactivate' ] = __( 'Deactivate', 'wallet-for-woocommerce' ) ;

			return $actions ;
		}

		/**
		 * Disable Month Dropdown.
		 * 
		 * @return bool
		 */
		public function disable_months_dropdown( $bool, $post_type ) {
			return true ;
		}

		/**
		 * Render extra tablenav.
		 *
		 */
		public function extra_tablenav( $which ) {
			if ( 'top' != $which ) {
				return ;
			}

			$export_url = add_query_arg( array( 'post_type' => $this->post_type, 'wal_export_csv' => 'wallet' ), admin_url( 'edit.php' ) ) ;
			?>
			<a href="<?php echo esc_url( $export_url ) ; ?>" class="wal-export-csv button button-primary"><?php esc_html_e( 'Export All Wallet as CSV', 'wallet-for-woocommerce' ) ; ?></a>
			<?php
		}

		/**
		 * Change messages when a post type is updated.
		 *
		 * @return array
		 */
		public function post_updated_messages( $messages ) {
			global $post ;

			$messages[ $this->post_type ] = array(
				0  => '', // Unused. Messages start at index 1.
				1  => __( 'Wallet Updated Successfully.', 'wallet-for-woocommerce' ),
				4  => __( 'Wallet Updated Successfully.', 'wallet-for-woocommerce' ),
				6  => __( 'Wallet Updated Successfully.', 'wallet-for-woocommerce' ),
				7  => __( 'Wallet Saved.', 'wallet-for-woocommerce' ),
				10 => __( 'Wallet Draft Updated.', 'wallet-for-woocommerce' ),
					) ;

			return $messages ;
		}

		/**
		 * Activate the wallet.
		 *
		 * @return void
		 */
		public function activate_wallet( $post_id ) {
			check_admin_referer( 'activate-post_' . $post_id ) ;

			if ( ! current_user_can( 'edit_post', $post_id ) ) {
				wp_die( esc_html__( 'Sorry, you are not allowed to edit this item.' ) ) ;
			}

			// Update the wallet status is active.
			wal_update_wallet( $post_id, array(), array( 'post_status' => 'wal_active' ) ) ;

			$sendback = add_query_arg( 'post_type', $this->post_type, admin_url( 'edit.php' ) ) ;

			wp_safe_redirect( add_query_arg( 'activated', 1, $sendback ) ) ;
			exit() ;
		}

		/**
		 * Deactivate the wallet.
		 *
		 * @return void
		 */
		public function deactivate_wallet( $post_id ) {
			check_admin_referer( 'deactivate-post_' . $post_id ) ;

			if ( ! current_user_can( 'edit_post', $post_id ) ) {
				wp_die( esc_html__( 'Sorry, you are not allowed to edit this item.' ) ) ;
			}

			// Update the wallet status is deactive.
			wal_update_wallet( $post_id, array(), array( 'post_status' => 'wal_inactive' ) ) ;

			$sendback = add_query_arg( 'post_type', $this->post_type, admin_url( 'edit.php' ) ) ;

			wp_safe_redirect( add_query_arg( 'deactivated', 1, $sendback ) ) ;
			exit() ;
		}

		/**
		 * Handle the bulk actions.
		 *
		 * @return string
		 */
		public function handle_bulk_actions( $redirect_to, $action, $post_ids ) {

			switch ( $action ) {
				case 'activate':
					$activated = 0 ;
					$locked    = 0 ;

					foreach ( ( array ) $post_ids as $post_id ) {
						if ( ! current_user_can( 'delete_post', $post_id ) ) {
							wp_die( esc_html__( 'Sorry, you are not allowed to move this item to the Trash.' ) ) ;
						}

						if ( wp_check_post_lock( $post_id ) ) {
							$locked++ ;
							continue ;
						}

						wal_update_wallet( $post_id, array(), array( 'post_status' => 'wal_active' ) ) ;

						$activated++ ;
					}

					$redirect_to = add_query_arg(
							array(
								'activated' => $activated,
								'ids'       => join( ',', $post_ids ),
								'locked'    => $locked,
							), $redirect_to
							) ;
					break ;

				case 'deactivate':
					$deactivated = 0 ;
					$locked      = 0 ;

					foreach ( ( array ) $post_ids as $post_id ) {
						if ( ! current_user_can( 'delete_post', $post_id ) ) {
							wp_die( esc_html__( 'Sorry, you are not allowed to move this item to the Trash.' ) ) ;
						}

						if ( wp_check_post_lock( $post_id ) ) {
							$locked++ ;
							continue ;
						}

						wal_update_wallet( $post_id, array(), array( 'post_status' => 'wal_inactive' ) ) ;

						$deactivated++ ;
					}

					$redirect_to = add_query_arg(
							array(
								'deactivated' => $deactivated,
								'ids'         => join( ',', $post_ids ),
								'locked'      => $locked,
							), $redirect_to
							) ;
					break ;
			}

			return esc_url_raw( $redirect_to ) ;
		}

		/**
		 * Get row actions to show in the list table.
		 *
		 * @return array
		 */
		protected function get_row_actions( $actions, $post ) {
			//Unset the Quick edit.
			unset( $actions[ 'inline hide-if-no-js' ] ) ;
			//Unset the trash.
			unset( $actions[ 'trash' ] ) ;
			//Unset the edit.
			unset( $actions[ 'edit' ] ) ;

			return $actions ;
		}

		/**
		 * Pre-fetch any data for the row each column has access to it.
		 */
		protected function prepare_row_data( $post_id ) {                        
			if ( empty( $this->object ) || $this->object->get_id() !== $post_id ) {
				$this->object = wal_get_wallet( $post_id ) ;
			}                       
		}

		/**
		 * Render the user name column.
		 *
		 * @return void
		 */
		public function render_user_name_column() {
			echo wp_kses_post( $this->object->get_user_name() ) ;
		}

		/**
		 * Render the balance column.
		 *
		 * @return void
		 */
		public function render_balance_column() {
			echo wp_kses_post( wal_price($this->object->get_balance()) ) ;
		}

		/**
		 * Render the expiry date column.
		 *
		 * @return void
		 */
		public function render_expiry_date_column() {
			echo wp_kses_post( $this->object->get_formatted_current_expiry_date() ) ;
		}

		/**
		 * Render the created date column.
		 *
		 * @return void
		 */
		public function render_created_date_column() {
			echo wp_kses_post( $this->object->get_formatted_created_date() ) ;
		}

		/**
		 * Render the last activity date column.
		 *
		 * @return void
		 */
		public function render_last_activity_date_column() {
			echo wp_kses_post( $this->object->get_formatted_last_activity_date() ) ;
		}

		/**
		 * Render the status column.
		 *
		 * @return void
		 */
		public function render_status_column() {
			echo wp_kses_post( wal_display_post_status( $this->object->get_status() ) ) ;
		}

		/**
		 * Render the actions column.
		 *
		 * @return void
		 */
		public function render_actions_column() {
			$actions = array() ;

			$actions['edit'] = wal_display_action('edit', $this->object->get_id(), admin_url('post.php'));
			$link      = wal_get_user_transaction_logs_page_url( array( 'wal_wallet_id' => $this->object->get_id() ) ) ;
			$actions['view_transactions'] = wal_display_action('view_transactions', $this->object->get_id(), $link);

			if ( $this->object->has_status( 'wal_active' ) ) {
				$link      = add_query_arg( array( 'action' => 'deactivate', 'post' => $this->object->get_id() ), admin_url( 'post.php' ) ) ;
				$actions['deactivate'] = wal_display_action('deactivate', $this->object->get_id(), $link);
			} else {
				$link      = add_query_arg( array( 'action' => 'activate', 'post' => $this->object->get_id() ), admin_url( 'post.php' ) ) ;
				$actions['activate'] = wal_display_action('activate', $this->object->get_id(), $link);
			}
			
			echo wp_kses_post( implode( ' ', $actions ) ) ;
		}

		/**
		 * Handle order by filters.
		 *
		 * @return array
		 */
		protected function orderby_filters( $query_vars ) {
			// return if the order by key is not exists.
			if ( ! isset( $query_vars[ 'orderby' ] ) ) {
				return $query_vars ;
			}

			switch ( strtolower( $query_vars[ 'orderby' ] ) ) {
				case 'balance':
					$query_vars[ 'meta_key' ] = 'wal_balance' ;
					$query_vars[ 'orderby' ]  = 'meta_value_num' ;
					break ;

				case 'expiry_date':
					$query_vars[ 'meta_key' ]  = 'wal_current_expiry_date' ;
					$query_vars[ 'orderby' ]   = 'meta_value' ;
					$query_vars[ 'meta_type' ] = 'DATETIME' ;
					break ;

				case 'last_activity_date':
					$query_vars[ 'meta_key' ]  = 'wal_last_activity_date' ;
					$query_vars[ 'orderby' ]   = 'meta_value' ;
					$query_vars[ 'meta_type' ] = 'DATETIME' ;
					break ;
			}

			return $query_vars ;
		}

		/**
		 * Get the search post IDs.
		 *
		 * @return array
		 */
		protected function get_search_post_ids( $terms ) {

			foreach ( $terms as $term ) {

				$term = $this->database->esc_like( ( $term ) ) ;

				// Get the user IDs.
				$user_query = new WAL_Query( $this->database->base_prefix . 'users', 'u' ) ;
				$user_query->select( 'DISTINCT `u`.ID' )
						->whereLike( '`u`.display_name', '%' . $term . '%' ) ;
				$user_ids   = $user_query->fetchCol( 'ID' ) ;

				// Get Transaction log IDs.
				$post_query = new WAL_Query( $this->database->prefix . 'posts', 'p' ) ;
				$post_query->select( 'DISTINCT `p`.ID' )
						->where( '`p`.post_type', $this->post_type )
						->whereIn( '`p`.post_status', wal_get_wallet_statuses() )
						->whereIn( '`p`.post_parent', $user_ids ) ;

				$transaction_log_ids = $post_query->fetchCol( 'ID' ) ;
			}

			return array_merge( $transaction_log_ids, array( 0 ) ) ;
		}

		/**
		 * Render the wallet status meta box.
		 *
		 * @return void
		 */
		public function render_status_meta_box( $post ) {
			global $wal_wallet ;

			$wallet_id  = ( isset( $post->ID ) ) ? $post->ID : '' ;
			$wal_wallet = wal_get_wallet( $wallet_id ) ;

			include_once WAL_PLUGIN_PATH . '/inc/admin/menu/views/meta-boxes/html-wallet-status.php' ;
		}

		/**
		 * Render the wallet settings meta box.
		 *
		 * @return void
		 */
		public function render_wallet_meta_box( $post ) {
			global $wal_wallet ;

			$wallet_id  = ( isset( $post->ID ) ) ? $post->ID : '' ;
			$wal_wallet = wal_get_wallet( $wallet_id ) ;

			include_once WAL_PLUGIN_PATH . '/inc/admin/menu/views/meta-boxes/html-wallet-data-panels.php' ;
		}
		
		/**
		 * Show the settings tab content/settings.
		 */
		private static function output_tabs() {
			global $wal_wallet ;

			include_once WAL_PLUGIN_PATH . '/inc/admin/menu/views/meta-boxes/html-wallet-data-general-panel.php' ;
			include_once WAL_PLUGIN_PATH . '/inc/admin/menu/views/meta-boxes/html-wallet-data-credit-debit-fund-panel.php' ;
		}

		/**
		 * Return array of tabs.
		 *
		 * @return array
		 */
		private static function get_wallet_data_tabs() {
			/**
			 * This hook is used to alter the wallet data tabs.
			 * 
			 * @since 1.0
			 */
			$tabs = apply_filters(
					'wal_wallet_data_tabs', array(
				'general'           => array(
					'label'      => __( 'General', 'wallet-for-woocommerce' ),
					'target'     => 'wal_wallet_data_general',
					'class'      => array(),
					'priority'   => 10,
					'icon_class' => 'dashicons-admin-tools',
					),
					'credit-debit-fund' => array(
					'label'      => __( 'Credit/Debit Funds', 'wallet-for-woocommerce' ),
					'target'     => 'wal_wallet_data_credit_debit_fund',
					'class'      => array( '' ),
					'priority'   => 50,
					'icon_class' => 'dashicons-edit',
					),
					) ) ;

			// Sort tabs based on priority.
			uasort( $tabs, array( __CLASS__, 'wallet_data_tabs_sort' ) ) ;

			return $tabs ;
		}

		/**
		 * Callback to sort tabs on priority.
		 *
		 * @return bool
		 */
		private static function wallet_data_tabs_sort( $a, $b ) {
			if ( ! isset( $a[ 'priority' ], $b[ 'priority' ] ) ) {
				return -1 ;
			}

			if ( $a[ 'priority' ] === $b[ 'priority' ] ) {
				return 0 ;
			}

			return $a[ 'priority' ] < $b[ 'priority' ] ? -1 : 1 ;
		}

		/**
		 * Save the meta box data.
		 *
		 * @return void
		 */
		public function save_current_meta_boxes( $post_id, $post ) {
			try {

				$expiry_day = isset( $_REQUEST[ 'wal_wallet_expiry_day' ] ) ? wc_clean( wp_unslash( $_REQUEST[ 'wal_wallet_expiry_day' ] ) ) : '' ;
				$status     = isset( $_REQUEST[ 'post_status' ] ) ? wc_clean( wp_unslash( $_REQUEST[ 'post_status' ] ) ) : 'wal_active' ;

				$expiry_day = empty( $expiry_day ) ? '' : WAL_Date_Time::get_mysql_date_time_format( $expiry_day, false, 'UTC' ) ;

				$meta_args = array( 'wal_current_expiry_date' => $expiry_day ) ;

				// Update the wallet data.
				wal_update_wallet( $post_id, $meta_args, array( 'post_status' => $status ) ) ;
			} catch ( Exception $ex ) {
				WC_Admin_Meta_Boxes::add_error( $ex->getMessage() ) ;
			}
		}
	}

}
	
