<?php
/**
 * Transaction Log List Table.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Post_List_Table')) {
	require_once WAL_PLUGIN_PATH . '/inc/abstracts/abstract-wal-admin-post-list-table.php';
}

if (!class_exists('WAL_Transaction_Log_List_Table')) {

	/**
	 * Class.
	 * */
	class WAL_Transaction_Log_List_Table extends WAL_Post_List_Table {

		/**
		 * Post Type.
		 * 
		 * @var String
		 */
		protected $post_type = WAL_Register_Post_Types::TRANSACTION_LOG_POSTTYPE;

		/**
		 * Plugin Slug.
		 * 
		 * @var String
		 */
		protected $plugin_slug = 'wal';

		/**
		 * Class initialization.
		 */
		public function __construct() {
			parent::__construct();
		}

		/**
		 * Define the which columns to show on this screen.
		 *
		 * @return array
		 */
		public function define_columns( $columns ) {
			if (empty($columns) && !is_array($columns)) {
				$columns = array();
			}

			unset($columns['comments'], $columns['date'], $columns['title'], $columns['cb']);

			$columns['user_name'] = __('Username', 'wallet-for-woocommerce');
			$columns['event'] = __('Event', 'wallet-for-woocommerce');
			$columns['amount'] = __('Amount', 'wallet-for-woocommerce');
			$columns['status'] = __('Status', 'wallet-for-woocommerce');
			$columns['total'] = __('Available Balance', 'wallet-for-woocommerce');
			$columns['created_date'] = __('Date', 'wallet-for-woocommerce');

			return $columns;
		}

		/**
		 * Define primary column.
		 *
		 * @return array
		 */
		protected function get_primary_column() {
			return 'user_name';
		}

		/**
		 * Define which columns are sortable.
		 *
		 * @return array
		 */
		public function define_sortable_columns( $columns ) {
			$custom_columns = array(
				'amount' => array( 'amount', false ),
				'created_date' => array( 'date', false ),
				'total' => array( 'total', false ),
			);

			return wp_parse_args($custom_columns, $columns);
		}

		/**
		 * Define bulk actions.
		 * 
		 * @return array
		 */
		public function define_bulk_actions( $actions ) {
			unset($actions['edit'], $actions['trash']);

			return $actions;
		}

		/**
		 * Disable the month dropdown.
		 * 
		 * @return bool
		 */
		public function disable_months_dropdown( $bool, $post_type ) {
			return true;
		}

		/**
		 * Render extra tablenav.
		 *
		 */
		public function extra_tablenav( $which ) {
			?>
			<div class="wal-transaction-filters-wrapper alignleft actions">
				<?php
				if ('top' === $which) {
					ob_start();
					$this->filter_type_dropdown();
					/**
					 * This hook used to display the extra rule table filters.
					 *
					 * @since 11.4.0
					 */
					do_action('wal_transaction_table_filters');
					$output = ob_get_clean();

					if (!empty($output)) {
						echo do_shortcode($output);
						submit_button(__('Filter', 'wallet-for-woocommerce'), '', 'filter_action', false, array( 'id' => 'post-query-submit' ));
					}
				}
				$export_url = add_query_arg(array( 'post_type' => $this->post_type, 'wal_export_csv' => 'transaction_logs' ), admin_url('edit.php'));
				?>
				<a href="<?php echo esc_url($export_url); ?>" class="wal-export-csv button button-primary"><?php esc_html_e('Export Transaction Log as CSV', 'wallet-for-woocommerce'); ?></a>
			</div>
			<?php
		}

		/**
		 * Display the filter type drop down.
		 * 
		 * @since 4.5.0
		 */
		private function filter_type_dropdown() {
			$filter_type = isset($_REQUEST['wal_filter_types']) ? wc_clean(wp_unslash($_REQUEST['wal_filter_types'])) : '';
			?>
			<select class='wal-filter-types' name='wal_filter_types'>
				<option value='' <?php selected($filter_type, ''); ?>><?php esc_html_e('Select a Filter Type', 'wallet-for-woocommerce'); ?></option>
				<?php foreach (wal_get_filter_types() as $type_id => $type_name) : ?>
					<option value='<?php echo esc_attr($type_id); ?>' <?php selected($filter_type, $type_id); ?>><?php echo esc_html($type_name); ?></option>
				<?php endforeach; ?>
			</select>
			<?php
		}

		/**
		 * Get row actions to show in the list table.
		 *
		 * @return array
		 */
		protected function get_row_actions( $actions, $post ) {
			//Unset the Quick edit.
			unset($actions['inline hide-if-no-js']);
			//Unset the edit.
			unset($actions['edit']);
			//Unset the trash.
			unset($actions['trash']);

			return $actions;
		}

		/**
		 * Pre-fetch any data for the row each column has access to it.
		 */
		protected function prepare_row_data( $post_id ) {
			if (empty($this->object) || $this->object->get_id() !== $post_id) {
				$this->object = wal_get_transaction_log($post_id);
			}
		}

		/**
		 * Render the user name column.
		 *
		 * @return void
		 */
		public function render_user_name_column() {
			echo wp_kses_post($this->object->get_user_name());
		}

		/**
		 * Render the event column.
		 *
		 * @return void
		 */
		public function render_event_column() {
			echo wp_kses_post($this->object->get_formatted_event_message());
		}

		/**
		 * Render the amount column.
		 *
		 * @return void
		 */
		public function render_amount_column() {   
			echo wp_kses_post(wal_price($this->object->get_amount() ));
		}

		/**
		 * Render the total column.
		 *
		 * @return void
		 */
		public function render_total_column() {
			echo wp_kses_post(wal_price($this->object->get_total()));
		}

		/**
		 * Render the status column.
		 *
		 * @return void
		 */
		public function render_status_column() {
			echo wp_kses_post(wal_display_post_status($this->object->get_status()));
		}

		/**
		 * Render the created date column.
		 *
		 * @return void
		 */
		public function render_created_date_column() {
			echo wp_kses_post($this->object->get_formatted_created_date());
		}

		/**
		 * Get the search post IDs.
		 *
		 * @return array
		 */
		protected function get_search_post_ids( $terms ) {

			foreach ($terms as $term) {

				$term = $this->database->esc_like(( $term ));

				// Get the user IDs.
				$user_query = new WAL_Query($this->database->base_prefix . 'users', 'u');
				$user_query->select('DISTINCT `u`.ID')
						->whereLike('`u`.display_name', '%' . $term . '%');
				$user_ids = $user_query->fetchCol('ID');

				// Get Transaction log IDs.
				$post_query = new WAL_Query($this->database->prefix . 'posts', 'p');
				$post_query->select('DISTINCT `p`.ID')
						->leftJoin($this->database->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`')
						->where('`p`.post_type', $this->post_type)
						->whereIn('`p`.post_status', wal_get_transaction_log_statuses())
						->where('`pm`.meta_key', 'wal_user_id')
						->whereIn('`pm`.meta_value', $user_ids);

				$transaction_log_ids = $post_query->fetchCol('ID');
			}

			return array_merge($transaction_log_ids, array( 0 ));
		}

		/**
		 * Handle order by filters.
		 *
		 * @return array
		 */
		protected function orderby_filters( $query_vars ) {

			// return if the order by key is not exists.
			if (!isset($query_vars['orderby'])) {
				return $query_vars;
			}

			switch (strtolower($query_vars['orderby'])) {

				case 'amount':
					$query_vars['meta_key'] = 'wal_amount';
					$query_vars['orderby'] = 'meta_value_num';
					break;

				case 'total':
					$query_vars['meta_key'] = 'wal_total';
					$query_vars['orderby'] = 'meta_value_num';
					break;
			}

			return $query_vars;
		}

		/**
		 * Handle query filters.
		 *
		 * @since 2.5.0
		 *
		 * @return array
		 */
		protected function query_filters( $query_vars ) {
			if (empty($query_vars['orderby']) && 'wal_transaction_log' == $this->post_type) {
				if (isset($_REQUEST['wal_filter_types'])) {
					$query_vars['meta_key'] = 'wal_event_id';
					$query_vars['meta_value'] = wc_clean(wp_unslash($_REQUEST['wal_filter_types']));
				}
				$query_vars['orderby'] = 'ID';
			}

			return $query_vars;
		}
	}

}
