<?php
/**
 * Admin Settings Class.
 * 
 * @since 1.0.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Settings')) {

	/**
	 * Class
	 * 
	 * @since 1.0.0
	 */
	class WAL_Settings {

		/**
		 * Setting pages.
		 * 
		 * @since 1.0.0
		 * @var array
		 */
		private static $settings = array();

		/**
		 * Errors.
		 * 
		 * @since 1.0.0
		 * @var array
		 */
		private static $errors = array();

		/**
		 * Plugin slug.
		 * 
		 * @since 1.0.0
		 * @var string
		 */
		private static $plugin_slug = 'wal';

		/**
		 * Messages.
		 * 
		 * @since 1.0.0
		 * @var array
		 */
		private static $messages = array();

		/**
		 * Include the settings page classes.
		 * 
		 * @since 1.0.0
		 */
		public static function get_settings_pages() {
			if (!empty(self::$settings)) {
				return self::$settings;
			}

			include_once WAL_PLUGIN_PATH . '/inc/abstracts/abstract-wal-settings-page.php';

			$settings = array();
			$tabs = self::settings_page_tabs();

			foreach ($tabs as $tab_name) {
				$settings[str_replace('-', '_', $tab_name)] = include 'tabs/' . sanitize_key($tab_name) . '.php';
			}
			/**
			 * This hook is used to alert the settings pages.
			 * 
			 * @since 1.0.0
			 * @param $settings
			 */
			self::$settings = apply_filters(sanitize_key(self::$plugin_slug . '_get_settings_pages'), $settings);

			return self::$settings;
		}

		/**
		 * Add a message.
		 * 
		 * @since 1.0.0
		 * @param string $text
		 */
		public static function add_message( $text ) {
			self::$messages[] = $text;
		}

		/**
		 * Add an error.
		 * 
		 * @since 1.0.0
		 * @param string $text
		 */
		public static function add_error( $text ) {
			self::$errors[] = $text;
		}

		/**
		 * Output messages + errors.
		 * 
		 * @since 1.0.0
		 */
		public static function show_messages() {
			if (count(self::$errors) > 0) {
				foreach (self::$errors as $error) {
					self::error_message($error);
				}
			} elseif (count(self::$messages) > 0) {
				foreach (self::$messages as $message) {
					self::success_message($message);
				}
			}
		}

		/**
		 * Show an success message.
		 * 
		 * @since 1.0.0
		 * @param string $text
		 * @param bool $echo
		 */
		public static function success_message( $text, $echo = true ) {
			ob_start();
			$contents = '<div id="message " class="updated inline ' . esc_html(self::$plugin_slug) . '_save_msg"><p><strong>' . esc_html($text) . '</strong></p></div>';
			ob_end_clean();

			if ($echo) {
				echo wp_kses_post($contents);
			} else {
				return $contents;
			}
		}

		/**
		 * Show an error message.
		 * 
		 * @since 1.0.0
		 * @param string $text
		 * @param bool $echo
		 * @return $html
		 */
		public static function error_message( $text, $echo = true ) {
			ob_start();
			$contents = '<div id="message" class="error inline"><p><strong>' . esc_html($text) . '</strong></p></div>';
			ob_end_clean();

			if ($echo) {
				echo wp_kses_post($contents);
			} else {
				return $contents;
			}
		}

		/**
		 * Settings page tabs.
		 * 
		 * @since 1.0.0
		 * @return array
		 */
		public static function settings_page_tabs() {

			return array(
				'general',
				'advanced',
				'notifications',
				'localization',
				'messages',
				'shortcode',
				'reset',
			);
		}

		/**
		 * Handles the display of the settings page in admin.
		 * 
		 * @since 1.0.0
		 * @global string $current_section
		 * @global string $current_tab
		 */
		public static function output() {
			global $current_section, $current_tab;

			$tabs = wal_get_allowed_setting_tabs();

			/* Include admin html settings */
			include_once 'views/html-settings.php';
		}

		/**
		 * Handles the display of the settings page buttons in page.
		 * 
		 * @since 1.0.0
		 * @param bool $reset
		 */
		public static function output_buttons( $reset = true ) {
			/* Include admin html settings buttons */
			include_once 'views/html-settings-buttons.php';
		}

		/**
		 * Output admin fields.
		 * 
		 * @since 1.0.0
		 * @param array $value
		 */
		public static function output_fields( $value ) {

			if (!isset($value['type']) || 'wal_custom_fields' != $value['type']) {
				return;
			}

			$value['id'] = isset($value['id']) ? $value['id'] : '';
			$value['css'] = isset($value['css']) ? $value['css'] : '';
			$value['desc'] = isset($value['desc']) ? $value['desc'] : '';
			$value['title'] = isset($value['title']) ? $value['title'] : '';
			$value['class'] = isset($value['class']) ? $value['class'] : '';
			$value['default'] = isset($value['default']) ? $value['default'] : '';
			$value['name'] = isset($value['name']) ? $value['name'] : $value['id'];
			$value['placeholder'] = isset($value['placeholder']) ? $value['placeholder'] : '';
			$value['without_label'] = isset($value['without_label']) ? $value['without_label'] : false;
			$value['custom_attributes'] = isset($value['custom_attributes']) ? $value['custom_attributes'] : '';

			// Custom attribute handling.
			$custom_attributes = wal_format_custom_attributes($value);

			// Description handling.
			$field_description = WC_Admin_Settings::get_field_description($value);
			$description = $field_description['description'];
			$tooltip_html = $field_description['tooltip_html'];

			// Switch based on type
			switch ($value['wal_field']) {

				case 'button':
					?>
					<tr valign="top">
						<?php if (!$value['without_label']) : ?>
							<th scope="row">
								<label for="<?php echo esc_attr($value['id']); ?>"><?php echo esc_html($value['title']); ?></label><?php echo wp_kses_post($tooltip_html); ?>
							</th>
						<?php endif; ?>
						<td>
							<button
								id="<?php echo esc_attr($value['id']); ?>"
								type="<?php echo esc_attr($value['wal_field']); ?>"
								class="<?php echo esc_attr($value['class']); ?>"
								<?php echo wp_kses_post(implode(' ', $custom_attributes)); ?>
								><?php echo esc_html($value['default']); ?> </button>
								<?php echo wp_kses_post($description); ?>
						</td>
					</tr>
					<?php
					break;

				case 'price':
					$option_value = get_option($value['id'], $value['default']);
					?>
					<tr valign="top">
						<th scope="row">
							<label for="<?php echo esc_attr($value['id']); ?>"><?php echo esc_html($value['title']); ?></label><?php echo wp_kses_post($tooltip_html); ?>
						</th>
						<td>
							<input
								id="<?php echo esc_attr($value['id']); ?>"
								name="<?php echo esc_attr($value['name']); ?>"
								type="text"
								class="wc_input_price <?php echo esc_attr($value['class']); ?>"
								<?php echo wp_kses_post(implode(' ', $custom_attributes)); ?>
								value="<?php echo esc_html(wc_format_localized_price($option_value)); ?>"/>
								<?php echo wp_kses_post($description); ?>
						</td>
					</tr>
					<?php
					break;

				case 'ajaxmultiselect':
					$option_value = get_option($value['id'], $value['default']);
					?>
					<tr valign="top">
						<th scope="row">
							<label for="<?php echo esc_attr($value['id']); ?>"><?php echo esc_html($value['title']); ?></label><?php echo wp_kses_post($tooltip_html); ?>
						</th>
						<td>
							<?php
							$value['options'] = $option_value;
							wal_select2_html($value);
							echo wp_kses_post($description);
							?>
						</td>
					</tr>
					<?php
					break;

				// datepicker.
				case 'datepicker':
					$value['value'] = get_option($value['id'], $value['default']);
					?>
					<tr valign="top">
						<th scope="row" class="titledesc">
							<label for="<?php echo esc_attr($value['id']); ?>"><?php echo esc_html($value['title']); ?> <?php echo wp_kses_post($tooltip_html); // WPCS: XSS ok. ?></label>
						</th>
						<td class="forminp">

							<?php
							wal_get_datepicker_html($value);
							echo wp_kses_post($description); // WPCS: XSS ok. 
							?>
						</td>
					</tr>
					<?php
					break;

				case 'wpeditor':
					$option_value = get_option($value['id'], $value['default']);
					?>
					<tr valign="top">
						<th scope="row">
							<label for="<?php echo esc_attr($value['id']); ?>"><?php echo esc_html($value['title']); ?></label><?php echo wp_kses_post($tooltip_html); ?>
						</th>
						<td>
							<?php
							wp_editor(
									$option_value, $value['id'], array(
								'media_buttons' => false,
								'editor_class' => esc_attr($value['class']),
									)
							);

							echo wp_kses_post($description);
							?>
						</td>
					</tr>
					<?php
					break;
				case 'topup_bonus':
					$topup_bonus_rules = get_option($value['id'], $value['default']);
					include_once WAL_PLUGIN_PATH . '/inc/admin/menu/views/html-topup-bonus-amount.php';
					break;
				case 'product_purchase_range':
					$product_purchase_range_rules = get_option($value['id'], $value['default']);
					include_once WAL_PLUGIN_PATH . '/inc/modules/views/html-product-purchase-range-fund.php';
					break;
				case 'category_total_range':
					$category_total_range_rules = get_option($value['id'], $value['default']);
					include_once WAL_PLUGIN_PATH . '/inc/modules/views/html-category-total-range-fund.php';
					break;
			}
		}

		/**
		 * Save the setting fields.
		 * 
		 * @since 1.0.0
		 * @param string $value
		 * @param array $option
		 * @param string $raw_value
		 * 
		 * @return mixed
		 */
		public static function prepare_field_value( $value, $option, $raw_value ) {
			if (!isset($option['type']) || 'wal_custom_fields' != $option['type']) {
				return $value;
			}

			$value = null;

			// Format the value based on option type.
			switch ($option['wal_field']) {
				case 'ajaxmultiselect':
					$value = array_filter((array) $raw_value, 'wal_array_filter');
					break;
				case 'datepicker':
				case 'wpeditor':
				case 'topup_bonus':
				case 'category_total_range':
					$value = isset($raw_value) ? $raw_value : array();
					break;
				case 'product_purchase_range':
					$value = isset($raw_value) ? $raw_value : array();
					break;
				case 'price':
					$value = wc_format_decimal($raw_value);
					break;
			}

			return $value;
		}

		/**
		 * Reset the setting fields.
		 * 
		 * @since 1.0.0
		 * @param array $options
		 * @return bool
		 */
		public static function reset_fields( $options ) {
			if (!is_array($options)) {
				return false;
			}

			// Loop options and get values to reset.
			foreach ($options as $option) {
				if (!isset($option['id']) || !isset($option['type']) || !isset($option['default'])) {
					continue;
				}

				update_option($option['id'], $option['default']);
			}

			return true;
		}
	}

}
