<?php

/**
 * Menu Management.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Menu_Management' ) ) {

	include_once 'class-wal-settings.php'  ;

	/**
	 * Class.
	 */
	class WAL_Menu_Management {

		/**
		 * Plugin slug.
		 * 
		 * @var string
		 */
		protected static $plugin_slug = 'wal' ;

		/**
		 * Menu slug.
		 * 
		 * @var string
		 */
		protected static $menu_slug = 'wal_wallet' ;

		/**
		 * Funds slug.
		 * 
		 * @var string
		 */
		protected static $funds_slug = 'wal_funds' ;

		/**
		 * Modules slug.
		 * 
		 * @var string
		 */
		protected static $modules_slug = 'wal_modules' ;

		/**
		 * Transaction log slug.
		 * 
		 * @var string
		 */
		protected static $transaction_logs_slug = 'wal_transaction_logs' ;

		/**
		 * Settings slug.
		 * 
		 * @var string
		 */
		protected static $settings_slug = 'wal_settings' ;

		/**
		 * Class initialization.
		 */
		public static function init() {
			add_action( 'admin_menu', array( __CLASS__, 'add_menu_pages' ) ) ;
			add_action( 'admin_head', array( __CLASS__, 'hide_submenu' ) ) ;
			add_filter( 'woocommerce_screen_ids', array( __CLASS__, 'add_custom_wc_screen_ids' ), 9, 1 ) ;
			add_filter( 'woocommerce_admin_settings_sanitize_option', array( 'WAL_Settings', 'prepare_field_value' ), 10, 3 ) ;
			add_filter( 'set-screen-option', array( __CLASS__, 'set_screen_option' ), 10, 3 ) ;
		}

		/**
		 * Add the custom screen IDs in WooCommerce.
		 * 
		 * @return array
		 */
		public static function add_custom_wc_screen_ids( $wc_screen_ids ) {
			$screen_ids = wal_page_screen_ids() ;

			$newscreenids = get_current_screen() ;
			$screenid     = str_replace( 'edit-', '', $newscreenids->id ) ;

			// Return if current page is not wallet page.
			if ( ! in_array( $screenid, $screen_ids ) ) {
				return $wc_screen_ids ;
			}

			$wc_screen_ids[] = $screenid ;

			return $wc_screen_ids ;
		}

		/**
		 * Add the custom menu pages.
		 * 
		 * @return void
		 */
		public static function add_menu_pages() {
			$url = WAL_PLUGIN_URL . '/assets/images/dash-icon.png' ;

			// Wallet Menu.
			add_menu_page( WAL()->menu_name(), WAL()->menu_name(), 'manage_woocommerce', self::$menu_slug, null, $url ) ;
			// Funds Submenu.
			add_submenu_page( self::$menu_slug, __( 'Credit/Debit Funds', 'wallet-for-woocommerce' ), __( 'Credit/Debit Funds', 'wallet-for-woocommerce' ), 'manage_woocommerce', self::$funds_slug, array( __CLASS__, 'funds_page' ) ) ;
			// Modules Submenu.
			$modules_page               = add_submenu_page( self::$menu_slug, __( 'Modules', 'wallet-for-woocommerce' ), __( 'Modules', 'wallet-for-woocommerce' ), 'manage_woocommerce', self::$modules_slug, array( __CLASS__, 'modules_page' ) ) ;
			// Transaction logs Submenu.
			$user_transaction_logs_page = add_submenu_page( self::$menu_slug, __( 'Transaction Logs', 'wallet-for-woocommerce' ), __( 'Transaction Logs', 'wallet-for-woocommerce' ), 'manage_woocommerce', self::$transaction_logs_slug, array( __CLASS__, 'transaction_logs_page' ) ) ;
			// Settings Submenu.
			$settings_page              = add_submenu_page( self::$menu_slug, __( 'Settings', 'wallet-for-woocommerce' ), __( 'Settings', 'wallet-for-woocommerce' ), 'manage_woocommerce', self::$settings_slug, array( 'WAL_Settings', 'output' ) ) ;

			add_action( 'load-' . $settings_page, array( __CLASS__, 'settings_page_init' ) ) ;
			add_action( 'load-' . $user_transaction_logs_page, array( __CLASS__, 'user_transaction_logs_page_init' ) ) ;
			add_action( 'load-' . $modules_page, array( __CLASS__, 'modules_page_init' ) ) ;
		}

		/**
		 * Hide the sub menu name from the wallet menu.
		 * 
		 * @return void
		 */
		public static function hide_submenu() {
			remove_submenu_page( self::$menu_slug, self::$transaction_logs_slug ) ;
		}

		/**
		 * Initialize the settings page.
		 */
		public static function settings_page_init() {
			global $current_tab, $current_section, $current_sub_section ;

			// Include settings pages.
			$settings = WAL_Settings::get_settings_pages() ;
			$tabs     = wal_get_allowed_setting_tabs() ;

			// prepare current tab/section.
			$current_tab = key( $tabs ) ;
			if ( ! empty( $_GET[ 'tab' ] ) ) {
				$sanitize_current_tab = sanitize_title( wp_unslash( $_GET[ 'tab' ] ) ) ; // @codingStandardsIgnoreLine.
				if ( array_key_exists( $sanitize_current_tab, $tabs ) ) {
					$current_tab = $sanitize_current_tab ;
				}
			}

			$section             = isset( $settings[ $current_tab ] ) ? $settings[ $current_tab ]->get_sections() : array() ;
			$current_section     = empty( $_REQUEST[ 'section' ] ) ? key( $section ) : sanitize_title( wp_unslash( $_REQUEST[ 'section' ] ) ) ; // @codingStandardsIgnoreLine.
			$current_section     = empty( $current_section ) ? $current_tab : $current_section ;
			$current_sub_section = empty( $_REQUEST[ 'subsection' ] ) ? '' : sanitize_title( wp_unslash( $_REQUEST[ 'subsection' ] ) ) ; // @codingStandardsIgnoreLine.

			/**
			 * This hook is used to do extra action after settings loaded.
			 * 
			 * @hooked WAL_Settings_Page->save - 10 (save the settings).
			 * @hooked WAL_Settings_Page->reset - 20 (reset the settings).
			 * @since 1.0
			 */
			do_action( sanitize_key( self::$plugin_slug . '_settings_loaded_' . $current_tab ), $current_section ) ;

			add_action( 'wal_settings_content', array( 'WAL_Settings', 'show_messages' ) ) ;
			add_action( 'woocommerce_admin_field_wal_custom_fields', array( 'WAL_Settings', 'output_fields' ) ) ;
		}

		/**
		 * Initialize the screen option in user transaction logs page.
		 */
		public static function user_transaction_logs_page_init() {

			// Add screen option.
			add_screen_option(
					'per_page', array(
				'default' => 20,
				'option'  => 'user_transaction_logs_per_page',
					)
			) ;
		}

		/**
		 * Initialize the modules page.
		 */
		public static function modules_page_init() {
			global $current_module, $current_tab, $current_section ;

			//Prepare current module.
			$current_module  = ( ! empty( $_GET[ 'module' ] ) ) ? sanitize_title( wp_unslash( $_GET[ 'module' ] ) ) : '' ;
			$current_tab     = ( ! empty( $_GET[ 'tab' ] ) ) ? sanitize_title( wp_unslash( $_GET[ 'tab' ] ) ) : '' ;
			$current_section = ( ! empty( $_GET[ 'section' ] ) ) ? sanitize_title( wp_unslash( $_GET[ 'section' ] ) ) : '' ;
						
			if ( ! empty( $current_module ) ) {
				$module      = WAL_Module_Instances::get_module_by_id( $current_module ) ;
				$tabs        = is_object( $module ) ? $module->setting_tabs() : array() ;
				$current_tab = ( array_key_exists( $current_tab, $tabs ) ) ? $current_tab : key( $tabs ) ;
			}

			/**
			 * This hook is used to do extra action after settings loaded.
			 * 
			 * @hooked WAL_Modules->save_settings - 10 (save the settings).
			 * @hooked WAL_Modules->reset_settings - 20 (reset the settings).
			 * @since 1.0
			 */
			do_action( sanitize_key( self::$plugin_slug . '_module_settings_loaded_' . $current_module ) ) ;
						

			add_action( 'wal_settings_content', array( 'WAL_Settings', 'show_messages' ) ) ;
			add_action( 'woocommerce_admin_field_wal_custom_fields', array( 'WAL_Settings', 'output_fields' ) ) ;
		}

		/**
		 * Output the modules page.
		 */
		public static function modules_page() {
			// Html for modules page.
			include_once WAL_PLUGIN_PATH . '/inc/admin/menu/views/html-module-page.php'  ;
		}

		/**
		 * Output the user transaction logs page.
		 */
		public static function transaction_logs_page() {
			global $wal_wallet ;

			$wallet_id = isset( $_REQUEST[ 'wal_wallet_id' ] ) ? wc_clean( wp_unslash( $_REQUEST[ 'wal_wallet_id' ] ) ) : '' ;
			if ( ! $wallet_id ) {
				return ;
			}

			$wal_wallet = wal_get_wallet( $wallet_id ) ;
			if ( ! $wal_wallet->exists() ) {
				return ;
			}

			if ( ! class_exists( 'WAL_Transaction_Logs_List_Table' ) ) {
				require_once WAL_PLUGIN_PATH . '/inc/admin/menu/wp-list-tables/class-wal-transaction-logs-list-table.php'  ;
			}

			// Html for user transaction logs page.
			include_once WAL_PLUGIN_PATH . '/inc/admin/menu/views/html-transaction-logs-page.php'  ;
		}

		/**
		 * Output the funds page.
		 */
		public static function funds_page() {
			// Html for funds page.
			include_once WAL_PLUGIN_PATH . '/inc/admin/menu/views/html-funds-page.php'  ;
		}

		/**
		 * Validate screen options on update.
		 *
		 * @param bool|int $status Screen option value. Default false to skip.
		 * @param string   $option The option name.
		 * @param int      $value  The number of rows to use.
		 */
		public static function set_screen_option( $status, $option, $value ) {
			if ( 'user_transaction_logs_per_page' === $option ) {
				return $value ;
			}

			return $status ;
		}
	}

	WAL_Menu_Management::init() ;
}
