<?php

/**
 * Admin User Handler.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Admin_User_Handler' ) ) {

	/**
	 * Class.
	 */
	class WAL_Admin_User_Handler {

		/**
		 * Class initialization.
		 */
		public static function init() {
			// Render the wallet details.
			add_action( 'show_user_profile', array( __CLASS__, 'render_wallet_data' ) );
			add_action( 'edit_user_profile', array( __CLASS__, 'render_wallet_data' ) );

			// Render wallet details in user table.
			add_filter( 'manage_users_columns', array( __CLASS__, 'define_columns' ) );
			add_filter( 'manage_users_custom_column', array( __CLASS__, 'render_columns' ), 10, 3 );

			// Delete the wallet data when user deleted.
			add_action( 'deleted_user', array( __CLASS__, 'delete_wallet_data' ), 10, 2 );
			// Render Admin Notice
			add_action( 'admin_notices', array( __CLASS__, 'wal_reset_alert_message' ) );
		}

		/**
		 * Render the wallet column title.
		 *
		 * @return array
		 */
		public static function define_columns( $columns ) {
			$columns['wallet_details'] = __( 'Wallet Balance', 'wallet-for-woocommerce' );

			return $columns;
		}

		/**
		 * Render the wallet column details.
		 *
		 * @return mixed
		 */
		public static function render_columns( $value, $column_name, $user_id ) {
			switch ( $column_name ) {
				case 'wallet_details':
					$balance   = 0;
					$wallet_id = wal_get_wallet_id_by_user_id( $user_id );
					$wallet    = wal_get_wallet( $wallet_id );

					if ( $wallet->exists() ) {
						$balance = $wallet->get_balance();
					}

					return wal_price( $balance );

					break;
			}

			return $value;
		}

		/**
		 * Render the wallet details.
		 *
		 * @return void
		 */
		public static function render_wallet_data( $user ) {
			if ( ! is_object( $user ) ) {
				return '';
			}

			$args = array();

			$default_args = array(
				'balance' => 0,
				'status'  => 'wal_active',
			);

			$wallet_id = wal_get_wallet_id_by_user_id( $user->ID );
			$wallet    = wal_get_wallet( $wallet_id );

			if ( $wallet->exists() ) {
				$args = array(
					'balance'  => $wallet->get_balance(),
					'status'   => $wallet->get_status(),
					'currency' => $wallet->get_currency(),
				);
			}

			$wallet_data = wp_parse_args( $args, $default_args );

			include WAL_PLUGIN_PATH . '/inc/admin/menu/views/html-edit-user.php';
		}

		/**
		 * Delete the wallet data when user deleted.
		 * */
		public static function delete_wallet_data( $user_id, $reassign ) {
			// Delete the transaction logs.
			$transaction_log_ids = wal_user_transaction_logs( $user_id );
			if ( wal_check_is_array( $transaction_log_ids ) ) {
				foreach ( $transaction_log_ids as $transaction_log_id ) {
					wp_delete_post( $transaction_log_id, true );
				}
			}

			// Delete the wallet.
			$wallet_id = wal_get_wallet_id_by_user_id( $user_id );
			if ( $wallet_id ) {
				wp_delete_post( $wallet_id, true );
			}
		}

		/**
		 * Reset alert message
		 *
		 * @since 3.5.0
		 */
		public static function wal_reset_alert_message() {
			if ( 'yes' === get_option( 'wal_processing_user_wallet_balance_reset' ) || 'yes' === get_option( 'wal_processing_user_wallet_transaction_log_reset' ) ) {
				$scheduler_url = add_query_arg(
					array(
						'page' => 'wc-status',
						'tab'  => 'action-scheduler',
					),
					admin_url( 'admin.php' )
				);
				echo '<div class="error">';
				/* translators: %s : Scheduled Action URL */
				echo '<p>' . wp_kses_post( __( sprintf( '<b>Note:</b> Wallet Data Reset is under process. You can switch to other tabs in the meantime. Also, you can check the processing ' . "<a href='%s' target='_blank'>status</a>" . ' by searching these hooks[wal_reset_user_balance, wal_reset_user_transaction_log].', esc_url( $scheduler_url ) ), 'wallet-for-woocommerce' ) ) . '</p>';
				echo '</div>';
			}

			if ( 'yes' === get_option( 'wal_user_wallet_reseted' ) ) {
				echo '<div class="notice wal-dismiss-notice is-dismissible">';
				echo '<p>' . wp_kses_post( __( 'Wallet Data Reset Successfully Completed.', 'wallet-for-woocommerce' ) ) . '</p>';
				echo '</div>';
			} else if ( 'no_data' === get_option( 'wal_user_wallet_reseted' ) ) {
				echo '<div class="notice wal-dismiss-notice is-dismissible">';
				echo '<p>' . wp_kses_post( __( 'No Data found to reset.', 'wallet-for-woocommerce' ) ) . '</p>';
				echo '</div>';
			}
		}
	}

	WAL_Admin_User_Handler::init();
}
