<?php

/**
 * Admin Post Type Handler.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Admin_Post_Type_Handler')) {

	/**
	 * Class.
	 */
	class WAL_Admin_Post_Type_Handler {

		/**
		 * Class initialization.
		 */
		public static function init() {
			// Load correct list table classes for current screen.
			add_action('current_screen', array( __CLASS__, 'setup_screen' ), 1);
			add_action('check_ajax_referer', array( __CLASS__, 'setup_screen' ), 1);

			// Export the CSV.
			add_action('admin_init', array( __CLASS__, 'export_csv' ));
			// May be add redeem wallet balance button
			add_action('woocommerce_order_item_add_action_buttons', array( __CLASS__, 'maybe_add_redeem_wallet_balance_button' ), 10, 1);
			
			add_filter( 'woocommerce_reports_get_order_report_query', array( __CLASS__, 'remove_topup_order_query' ) );
			// add_filter( 'woocommerce_analytics_revenue_query_args', array( __CLASS__, 'remove_topup_order_analytics' ) );
			// add_filter( 'woocommerce_analytics_orders_stats_query_args', array( __CLASS__, 'remove_topup_order_analytics' ) );
		}

		/**
		 * Looks at the current screen and loads the correct list table handler.
		 *
		 * @return void
		 */
		public static function setup_screen() {
			global $wal_list_table;

			$screen_id = false;

			if (!empty($_REQUEST['screen'])) {
				$screen_id = wc_clean(wp_unslash($_REQUEST['screen']));
			} elseif (function_exists('get_current_screen')) {
				$screen = get_current_screen();
				$screen_id = isset($screen, $screen->id) ? $screen->id : '';
			}

			$screen_id = str_replace('edit-', '', $screen_id);

			switch ($screen_id) {
				case 'wal_wallet':
					include_once WAL_PLUGIN_PATH . '/inc/admin/menu/post-tables/class-wal-wallet-list-table.php';

					$wal_list_table = new WAL_Wallet_List_Table();
					break;

				case 'wal_transaction_log':
					include_once WAL_PLUGIN_PATH . '/inc/admin/menu/post-tables/class-wal-transaction-log-list-table.php';

					$wal_list_table = new WAL_Transaction_Log_List_Table();
					break;

				case 'wal_wallet_withdraw':
					include_once WAL_PLUGIN_PATH . '/inc/admin/menu/post-tables/class-wal-wallet-withdrawal-list-table.php';

					$wal_list_table = new WAL_Wallet_Withdrawal_List_Table();
					break;
			}

			// Ensure the table handler is only loaded once. Prevents multiple loads if a plugin calls check_ajax_referer many times.
			remove_action('current_screen', array( __CLASS__, 'setup_screen' ));
			remove_action('check_ajax_referer', array( __CLASS__, 'setup_screen' ));
		}

		/**
		 * Export the CSV.
		 * */
		public static function export_csv() {

			if (!isset($_REQUEST['wal_export_csv'])) {
				return;
			}

			$action = wc_clean(wp_unslash($_REQUEST['wal_export_csv']));

			switch ($action) {
				case 'transaction_logs':
					include_once WAL_ABSPATH . 'inc/exports/class-wal-transaction-logs-export-csv.php';

					$exporter = new WAL_Transaction_Logs_Export_CSV();

					// Set wallet if exists.
					$wallet_id = isset($_REQUEST['wallet_id']) ? wc_clean(wp_unslash($_REQUEST['wallet_id'])) : '';
					if ($wallet_id) {
						$wallet = wal_get_wallet($wallet_id);
						if ($wallet->exists()) {
							$exporter->set_wallet($wallet);
						}
					}

					$exporter->export();
					break;

				case 'wallet':
					include_once WAL_ABSPATH . 'inc/exports/class-wal-wallet-export-csv.php';

					$exporter = new WAL_Wallet_Export_CSV();
					$exporter->export();
					break;
			}
		}

		/**
		 * Maybe add redeem wallet balance button
		 *
		 * @since 3.4.0
		 * @param object $order
		 * 
		 * @return void
		 */
		public static function maybe_add_redeem_wallet_balance_button( $order ) {
			// Return if not object
			if (!is_object($order)) {
				return;
			}
			
			// Return if not editable
			if ( !$order->is_editable() ) {
				return;
			}
		  
			include_once WAL_PLUGIN_PATH . '/inc/admin/menu/views/html-redeem-wallet-balance-button.php';
		}

		/**
		 * Exclude Wallet Topup orders from admin report
		 *
		 * @since 4.6.0
		 * @return array
		 */
		public static function remove_topup_order_query( $query ) {
			$order_ids = wc_get_orders(
				array(
					'limit'      => -1,
					'meta_query' => array(
						'relation' => 'OR',
						array(
							'key'   => 'wal_topup_mode',
							'value' => 'manual',
						),
						array(
							'key'   => 'wal_topup_mode',
							'value' => 'auto',
						),
					),
					'return'     => 'ids',
				)
			);

			if ( wal_check_is_array( $order_ids ) ) {
				$exclude_orders  = implode( ', ', $order_ids );
				$query['where'] .= " AND posts.ID NOT IN ({$exclude_orders})";
			}

			return $query;
		}
		/**
		 * Exclude Topup orders from admin analytics
		 *
		 * @param array $query_vars query_vars.
		 * @return array
		 */
		public function remove_topup_order_analytics( $query_vars ) {
			$topup_product = array_filter( ( array ) get_option( 'wal_general_topup_product_id', array() ), 'wal_array_filter' ) ;
			if ( get_wallet_rechargeable_product() ) {
				$query_vars['product_excludes'][] = $topup_product;
			}

			return $query_vars;
		}
	}

	WAL_Admin_Post_Type_Handler::init();
}
