<?php

/**
 * Admin Assets.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}
if (!class_exists('WAL_Admin_Assets')) {

	/**
	 * Class.
	 */
	class WAL_Admin_Assets {

		/**
		 * Suffix.
		 * 
		 * @var string
		 */
		private static $suffix;

		/**
		 * Scripts.
		 *
		 * @since 4.1.0
		 * @var array
		 */
		private static $scripts = array();

		/**
		 * Styles.
		 *
		 * @since 4.1.0
		 * @var array
		 */
		private static $styles = array();

		/**
		 * Localized scripts.
		 *
		 * @since 4.1.0
		 * @var array
		 */
		private static $wp_localized_scripts = array();

		/**
		 * Class Initialization.
		 */
		public static function init() {
			self::$suffix = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';

			add_action('admin_enqueue_scripts', array( __CLASS__, 'load_scripts' ));
			add_action('wp_print_scripts', array( __CLASS__, 'localize_printed_scripts' ), 5);
			add_action('wp_print_footer_scripts', array( __CLASS__, 'localize_printed_scripts' ), 5);
		}

		/**
		 * Register and enqueue frontend scripts.
		 * 
		 * @since 4.1.0
		 */
		public static function load_scripts() {
			
			self::register_scripts();
			self::register_styles();
			self::enqueue_registered_scripts();
			self::enqueue_registered_styles();
		}

		/**
		 * Localize scripts only when enqueued.
		 * 
		 * @since 4.1.0
		 */
		public static function localize_printed_scripts() {
			foreach (self::$scripts as $handle) {
				self::localize_script($handle);
			}
		}

		/**
		 * Register all scripts.
		 * 
		 * @since 4.1.0
		 */
		private static function register_scripts() {
			$default_scripts = self::get_default_scripts();
			// Returns if there is no scripts to register.
			if (!wal_check_is_array($default_scripts)) {
				return;
			}
			
			foreach ($default_scripts as $handle => $script) {
				if (!isset($script['src'])) {
					continue;
				}

				if (!$script['can_register']) {
					continue;
				}

				$deps = isset($script['deps']) ? array_merge(array( 'jquery' ), $script['deps']) : array( 'jquery' );
				$version = isset($script['version']) ? $script['version'] : WAL_VERSION;
				if (!wp_register_script($handle, $script['src'], $deps, $version)) {
					continue;
				}

				self::$scripts[] = $handle;
			}
		}

		/**
		 * Register all styles.
		 * 
		 * @since 4.1.0
		 */
		private static function register_styles() {
			$default_styles = self::get_default_styles();
			// Returns if there is no styles to register.
			if (!wal_check_is_array($default_styles)) {
				return;
			}

			foreach ($default_styles as $handle => $style) {
				if (!isset($style['src'])) {
					continue;
				}

				if (!$style['can_register']) {
					continue;
				}

				$deps = isset($style['deps']) ? $style['deps'] : array();
				$version = isset($style['version']) ? $style['version'] : WAL_VERSION;
				$media = isset($style['media']) ? $style['media'] : 'all';
				$has_rtl = isset($style['has_rtl']) ? $style['has_rtl'] : false;
				if (!wp_register_style($handle, $style['src'], $deps, $version, $media)) {
					continue;
				}

				self::$styles[] = $handle;

				if ($has_rtl) {
					wp_style_add_data($handle, 'rtl', 'replace');
				}
			}
		}

		/**
		 * Get the default scripts to register.
		 * 
		 * @since 4.1.0
		 * @return array
		 */
		private static function get_default_scripts() {
			$screen_ids = wal_page_screen_ids();
			$newscreenids = get_current_screen();
			$screenid = str_replace('edit-', '', $newscreenids->id);
			
			if ( !in_array( $screenid , $screen_ids ) ) {
				return array();
			}
				$register_scripts = array(
						'lightcase' => array(
							'src' => self::get_asset_url('assets/lib/lightcase/lightcase' . self::$suffix . '.js'),
							'can_register' => true,
						),
						'wal-lightcase' => array(
							'src' => self::get_asset_url('assets/js/wal-lightcase-enhanced.js'),
							'deps' => array( 'lightcase' ),
							'can_register' => true,
						),
						'wal-admin' => array(
							'src' => self::get_asset_url('assets/js/admin/admin.js'),
							'deps' => array( 'jquery-blockui', 'wc-admin-meta-boxes' ),
							'can_register' => true,
						),
						'wal-enhanced' => array(
							'src' => self::get_asset_url('assets/js/wal-enhanced.js'),
							'deps' => array( 'select2' ),
							'can_register' => true,
						),
					); 

				if ('wal_wallet' == $screenid) {
					$register_scripts[] = array(
						'jquery-ui-timepicker-addon' => array(
							'src' => self::get_asset_url('assets/lib/timepicker-addon/jquery-ui-timepicker-addon' . self::$suffix . '.js'),
							'deps' => array( 'jquery-ui-datepicker' ),
							'can_register' => true,
						),
					);
				}
			
				if ( 'shop_order' === $screenid ) {
					$register_scripts = array(
						'wal-refund' => array(
							'src' => self::get_asset_url('assets/js/admin/refund.js'),
							'deps' => array( 'jquery-blockui' ),
							'can_register' => true,
						),
					);
				}
				/**
				 * This hook is used to alter the admin default register scripts.
				 * 
				 * @since 4.1.0
				 */
				return apply_filters('wal_admin_default_register_scripts', $register_scripts);
		}

		/**
		 * Get the default styles to register.
		 * 
		 * @since 4.1.0
		 * @return array
		 */
		private static function get_default_styles() {
			$screen_ids = wal_page_screen_ids();
			$newscreenids = get_current_screen();
			$screenid = str_replace('edit-', '', $newscreenids->id);

			$screen_ids[] = 'user-edit';
			$screen_ids[] = 'profile';

			if ( !in_array( $screenid , $screen_ids ) ) {
				return array();
			}

			$register_scripts = array(
				'lightcase' => array(
					'src' => self::get_asset_url('assets/lib/lightcase/lightcase' . self::$suffix . '.css'),
					'can_register' => true,
				),
				'wal-admin' => array(
					'src' => self::get_asset_url('assets/css/admin.css'),
					'deps' => array( 'wc-admin-layout' ),
					'can_register' => true,
				),
			);

			if ('wal_wallet' == $screenid) {
				$register_scripts[] = array(
					'jquery-ui-datepicker-addon' => array(
						'src' => self::get_asset_url('assets/lib/timepicker-addon/jquery-ui-timepicker-addon'),
						'can_register' => true,
					),
				);
			}

			/**
			 * This hook is used to alter the admin default register styles.
			 * 
			 * @since 4.1.0
			 */
			return apply_filters('wal_admin_default_register_styles', $register_scripts);
		}

		/**
		 * Get script data.
		 * 
		 * @since 4.1.0
		 * @param string $handle
		 * @return array/false
		 */
		public static function get_script_data( $handle ) {
			switch ($handle) {
				case 'wal-admin':
					$params = array(
						'topup_nonce' => wp_create_nonce('wal-topup-nonce'),
						'module_nonce' => wp_create_nonce('wal-module-nonce'),
						'reset_user_wallet_nonce' => wp_create_nonce('wal-reset-user-wallet-data'),
						'remove_notice_nonce' => wp_create_nonce('wal-reset-notice-nonce'),
						'topup_product_name_empty_msg' => __('Please Enter the Product Name', 'wallet-for-woocommerce'),
						'confirm_product_creation_msg' => __('Are you sure you want to create a product?', 'wallet-for-woocommerce'),
						'confirm_manual_fund_msg' => __('Are you sure you want to proceed?', 'wallet-for-woocommerce'),
						'wallet_reset_confirm_msg' => __('Are you sure you want to proceed?', 'wallet-for-woocommerce'),
						'delete_confirm_msg' => __('Are you sure you want to delete?', 'wallet-for-woocommerce'),
						'redeem_wallet_funds_nonce' => wp_create_nonce('wal-redeem-wallet-funds'),
						'gateway_funds_nonce' => wp_create_nonce('wal-gateway-funds'),
					);
					break;

				case 'wal-refund':
					$gateways = WC()->payment_gateways()->payment_gateways();
					$params = array(
						/* translators: %1$s- amount, %2$s - gateway name */
						'refund_button' => '<button type="button" class="button button-primary do-api-wallet-refund">' . sprintf(__('Refund %1$s via %2$s', 'woocommerce'), '<span class="wc-order-refund-amount">' . wc_price(0) . '</span>', esc_html($gateways['wal_wallet']->get_title())) . '</button>',
						'confirm_refund_msg' => __('Are you sure you want to proceed?', 'wallet-for-woocommerce'),
						);
					break;

				case 'wal-enhanced':
					$params = array(
						'i18n_no_matches' => __('No matches found', 'wallet-for-woocommerce'),
						'i18n_input_too_short_1' => __('Please enter 1 or more characters', 'wallet-for-woocommerce'),
						'i18n_input_too_short_n' => __('Please enter %qty% or more characters', 'wallet-for-woocommerce'),
						'i18n_input_too_long_1' => __('Please delete 1 character', 'wallet-for-woocommerce'),
						'i18n_input_too_long_n' => __('Please delete %qty% characters', 'wallet-for-woocommerce'),
						'i18n_selection_too_long_1' => __('You can only select 1 item', 'wallet-for-woocommerce'),
						'i18n_selection_too_long_n' => __('You can only select %qty% items', 'wallet-for-woocommerce'),
						'i18n_load_more' => __('Loading more results&hellip;', 'wallet-for-woocommerce'),
						'i18n_searching' => __('Searching&hellip;', 'wallet-for-woocommerce'),
						'search_nonce' => wp_create_nonce('wal-search-nonce'),
						'ajaxurl' => WAL_ADMIN_AJAX_URL,
						'calendar_image' => WC()->plugin_url() . '/assets/images/calendar.png',
					);
					break;

				default:
					$params = false;
					break;
			}

			return $params;
		}

		/**
		 * Enqueue all registered scripts.
		 * 
		 * @since 4.1.0
		 */
		private static function enqueue_registered_scripts() {
			foreach (self::$scripts as $handle) {
				self::enqueue_script($handle);
			}
		}

		/**
		 * Enqueue script.
		 * 
		 * @param string $handle
		 * @since 4.1.0
		 */
		private static function enqueue_script( $handle ) {
			if (!wp_script_is($handle, 'registered')) {
				return;
			}

			if ( 'wal-refund' === $handle ) {
				global $post;
				if (!$post->ID) {
					return;
				}
	
				$order = wc_get_order($post->ID);
				if (!is_object($order)) {
					return;
				}
	
				// Return if the order payment method is a wallet.
				if ('wal_wallet' === $order->get_payment_method()) {
					return;
				}
	
				// Return if the order is topup.
				if ('yes' === $order->get_meta('wal_topup_fund_credited')) {
					return;
				}
				// Refund.   
				wp_enqueue_script('wal-refund', WAL_PLUGIN_URL . '/assets/js/admin/refund.js', array( 'jquery', 'jquery-blockui' ), WAL_VERSION);
			}
			
			wp_enqueue_script($handle);
		}

		/**
		 * Enqueue all registered styles.
		 * 
		 * @since 4.1.0
		 */
		private static function enqueue_registered_styles() {
			foreach (self::$styles as $handle) {
				self::enqueue_style($handle);
			}
		}

		/**
		 * Enqueue style.
		 * 
		 * @param string $handle
		 * @since 4.1.0
		 */
		private static function enqueue_style( $handle ) {
			if (!wp_style_is($handle, 'registered')) {
				return;
			}
			
			wp_enqueue_style($handle);
		}

		/**
		 * Localize the enqueued script.
		 * 
		 * @since 4.1.0
		 * @param string $handle
		 * @return null
		 */
		private static function localize_script( $handle ) {
			// Return if already localized script or not enqueued script.
			if (in_array($handle, self::$wp_localized_scripts, true) || !wp_script_is($handle)) {
				return;
			}

			// Get the data for current script.
			$data = self::get_script_data($handle);
			if (!$data) {
				return;
			}

			$name = str_replace('-', '_', $handle) . '_params';

			/**
			 * This hook is used to alter the script data.
			 * 
			 * @since 4.1.0
			 */
			if (wp_localize_script($handle, $name, apply_filters($name, $data))) {
				self::$wp_localized_scripts[] = $handle;
			}
		}

		/**
		 * Get asset URL.
		 *
		 * @since 4.1.0
		 * @param string $path Assets path.
		 * @return string
		 */
		private static function get_asset_url( $path ) {
			/**
			 * This hook is used to alter the asset URL.
			 * 
			 * @since 3.9.0
			 */
			return apply_filters('wal_get_asset_url', WAL_PLUGIN_URL . '/' . $path, $path);
		}
	}

	WAL_Admin_Assets::init();
}
