<?php

/**
 * Admin Ajax.
 *
 * @since 1.0.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}
if (!class_exists('WAL_Admin_Ajax')) {

	/**
	 * Class.
	 *
	 * @since 1.0.0
	 */
	class WAL_Admin_Ajax {

		/**
		 *  Class initialization.
		 *
		 *  @since 1.0.0
		 */
		public static function init() {

			$actions = array(
				'json_search_products_and_variations' => false,
				'json_search_products' => false,
				'json_search_customers' => false,
				'enable_module' => false,
				'manual_credit_debit_fund' => false,
				'manual_credit_debit_fund_for_user_wallet' => false,
				'apply_partial_fund' => false,
				'remove_partial_fund' => false,
				'transaction_logs_pagination' => false,
				'get_wallet_fund_redeem_popup_content' => false,
				'redeem_wallet_balance_manually' => false,
				'validate_create_order_by_wallet_gateway' => false,
				'pagination_action' => false,
				'reset_user_wallet_data' => false,
				'remove_user_wallet_reseted_success_notice' => false,
				'get_valid_shipping_methods_html' => false,
			);

			foreach ($actions as $action => $nopriv) {
				add_action('wp_ajax_wal_' . $action, array( __CLASS__, $action ));

				if ($nopriv) {
					add_action('wp_ajax_nopriv_wal_' . $action, array( __CLASS__, $action ));
				}
			}
		}

		/**
		 * Search for products.
		 *
		 * @since 1.0.0
		 * @param string $term
		 * @param bool   $include_variations
		 * @return void
		 */
		public static function json_search_products( $term = '', $include_variations = false ) {
			check_ajax_referer('search-products', 'wal_security');

			try {

				if (empty($term) && isset($_GET['term'])) {
					$term = isset($_GET['term']) ? wc_clean(wp_unslash($_GET['term'])) : '';
				}

				if (empty($term)) {
					throw new exception(__('No Products found', 'wallet-for-woocommerce'));
				}

				if (!empty($_GET['limit'])) {
					$limit = absint($_GET['limit']);
				} else {
					/**
					 * This hook is used to alert the WooCommerce JSON search limit.
					 *
					 * @since 1.0.0
					 */
					$limit = absint(apply_filters('woocommerce_json_search_limit', 30));
				}

				$data_store = WC_Data_Store::load('product');
				$ids = $data_store->search_products($term, '', (bool) $include_variations, false, $limit);

				$product_objects = wal_filter_readable_products($ids);
				$products = array();

				$exclude_global_variable = isset($_GET['exclude_global_variable']) ? wc_clean(wp_unslash($_GET['exclude_global_variable'])) : 'no'; // @codingStandardsIgnoreLine.
				foreach ($product_objects as $product_object) {
					if ('yes' == $exclude_global_variable && $product_object->is_type('variable')) {
						continue;
					}

					$products[$product_object->get_id()] = rawurldecode(strip_tags($product_object->get_formatted_name()));
				}
				/**
				 * This hook is used to alert the WooCommerce JSON search found products.
				 *
				 * @since 1.0.0
				 * @param array $objects
				 */
				wp_send_json(apply_filters('woocommerce_json_search_found_products', $products));
			} catch (Exception $ex) {
				wp_die();
			}
		}

		/**
		 * Search for product variations.
		 *
		 * @since 1.0.0
		 * @param string $term
		 * @param bool   $include_variations
		 *
		 * @return void
		 */
		public static function json_search_products_and_variations( $term = '', $include_variations = false ) {
			self::json_search_products('', true);
		}

		/**
		 * Customers search.
		 *
		 * @since 1.0.0
		 *
		 * @return void
		 */
		public static function json_search_customers() {
			check_ajax_referer('wal-search-nonce', 'wal_security');

			try {
				$term = isset($_GET['term']) ? wc_clean(wp_unslash($_GET['term'])) : ''; // @codingStandardsIgnoreLine.

				if (empty($term)) {
					throw new exception(__('No Customer found', 'wallet-for-woocommerce'));
				}

				if (!empty($_GET['limit'])) {
					$limit = absint($_GET['limit']);
				} else {
					/**
					 * This hook is used to alert the WooCommerce JSON search limit.
					 *
					 * @since 1.0.0
					 */
					$limit = absint(apply_filters('woocommerce_json_search_limit', 30));
				}

				$found_customers = array();
				$include = isset($_GET['include']) ? wc_clean(wp_unslash($_GET['include'])) : array();
				$include_roles = isset($_GET['include_roles']) ? wc_clean(wp_unslash($_GET['include_roles'])) : array();
				$exclude = isset($_GET['exclude']) ? wc_clean(wp_unslash($_GET['exclude'])) : array();
				$exclude_roles = isset($_GET['exclude_roles']) ? wc_clean(wp_unslash($_GET['exclude_roles'])) : array();
				$show_user_name_only = isset($_GET['show_user_name_only']) ? wc_clean(wp_unslash($_GET['show_user_name_only'])) : '';

				$customers_query = new WP_User_Query(
						array(
					'fields' => 'all',
					'orderby' => 'display_name',
					'exclude' => $exclude,
					'search' => '*' . $term . '*',
					'number' => $limit,
					'include' => $include,
					'role__in' => $include_roles,
					'role__not_in' => $exclude_roles,
					'search_columns' => array( 'ID', 'user_login', 'user_email', 'user_nicename' ),
						)
				);
				$customers = $customers_query->get_results();

				if (wal_check_is_array($customers)) {
					foreach ($customers as $customer) {
						$found_customers[$customer->ID] = $customer->display_name;
						if ('yes' != $show_user_name_only) {
							$found_customers[$customer->ID] .= ' (#' . $customer->ID . ' &ndash; ' . sanitize_email($customer->user_email) . ')';
						}
					}
				}

				wp_send_json($found_customers);
			} catch (Exception $ex) {
				wp_die();
			}
		}

		/**
		 * Enable/ Disable the module.
		 *
		 * @since 1.0.0
		 *
		 * @return void
		 * */
		public static function enable_module() {
			check_ajax_referer('wal-module-nonce', 'wal_security');

			try {
				if (!isset($_POST)) {
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				$module_id = isset($_POST['module']) ? wc_clean(wp_unslash($_POST['module'])) : '';
				$status = isset($_POST['status']) ? wc_clean(wp_unslash($_POST['status'])) : '';
				// Return if the data is empty.
				if (!$module_id || !$status) {
					throw new exception(__('Invalid request', 'wallet-for-woocommerce'));
				}

				$module = WAL_Module_Instances::get_module_by_id($module_id);
				// Return if the module does not exists.
				if (!is_object($module)) {
					throw new exception(__('Invalid request', 'wallet-for-woocommerce'));
				}

				// Return if the current user does not have permission.
				if (!current_user_can('edit_posts')) {
					throw new exception(__("You don't have permission to do this action", 'wallet-for-woocommerce'));
				}

				if ('disable' == $status) {
					$action = 'no';
					$msg = __('Module Disabled Successfully', 'wallet-for-woocommerce');
				} else {
					$action = 'yes';
					$msg = __('Module Enabled Successfully', 'wallet-for-woocommerce');
				}

				update_option($module->get_enable_key(), $action);

				wp_send_json_success(array( 'msg' => $msg ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Manually credit/debit the fund for a user to/from the wallet.
		 *
		 * @since 2.6.0
		 * @return void
		 * */
		public static function manual_credit_debit_fund_for_user_wallet() {
			check_ajax_referer('wal-topup-nonce', 'wal_security');

			try {
				$user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : '';
				if (!$user_id) {
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				$fund = isset($_POST['fund']) ? wc_clean(wp_unslash($_POST['fund'])) : '';
				// Return if the fund is empty.
				if (!$fund) {
					throw new exception(__('Please enter the Funds', 'wallet-for-woocommerce'));
				}

				// Return if the current user does not have permission to do action.
				if (!current_user_can('edit_posts')) {
					throw new exception(__("You don't have permission to do this action", 'wallet-for-woocommerce'));
				}

				$type = isset($_POST['type']) ? absint($_POST['type']) : '';
				$reason = isset($_POST['reason']) ? wc_clean(wp_unslash($_POST['reason'])) : '';

				$args = array(
					'user_id' => $user_id,
					'amount' => wc_format_decimal($fund),
					'event_message' => $reason,
					'mode' => 'manual',
				);

				switch ($type) {
					case '1':
						$args['event_id'] = 1;
						wal_credit_wallet_fund($args);
						break;

					case '2':
						$args['event_id'] = 2;
						wal_debit_wallet_fund($args);
						break;
				}

				wp_send_json_success(array( 'msg' => self::get_credit_debit_success_message($type) ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Manually credit/debit the fund for specific users.
		 *
		 * @since 1.0.0
		 *
		 * @return void
		 * */
		public static function manual_credit_debit_fund() {
			check_ajax_referer('wal-topup-nonce', 'wal_security');

			try {
				if (!isset($_POST)) {
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				$user_type = isset($_POST['user_type']) ? absint($_POST['user_type']) : '';
				if (!$user_type) {
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				$user_ids = isset($_POST['user_ids']) ? wc_clean(wp_unslash($_POST['user_ids'])) : array();
				$user_roles = isset($_POST['user_roles']) ? wc_clean(wp_unslash($_POST['user_roles'])) : array();
				$type = isset($_POST['type']) ? absint($_POST['type']) : '';
				$fund = isset($_POST['fund']) ? wc_clean(wp_unslash($_POST['fund'])) : '';
				$reason = isset($_POST['reason']) ? wc_clean(wp_unslash($_POST['reason'])) : '';

				switch ($user_type) {
					case '2':
					case '3':
						// Return if the user ids is empty.
						if (!wal_check_is_array($user_ids)) {
							throw new exception(__('Please select a User', 'wallet-for-woocommerce'));
						}
						break;

					case '4':
					case '5':
						// Return if the user ids is empty.
						if (!wal_check_is_array($user_roles)) {
							throw new exception(__('Please select a User roles', 'wallet-for-woocommerce'));
						}
						break;

					default:
						break;
				}

				// Return if the fund is empty.
				if (!$fund) {
					throw new exception(__('Please enter the Funds', 'wallet-for-woocommerce'));
				}

				// Return if the current user does not have permission.
				if (!current_user_can('edit_posts')) {
					throw new exception(__("You don't have permission to do this action", 'wallet-for-woocommerce'));
				}

				$valid_user_ids = self::get_valid_user_ids($user_type, $user_ids, $user_roles);
				if (!wal_check_is_array($valid_user_ids)) {
					throw new exception(__('There is no valid user to credit/debit.', 'wallet-for-woocommerce'));
				}

				$action_count = 0;
				foreach ($valid_user_ids as $user_id) {
					$args = array(
						'user_id' => $user_id,
						'amount' => wc_format_decimal($fund),
						'event_message' => $reason,
						'mode' => 'manual',
					);

					switch ($type) {
						case '1':
							$args['event_id'] = 1;
							if (wal_credit_wallet_fund($args)) {
								++$action_count;
							}

							break;

						case '2':
							$args['event_id'] = 2;
							if (wal_debit_wallet_fund($args)) {
								++$action_count;
							}

							break;
					}
				}

				$user_count = count($valid_user_ids);

				wp_send_json_success(array( 'msg' => self::get_credit_debit_success_message($type, $user_count, $action_count) ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Get the credit/debit success message.
		 *
		 * @since 2.4.0
		 * @param string $type
		 * @param int    $action_count
		 * @param int    $user_count
		 * @return string
		 */
		private static function get_credit_debit_success_message( $type, $action_count = 1, $user_count = 1 ) {
			$msg = '';
			switch ($type) {
				case '2':
					if (1 != $user_count && !$action_count) {
						/* translators: %s - user count */
						$msg = sprintf(__('Unable to debit the funds from %s users wallet due to insufficient balance.', 'wallet-for-woocommerce'), $user_count);
					} elseif (!$action_count) {
						$msg = __("Unable to debit the funds from the selected user's wallet due to insufficient balance.", 'wallet-for-woocommerce');
					} elseif (1 == $user_count) {
						$msg = __("Successfully debited the funds from the selected user's wallet.", 'wallet-for-woocommerce');
					} elseif ($user_count != $action_count) {
						/* translators: %1$s - debit count, %2$s - non debit count */
						$msg = sprintf(__(' Successfully debited the funds from %1$s users wallet & Unable to debit the funds from %2$s users wallet due to insufficient balance.', 'wallet-for-woocommerce'), $action_count, $user_count - $action_count);
					} else {
						/* translators: %s - debit count */
						$msg = sprintf(__('Successfully debited the funds from %s users wallet.', 'wallet-for-woocommerce'), $action_count);
					}
					break;

				case '1':
					if (1 == $user_count) {
						$msg = __("Successfully credited the funds to the selected user's wallet.", 'wallet-for-woocommerce');
					} else {
						/* translators: %s - credit count */
						$msg = sprintf(__('Successfully credited the funds to %s users wallet.', 'wallet-for-woocommerce'), $action_count);
					}
					break;
			}

			return $msg;
		}

		/**
		 * Get the valid user IDs for the credit or debit.
		 *
		 * @since 2.4.0
		 * @param string $user_type
		 * @param array  $user_ids
		 * @param array  $user_roles
		 * @return array
		 */
		private static function get_valid_user_ids( $user_type, $user_ids, $user_roles ) {
			$args = array( 'fields' => 'ids' );

			switch ($user_type) {
				case '2':
					$args['include'] = $user_ids;
					break;

				case '3':
					$args['exclude'] = $user_ids;
					break;

				case '4':
					$args['role__in'] = $user_roles;
					break;

				case '5':
					$args['role__not_in'] = $user_roles;
					break;
			}

			return get_users($args);
		}

		/**
		 * Apply the partial fund in the cart.
		 *
		 * @since 1.0.0
		 *
		 * @return void
		 * */
		public static function apply_partial_fund() {
			check_ajax_referer('wal-partial-fund-usage-nonce', 'wal_security');

			try {
				if (!isset($_POST)) {
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				$fund = isset($_POST['fund']) ? wc_clean(wp_unslash($_POST['fund'])) : '';

				// Return if the fund is empty.
				if (!$fund) {
					throw new exception(get_option('wal_messages_partial_fund_empty'));
				}

				// Return if the fund is not numeric.
				if (!is_numeric($fund)) {
					throw new exception(get_option('wal_messages_partial_fund_numeric'));
				}

				// Return if the current wallet not valid fund usage.
				if (!WAL_Current_User_Wallet::is_valid_fund_usage()) {
					throw new exception(get_option('wal_messages_partial_fund_user_restricted'));
				}

				$fund = floatval(wal_convert_price($fund, true));

				// Return if the fund is greater than wallet balance.
				if (WAL_Current_User_Wallet::get_balance() < $fund) {
					/* translators: %s wallet balance */
					throw new exception(str_replace('{wallet_balance}', wal_convert_price_by_currency(WAL_Current_User_Wallet::get_balance()), get_option('wal_messages_partial_wallet_insufficient_fund')));
				}

				$fund_to_apply = WAL_Current_User_Wallet::get_fund_to_apply($fund);

				if ( empty($fund_to_apply) ) {
					throw new exception( esc_html__( get_option('wal_messages_product_filter_error_for_usage') ) );
				}

				$maximum_amount = wal_get_maximum_fund_usage_limit();
				// Return if the fund is greater than maximum amount.
				if (false !== $maximum_amount && $maximum_amount < $fund_to_apply) {
					/* translators: %s maximum amount */
					throw new exception(str_replace('{restricted_funds}', wal_convert_price_by_currency($maximum_amount), get_option('wal_messages_partial_allow_maximum_fund')));
				}

				// Return if the wallet balance is less than order total.
				if ('2' != get_option('wal_general_partial_payments_mode') && WAL_Current_User_Wallet::get_balance() >= wal_get_wc_cart_total()) {
					throw new exception(__('Sorry, You are not valid to use fund.', 'wallet-for-woocommerce'));
				}

				// Set the partial fund in the WC session.
				WC()->session->set('wal_partial_fund', $fund_to_apply);

				wal_add_wc_notice(get_option('wal_messages_partial_fund_added'));

				wp_send_json_success();
			} catch (Exception $ex) {
				wal_add_wc_notice($ex->getMessage(), 'error');
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Remove the partial fund from the cart.
		 *
		 * @since 1.0.0
		 *
		 * @return void
		 * */
		public static function remove_partial_fund() {
			check_ajax_referer('wal-partial-fund-usage-nonce', 'wal_security');

			try {
				if (!isset($_POST)) {
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				if ('yes' == get_option('wal_general_auto_deduct_wallet_for_partial_payments')) {
					// Remove the auto deduct partial fund from the WC session.
					WC()->session->set('wal_auto_deduct_partial_fund', 'no');
				}

				// Return if the partial fund is not applied inf the session.
				if (!WC()->session->get('wal_partial_fund')) {
					throw new exception(__('You have already removed the applied funds.', 'wallet-for-woocommerce'));
				}

				// Remove the partial fund from the WC session.
				WC()->session->set('wal_partial_fund', null);

				wal_add_wc_notice(get_option('wal_messages_partial_fund_removed'));

				wp_send_json_success();
			} catch (Exception $ex) {
				wal_add_wc_notice($ex->getMessage(), 'error');
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Transaction logs for a selected page.
		 *
		 * @since 1.0.0
		 *
		 * @return void
		 */
		public static function transaction_logs_pagination() {
			check_ajax_referer('wal-pagination-nonce', 'wal_security');

			try {

				if (!isset($_POST) || !isset($_POST['page_number'])) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				// Sanitize post values
				$current_page = !empty($_POST['page_number']) ? absint($_POST['page_number']) : 0; // @codingStandardsIgnoreLine.

				$per_page = wal_transaction_logs_pagination_count();
				$offset = ( $current_page - 1 ) * $per_page;

				// Get the transaction logs based on per page count
				$transaction_log_ids = wal_user_transaction_logs();
				$transaction_log_ids = array_slice($transaction_log_ids, $offset, $per_page);

				// Get the transaction logs content.
				$html = wal_get_template_html(
						'dashboard/activity.php',
						array(
							'transaction_log_ids' => $transaction_log_ids,
							'columns' => wal_transaction_log_columns(),
						)
				);

				wp_send_json_success(array( 'html' => $html ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Get wallet fund redeem popup content
		 *
		 * @since 3.3.0
		 * @return void
		 */
		public static function get_wallet_fund_redeem_popup_content() {
			check_ajax_referer('wal-redeem-wallet-funds', 'wal_security');

			try {
				if (!isset($_POST)) {
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				$order_id = !empty($_POST['order_id']) ? absint($_POST['order_id']) : 0;
				if (!$order_id) {
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				$order = wc_get_order($order_id);
				if (!is_object($order)) {
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				// Return if order items empty.
				if (!wal_check_is_array($order->get_items())) {
					throw new exception(__('No product(s) added. Please add an item(s) to place the order.', 'wallet-for-woocommerce'));
				}

				$user_id = isset($_POST['user_id']) ? absint($_POST['user_id']) : '';
				if (!$user_id) {
					throw new exception(__('Please Select a Customer', 'wallet-for-woocommerce'));
				}

				$wallet_id = wal_get_wallet_id_by_user_id($user_id);
				$wallet = wal_get_wallet($wallet_id);
				if (!$wallet->exists()) {
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				$payment_method = isset($_POST['payment_method']) ? wc_clean(wp_unslash($_POST['payment_method'])) : '';
				if ('wal_wallet' === $payment_method) {
					throw new exception(__('Since you have selected wallet gateway, you cannot apply wallet funds in this order.', 'wallet-for-woocommerce'));
				}

				ob_start();
				include_once WAL_PLUGIN_PATH . '/inc/admin/menu/views/html-redeem-wallet-funds-popup.php';
				$html = ob_get_contents();
				ob_end_clean();

				wp_send_json_success(array( 'html' => $html ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Redeem wallet balance manually
		 *
		 * @since 3.4.0
		 */
		public static function redeem_wallet_balance_manually() {
			check_ajax_referer('wal-redeem-wallet-funds', 'wal_security');

			try {
				if (!isset($_POST)) {
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				$fund = isset($_POST['fund']) ? floatval($_POST['fund']) : '';
				$order_id = isset($_POST['order_id']) ? absint($_POST['order_id']) : '';
				$user_id = isset($_POST['user_id']) ? absint($_POST['user_id']) : '';

				$order = wc_get_order($order_id);
				if (!is_object($order) || !$fund || !is_numeric($fund)) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				$fund = wal_convert_price($fund, true);
				// Return if the wallet balance is less than order total.
				if (empty($fund)) {
					throw new exception(__('Sorry, You are not valid to use fund.', 'wallet-for-woocommerce'));
				}

				$wallet_id = wal_get_wallet_id_by_user_id($user_id);
				if (!$wallet_id) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				$wallet = wal_get_wallet($wallet_id);
				if (!is_object($wallet)) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				// Return if point entered is more than availabel balance.
				if ($fund > (float) $wallet->get_balance()) {
					throw new exception(__('You have entered more than available balance for selected user', 'wallet-for-woocommerce'));
				}

				$order->calculate_totals();
				// Return if the fund is greater than cart total.
				if ($order->get_total() && $order->get_total() < $fund) {
					throw new exception(__('You have entered the funds more than your cart total.', 'wallet-for-woocommerce'));
				}

				self::add_order_item_fee($order, $fund);

				// Update the partial fund details.
				$order->update_meta_data('wal_wallet_fund', $fund);
				$order->update_meta_data('wal_partial_fund', $fund);
				$order->update_meta_data('wal_partial_fund_applied', 'yes');
				$order->save();

				wp_send_json_success(array( 'success' => __('Wallet Balance Applied Successfully.', 'wallet-for-woocommerce') ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Add order item fee
		 *
		 * @since 3.4.0
		 * @param object $order
		 * @param float  $fund
		 * @return void
		 */
		public static function add_order_item_fee( $order, $fund ) {
			$fee = new WC_Order_Item_Fee();
			$fee->set_amount('-' . $fund);
			$fee->set_total('-' . $fund);
			$fee->set_name(__('Wallet Fund', 'wallet-for-woocommerce'));
			$order->add_item($fee);
			$order->calculate_totals();
			$order->save();
		}

		/**
		 * Validate create order by wallet gateway
		 *
		 * @since 3.4.0
		 * @return void
		 */
		public static function validate_create_order_by_wallet_gateway() {
			check_ajax_referer('wal-gateway-funds', 'wal_security');

			try {
				if (!isset($_POST)) {
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				$order_id = isset($_POST['order_id']) ? absint($_POST['order_id']) : '';
				$order = wc_get_order($order_id);
				$order->calculate_totals();

				$user_id = !empty($_POST['user_id']) ? absint($_POST['user_id']) : 0;
				if (!$user_id) {
					throw new exception(__('Please select the user', 'wallet-for-woocommerce'));
				}

				$wallet_id = wal_get_wallet_id_by_user_id($user_id);
				$wallet = wal_get_wallet($wallet_id);

				// Return if order items empty.
				if (!wal_check_is_array($order->get_items())) {
					throw new exception(__('No product(s) added. Please add an item(s) to place the order using the "Wallet Payment Gateway".', 'wallet-for-woocommerce'));
				}
				// Return if wallet balance empty
				if (empty($wallet->get_balance())) {
					throw new exception(__("The selected user doesn't have funds in their account. Hence, you cannot create the order.", 'wallet-for-woocommerce'));
				}

				$payment_method = isset($_POST['payment_method']) ? wc_clean(wp_unslash($_POST['payment_method'])) : '';
				if ('wal_wallet' === $payment_method && 'yes' === $order->get_meta('wal_partial_fund_applied')) {
					throw new exception(__('Since you have applied the wallet funds, you cannot use Wallet Payment Gateway in this order.', 'wallet-for-woocommerce'));
				}

				wp_send_json_success();
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Pagination action.
		 *
		 * @since 3.4.0
		 *
		 * @return void
		 * */
		public static function pagination_action() {
			check_ajax_referer('wal-pagination-action-nonce', 'wal_security');
			try {
				if (!isset($_POST) || !isset($_POST['selected_page'])) { // @codingStandardsIgnoreLine.
					throw new exception(__('Invalid Request', 'wallet-for-woocommerce'));
				}

				$log_ids = array();
				$table_name = isset($_POST['table_name']) && !empty($_POST['table_name']) ? wc_clean(wp_unslash($_POST['table_name'])) : '';
				switch ($table_name) {
					case 'activity':
						$log_ids = wal_user_transaction_logs();
						break;
					case 'wallet_withdrawal':
						$log_ids = wal_user_wallet_withdrawals();
						break;
					case 'fund_transfer':
						$log_ids = wal_user_fund_transfers();
						break;
				}

				$current_page = isset($_POST['selected_page']) && !empty($_POST['selected_page']) ? absint($_POST['selected_page']) : 1;
				$per_page = wal_transaction_logs_pagination_count();
				$offset = ( $per_page * $current_page ) - $per_page;
				$page_count = ceil(count($log_ids) / $per_page);
				$wallet_log_ids = array_slice($log_ids, $offset, $per_page);
				$pagination_range = wal_transaction_logs_pagination_range();

				$table_args = array(
					'offset' => $offset,
					'per_page' => $per_page,
					'count' => count($log_ids),
					'pagination' => array(
						'page_count' => $page_count,
						'current_page' => $current_page,
						'pagination_range' => $pagination_range,
						'prev_page_count' => ( ( $current_page - 1 ) == 0 ) ? ( $current_page ) : ( $current_page - 1 ),
						'next_page_count' => ( ( $current_page + 1 ) <= ( $page_count ) ) ? ( $current_page + 1 ) : ( $current_page ),
						'prev_page_row' => ( ( $current_page - $pagination_range ) > 0 ) ? $current_page - $pagination_range : $current_page,
						'next_page_row' => ( ( $current_page + $pagination_range ) <= $page_count ) ? $current_page + $pagination_range : $current_page,
					),
				);

				ob_start();
				switch ($table_name) {
					case 'activity':
						$table_args['transaction_log_ids'] = $wallet_log_ids;
						wal_get_template('dashboard/activity-wrapper.php', $table_args);
						break;
					case 'wallet_withdrawal':
						$table_args['withdrawal_ids'] = $wallet_log_ids;
						wal_get_template('modules/wallet-withdrawal/transactions-wrapper.php', $table_args);
						break;
					case 'fund_transfer':
						$table_args['fund_transfer_ids'] = $wallet_log_ids;
						wal_get_template('modules/fund-transfer/transaction-wrapper.php', $table_args);
						break;
				}

				$contents = ob_get_contents();
				ob_end_clean();

				wp_send_json_success(array( 'html' => $contents ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Reset user wallet data.
		 *
		 * @since 3.5.0
		 * @throws exception
		 */
		public static function reset_user_wallet_data() {
			check_ajax_referer('wal-reset-user-wallet-data', 'wal_security');

			try {
				if (!isset($_POST)) {
					throw new exception(esc_html__('Invalid Request', 'wallet-for-woocommerce'));
				}

				// Return if the current user does not have permission.
				if (!current_user_can('edit_posts')) {
					throw new exception(esc_html__("You don't have permission to do this action", 'wallet-for-woocommerce'));
				}

				$reset_data = isset($_POST['serialized_data']) ? wp_parse_args(wc_clean(wp_unslash($_POST['serialized_data']))) : '';
				if (!wal_check_is_array($reset_data)) {
					throw new exception(esc_html__('Invalid Request', 'wallet-for-woocommerce'));
				}

				if (!isset($reset_data['wal_reset_wallet_balance']) && !isset($reset_data['wal_reset_transaction_log'])) {
					throw new exception(esc_html__('Please select either the Reset Wallet Balance checkbox / Reset Transaction Log checkbox', 'wallet-for-woocommerce'));
				}

				$user_ids = self::get_valid_wallet_user_ids_to_reset($reset_data);
				if (!wal_check_is_array($user_ids)) {
					throw new exception(esc_html__('No User(s) Found', 'wallet-for-woocommerce'));
				}

				delete_option('wal_user_wallet_reseted');
				// Set processing user wallet balance reset notice.
				if (isset($reset_data['wal_reset_wallet_balance'])) {
					update_option('wal_processing_user_wallet_balance_reset', 'yes');
				}

				// Set processing user wallet transaction log notice.
				if (isset($reset_data['wal_reset_transaction_log'])) {
					update_option('wal_processing_user_wallet_transaction_log_reset', 'yes');
				}

				$chunked_user_ids = array_filter(array_chunk($user_ids, 100), 'wal_array_filter');
				foreach ($chunked_user_ids as $key => $user_ids) {
					// Reset wallet balance.
					if (isset($reset_data['wal_reset_wallet_balance'])) {
						as_schedule_single_action(
								time(),
								'wal_reset_user_balance',
								array(
									'user_ids' => $user_ids,
									'completed' => count($chunked_user_ids) - 1 === $key,
								)
						);
					}

					// Reset transaction log.
					if (isset($reset_data['wal_reset_transaction_log'])) {
						$chunked_transaction_log_ids = array_filter(array_chunk(wal_users_transaction_logs($user_ids), 100), 'wal_array_filter');
						if ( ! wal_check_is_array( $chunked_transaction_log_ids ) ) {
							as_schedule_single_action(
								time(),
								'wal_reset_user_transaction_log',
								array(
									'transaction_log_ids' => array(),
									'completed' => true,
								)
							);
						} else {
							foreach ($chunked_transaction_log_ids as $key => $transaction_log_ids) {
								as_schedule_single_action(
									time(),
									'wal_reset_user_transaction_log',
									array(
										'transaction_log_ids' => $transaction_log_ids,
										'completed' => count($chunked_transaction_log_ids) - 1 === $key,
									)
								);
							}
						}
					}
				}

				wp_send_json_success();
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Get valid wallet user ids to reset
		 *
		 * @since 3.5.0
		 * @param type $serialized_data
		 * @return array
		 */
		public static function get_valid_wallet_user_ids_to_reset( $serialized_data ) {
			$args = array();
			$user_selection_type = isset($serialized_data['wal_reset_user_selection_type']) ? $serialized_data['wal_reset_user_selection_type'] : 1;
			$args['fields'] = 'ids';

			switch ($user_selection_type) {
				case '2':
					$include_users = isset($serialized_data['wal_reset_include_users']) ? $serialized_data['wal_reset_include_users'] : array();
					// Return if empty include user
					if (empty($include_users)) {
						return $args;
					}

					$args['include'] = $include_users;
					break;
				case '3':
					$exclude_users = isset($serialized_data['wal_reset_exclude_users']) ? $serialized_data['wal_reset_exclude_users'] : array();
					// Return if empty exclude user
					if (empty($exclude_users)) {
						return $args;
					}

					$args['exclude'] = $exclude_users;
					break;
				case '4':
					$include_user_roles = isset($serialized_data['wal_reset_include_user_roles']) ? $serialized_data['wal_reset_include_user_roles'] : array();
					// Return if empty include user roles
					if (empty($include_user_roles)) {
						return $args;
					}

					$args['role__in'] = $include_user_roles;
					break;
				case '5':
					$exclude_user_roles = isset($serialized_data['wal_reset_exclude_user_roles']) ? $serialized_data['wal_reset_exclude_user_roles'] : array();
					// Return if empty exclude user roles
					if (empty($exclude_user_roles)) {
						return $args;
					}

					$args['role__not_in'] = $exclude_user_roles;
					break;
			}

			return get_users($args);
		}

		/**
		 * Remove user wallet reseted success notice
		 *
		 * @since 3.5.0
		 * @return void
		 */
		public static function remove_user_wallet_reseted_success_notice() {
			check_ajax_referer('wal-reset-notice-nonce', 'wal_security');

			try {
				delete_option('wal_user_wallet_reseted');
				wp_send_json_success();
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Get valid shipping methods html
		 *
		 * @since 4.3.0
		 * @return void
		 */
		public static function get_valid_shipping_methods_html() {
			check_ajax_referer('wal-shipping-list', 'wal_security');

			try {
				$shipping_list = isset( $_POST['shipping_lists'] ) ?  wc_clean( $_POST['shipping_lists'] ) : array();
				$html = wal_get_template_html( 'modules/product-purchase/popup-shipping-methods.php', array( 'shipping_list' => $shipping_list ) );

				wp_send_json_success( array( 'html' => $html ) );
			} catch ( Exception $ex ) {
				wp_send_json_error( array( 'error' => $ex->getMessage() ) );
			}
		}
	}

	WAL_Admin_Ajax::init();
}
