<?php

/**
 * Abstract Modules Class.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Modules' ) ) {

	/**
	 * Class.
	 */
	abstract class WAL_Modules {

		/**
		 * ID.
		 * 
		 * @var string
		 */
		protected $id ;

		/**
		 * Title.
		 * 
		 * @var string
		 */
		protected $title ;

		/**
		 * Description.
		 * 
		 * @var string
		 */
		protected $description ;

		/**
		 * Tool tip.
		 * 
		 * @var string
		 */
		protected $tooltip ;

		/**
		 * Suffix.
		 * 
		 * @var string
		 */
		protected $suffix ;

		/**
		 * Show buttons.
		 * 
		 * @var bool
		 */
		protected $show_buttons = true ;

		/**
		 * Show reset button.
		 * 
		 * @var bool
		 */
		protected $show_reset_button = true ;

		/**
		 * Plugin slug.
		 * 
		 * @var string
		 */
		protected $plugin_slug = 'wal' ;

		/**
		 * Class Constructor.
		 */
		public function __construct() {
			$this->process_actions() ;
		}

		/**
		 * Get the id.
		 * 
		 * @return string
		 */
		public function get_id() {
			return $this->id ;
		}

		/**
		 * Get the title.
		 * 
		 * @return string
		 */
		public function get_title() {
			return $this->title ;
		}

		/**
		 * Get the description.
		 * 
		 * @return string
		 */
		public function get_description() {
			return $this->description ;
		}

		/**
		 * Get the tool tip.
		 * 
		 * @return string
		 */
		public function get_tooltip() {
			return $this->tooltip ;
		}

		/**
		 * Get the suffix.
		 * 
		 * @return string
		 */
		public function get_suffix() {
			return $this->suffix ;
		}

		/**
		 * Get the plugin slug.
		 * 
		 * @return string
		 */
		public function get_plugin_slug() {
			return $this->plugin_slug ;
		}

		/**
		 * Show reset button?.
		 * 
		 * @return string
		 */
		public function show_reset_button() {
			return $this->show_reset_button ;
		}

		/**
		 * Show buttons?.
		 * 
		 * @return string
		 */
		public function show_buttons() {
			return $this->show_buttons ;
		}

		/**
		 * Is module enabled?.
		 * 
		 * @return bool
		 */
		public function is_enabled() {
			/**
			 * This hook is used to alter the current module enable status.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->get_plugin_slug() . '_' . $this->get_id() . '_enabled', 'yes' === get_option( $this->get_enable_key() ), $this ) ;
		}

		/**
		 * Get the enable key.
		 * 
		 * @return string
		 */
		public function get_enable_key() {
			return $this->get_option_key( $this->get_id() . '_enabled' ) ;
		}

		/**
		 * Get the option key.
		 * 
		 * @return string
		 */
		public function get_option_key( $key ) {
			return sanitize_key( $this->get_plugin_slug() . '_module_' . $key ) ;
		}

		/**
		 * Get the setting tabs.
		 * 
		 * @return array
		 */
		public function setting_tabs() {
			/**
			 * This hook is used to alter the current module settings tabs.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->get_plugin_slug() . '_get_module_tabs_' . $this->get_id(), array(), $this ) ;
		}

		/**
		 * Get the setting options.
		 * 
		 * @return array
		 */
		public function setting_options( $current_section = '' ) {
			$settings = array() ;
			$function = $current_section . '_section_array' ;

			if ( method_exists( $this, $function ) ) {
				$settings = $this->$function() ;
			}
			/**
			 * This hook is used to alter the current module settings fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters( $this->get_plugin_slug() . '_' . $this->get_id() . '_setting_options', $settings, $current_section, $this ) ;
		}

		/**
		 * Process actions.
		 */
		protected function process_actions() {
			$this->global_actions() ;
			$this->include_files() ;

			if ( ! $this->is_enabled() ) {
				return ;
			}

			$this->actions() ;

			$this->suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min' ;

			if ( is_admin() ) {
				$this->admin_action() ;
			}

			if ( ! is_admin() || defined( 'DOING_AJAX' ) ) {
				$this->frontend_action() ;

				//add action for external js files in backend
				add_action( 'wp_enqueue_scripts', array( $this, 'frontend_enqueue_scripts' ), 20 ) ;
			}
		}

		/**
		 * Global actions.
		 */
		private function global_actions() {

			if ( is_admin() ) {
				add_action( sanitize_key( $this->get_plugin_slug() . '_module_settings_' . $this->get_id() ), array( $this, 'output_settings' ), 10 ) ;
				add_action( sanitize_key( $this->get_plugin_slug() . '_module_settings_' . $this->get_id() ), array( $this, 'output_setting_buttons' ), 20 ) ;
				add_action( sanitize_key( $this->get_plugin_slug() . '_module_settings_loaded_' . $this->get_id() ), array( $this, 'save_settings' ), 10 ) ;
				add_action( sanitize_key( $this->get_plugin_slug() . '_module_settings_loaded_' . $this->get_id() ), array( $this, 'reset_settings' ), 20 ) ;
				//add action for external js files in backend
				add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ), 20 ) ;
			}
		}

		/**
		 * Output the settings.
		 */
		public function output_settings() {
			global $current_tab ;

			$settings = $this->setting_options( $current_tab ) ;
			if ( wal_check_is_array( $settings ) ) {
				WC_Admin_Settings::output_fields( $settings ) ;
			} else {
				$this->output_tab_content() ;
			}
		}

		/**
		 * Output the tab content.
		 */
		public function output_tab_content() {
		}

		/**
		 * Output the setting buttons.
		 */
		public function output_setting_buttons() {
			if ( ! $this->show_buttons() ) {
				return ;
			}

			WAL_Settings::output_buttons( $this->show_reset_button() ) ;
		}

		/**
		 * Save the settings.
		 */
		public function save_settings() {
			global $current_tab ;
			if ( ! isset( $_POST[ 'save' ] ) || empty( $_POST[ 'save' ] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.NoNonceVerification
				return ;
			}

			check_admin_referer( 'wal_save_settings', '_wal_nonce' ) ;

			$settings = $this->setting_options( $current_tab ) ;
			WC_Admin_Settings::save_fields( $settings ) ;

			WAL_Settings::add_message( __( 'Your settings have been saved', 'wallet-for-woocommerce' ) ) ;
						
						/**
			 * This hook is used to do extra action after module settings saved.
			 * 
			 * @hooked WAL_Wallet_Withdrawal_Module->save_gateway_settings - 10 (save the gateway settings).
			 * @since 1.0
			 */
			do_action( sanitize_key( $this->get_plugin_slug() . '_after_module_' . $this->get_id() . '_settings_saved' ) ) ;
		}

		/**
		 * Reset the settings.
		 */
		public function reset_settings() {
			global $current_tab ;
			if ( ! isset( $_POST[ 'reset' ] ) || empty( $_POST[ 'reset' ] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.NoNonceVerification
				return ;
			}

			check_admin_referer( 'wal_reset_settings', '_wal_nonce' ) ;

			$settings = $this->setting_options( $current_tab ) ;
			WAL_Settings::reset_fields( $settings ) ;

			WAL_Settings::add_message( __( 'Your settings have been reset', 'wallet-for-woocommerce' ) ) ;
						
						/**
			 * This hook is used to do extra action after module settings reset.
			 * 
			 * @hooked WAL_Wallet_Withdrawal_Module->reset_gateway_settings - 10 (reset the gateway settings).
			 * @since 1.0
			 */
			do_action( sanitize_key( $this->get_plugin_slug() . '_after_module_' . $this->get_id() . '_settings_reset' ) ) ;
		}

		/**
		 * Include files.
		 */
		protected function include_files() {
		}

		/**
		 * Actions.
		 */
		protected function actions() {
		}

		/**
		 * Admin actions.
		 */
		protected function admin_action() {
		}

		/**
		 * Front end actions.
		 */
		protected function frontend_action() {
		}

		/**
		 * Enqueue admin scripts.
		 */
		public function admin_enqueue_scripts() {
			$this->admin_external_js_files() ;
			$this->admin_external_css_files() ;
		}

		/**
		 * Enqueue front end scripts.
		 */
		public function frontend_enqueue_scripts() {
			$this->frontend_external_js_files() ;
			$this->frontend_external_css_files() ;
		}

		/**
		 * Enqueue front end JS files.
		 */
		protected function frontend_external_js_files() {
		}

		/**
		 * Enqueue front end CSS files.
		 */
		protected function frontend_external_css_files() {
		}

		/**
		 * Enqueue admin JS files.
		 */
		protected function admin_external_js_files() {
		}

		/**
		 * Enqueue admin CSS files.
		 */
		protected function admin_external_css_files() {
		}
	}

}
