<?php

use Automattic\WooCommerce\Blocks\Integrations\IntegrationInterface;

if ( ! class_exists( 'TC_WB_Blocks_Integration' ) ) {

    class TC_WB_Blocks_Integration implements IntegrationInterface {

        /**
         * Plugin's name
         * @var string
         */
        private $name;

        /** Plugin's base directory name
         * @var string
         */
        private $dir_name;

        /**
         * Plugin's version
         * @var string
         */
        private $version;

        /**
         * Plugin's General Settings
         * @var array
         */
        private $settings;

        /**
         * The name of the integration.
         * @return string
         */
        public function get_name() {
            return 'tickera/checkout-fields';
        }

        /**
         * When called invokes any initialization/setup for the integration.
         */
        public function initialize() {
            $this->name = 'tc_woobridge';
            $this->dir_name = 'bridge-for-woocommerce';
            $this->version = 1.2;
            $this->settings = get_option( 'tickera_general_setting' );
            $this->register_scripts_styles();
        }

        function register_scripts_styles() {

            // Frontend Scripts
            $script_path = '../../assets/blocks/build/frontend.js';
            $script_url = plugins_url( $script_path, __FILE__ );

            $script_asset_path = trailingslashit( WP_PLUGIN_DIR ) . $this->dir_name . '/assets/blocks/build/frontend.asset.php';
            $script_asset = file_exists( $script_asset_path ) ? require $script_asset_path : [ 'dependencies' => [], 'version' => $this->get_file_version( $script_asset_path ) ];
            wp_register_script( $this->name . '-block-frontend', $script_url, $script_asset[ 'dependencies' ], $script_asset[ 'version' ], true );
            wp_localize_script( $this->name . '-block-frontend', $this->name . '_block', self::get_translations() );

            // Editor Scripts
            $script_path = '../../assets/blocks/build/editor.js';
            $script_url = plugins_url( $script_path, __FILE__ );
            $script_asset_path = trailingslashit( WP_PLUGIN_DIR ) . $this->dir_name . '/assets/blocks/build/editor.asset.php';
            $script_asset = file_exists( $script_asset_path ) ? require $script_asset_path : [ 'dependencies' => [], 'version' => $this->get_file_version( $script_asset_path ) ];
            wp_register_script( $this->name . '-block-editor', $script_url, $script_asset[ 'dependencies' ], $script_asset[ 'version' ], true );
            wp_localize_script( $this->name . '-block-editor', $this->name . '_block', self::get_translations() );

            // Frontend Styles
            $style_path = '../../assets/blocks/build/style-frontend.css';
            $style_url = plugins_url( $style_path, __FILE__ );
            wp_enqueue_style( $this->name . '-block', $style_url, [], $this->get_file_version( $style_path ) );

            // Register main integration
            $script_path = '../../assets/blocks/build/index.js';
            $script_url = plugins_url( $script_path, __FILE__ );

            $script_asset_path = trailingslashit( WP_PLUGIN_DIR ) . $this->dir_name . '/assets/blocks/build/index.asset.php';
            $script_asset = file_exists( $script_asset_path ) ? require $script_asset_path : [ 'dependencies' => [], 'version' => $this->get_file_version( $script_path ), ];
            wp_register_script( $this->name . '-blocks-integration', $script_url, $script_asset[ 'dependencies' ], $script_asset[ 'version' ], true );
        }

        /**
         * Returns an array of script handles to enqueue in the frontend context.
         *
         * @return string[]
         */
        public function get_script_handles() {
            return [ $this->name . '-blocks-integration', $this->name . '-block-frontend' ];
        }

        /**
         * Returns an array of script handles to enqueue in the editor context.
         *
         * @return string[]
         */
        public function get_editor_script_handles() {
            return [ $this->name . '-blocks-integration', $this->name . '-block-editor', ];
        }

        /**
         * Get the file modified time as a cache buster if we're in dev mode.
         *
         * @param string $file Local path to the file.
         * @return string The cache buster value to use for the given file.
         */
        protected function get_file_version( $file ) {

            if ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG && file_exists( $file ) ) {
                return filemtime( $file );
            }

            return $this->version;
        }

        /**
         * An array of key, value pairs of data made available to the block on the client side.
         *
         * @return array
         */
        public function get_script_data() {

            global $tc;

            $checkout_owner_fields = $tc->session->get( 'tcwb-checkout-fields' );
            $checkout_owner_fields = $checkout_owner_fields ? $checkout_owner_fields : [];

            $checkout_owner_fields[ 'cart_items' ] = self::get_cart_session();
            $tc->session->set( 'tcwb-checkout-fields', $checkout_owner_fields );
            $cart_data = self::get_cart_data();

            return apply_filters( 'tcwb_checkout_fields_script_data', [
                'cart_items' => ( isset( $cart_data[ 'items' ] ) ? array_values( $cart_data[ 'items' ] ) : [] ),
                'owner_fields' => ( isset( $cart_data[ 'owner_fields' ] ) ? $cart_data[ 'owner_fields' ] : [] ),
                'buyer_fields' => ( isset( $cart_data[ 'buyer_fields' ] ) ? $cart_data[ 'buyer_fields' ] : [] ),
                'extra_fields' => ( isset( $cart_data[ 'extra_fields' ] ) ? $cart_data[ 'extra_fields' ] : [] )
            ], self::get_cart_session() );
        }

        /**
         * Format cart by [ticket_type_id => quantity]
         * Collection of Buyer and Owner Fields
         *
         * @return array
         */
        public static function get_cart_data() {

            $data = [];
            $settings = get_option( 'tickera_general_setting' );

            // Show Attendees Fields
            if ( isset( $settings[ 'show_owner_fields' ] ) ) {
                $show_owner_fields = ( 'yes' == $settings[ 'show_owner_fields' ] ) ? true : false;

            } else {
                $show_owner_fields = true;
            }

            $items = self::get_cart_session();
            foreach ( $items as $item => $values ) {

                $is_ticket_meta = get_post_meta( TC_WooCommerce_Bridge::maybe_get_original_wpml_product_id( (int) $values[ 'product_id' ] ), '_tc_is_ticket', true );
                $is_ticket = $is_ticket_meta == 'yes' ? true : false;

                if ( $is_ticket ) {
                    $product_id = isset( $values[ 'variation_id' ] ) && is_numeric( $values[ 'variation_id' ] ) && $values[ 'variation_id' ] > 0 ? (int) $values[ 'variation_id' ] : (int) $values[ 'product_id' ];
                    $data[ 'items' ][ $product_id ] = [
                        'id' => $product_id,
                        'name' => html_entity_decode( get_the_title( $product_id ) ),
                        'quantity' => (int) @$data[ $product_id ][ 'quantity' ] + (int) $values[ 'quantity' ]
                    ];

                    $owner_fields = new Tickera\TC_Cart_Form();

                    if ( $show_owner_fields ) {
                        $data[ 'owner_fields' ][ $product_id ] = array_values( $owner_fields->get_owner_info_fields( $product_id ) ); // array_values() to reindex

                    } else {
                        $fields = $owner_fields->get_owner_info_fields( $product_id );
                        $data[ 'owner_fields' ][ $product_id ] = array_values( array_filter( ( array_map( function( $field ) { if ( ! $field[ 'form_visibility' ] ) return $field; }, $fields ) ) ) );
                    }

                    $data[ 'buyer_fields' ] = $owner_fields->get_buyer_info_fields();
                }
            }

            // Show Age Confirmation
            if ( isset( $settings[ 'show_age_check' ] ) && 'yes' == $settings[ 'show_age_check' ] ) {

                $data[ 'extra_fields' ][] = [
                    'field_name' => 'confirm_age',
                    'field_title' => sprintf(
                        /* translators: %s: Tickera Age confirmation message */
                        __( '%s', 'woocommerce-tickera-bridge' ),
                        esc_html( isset( $settings[ 'age_text' ] ) ? $settings[ 'age_text' ] : 'I hereby declare that I am 16 years or older' )
                    ),
                    'field_type' => 'script',
                    'post_meta' => false,
                    'error_message' => sprintf(
                        /* translators: %s: Tickera Failed age check message */
                        __( '%s', 'woocommerce-tickera-bridge' ),
                        esc_html( isset( $settings[ 'age_error_text' ] ) ? $settings[ 'age_error_text' ] : 'Only customers aged 16 or older are permitted for purchase on this website' )
                    )
                ];
            }

            return apply_filters( 'tcwb_checkout_blocks_cart_data', $data );
        }

        /**
         * Collection of Cart Session Data
         *
         * @return array
         */
        public static function get_cart_session() {

            global $tc;
            $items = [];

            /**
             * Collect from cart session/cookie
             */
            if ( WC()->cart && method_exists( WC()->cart, 'get_cart' ) ) {
                $items = WC()->cart->get_cart();
            }

            /**
             * Collect from cart persistent session
             */
            if ( ! isset( $items ) ) {
                $saved_cart_meta = get_user_meta( get_current_user_id(), '_woocommerce_persistent_cart_' . get_current_blog_id(), true );
                if ( isset( $saved_cart_meta['cart'] ) ) {
                    $items = array_filter( (array) $saved_cart_meta['cart'] );
                }
            }

            /**
             * Collect from Tickera session
             */
            if ( ! $items && isset( $tc->session ) ) {
                $checkout_owner_fields = $tc->session->get( 'tcwb-checkout-fields' );
                $items = isset( $checkout_owner_fields[ 'cart_items' ] ) ? $checkout_owner_fields[ 'cart_items' ] : [];
            }

            return $items ? $items : [];
        }

        protected function get_translations() {

            return [
                'general_admission' => __( 'General Admission', 'woocommerce-tickera-bridge' ),
                'attendee_info' => __( 'Attendee Info', 'woocommerce-tickera-bridge' ),
                'first_name' => __( 'First Name', 'woocommerce-tickera-bridge' ),
                'last_name' => __( 'Last Name', 'woocommerce-tickera-bridge' ),
                'email_address' => __( 'Email Address', 'woocommerce-tickera-bridge' ),
                'confirm_email_address' => __( 'Confirm Email Address', 'woocommerce-tickera-bridge' ),
                'confirmation_email_address' => __( 'Confirmation Email Address', 'woocommerce-tickera-bridge' ),
                'please_enter_a_valid' => __( 'Please enter a valid', 'woocommerce-tickera-bridge' ),
                'something_went_wrong_please_try_again' => __( 'Something went wrong, please try again.', 'woocommerce-tickera-bridge' ),
                'the_following_problems_were_found' => __( 'The following problems were found:', 'woocommerce-tickera-bridge' ),
                'please_enter_number' => __( 'Please enter a number at least', 'woocommerce-tickera-bridge' ),
                'and_below' => __( 'and below.', 'woocommerce-tickera-bridge' ),
                'and_above' => __( 'and above.', 'woocommerce-tickera-bridge' ),
                'enter_divisible_by' => __( 'Please enter a number divisible by', 'woocommerce-tickera-bridge' ),
                'please_select_a_value' => __( 'Please select a value', 'woocommerce-tickera-bridge' ),
            ];
        }
    }
}
