<?php

/**
 * Plugin Name: Tickera Bridge for WooCommerce
 * Plugin URI: https://tickera.com/
 * Description: Leverage the power of both WooCommerce and Tickera to manage events and sell tickets
 * Version: 1.6.7
 * Update URI: https://api.freemius.com
 * Author: Tickera
 * Author URI: https://tickera.com/
 * Developer: Tickera
 * Developer URI: https://tickera.com/
 * Text Domain: woocommerce-tickera-bridge
 * Domain Path: /languages
 */
if ( !defined( 'ABSPATH' ) ) {
    exit;
}
// Exit if accessed directly
if ( class_exists( 'Tickera\\TC' ) ) {
} else {
    add_action( 'admin_notices', 'tc_tickera_plugin_installation_message' );
}
if ( class_exists( 'Tickera\\TC' ) && class_exists( 'WooCommerce' ) ) {
} else {
    add_action( 'admin_notices', 'tc_tickera_plugin_installation_message' );
}
if ( !function_exists( 'woo_bridge_fs' ) ) {
    if ( $_POST && isset( $_POST['fs_activation'] ) && $_POST['fs_activation'] === 'false' || defined( 'FS_ACTIVATION' ) && !FS_ACTIVATION ) {
        // Do nothing if freemius activation is disabled
        define( 'FS_ACTIVATION', false );
    } else {
        // Create a helper function for easy SDK access.
        function woo_bridge_fs() {
            global $woo_bridge_fs;
            if ( !isset( $woo_bridge_fs ) ) {
                // Activate multisite network integration.
                if ( !defined( 'WP_FS__PRODUCT_3166_MULTISITE' ) ) {
                    define( 'WP_FS__PRODUCT_3166_MULTISITE', true );
                }
                // Include Freemius SDK.
                if ( file_exists( dirname( dirname( __FILE__ ) ) . '/tickera-event-ticketing-system/freemius/start.php' ) ) {
                    // Try to load SDK from parent plugin folder.
                    require_once dirname( dirname( __FILE__ ) ) . '/tickera-event-ticketing-system/freemius/start.php';
                } else {
                    if ( file_exists( dirname( dirname( __FILE__ ) ) . '/tickera/freemius/start.php' ) ) {
                        // Try to load SDK from premium parent plugin folder.
                        require_once dirname( dirname( __FILE__ ) ) . '/tickera/freemius/start.php';
                    } else {
                        require_once dirname( __FILE__ ) . '/freemius/start.php';
                    }
                }
                $woo_bridge_fs = fs_dynamic_init( array(
                    'id'               => '3166',
                    'slug'             => 'bridge-for-woocommerce',
                    'premium_slug'     => 'bridge-for-woocommerce',
                    'type'             => 'plugin',
                    'public_key'       => 'pk_3114ed61c5e7603ae5b8c7ee5801f',
                    'is_premium'       => true,
                    'is_premium_only'  => true,
                    'has_paid_plans'   => true,
                    'is_org_compliant' => false,
                    'parent'           => array(
                        'id'         => '3102',
                        'slug'       => 'tickera-event-ticketing-system',
                        'public_key' => 'pk_7a38a2a075ec34d6221fe925bdc65',
                        'name'       => 'Tickera',
                    ),
                    'menu'             => array(
                        'first-path' => 'plugins.php',
                        'support'    => false,
                    ),
                    'is_live'          => true,
                ) );
            }
            return $woo_bridge_fs;
        }

    }
}
/**
 * Check if the parent's init SDK method exists.
 * @return bool
 */
function woo_bridge_fs_is_parent_active_and_loaded() {
    return function_exists( '\\Tickera\\tets_fs' );
}

function woo_bridge_fs_is_parent_active() {
    $active_plugins = get_option( 'active_plugins', array() );
    if ( is_multisite() ) {
        $network_active_plugins = get_site_option( 'active_sitewide_plugins', array() );
        $active_plugins = array_merge( $active_plugins, array_keys( $network_active_plugins ) );
    }
    foreach ( $active_plugins as $basename ) {
        if ( 0 === strpos( $basename, 'tickera-event-ticketing-system/' ) || 0 === strpos( $basename, 'tickera/' ) ) {
            return true;
        }
    }
    return false;
}

function woo_bridge_fs_init() {
    if ( woo_bridge_fs_is_parent_active_and_loaded() ) {
        // Init Freemius.
        woo_bridge_fs();
        // Signal that the add-on's SDK was initiated.
        do_action( 'woo_bridge_fs_loaded' );
        // Parent is active, add your init code here.
    } else {
        // Parent is inactive, add your error handling here.
    }
}

if ( woo_bridge_fs_is_parent_active_and_loaded() ) {
    // If parent already included, init add-on.
    woo_bridge_fs_init();
} else {
    if ( woo_bridge_fs_is_parent_active() ) {
        // Init add-on only after the parent is loaded.
        add_action( 'tets_fs_loaded', 'woo_bridge_fs_init' );
    } else {
        // Even though the parent is not activated, execute add-on for activation / uninstall hooks.
        woo_bridge_fs_init();
    }
}
/**
 * Show admin notices if Tickera of WooCommerce are not installed
 */
function tc_tickera_plugin_installation_message() {
    $url_tickera = add_query_arg( array(
        'tab'       => 'plugin-information',
        'plugin'    => 'tickera-event-ticketing-system',
        'TB_iframe' => 'true',
    ), network_admin_url( 'plugin-install.php' ) );
    $url_woocommerce = add_query_arg( array(
        'tab'       => 'plugin-information',
        'plugin'    => 'woocommerce',
        'TB_iframe' => 'true',
    ), network_admin_url( 'plugin-install.php' ) );
    $title_tickera = __( 'Tickera', 'woocommerce-tickera-bridge' );
    $title_woocommerce = __( 'WooCommerce', 'woocommerce-tickera-bridge' );
    if ( !class_exists( 'Tickera\\TC' ) ) {
        echo '<div class="error"><p>' . sprintf(
            __( 'To begin using WooCommerce Bridge for Tickera, please install and activate the latest version of <a href="%s" class="thickbox" title="%s">%s</a>.', 'woocommerce-tickera-bridge' ),
            esc_url( $url_tickera ),
            $title_tickera,
            $title_tickera
        ) . '</p></div>';
    }
    if ( !class_exists( 'WooCommerce' ) ) {
        echo '<div class="error"><p>' . sprintf(
            __( 'To begin using WooCommerce Bridge for Tickera, please install and activate the latest version of <a href="%s" class="thickbox" title="%s">%s</a>.', 'woocommerce-tickera-bridge' ),
            esc_url( $url_woocommerce ),
            $title_woocommerce,
            $title_woocommerce
        ) . '</p></div>';
    }
}

if ( !class_exists( 'TC_WooCommerce_Bridge' ) ) {
    class TC_WooCommerce_Bridge {
        var $version = '1.2';

        var $tc_version_required = '3.5.1.7';

        // Prev: 3.4.9.3
        var $title = 'Bridge for WooCommerce';

        var $name = 'tc_woobridge';

        var $dir_name = 'bridge-for-woocommerce';

        var $location = 'plugins';

        var $plugin_dir = '';

        var $plugin_url = '';

        var $tc_general_settings = '';

        function __construct() {
            global $tc, $pagenow;
            $this->init_vars();
            require_once $this->plugin_dir . 'includes/functions.php';
            require_once $this->plugin_dir . 'includes/classes/class.admin-duplicate-product.php';
            add_action( 'admin_notices', array($this, 'admin_notices') );
            add_action( 'activated_plugin', array($this, 'plugin_dependencies'), 99 );
            add_action( 'woocommerce_product_options_general_product_data', array($this, 'tc_add_custom_settings') );
            add_action(
                'woocommerce_process_product_meta',
                array($this, 'tc_custom_settings_fields_save'),
                10,
                1
            );
            // TO DO: add_action( 'woocommerce_save_product_variation', array( &$this, 'tc_custom_additional_meta_save_variations' ), 10, 2 );
            if ( !function_exists( 'is_plugin_active' ) ) {
                include_once ABSPATH . 'wp-admin/includes/plugin.php';
            }
            /**
             * Remove "tc_woo_bridge_alternative_hook_for_checkout_to_billing" filter hook.
             * @since 1.5.5
             */
            add_action( 'woocommerce_after_checkout_billing_form', array($this, 'render_tc_buyer_fields') );
            $checkout_owner_fields_placement_hook = ( isset( $this->tc_general_settings['tc_woo_checkout_owner_fields_placement'] ) ? $this->tc_general_settings['tc_woo_checkout_owner_fields_placement'] : 'woocommerce_checkout_after_customer_details' );
            add_action( $checkout_owner_fields_placement_hook, array($this, 'render_tc_owner_fields'), 20 );
            add_action( 'woocommerce_new_order', array($this, 'tc_order_created') );
            add_action( 'woocommerce_resume_order', array($this, 'tc_order_created') );
            /*
             * Redundant process in woocommerce_new_order hook.
             *
             * Disabled.
             * @since 1.4.4
             *
             * add_action( 'woocommerce_api_create_order', array( &$this, 'tc_api_create_order' ), 10, 2 );
             */
            add_action(
                'woocommerce_order_details_after_order_table',
                array(&$this, 'tc_add_tickets_table_on_woo_order_details_page'),
                10,
                2
            );
            add_action(
                'woocommerce_view_order',
                array(&$this, 'tc_add_tickets_table_on_woo_order_details_page'),
                10,
                2
            );
            add_action(
                'woocommerce_email_after_order_table',
                array(&$this, 'tc_add_content_email_after_order_table'),
                99,
                4
            );
            add_action(
                'woocommerce_email_before_order_table',
                array(&$this, 'tc_add_content_email_after_order_table'),
                99,
                4
            );
            add_action(
                'woocommerce_checkout_process',
                array(&$this, 'tc_validate_tickera_fields'),
                10,
                1
            );
            add_action( 'woocommerce_admin_order_data_after_order_details', array($this, 'tc_woocommerce_admin_order_data_after_order_details') );
            add_action(
                'woocommerce_order_status_changed',
                array(&$this, 'check_tickets_action'),
                10,
                3
            );
            add_action(
                'woocommerce_order_status_completed',
                array($this, 'send_attendee_order_email_notification'),
                10,
                2
            );
            add_filter( 'tc_bridge_for_woocommerce_is_active', array(&$this, 'tc_bridge_for_woocommerce_is_active') );
            add_filter(
                'tc_event_shortcode_column',
                array(&$this, 'tc_event_shortcode_column_modify'),
                10,
                2
            );
            add_filter(
                'tc_plugin_admin_menu_items',
                array(&$this, 'tc_modify_plugin_admin_menu_items'),
                10,
                1
            );
            add_filter(
                'tc_ticket_instance_order_admin_url',
                array(&$this, 'tc_modify_ticket_instance_order_admin_url'),
                10,
                3
            );
            add_filter(
                'tc_cart_contents',
                array($this, 'tc_modify_cart_contents'),
                10,
                1
            );
            add_filter(
                'tc_event_name_field_name',
                array($this, 'tc_modify_event_name_field_name'),
                10,
                1
            );
            add_filter(
                'tc_ticket_template_field_name',
                array($this, 'tc_modify_ticket_template_field_name'),
                99,
                1
            );
            add_filter( 'tc_allow_ticket_checkout_field_name', array($this, 'tc_modify_allow_ticket_checkout') );
            add_filter(
                'tc_order_is_paid',
                array($this, 'tc_modify_order_is_paid'),
                10,
                2
            );
            add_filter( 'tc_paid_post_statuses', array($this, 'tc_modify_order_paid_statuses') );
            add_filter(
                'tc_order_post_type_name',
                array($this, 'tc_modify_order_post_type_name'),
                10,
                1
            );
            add_filter(
                'tc_download_ticket_url_front',
                array($this, 'tc_modify_download_ticket_url_front'),
                10,
                3
            );
            add_filter(
                'tc_order_details_post_status',
                array($this, 'tc_modify_order_details_post_status'),
                10,
                2
            );
            add_filter(
                'tc_ticket_checkin_order_date',
                array(&$this, 'modify_checkin_order_date'),
                10,
                2
            );
            add_filter(
                'tc_ticket_checkin_buyer_first_name',
                array(&$this, 'modify_checkin_buyer_first_name'),
                10,
                2
            );
            add_filter(
                'tc_ticket_checkin_buyer_last_name',
                array(&$this, 'modify_checkin_buyer_last_name'),
                10,
                2
            );
            add_filter(
                'tc_ticket_checkin_buyer_full_name',
                array(&$this, 'modify_checkin_buyer_full_name'),
                10,
                2
            );
            add_filter(
                'tc_ticket_buyer_name_element',
                array(&$this, 'modify_checkin_buyer_full_name'),
                10,
                2
            );
            add_filter(
                'tc_ticket_checkin_buyer_email',
                array(&$this, 'modify_checkin_buyer_email'),
                10,
                2
            );
            add_filter(
                'tc_general_settings_store_fields',
                array(&$this, 'modify_general_settings_store_fields'),
                10,
                1
            );
            add_filter(
                'tc_get_event_ticket_types',
                array(&$this, 'modify_get_event_ticket_types'),
                10,
                3
            );
            add_filter( 'tc_settings_new_menus', array(&$this, 'tc_settings_new_menus') );
            add_filter( 'tc_settings_general_sections', array(&$this, 'modify_settings_general_sections') );
            add_filter( 'tc_general_settings_miscellaneous_fields', array(&$this, 'modify_settings_miscellaneous_fields') );
            add_filter( 'tc_settings_email_sections', array(&$this, 'modify_settings_email_sections') );
            add_filter( 'tc_settings_email_fields', array(&$this, 'modify_settings_email_fields') );
            add_filter( 'tc_add_network_admin_menu', array(&$this, 'modify_add_network_admin_menu') );
            add_filter(
                'tc_custom_forms_owner_form_template_meta',
                array(&$this, 'modify_custom_forms_owner_form_template_meta'),
                10,
                1
            );
            add_filter(
                'tc_order_fields',
                array($this, 'modify_order_fields'),
                9,
                1
            );
            add_filter(
                'tc_buyer_info_fields',
                array(&$this, 'modify_buyer_info_fields'),
                9,
                1
            );
            add_filter(
                'tc_custom_forms_show_custom_fields_as_order_columns',
                array(&$this, 'modify_show_custom_fields_as_order_columns'),
                10,
                1
            );
            add_filter(
                'tc_ticket_type_id',
                array(&$this, 'modify_ticket_type_id'),
                10,
                1
            );
            add_filter(
                'tc_checkout_owner_info_ticket_title',
                array(&$this, 'modify_checkout_owner_info_ticket_title'),
                10,
                2
            );
            add_filter(
                'tc_csv_payment_statuses',
                array(&$this, 'modify_csv_payment_statuses'),
                10,
                1
            );
            add_filter(
                'tc_export_order_number_column_value',
                array(&$this, 'modify_export_order_number_column_value'),
                10,
                2
            );
            add_filter(
                'tc_order_payment_gateway_name',
                array(&$this, 'modify_order_payment_gateway_name'),
                10,
                2
            );
            add_filter( 'tc_csv_admin_fields', array(&$this, 'tc_csv_admin_fields_woo_billing') );
            add_filter(
                'tc_order_status_title',
                array($this, 'modify_order_status_title'),
                10,
                3
            );
            add_filter(
                'tc_shortcodes',
                array(&$this, 'modify_shortcode_builder_list'),
                10,
                1
            );
            add_filter( 'tc_gutenberg_blocks', array($this, 'modify_gutenberg_blocks_list') );
            add_filter(
                'tc_event_shortcode',
                array(&$this, 'modify_event_shortcode'),
                10,
                1
            );
            add_filter(
                'tc_order_found',
                array(&$this, 'modify_order_found'),
                10,
                2
            );
            add_filter(
                'tc_ticket_type_admin_url',
                array(&$this, 'modify_ticket_type_admin_url'),
                10,
                1
            );
            add_filter(
                'tc_csv_array',
                array(&$this, 'tc_csv_array_woo_billing'),
                10,
                4
            );
            add_action( 'admin_enqueue_scripts', array($this, 'admin_header') );
            add_action( 'wp_enqueue_scripts', array(&$this, 'front_header') );
            add_action(
                'add_meta_boxes',
                array($this, 'add_meta_boxes'),
                30,
                2
            );
            add_action( 'pre_get_posts', array(&$this, 'tc_custom_pre_get_posts_query'), 10 );
            add_action( 'template_redirect', array(&$this, 'tc_redirect_ticket_single_to_event') );
            add_action( 'init', array(&$this, 'replace_shortcodes'), 11 );
            add_action( 'init', array(&$this, 'load_plugin_textdomain'), 11 );
            add_action( 'admin_init', array(&$this, 'check_order_deletion') );
            add_action( 'tc_load_ticket_template_elements', array($this, 'load_ticket_template_elements') );
            // Seat Charts
            add_filter( 'tc_seat_chart_add_to_cart_url', array(&$this, 'modify_seat_chart_add_to_cart_url') );
            add_filter( 'tc_seat_chart_checkout_url', array(&$this, 'modify_seat_chart_checkout_url') );
            add_filter( 'tc_seat_chart_in_cart_count', array(&$this, 'modify_seat_chart_in_cart_count') );
            add_filter( 'tc_seat_chart_cart_subtotal', array(&$this, 'modify_seat_chart_cart_subtotal') );
            add_filter( 'tc_is_woo', array(&$this, 'modify_tc_seat_charts_is_woo') );
            add_filter(
                'tc_seat_charts_get_reserved_seats_order_statuses',
                array($this, 'modify_seat_charts_get_reserved_seats_post_status'),
                10,
                2
            );
            add_filter(
                'tc_seat_chart_shortcode_price',
                array(&$this, 'modify_price_per_ticket'),
                10,
                4
            );
            add_action(
                'manage_product_posts_custom_column',
                array(&$this, 'change_product_post_link_in_the_admin'),
                10,
                2
            );
            // Event Level Identifiers
            add_action( 'woocommerce_before_add_to_cart_form', array($this, 'manage_before_add_to_cart_form') );
            add_filter(
                'woocommerce_is_purchasable',
                array($this, 'manage_product_availability'),
                10,
                2
            );
            add_filter(
                'woocommerce_admin_stock_html',
                array($this, 'manage_product_availability'),
                10,
                2
            );
            add_filter(
                'woocommerce_get_availability_text',
                array($this, 'manage_product_availability'),
                10,
                2
            );
            add_action(
                'woocommerce_single_product_summary',
                array($this, 'manage_product_availability'),
                30,
                2
            );
            // Search product filter
            add_filter(
                'woocommerce_product_filters',
                array(&$this, 'tc_ticket_filter_products'),
                10,
                2
            );
            add_filter( 'parse_query', array(&$this, 'tc_ticket_filter_get_products'), 10 );
            add_filter( 'request', array(&$this, 'tc_ticket_search_get_products'), 20 );
            if ( !defined( 'TC_WOO_DISABLE_FRAGMENTS_UPDATE' ) ) {
                add_action( 'woocommerce_update_order_review_fragments', array($this, 'update_order_review_fragments'), 99 );
            }
            // Execute when e-mail verification is enabled
            if ( isset( $this->tc_general_settings['email_verification_buyer_owner'] ) && 'yes' == $this->tc_general_settings['email_verification_buyer_owner'] ) {
                add_filter( 'woocommerce_checkout_fields', array(&$this, 'tc_woo_email_verification_buyer') );
                add_action( 'woocommerce_checkout_process', array(&$this, 'tc_woo_matching_email') );
            }
            add_action(
                'woocommerce_saved_order_items',
                array($this, 'update_order_ticket_instances'),
                10,
                2
            );
            add_action(
                'woocommerce_new_order_item',
                array($this, 'create_order_ticket_instances'),
                11,
                3
            );
            add_action(
                'woocommerce_before_delete_order_item',
                array($this, 'remove_order_ticket_instances'),
                10,
                1
            );
            // Allow Checkins for Woocommerce Subscription Products.
            if ( is_plugin_active( 'woocommerce-subscriptions/woocommerce-subscriptions.php' ) && is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
                add_filter( 'tc_checkin_output_data', array(&$this, 'additional_subscription_checkin_validation'), 20 );
            }
            // Email Notifications
            add_action(
                'woocommerce_order_status_refunded',
                array(&$this, 'tc_send_admin_order_email_notification'),
                11,
                1
            );
            add_action(
                'woocommerce_order_status_completed',
                array(&$this, 'tc_send_admin_order_email_notification'),
                11,
                1
            );
            // Downloadable tickets validation. If invalid, redirect user to order received/details page.
            add_filter(
                'tc_validate_downloadable_ticket_order_status',
                array($this, 'woo_validate_downloadable_ticket_order_status'),
                10,
                2
            );
            add_action( 'tc_after_invalid_downloadable_ticket', array(&$this, 'woo_after_invalid_downloadable_ticket') );
            // Hide attendees tickets that are not checkin eligible. Applies in Attendees & Tickets page
            add_filter( 'tc_checkin_eligible_order_statuses', array($this, 'modify_checkin_eligible_order_statuses') );
            add_filter(
                'tc_tickets_instances_column_value',
                array($this, 'tc_tickets_instances_column_value'),
                11,
                3
            );
            add_filter(
                'tc_tickets_instances_order_status_where_clause',
                array($this, 'tc_tickets_instances_order_status_where_clause'),
                11,
                2
            );
            add_filter(
                'tc_tickets_instances_order_status_where',
                array($this, 'tc_tickets_instances_order_status_where_clause'),
                11,
                3
            );
            // Render add to cart button.
            add_action( 'wp_ajax_nopriv_tc_wb_add_to_cart', array(&$this, 'render_add_to_cart_button') );
            add_action( 'wp_ajax_tc_wb_add_to_cart', array(&$this, 'render_add_to_cart_button') );
            // Available Checkins
            add_filter( 'tc_available_checkins_per_ticket_field_name', array($this, 'tc_modify_available_checkins_per_ticket_field_name') );
            // Checkins on time basis
            add_filter( 'tc_checkins_time_basis_field_name', array($this, 'tc_modify_checkins_time_basis_field_name') );
            add_filter( 'tc_allowed_checkins_per_time_basis_field_name', array($this, 'tc_modify_allowed_checkins_per_time_basis_field_name') );
            add_filter( 'tc_checkins_time_basis_type_field_name', array($this, 'tc_modify_checkins_time_basis_type_field_name') );
            add_filter( 'tc_checkins_time_calendar_basis_field_name', array($this, 'tc_modify_checkins_time_calendar_basis_field_name') );
            // Add to cart shortcode
            add_filter(
                'woocommerce_loop_add_to_cart_link',
                array($this, 'render_add_to_cart_shortcode'),
                20,
                3
            );
            add_filter(
                'shortcode_atts_product_add_to_cart',
                array($this, 'modify_add_to_cart_shortcode_attributes'),
                20,
                4
            );
            // Cart/Checkout Validation
            add_action( 'woocommerce_check_cart_items', array($this, 'woocommerce_cart_items_validation') );
            add_action( 'woocommerce_before_checkout_process', array($this, 'woocommerce_checkout_validation') );
            /**
             * Woocommerce blocks Integration
             * Register Inner Blocks for Woocommerce Blocks.
             * Generate tickets from the store api data.
             * Triggers on newly created orders.
             *
             * @since 1.4.5
             */
            add_action( 'woocommerce_blocks_loaded', array($this, 'init_block_integration') );
            add_action(
                'woocommerce_store_api_checkout_update_order_from_request',
                array($this, 'create_order_ticket_instances_from_store_api'),
                10,
                2
            );
            add_filter(
                'tcwb_checkout_fields_script_data',
                array($this, 'checkout_block_confirmation_email_field'),
                10,
                2
            );
            add_action( 'woocommerce_store_api_checkout_order_processed', array($this, 'woocommerce_checkout_block_validation') );
            add_action( 'woocommerce_store_api_cart_errors', array($this, 'woocommerce_cart_block_error') );
            // High-performance order storage is currently not supported.
            // add_action( 'before_woocommerce_init', array( $this, 'hpos_compatibility_notice' ) );
            add_action(
                'tc_order_status',
                array($this, 'get_woo_order_status'),
                10,
                2
            );
            add_action( 'tc_order_types', array($this, 'get_woo_order_types') );
            // Attendees & Tickets search meta keys
            add_filter( 'tc_tickets_instances_extended_search_meta_keys', array($this, 'additional_tickets_instances_extended_search_meta_keys') );
            // Remove session data
            add_action( 'woocommerce_cart_emptied', array($this, 'remove_session_data'), 10 );
            add_action( 'woocommerce_cart_is_empty', array($this, 'remove_session_data'), 10 );
            add_filter(
                'tcsc_ticket_type_is_used_for_seatings',
                array($this, 'ticket_type_is_used_for_seatings'),
                10,
                2
            );
            remove_action( 'plugins_loaded', array($tc, 'load_payment_gateway_addons'), 8 );
            add_action( 'tc_after_bulk_delete_ticket_order', array($this, 'delete_ticket_order') );
            // TO DO: add_action('woocommerce_product_after_variable_attributes', array($this, 'woo_ticket_type_additional_meta_variations'), 10, 3);
            // On activate or decativate plugin run function using this hook
            register_activation_hook( __FILE__, array(&$this, 'tc_bridge_is_activate') );
            register_deactivation_hook( __FILE__, array(&$this, 'tc_bridge_is_deactivate') );
        }

        /**
         * Load this plugin the very last to ensure all the dependencies loaded first.
         */
        function plugin_dependencies() {
            $plugin_slug = basename( __DIR__ ) . '/' . basename( __FILE__ );
            $active_plugins = get_option( 'active_plugins', [] );
            $index = array_search( $plugin_slug, $active_plugins );
            if ( $index !== false ) {
                unset($active_plugins[$index]);
                $active_plugins[] = $plugin_slug;
                update_option( 'active_plugins', array_values( $active_plugins ) );
            }
        }

        /**
         * @param $order_id
         * @return void
         */
        function delete_ticket_order( $order_id ) {
            try {
                $order = new WC_Order($order_id);
                $order->delete( true );
            } catch ( Exception $e ) {
                // Order might no longer exists.
            }
        }

        /**
         * Retrieve WooCommerce order types
         * @return array
         */
        function get_woo_order_types() {
            return ['shop_order', 'shop_order_placehold'];
        }

        /**
         * Retrieve Wocoommerce order status.
         * @param $order_status
         * @param $order_id
         * @return string
         */
        function get_woo_order_status( $order_status, $order_id ) {
            if ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ) {
                try {
                    $order = new WC_Order($order_id);
                    return $order->get_status();
                } catch ( Exception $e ) {
                    // Do nothing for now.
                }
            }
            return $order_status;
        }

        /**
         * Identify a variation's parent `Used for Seatings` option whether it is set or not.
         *
         * @param $is_seat
         * @param $ticket_type_id
         * @return mixed
         *
         * @since 1.6.3
         */
        function ticket_type_is_used_for_seatings( $is_seat, $ticket_type_id ) {
            if ( 'variation' == tc_wb_get_product_type( $ticket_type_id ) ) {
                $variation_product = wc_get_product( $ticket_type_id );
                $product_id = $variation_product->get_parent_id();
                $is_seat = get_post_meta( $product_id, '_tc_used_for_seatings', true );
                $is_seat = ( 'yes' == $is_seat ? true : false );
            }
            return $is_seat;
        }

        /**
         * Remove cart/checkout block session data.
         * @since 1.5.7
         */
        function remove_session_data() {
            global $tc;
            $tc->session->drop( 'tcwb-checkout-fields' );
        }

        /**
         * Extra validation processes in Woocommerce cart/checkout blocks.
         * Validate event limit
         * @param $errors
         */
        function woocommerce_cart_block_error( $errors ) {
            self::event_limit_cart_validation();
        }

        /**
         * @param $order
         */
        function woocommerce_checkout_block_validation( $order ) {
            // self::event_limit_cart_validation();
            $order_id = $order->get_id();
            foreach ( $order->get_items() as $item ) {
                /**
                 * Use this hook to skip cart item loop.
                 * @since 1.6.2
                 */
                if ( !apply_filters( 'tc_skip_cart_item_loop', false, $item ) ) {
                    $product_id = $item->get_product_id();
                    $is_ticket = get_post_meta( $product_id, '_tc_is_ticket', true );
                    if ( 'yes' == $is_ticket && !metadata_exists( 'post', $order_id, 'tc_cart_contents' ) ) {
                        wc_add_notice( __( 'Sorry, something went wrong. Please refresh the page.', 'woocommerce-tickera-bridge' ), 'error' );
                        break;
                    }
                }
            }
        }

        /**
         * Extra validation process in Woocommerce checkout after clicking the place order button.
         * Validate event limit
         */
        function woocommerce_checkout_validation() {
            // Validate event limit
            $error = self::event_limit_cart_validation();
            if ( $error ) {
                throw new Exception(__( $error, "woocommerce-tickera-bridge" ));
            }
        }

        /**
         * Extra validation process in Woocommerce cart items.
         * Validate event limit
         *
         * Action(s):
         * Hide proceed to checkout button if error found.
         */
        function woocommerce_cart_items_validation() {
            if ( is_cart() || is_checkout() ) {
                $error = self::event_limit_cart_validation();
                if ( $error ) {
                    remove_action( 'woocommerce_proceed_to_checkout', 'wc_get_pay_buttons', 20 );
                    remove_action( 'woocommerce_proceed_to_checkout', 'woocommerce_button_proceed_to_checkout', 20 );
                }
            }
        }

        /**
         * Additional meta keys to search for in Tickera > Attendees & Tickets admin page
         *
         * @param $keys
         * @return array
         *
         * @since 1.5.3
         */
        function additional_tickets_instances_extended_search_meta_keys( $keys ) {
            $keys[] = '_billing_first_name';
            $keys[] = '_billing_last_name';
            $keys[] = '_billing_email';
            return $keys;
        }

        /**
         * High-performance order storage is currently not supported.
         * Show notice at Woocommerce > Settings > Advanced > Features.
         */
        function hpos_compatibility_notice() {
            if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
                \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, false );
            }
        }

        /**
         * Populate Confirmation Field onto the checkout block script data.
         *
         * @param $data
         * @param $session
         * @return mixed
         *
         * @since 1.4.5
         */
        function checkout_block_confirmation_email_field( $data, $session ) {
            $general_settings = get_option( 'tickera_general_setting' );
            $show_attendee_fields = ( isset( $general_settings['show_owner_fields'] ) && 'yes' == $general_settings['show_owner_fields'] ? true : false );
            $owner_email = ( isset( $general_settings['show_owner_email_field'] ) && 'yes' == $general_settings['show_owner_email_field'] ? true : false );
            $confirm_email = ( isset( $general_settings['email_verification_buyer_owner'] ) && 'yes' == $general_settings['email_verification_buyer_owner'] ? true : false );
            // Bypass the process if attendee fields is disabled.
            if ( !$show_attendee_fields ) {
                return $data;
            }
            foreach ( $data['owner_fields'] as $ticket_type_id => $fields ) {
                foreach ( $fields as $key => $field ) {
                    if ( $confirm_email && $owner_email ) {
                        switch ( $field['field_name'] ) {
                            case 'owner_email':
                                unset($data['owner_fields'][$ticket_type_id][$key]);
                                break;
                            case 'owner_confirm_email':
                                $data['owner_fields'][$ticket_type_id][$key]['field_type'] = 'confirm_email';
                                break;
                        }
                    } elseif ( 'owner_email' == $field['field_name'] ) {
                        $data['owner_fields'][$ticket_type_id][$key]['field_type'] = 'email';
                    }
                }
                $data['owner_fields'][$ticket_type_id] = array_values( $data['owner_fields'][$ticket_type_id] );
            }
            return $data;
        }

        /**
         * Woocommerce block integration.
         *
         * @since 1.4.5
         */
        function init_block_integration() {
            /**
             * Register Block Integration
             * @param $integration_registry
             */
            require_once $this->plugin_dir . 'includes/classes/class.block-integration.php';
            add_action( 'woocommerce_blocks_checkout_block_registration', function ( $integration_registry ) {
                $integration_registry->register( new TC_WB_Blocks_Integration() );
            } );
            /**
             * Initialize store endpoint extension
             */
            require_once $this->plugin_dir . 'includes/classes/class.store-endpoint.php';
            TC_WB_Extend_Store_Endpoint::init();
        }

        /**
         * Woocommerce block - Create Ticket Instances
         *
         * @param WC_Order $order
         * @param WP_REST_Request $request
         * @throws Exception
         */
        function create_order_ticket_instances_from_store_api( \WC_Order $order, \WP_REST_Request $request ) {
            $extension_checkout_fields = ( isset( $request['extensions']['tickera/checkout-fields'] ) ? $request['extensions']['tickera/checkout-fields'] : [] );
            if ( $extension_checkout_fields ) {
                remove_action( 'woocommerce_new_order', array($this, 'tc_order_created') );
                remove_action( 'woocommerce_resume_order', array($this, 'tc_order_created') );
                // Delete previously created tickets. This function is being executed multiple times on failed checkout process.
                $this->delete_associated_tickets( $order->get_id(), false );
                /**
                 * Update order tickets meta
                 * @since 1.5.8
                 *
                 * As per Woocommerce 9.2.3 `woocommerce_new_order` and `woocommerce_resume_order` are being executed prior to `woocommerce_store_api_checkout_update_order_from_request`.
                 * Therefore, there's no need to execute `tc_order_created`.
                 * @since 1.5.9
                 */
                // $this->tc_order_created( $order->get_id(), false );
                $buyer_data = [];
                $owner_data = [];
                foreach ( $order->get_items() as $item ) {
                    /**
                     * Use this hook to skip cart item loop.
                     * @since 1.6.2
                     */
                    if ( !apply_filters( 'tc_skip_cart_item_loop', false, $item ) ) {
                        $product_id = $item->get_product_id();
                        $variation_id = $item->get_variation_id();
                        $ticket_type_id = ( !$variation_id ? $product_id : $variation_id );
                        $data = [];
                        foreach ( $extension_checkout_fields as $key => $value ) {
                            if ( false === strpos( $key, 'buyer_data_' ) ) {
                                $exploded_key = explode( '_', $key );
                                if ( $exploded_key[0] == $ticket_type_id ) {
                                    $occurrence = $exploded_key[1];
                                    // Get field name from key string
                                    $field_name = str_replace( ['buyer_data_', '_post_meta'], ['', ''], $key );
                                    $exploded_field_name = explode( '_', $field_name );
                                    $field_name = implode( '_', array_slice( $exploded_field_name, 2 ) );
                                    // Option exists if $option is greater than -1
                                    $option = ( 'option' == $exploded_field_name[count( $exploded_field_name ) - 2] ? $exploded_field_name[count( $exploded_field_name ) - 1] : -1 );
                                    $key = ( $option > -1 ? 'owner_data_' . implode( '_', array_slice( $exploded_field_name, 2, -2 ) ) . '_post_meta' : 'owner_data_' . $field_name . '_post_meta' );
                                    // Only applies to owner data
                                    $data[$key][$ticket_type_id][$occurrence] = ( isset( $data[$key][$ticket_type_id][$occurrence] ) && $data[$key][$ticket_type_id][$occurrence] ? ( $value ? $data[$key][$ticket_type_id][$occurrence] . ',' . $value : $data[$key][$ticket_type_id][$occurrence] ) : $value );
                                    $owner_data[$key][$ticket_type_id][$occurrence] = $data[$key][$ticket_type_id][$occurrence];
                                }
                            } else {
                                // Get field name from key string
                                $field_name = str_replace( ['buyer_data_', '_post_meta'], ['', ''], $key );
                                $exploded_field_name = explode( '_', $field_name );
                                // Option exists if $option is greater than -1
                                $option = ( 'option' == $exploded_field_name[count( $exploded_field_name ) - 2] ? $exploded_field_name[count( $exploded_field_name ) - 1] : -1 );
                                $key = ( $option > -1 ? implode( '_', array_slice( $exploded_field_name, 0, -2 ) ) . '_post_meta' : $field_name . '_post_meta' );
                                // Applies to buyer data
                                $data[$key] = ( isset( $data[$key] ) && $data[$key] ? ( $value ? $data[$key] . ',' . $value : $data[$key] ) : $value );
                                $buyer_data[$key] = $data[$key];
                            }
                        }
                        // Create tickets
                        $this->create_order_ticket_instances(
                            $item->get_id(),
                            null,
                            $order->get_id(),
                            $data,
                            true
                        );
                    }
                }
                // Update Order Cart Info
                $order->update_meta_data( 'tc_cart_info', [
                    'buyer_data' => tickera_sanitize_array( $buyer_data, false, true ),
                    'owner_data' => tickera_sanitize_array( $owner_data, false, true ),
                ] );
            }
        }

        /**
         * Send attendee order email notification when order status is updated to completed.
         *
         * @param $order_id
         * @param $order
         *
         * @since 1.4.4
         */
        function send_attendee_order_email_notification( $order_id, $order ) {
            if ( $this->allowed_tickets_download( $order ) ) {
                do_action( 'tc_wb_maybe_send_attendee_order_completed_email', $order );
            }
        }

        /**
         * Insert additional Quantity Selector within Woocommerce add_to_cart shortcode.
         * Recreates the shortcode content using Woocommerce core methods to print the quantity selector and the product price.
         *
         * @param $html
         * @param $product
         * @param $args
         * @return string
         *
         * @since 1.4.3
         */
        function render_add_to_cart_shortcode( $html, $product, $args = '' ) {
            if ( isset( $args['quantity'] ) && 'true' == $args['quantity'] ) {
                $is_ticket = $product->get_meta( '_tc_is_ticket', true );
                if ( 'yes' == $is_ticket && 'variable' != $product->get_type() && 'instock' == $product->get_stock_status() ) {
                    $quantity_selector = woocommerce_quantity_input( array(
                        'min_value'   => $product->get_min_purchase_quantity(),
                        'max_value'   => $product->get_max_purchase_quantity(),
                        'input_value' => $product->get_min_purchase_quantity(),
                        'classes'     => 'tc-wb-quantity-selector',
                    ), $product, false );
                    $add_to_cart = sprintf(
                        '<a href="%s" data-quantity="1" class="%s" %s>%s</a>',
                        esc_url( $product->add_to_cart_url() ),
                        esc_attr( ( isset( $args['class'] ) ? $args['class'] : 'button' ) ),
                        ( isset( $args['attributes'] ) ? wc_implode_html_attributes( $args['attributes'] ) : '' ),
                        esc_html( $product->add_to_cart_text() )
                    );
                    $html = '<div class="tc-wb-event-dropdown-wrap"><div class="inner-wrap">' . $product->get_price_html() . $quantity_selector . '<div class="tc-wb-add-to-cart">' . $add_to_cart . '</div></div></div>';
                }
            }
            return $html;
        }

        /**
         * Hide the default Woocommerce add_to_cart shortcode price element if quantity is set to true or enabled.
         * The price is being recreated at render_add_to_cart_shortcode function.
         * A "tc-amount-hidden" class added to hide the amount via css if the quantity is enabled while show_price is disabled.
         *
         * @param $out
         * @param $pairs
         * @param $atts
         * @param $shortcode
         * @return mixed
         *
         * @since 1.4.3
         */
        function modify_add_to_cart_shortcode_attributes(
            $out,
            $pairs,
            $atts,
            $shortcode
        ) {
            $product_id = (int) $out['id'];
            $product = wc_get_product( $product_id );
            if ( $product ) {
                $is_ticket = $product->get_meta( '_tc_is_ticket', true );
                if ( 'yes' == $is_ticket ) {
                    if ( 'variable' != $product->get_type() ) {
                        if ( isset( $out['quantity'] ) && 'true' == $out['quantity'] ) {
                            if ( isset( $out['show_price'] ) && 'true' == $out['show_price'] ) {
                                $out['show_price'] = false;
                            } else {
                                $out['class'] .= 'tc-amount-hidden';
                            }
                        }
                    }
                }
            }
            return $out;
        }

        /**
         * Hide Attendees Tickets that are not checkin eligible.
         * Applies in Attendees & Tickets page
         *
         * @param $order_statuses
         * @return array|mixed
         */
        function modify_checkin_eligible_order_statuses( $order_statuses ) {
            $hide_checkin_ineligible = ( isset( $this->tc_general_settings['hide_checkin_ineligible_tickets'] ) ? $this->tc_general_settings['hide_checkin_ineligible_tickets'] : 'no' );
            if ( class_exists( 'WooCommerce' ) && function_exists( 'wc_get_order_statuses' ) ) {
                return ( 'no' == $hide_checkin_ineligible ? wc_get_order_statuses() : [
                    'wc-completed'  => wc_get_order_statuses()['wc-completed'],
                    'wc-processing' => wc_get_order_statuses()['wc-processing'],
                ] );
            }
            return $order_statuses;
        }

        /**
         * Collect order status from
         * Hide Attendees Tickets that are not checkin eligible.
         * Applies in Attendees & Tickets page
         *
         * @param $where
         * @param $order_statuses
         * @param bool $filter
         * @return array
         * @since 1.6.4
         */
        function tc_tickets_instances_order_status_where_clause( $where, $order_statuses, $filter = false ) {
            global $wpdb;
            $hpos = tc_wb_hpos();
            $order_statuses = ( $filter ? '\'trash\',' . $order_statuses : $order_statuses );
            if ( $hpos->status ) {
                return "{$wpdb->posts}.post_parent IN ( SELECT wco.id FROM {$wpdb->prefix}wc_orders wco WHERE wco.status IN (" . $order_statuses . ") )";
            } else {
                return "{$wpdb->posts}.post_parent IN ( SELECT ID FROM {$wpdb->posts} WHERE post_status IN (" . $order_statuses . ") )";
            }
        }

        /**
         * Attendees & Tickets Order Status from wc_orders table.
         *
         * @param $value
         * @param $field_id
         * @param $initial_value
         * @return mixed|string
         * @since 1.6.4
         */
        function tc_tickets_instances_column_value( $value, $field_id, $initial_value ) {
            global $wpdb;
            $hpos = tc_wb_hpos();
            if ( $hpos->status && 'order_status' == $field_id ) {
                $status_color = array(
                    'cancelled'  => 'tc_order_cancelled',
                    'completed'  => 'tc_order_paid',
                    'processing' => 'tc_order_received',
                    'pending'    => 'tc_order_received',
                    'on-hold'    => 'tc_order_hold',
                    'refunded'   => 'tc_order_fraud',
                    'failed'     => 'tc_order_fraud',
                );
                $order_id = wp_get_post_parent_id( $initial_value );
                $order = wc_get_order( $order_id );
                $status = $order->get_status();
                $color = ( isset( $status_color[$status] ) ? $status_color[$status] : 'tc_order_received' );
                $value = ( strrpos( $status, '_' ) ? str_replace( '_', ' ', $status ) : str_replace( '-', ' ', $status ) );
                $value = ucwords( $value );
                $value = sprintf( 
                    /* translators: 1: Order status color identifier 2: Order status name */
                    __( '<span class="%1$s">%2$s</span>', 'woocommerce-tickera-bridge' ),
                    esc_attr( $color ),
                    esc_html( ucwords( $value ) )
                 );
            }
            return $value;
        }

        /**
         * Validate downloadable tickets in frontend.
         * (e.g tickets from email or any ticket links from the website pages )
         *
         * @param $order_status
         * @param $order
         * @return array
         */
        function woo_validate_downloadable_ticket_order_status( $order_status, $order ) {
            if ( !in_array( $order->details->post_type, self::get_woo_order_types() ) ) {
                return $order_status;
            }
            $wc_order = wc_get_order( $order->details->ID );
            $order_status[] = 'completed';
            if ( $this->allowed_tickets_download( $wc_order ) ) {
                $order_status[] = 'processing';
            }
            return $order_status;
        }

        /**
         * Redirect customers to Order Received/Details page if the downloadable tickets failed the validation.
         * Validates order status
         *
         * @param $order_id
         */
        function woo_after_invalid_downloadable_ticket( $order_id ) {
            $order = new WC_Order($order_id);
            $order_details_page = $order->get_checkout_order_received_url();
            ob_start();
            @wp_redirect( $order_details_page );
            @tc_js_redirect( $order_details_page );
            exit;
        }

        /**
         * Validates customer subscription.
         * Additional validation before check-in tickets.
         *
         * @param $data
         * @return mixed
         */
        function additional_subscription_checkin_validation( $data ) {
            if ( $data['status'] && $data['pass'] ) {
                $checksum = sanitize_text_field( $data['checksum'] );
                $ticket_instance_id = tickera_ticket_code_to_id( $checksum );
                $order_id = wp_get_post_parent_id( $ticket_instance_id );
                $tc_checkin = get_post_meta( $ticket_instance_id, 'tc_checkins', true );
                if ( $tc_checkin && 'Pass' == $tc_checkin[count( $tc_checkin ) - 1]['status'] ) {
                    $ticket_type_id = get_post_meta( $ticket_instance_id, 'ticket_type_id', true );
                    $subsciptions_instances = get_posts( [
                        'posts_per_page' => -1,
                        'post_parent'    => $order_id,
                        'post_type'      => 'shop_subscription',
                        'post_status'    => 'any',
                    ] );
                    foreach ( $subsciptions_instances as $instance ) {
                        $subscription_id = $instance->ID;
                        $subscription = new WC_Subscription($subscription_id);
                        foreach ( $subscription->get_items() as $item ) {
                            /**
                             * Use this hook to skip cart item loop.
                             * @since 1.6.2
                             */
                            if ( !apply_filters( 'tc_skip_cart_item_loop', false, $item ) ) {
                                $_product = $item->get_product();
                                if ( $ticket_type_id == $_product->get_id() && 'active' != $subscription->get_status() ) {
                                    $data['status'] = false;
                                    $tc_checkin[count( $tc_checkin ) - 1]['status'] = 'Fail';
                                    update_post_meta( $ticket_instance_id, 'tc_checkins', $tc_checkin );
                                }
                            }
                        }
                    }
                }
            }
            return $data;
        }

        /**
         * Send Admin Order Email Notification
         *
         * @param $order_id
         * @return bool
         */
        function tc_send_admin_order_email_notification( $order_id ) {
            $tc_tickets_instances = Tickera\TC_Orders::get_tickets_ids( $order_id );
            if ( !$tc_tickets_instances ) {
                return false;
            }
            $order = new WC_Order($order_id);
            $order_status = $order->get_status();
            $tc_email_settings = get_option( 'tickera_email_setting', false );
            $enabled = ( isset( $tc_email_settings["send_admin_woo_order_{$order_status}_message"] ) ? $tc_email_settings["send_admin_woo_order_{$order_status}_message"] : 'yes' );
            $subject = ( isset( $tc_email_settings["admin_woo_order_{$order_status}_subject"] ) ? $tc_email_settings["admin_woo_order_{$order_status}_subject"] : __( 'Order ' . ucfirst( $order_status ), 'woocommerce-tickera-bridge' ) );
            $to_email = ( isset( $tc_email_settings["admin_woo_order_{$order_status}_to_email"] ) ? $tc_email_settings["admin_woo_order_{$order_status}_to_email"] : get_option( 'admin_email' ) );
            $email_content = ( isset( $tc_email_settings["admin_woo_order_{$order_status}_message_content"] ) ? $tc_email_settings["admin_woo_order_{$order_status}_message_content"] : "Hello, <br /><br />An order (ORDER_ID) totalling <strong>ORDER_TOTAL</strong> has been {$order_status}. <br /><br />You can check the order details here ORDER_DETAILS_URL" );
            if ( 'yes' == $enabled ) {
                /**
                 * Declare WP Mail Header Email From
                 */
                add_filter( 'wp_mail_from', function () {
                    return get_option( 'admin_email' );
                }, 999 );
                /**
                 * Declare WP Mail Header From Name
                 */
                if ( 'refunded' == $order_status ) {
                    add_filter( 'wp_mail_from_name', function () {
                        $tc_email_settings = get_option( 'tickera_email_setting', false );
                        return ( isset( $tc_email_settings['admin_woo_order_refunded_from_name'] ) ? $tc_email_settings['admin_woo_order_refunded_from_name'] : get_option( 'blogname' ) );
                    }, 999 );
                } elseif ( 'completed' == $order_status ) {
                    add_filter( 'wp_mail_from_name', function () {
                        $tc_email_settings = get_option( 'tickera_email_setting', false );
                        return ( isset( $tc_email_settings['admin_woo_order_completed_from_name'] ) ? $tc_email_settings['admin_woo_order_completed_from_name'] : get_option( 'blogname' ) );
                    }, 999 );
                }
                // Get Ticket Event Details
                $event_titles = [];
                $event_locations = [];
                foreach ( $tc_tickets_instances as $ticket_instance_id ) {
                    $ticket_type_id = get_post_meta( $ticket_instance_id, 'ticket_type_id', true );
                    $product = wc_get_product( $ticket_type_id );
                    if ( 'variation' == $product->get_type() ) {
                        $product = wc_get_product( $product->get_parent_id() );
                    }
                    $is_ticket = $product->get_meta( '_tc_is_ticket', true );
                    if ( $is_ticket ) {
                        $event_id = get_post_meta( $ticket_instance_id, 'event_id', true );
                        $event_title = get_the_title( $event_id );
                        $event_location = get_post_meta( $event_id, 'event_location', true );
                        // Store event title in temporary storage for later use
                        if ( !in_array( $event_title, $event_titles ) ) {
                            $event_titles[] = $event_title;
                        }
                        // Store event location in temporary storage for later use
                        if ( !in_array( $event_location, $event_locations ) ) {
                            $event_locations[] = $event_location;
                        }
                    }
                }
                // Declare Placeholders and Values
                $placeholders = array(
                    'ORDER_ID',
                    'ORDER_TOTAL',
                    'ORDER_DETAILS_URL',
                    'BUYER_NAME',
                    'EVENT_NAME',
                    'EVENT_LOCATION',
                    'TICKETS_TABLE'
                );
                $placeholder_values = array(
                    $order->get_id(),
                    $order->get_currency() . $order->get_total(),
                    $order->get_view_order_url(),
                    $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                    implode( ' | ', $event_titles ),
                    implode( ' | ', $event_locations ),
                    tickera_get_tickets_table_email( $order_id )
                );
                // Replace Placeholders with actual order details
                $email_content = str_replace( apply_filters( "tc_order_{$order_status}_admin_email_placeholders", $placeholders ), apply_filters( "tc_order_{$order_status}_admin_email_placeholder_values", $placeholder_values ), $email_content );
                wp_mail(
                    $to_email,
                    $subject,
                    html_entity_decode( stripcslashes( apply_filters( "tc_order_{$order_status}_admin_email_message", wpautop( $email_content ) ) ) ),
                    apply_filters( "tc_order_{$order_status}_admin_email_headers", array('Content-Type: text/html; charset=UTF-8') )
                );
            }
        }

        /**
         * Added Confirm Email Address field
         *
         * @param $fields
         * @return mixed
         */
        function tc_woo_email_verification_buyer( $fields ) {
            $fields['billing']['billing_email']['class'] = array('form-row-first');
            $fields['billing']['billing_em_ver'] = array(
                'label'    => __( 'Confirm Email Address', 'woocommerce-tickera-bridge' ),
                'required' => true,
                'type'     => 'email',
                'class'    => array('form-row-last'),
                'clear'    => true,
                'priority' => 999,
            );
            return $fields;
        }

        /**
         * Show error when email addresses don't match
         */
        function tc_woo_matching_email() {
            $email1 = sanitize_email( $_POST['billing_email'] );
            $email2 = sanitize_email( $_POST['billing_em_ver'] );
            if ( !filter_var( $email2, FILTER_VALIDATE_EMAIL ) ) {
                // wc_add_notice( __( "Invalid billing confirmation email address", "woocommerce-tickera-bridge" ), "error" );
                throw new Exception(__( "Invalid billing confirmation email address", "woocommerce-tickera-bridge" ));
            }
            if ( $email2 != $email1 || !$email2 ) {
                // wc_add_notice( __( "Buyer email addresses don't match", "woocommerce-tickera-bridge" ), "error" );
                throw new Exception(__( "Buyer email addresses don't match", "woocommerce-tickera-bridge" ));
            }
        }

        /**
         * Remove csv remember option bridge active or not
         */
        function tc_bridge_is_activate() {
            delete_option( 'tc_atteende_keep_selection' );
        }

        function tc_bridge_is_deactivate() {
            delete_option( 'tc_atteende_keep_selection' );
        }

        /**
         * Show filter in product page
         *
         * @param $output
         * @return string
         */
        function tc_ticket_filter_products( $output ) {
            ob_start();
            $current_ticket_status = ( isset( $_REQUEST['tc_ticker_type'] ) ? wc_clean( wp_unslash( $_REQUEST['tc_ticker_type'] ) ) : false );
            // WPCS: input var
            $ticket_statuses = array(
                'no'  => __( 'Show all products', 'woocommerce' ),
                'yes' => __( 'Show tickets only', 'woocommerce' ),
            );
            $output .= '<select name="tc_ticker_type">';
            foreach ( $ticket_statuses as $status => $label ) {
                $output .= '<option ' . selected( $status, $current_ticket_status, false ) . ' value="' . esc_attr( $status ) . '">' . esc_html( $label ) . '</option>';
            }
            $output .= '</select>';
            $output .= ob_get_clean();
            return $output;
        }

        /**
         * Selected ticket filer
         *
         * @param $query
         */
        function tc_ticket_filter_get_products( $query ) {
            global $pagenow;
            // Get the post type
            $post_type = ( isset( $_GET['post_type'] ) ? $_GET['post_type'] : '' );
            if ( is_admin() && $pagenow == 'edit.php' && $post_type == 'product' && isset( $_GET['tc_ticker_type'] ) && $_GET['tc_ticker_type'] == 'yes' ) {
                $query->query_vars['meta_key'] = '_tc_is_ticket';
                $query->query_vars['meta_value'] = $_GET['tc_ticker_type'];
                $query->query_vars['meta_compare'] = '=';
            }
        }

        /**
         * Search tickets through Admin Woocommerce Product Lists
         *
         * @param $query_vars
         * @return mixed
         */
        function tc_ticket_search_get_products( $query_vars ) {
            global $typenow, $wpdb, $pagenow;
            if ( 'product' === $typenow && isset( $_GET['s'] ) && 'edit.php' === $pagenow ) {
                $meta_key = '_event_name';
                $search_term = esc_sql( sanitize_text_field( $_GET['s'] ) );
                // Split the search term by comma.
                $search_terms = explode( ',', $search_term );
                // If there are more terms make sure we also search for the whole thing, maybe it's not a list of terms.
                if ( count( $search_terms ) > 1 ) {
                    $search_terms[] = $search_term;
                }
                // Cleanup the array manually to avoid issues with quote escaping.
                array_walk( $search_terms, 'trim' );
                array_walk( $search_terms, 'esc_sql' );
                $search_term = implode( "','", $search_terms );
                $prepare_query = $wpdb->prepare( "SELECT DISTINCT pm.post_id as ticket_id FROM {$wpdb->posts} p LEFT JOIN {$wpdb->postmeta} AS pm ON p.ID = pm.meta_value WHERE p.post_title LIKE \"%{%s}%\" AND pm.meta_key = %s ORDER BY p.post_parent ASC, p.post_title ASC", $search_term, $meta_key );
                $search_results = $wpdb->get_results( $prepare_query );
                $product_ids = wp_parse_id_list( array_merge( wp_list_pluck( $search_results, 'ticket_id' ) ) );
                if ( isset( $query_vars['post__in'] ) ) {
                    $query_vars['post__in'] = array_merge( $product_ids, (array) $query_vars['post__in'] );
                }
            }
            return $query_vars;
        }

        function woo_ticket_type_additional_meta_variations( $loop, $variation_data, $variation ) {
            $variation_id = $variation->ID;
            $_ticket_type_variation_override_checkins = get_post_meta( $variation_id, '_ticket_type_variation_override_checkins', true );
            $_ticket_type_variation_override_checkins = ( !empty( $_ticket_type_variation_override_checkins ) ? $_ticket_type_variation_override_checkins : 'no' );
            $parent_product_id = wp_get_post_parent_id( $variation_id );
            $is_ticket_meta = get_post_meta( TC_WooCommerce_Bridge::maybe_get_original_wpml_product_id( (int) $parent_product_id ), '_tc_is_ticket', true );
            $is_ticket = ( $is_ticket_meta == 'yes' ? true : false );
            if ( $is_ticket ) {
                $display = 'style="display: block"';
            } else {
                $display = 'style="display: none"';
            }
            ?>
            <div class="tc_ticket_type_pd_variation_holder show_if_tc_ticket" <?php 
            echo $display;
            ?>>
                <p class="form-field form-row form-row-first form-row-first tc_ticket_type_pd_variation">
                    <label><?php 
            _e( 'Override "Check-ins per ticket" count', 'woocommerce-tickera-bridge' );
            ?></label>
                    <span class="woocommerce-help-tip" data-tip="<?php 
            echo esc_attr( __( 'Set "Yes" if number of check-ins per ticket variation is different from the parent ticket product', 'woocommerce-tickera-bridge' ) );
            ?>"></span>
                    <br />
                    <input type="radio" name="_ticket_type_variation_override_checkins[<?php 
            echo esc_attr( $loop );
            ?>]" value="no" <?php 
            checked( $_ticket_type_variation_override_checkins, 'no', true );
            ?>  /><?php 
            _e( 'No', 'woocommerce-tickera-bridge' );
            ?>
                    <input type="radio" name="_ticket_type_variation_override_checkins[<?php 
            echo esc_attr( $loop );
            ?>]" value="yes" <?php 
            checked( $_ticket_type_variation_override_checkins, 'yes', true );
            ?>  /><?php 
            _e( 'Yes', 'woocommerce-tickera-bridge' );
            ?>
                </p>
                <p class="form-field form-row form-row-last tc_ticket_type_pd_variation_available_checkins_per_ticket_field">
                    <label for="tc_ticket_type_pd_variation_available_checkins_per_ticket_field"><?php 
            _e( 'Check-ins per ticket', 'woocommerce-tickera-bridge' );
            ?></label>
                    <span class="woocommerce-help-tip" data-tip="<?php 
            echo esc_attr( __( 'Set number of check-ins allowed per ticket variation', 'woocommerce-tickera-bridge' ) );
            ?>"></span>
                    <input type="number" class="short" style="" name="tc_ticket_type_pd_variation_available_checkins_per_ticket_field" id="_available_checkins_per_ticket" value="" placeholder="<?php 
            echo esc_attr( __( 'Unlimited', 'woocommerce-tickera-bridge' ) );
            ?>">
                </p>
            </div>
            <?php 
        }

        /**
         * Additional Identifiers/Labels in both admin and frontend sides.
         * Display Ticket Quantity Limitation ( Event Level ) in admin product list page and product page.
         *
         * @param string $value
         * @param string $product
         * @return bool|string
         */
        function manage_product_availability( $value = '', $product = '' ) {
            if ( !$product ) {
                global $product;
            }
            $is_variation = ( 'variation' == $product->get_type() ? true : false );
            $product_id = ( $is_variation ? $product->get_parent_id() : $product->get_ID() );
            $ticket = new Tickera\TC_Ticket($product_id);
            $is_ticket = ( isset( $ticket->details->_tc_is_ticket ) ? $ticket->details->_tc_is_ticket : 'no' );
            // Make sure to process ticket types only.
            if ( 'no' == $is_ticket ) {
                return $value;
            }
            // If ticket is not saleable, mark the product out of stock.
            if ( doing_filter( 'woocommerce_is_purchasable' ) && !Tickera\TC_Ticket::is_sales_available( $product_id ) ) {
                // Remove the variation add to cart button in frontend product page.
                remove_action( 'woocommerce_single_variation', 'woocommerce_single_variation_add_to_cart_button', 20 );
                return false;
            }
            $event_id = get_post_meta( $product_id, '_event_name', true );
            $limit_level = get_post_meta( $event_id, 'limit_level', true );
            // Event Limit Level
            if ( $limit_level ) {
                $event_ticket_sold_count = $this->get_event_tickets_count_sold( $event_id );
                $limit_level_value = get_post_meta( $event_id, 'limit_level_value', true );
                $quantity_left = (int) $limit_level_value - (int) $event_ticket_sold_count;
                // Update product stock status to instock. Retain the quantity.
                if ( $product->managing_stock() ) {
                    update_post_meta( $product_id, '_manage_stock', 'no' );
                }
                // Make sure to inherit in stock functionality
                if ( 'instock' != $product->get_stock_status() ) {
                    update_post_meta( $product_id, '_stock_status', 'instock' );
                    wp_set_post_terms(
                        $product_id,
                        'instock',
                        'product_visibility',
                        true
                    );
                }
                // Not unlimited
                if ( '' !== $limit_level_value ) {
                    if ( doing_filter( 'woocommerce_is_purchasable' ) ) {
                        // Handles Add to cart button in the frontend
                        $value = ( $quantity_left > 0 ? true : false );
                    } else {
                        // Modify the stock status label in the frontend product page.
                        $value = ( $quantity_left > 0 ? '<p id="tc_before_variation_form" class="stock in-stock">' . $quantity_left . ' ' . __( 'in stock', 'woocommerce-tickera-bridge' ) . '</p>' : '<p id="tc_before_variation_form" class="stock out-of-stock">' . __( 'Out of stock', 'woocommerce-tickera-bridge' ) . '</p>' );
                        if ( $quantity_left <= 0 && did_action( 'woocommerce_single_product_summary' ) ) {
                            /*
                             * Print "Out of stock" status in frontend product page.
                             * Only applies to those none variation type ( e.g Simple ), to make sure it is being printed once.
                             */
                            if ( 'variation' != $product->get_type() ) {
                                echo $value;
                            }
                            // Remove the variation add to cart button in frontend product page.
                            remove_action( 'woocommerce_single_variation', 'woocommerce_single_variation_add_to_cart_button', 20 );
                        }
                    }
                    // Update inventory to out of stock
                    if ( $quantity_left <= 0 ) {
                        update_post_meta( $product_id, '_stock_status', 'outofstock' );
                        wp_set_post_terms(
                            $product_id,
                            'outofstock',
                            'product_visibility',
                            true
                        );
                    }
                }
                // Modify the "Stock" column in the admin product posts. Flag the status with "Event Level".
                if ( doing_filter( 'woocommerce_admin_stock_html' ) ) {
                    // Not unlimited
                    if ( '' !== $limit_level_value ) {
                        $value = ( $quantity_left > 0 ? '<mark class="instock">' . __( 'In stock', 'woocommerce-tickera-bridge' ) . '</mark> (' . $quantity_left . ')' : '<mark class="outofstock">' . __( 'Out of stock', 'woocommerce-tickera-bridge' ) . '</mark>' );
                    }
                    $value .= '<br><a href="' . admin_url( "post.php?post=" . esc_attr( $event_id ) . "&action=edit#limit_level-tc-metabox-wrapper", ( is_ssl() ? 'https' : 'http' ) ) . '"><small>' . __( 'Event level', 'woocommerce-tickera-bridge' ) . '</small></a>';
                }
            }
            return $value;
        }

        /**
         * Insert identifier for single variation page.
         * This identifier is being used in css to hide variation form.
         */
        function manage_before_add_to_cart_form() {
            global $product;
            if ( is_product() && $product && method_exists( $product, 'get_id' ) ) {
                $product_id = ( 'variation' == $product->get_type() ? $product->get_parent_id() : $product->get_id() );
                if ( !Tickera\TC_Ticket::is_sales_available( $product_id ) ) {
                    echo '<div id="tc_before_variation_form"></div>';
                }
            }
        }

        /**
         * Validate cart for event limit.
         * Return error if event limit has been reached.
         *
         * @return array
         */
        function event_limit_cart_validation() {
            global $tc;
            $items = [];
            $errors = [];
            $qty_count_per_event = [];
            /**
             * Collect from cart session/cookie
             */
            if ( WC()->cart && method_exists( WC()->cart, 'get_cart' ) ) {
                $items = WC()->cart->get_cart();
            }
            /**
             * Collect from Tickera session
             */
            if ( !$items && isset( $tc->session ) ) {
                $checkout_owner_fields = $tc->session->get( 'tcwb-checkout-fields' );
                $items = ( isset( $checkout_owner_fields['cart_items'] ) ? $checkout_owner_fields['cart_items'] : [] );
            }
            foreach ( $items as $key => $val ) {
                /**
                 * Use this hook to skip cart item loop.
                 * @since 1.6.2
                 */
                if ( !apply_filters( 'tc_skip_cart_item_loop', false, $val ) ) {
                    $product_id = (int) $val['product_id'];
                    $quantity = (int) $val['quantity'];
                    if ( $product_id ) {
                        $product_meta = get_post_meta( $product_id );
                        $is_ticket = ( isset( $product_meta['_tc_is_ticket'] ) ? $product_meta['_tc_is_ticket'][0] : 'no' );
                        if ( $is_ticket ) {
                            $event_id = get_post_meta( $product_id, '_event_name', true );
                            $limit_level = get_post_meta( $event_id, 'limit_level', true );
                            if ( $limit_level ) {
                                $limit_level_value = get_post_meta( $event_id, 'limit_level_value', true );
                                // Not unlimited
                                if ( '' !== $limit_level_value ) {
                                    // Count all committed ticket quantity per event
                                    $qty_count_per_event[$event_id] = @$qty_count_per_event[$event_id] + $quantity;
                                    $event_ticket_sold_count = $this->get_event_tickets_count_sold( $event_id );
                                    $quantity_left = ( $limit_level_value ? (int) $limit_level_value - (int) $event_ticket_sold_count : 9999 );
                                    // Retrieve the remaining available quantity
                                    if ( $qty_count_per_event[$event_id] >= $quantity_left ) {
                                        $quantity_left = $quantity - ($qty_count_per_event[$event_id] - $quantity_left);
                                    }
                                    if ( $quantity_left < $quantity ) {
                                        $tc_cart_errors = ( $quantity_left > 0 ? sprintf(
                                            __( 'Only %d "%s" %s left', 'woocommerce-tickera-bridge' ),
                                            $quantity_left,
                                            get_the_title( $product_id ),
                                            ( $quantity_left > 1 ? __( 'tickets', 'woocommerce-tickera-bridge' ) : __( 'ticket', 'woocommerce-tickera-bridge' ) )
                                        ) : sprintf( __( '"%s" tickets are sold out', 'woocommerce-tickera-bridge' ), get_the_title( $product_id ) ) );
                                        wc_add_notice( $tc_cart_errors, 'error' );
                                        $errors[] = $tc_cart_errors;
                                    }
                                }
                            }
                        }
                    }
                }
            }
            return $errors;
        }

        /**
         * Count the total number of sold event tickets.
         *
         * @param $event_id
         * @return int
         *
         * @since 1.4.3 Updated Query.
         */
        function get_event_tickets_count_sold( $event_id ) {
            global $wpdb;
            $event = new Tickera\TC_Event($event_id);
            $ticket_types = $event->get_event_ticket_types();
            $sold_count = 0;
            if ( count( $ticket_types ) > 0 ) {
                $query = "\n                        SELECT COUNT(*) as cnt\n                        FROM {$wpdb->posts} p\n                        INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id\n                        WHERE p.post_type='tc_tickets_instances'\n                            AND pm.meta_key = 'ticket_type_id'\n                            AND pm.meta_value IN (" . implode( ',', $ticket_types ) . ")\n                            AND (\n                                ( SELECT p2.post_status FROM {$wpdb->posts} p2 WHERE p2.ID=p.post_parent ) IN ( 'wc-completed', 'wc-processing', 'wc-pending' )\n                                OR ( SELECT wco.status FROM {$wpdb->prefix}wc_orders wco WHERE wco.id=p.post_parent ) IN ( 'wc-completed', 'wc-processing', 'wc-pending' )\n                            )\n                    ";
                $sold_count = $wpdb->get_var( $query );
            }
            return $sold_count;
        }

        /**
         * Handles and validates $_POST data.
         *
         * @param $_post
         * @param $item_id
         * @return array
         * @throws Exception
         */
        function handle_post( $_post, $item_id = null ) {
            /**
             * Maybe collect raw data
             * @since 1.4.4
             */
            if ( !$_post ) {
                $payload = @file_get_contents( 'php://input' );
                $payload = json_decode( $payload, true );
                $_post = $payload;
            }
            $new_post = [];
            /**
             * Compatibility for Woocommerce Paypal Payment plugin.
             *
             * Issue: When an order is created via paypal checkout, $_POST data is being modified.
             * Resolution: Parse and validate the form data before creating ticket instances.
             */
            foreach ( (array) $_post as $key => $value ) {
                if ( strpos( $key, '[' ) !== false ) {
                    if ( strpos( $key, 'owner_data' ) !== false || strpos( $key, 'buyer_data' ) !== false ) {
                        if ( strpos( $key, 'ticket_type_id' ) === false ) {
                            $key = str_replace( ']', '', $key );
                            $assoc_arr_levels = array_filter( explode( '[', $key ) );
                            $new_post[$assoc_arr_levels[0]][$assoc_arr_levels[1]][] = $value;
                        }
                    }
                } else {
                    $new_post[$key] = $value;
                }
            }
            /**
             * Compatibility for the following plugins:
             * Extendons: WooCommerce Product Add-ons - Custom Product Options Plugin
             * WPC Product Bundles for WooCommerce
             *
             * WooCommerce Product Bundles
             * @since 1.5.7
             *
             * YITH Booking and Appointment for WooCommerce
             * @since 1.6.0
             *
             * Issue: Incorrect allocation of Attendees' meta data.
             * Solution: Remove processed ticket instance and meta from the POST variable
             */
            if ( $item_id && (is_plugin_active( 'extendons_product_addons/extendons-product-addons.php' ) || is_plugin_active( 'woocommerce-product-bundles/woocommerce-product-bundles.php' ) || is_plugin_active( 'woo-product-bundle/wpc-product-bundles.php' ) || is_plugin_active( 'woo-product-bundle-premium/wpc-product-bundles.php' ) || is_plugin_active( 'yith-woocommerce-booking-premium/init.php' )) ) {
                $order_id = wc_get_order_id_by_order_item_id( $item_id );
                // Collect all existing tickets. To be compared in the next step.
                $_new_post = $new_post;
                $ticket_instances = Tickera\TC_Orders::get_tickets_ids( $order_id );
                foreach ( $ticket_instances as $ticket_instance_id ) {
                    $ticket_type_id = get_post_meta( $ticket_instance_id, 'ticket_type_id', true );
                    if ( $ticket_type_id ) {
                        foreach ( $new_post as $key => $value ) {
                            // Remove existing Ticket Instance IDs from the newly created POST/$new_post
                            if ( false !== strpos( $key, 'owner_data' ) && isset( $_new_post[$key] ) && isset( $_new_post[$key][$ticket_type_id] ) ) {
                                array_shift( $_new_post[$key][$ticket_type_id] );
                            }
                        }
                    }
                }
                $new_post = $_new_post;
            }
            return $new_post;
        }

        /**
         * Create Ticket Instances based on recently created order line items
         * @param int $item_id
         *
         * @param $item
         * @param int $order_id
         * @param bool|array $extension_data field keys, values
         * @since 1.4.5
         *
         * Don't create tickets until it is being processed in the checkout page.
         * Woocommerce Block, create the order cart page and triggers woocommerce_new_order_item hook.
         * @since 1.4.8
         *
         * Prevent store api process from creating tickets instances (e.g updating cart items and applying discount codes).
         * @since 1.5.0
         *
         * @return bool
         * @throws Exception
         */
        function create_order_ticket_instances(
            $item_id,
            $item,
            $order_id,
            $extension_data = [],
            $blocks = false
        ) {
            $_post = ( $extension_data ? $extension_data : $_POST );
            $_post = $this->handle_post( $_post, $item_id );
            /*
             * Don't create tickets until it is being processed in the checkout page.
             * Prevent store api process from creating tickets instances (e.g updating cart items and applying discount codes)
             */
            if ( !$extension_data && (!$_post || $_post && isset( $_post['requests'] )) ) {
                return;
            }
            // Collect Owner and Buyer Data
            $metas = [];
            foreach ( $_post as $field => $value ) {
                if ( preg_match( '/owner_data_/', $field ) ) {
                    $field_name = str_replace( 'owner_data_', '', $field );
                    // Collect Post Field Values
                    if ( preg_match( '/_post_title/', $field_name ) ) {
                        $value = sanitize_text_field( $value );
                        $title = $value;
                    } elseif ( preg_match( '/_post_excerpt/', $field_name ) ) {
                        $value = sanitize_textarea_field( $value );
                        $excerpt = $value;
                    } elseif ( preg_match( '/_post_content/', $field_name ) ) {
                        $value = sanitize_textarea_field( $value );
                        $content = $value;
                    } elseif ( preg_match( '/_post_meta/', $field_name ) ) {
                        $metas[] = str_replace( '_post_meta', '', $field_name );
                        $value = ( is_array( $value ) ? tickera_sanitize_array( $value, false, true ) : sanitize_text_field( $value ) );
                        ${str_replace( '_post_meta', '', $field_name )} = $value;
                    }
                }
            }
            $metas = array_merge( ['ticket_code', 'event_id', 'item_id'], $metas );
            // Replacing $item object. $item parameter returns both the order line product and shipping fees
            $item = new WC_Order_Item_Product($item_id);
            // Collection of IDs
            $variation_id = (int) $item->get_variation_id();
            $product_id = (int) $item->get_product_id();
            $tc_ticket_type_id = ( !$variation_id ? $product_id : $variation_id );
            $quantity = (int) $item->get_quantity();
            $user_id = (int) get_current_user_id();
            $event_id = (int) get_post_meta( $product_id, '_event_name', true );
            // Exit if not a ticket instance
            $is_ticket_instance = get_post_meta( $product_id, '_tc_is_ticket', true );
            $is_shop_order = ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ? true : false );
            if ( !$is_ticket_instance || !$product_id || !$is_shop_order ) {
                return false;
            }
            // Make sure to include ticket_type_id meta
            if ( !in_array( 'ticket_type_id', $metas ) ) {
                $metas[] = 'ticket_type_id';
            }
            $owner_form = new Tickera\TC_Cart_Form($tc_ticket_type_id);
            $owner_fields = $owner_form->get_owner_info_fields();
            $cart_info = get_post_meta( $order_id, 'tc_cart_info', true );
            $cart_info = ( $cart_info ? $cart_info : [] );
            // Create Ticket Instance
            for ($x = 0; $x < $quantity; $x++) {
                $tc_ticket_code_count = [];
                $tc_tickets_instances = Tickera\TC_Orders::get_tickets_ids( $order_id );
                // Collect Ticket Codes
                foreach ( $tc_tickets_instances as $ticket_instance_id ) {
                    $ticket_instance_meta = get_post_meta( $ticket_instance_id );
                    // Collect Ticket Codes
                    $tc_ticket_code = ( isset( $ticket_instance_meta['ticket_code'] ) ? $ticket_instance_meta['ticket_code'][0] : '' );
                    $tc_ticket_code = explode( '-', $tc_ticket_code );
                    $tc_ticket_code = ( @$tc_ticket_code[1] ? $tc_ticket_code[1] : '' );
                    if ( $tc_ticket_code ) {
                        $tc_ticket_code_count[] = (int) $tc_ticket_code;
                    }
                }
                // Identify available slot in ticket_code
                $available_slot = [];
                if ( $tc_ticket_code_count ) {
                    sort( $tc_ticket_code_count );
                    $temp = range( 1, max( $tc_ticket_code_count ) );
                    $available_slot = array_diff( $temp, $tc_ticket_code_count );
                }
                // Insert post and postmeta
                $owner_record_id = @wp_insert_post( array(
                    'post_author'  => (int) $user_id,
                    'post_parent'  => (int) $order_id,
                    'post_excerpt' => ( isset( $excerpt ) ? $excerpt : '' ),
                    'post_content' => ( isset( $content ) ? $content : '' ),
                    'post_status'  => 'publish',
                    'post_title'   => ( isset( $title ) ? $title : '' ),
                    'post_type'    => 'tc_tickets_instances',
                ), true );
                $available_slot = ( $available_slot ? reset( $available_slot ) : end( $tc_ticket_code_count ) + 1 );
                $ticket_code = ( apply_filters( 'tc_use_only_digit_order_number', false ) == true ? apply_filters(
                    'tc_ticket_code',
                    $order_id . '' . $available_slot,
                    $tc_ticket_type_id,
                    $owner_record_id
                ) : apply_filters(
                    'tc_ticket_code',
                    $order_id . '-' . $available_slot,
                    $tc_ticket_type_id,
                    $owner_record_id
                ) );
                /*
                 * Update Ticket Instance Postmeta Values
                 * Make sure to map ticket instance with WC item ID
                 */
                foreach ( $metas as $key ) {
                    if ( isset( ${$key} ) ) {
                        if ( 'ticket_type_id' == $key ) {
                            add_post_meta( $owner_record_id, $key, $tc_ticket_type_id );
                        } else {
                            if ( is_array( ${$key} ) && isset( ${$key}[$tc_ticket_type_id] ) ) {
                                $value = ( isset( ${$key}[$tc_ticket_type_id][$x] ) ? ${$key}[$tc_ticket_type_id][$x] : '' );
                            } else {
                                $value = ${$key};
                            }
                            add_post_meta( $owner_record_id, $key, sanitize_text_field( $value ) );
                        }
                    } elseif ( isset( ${'tc_' . $key} ) ) {
                        add_post_meta( $owner_record_id, $key, sanitize_text_field( ${'tc_' . $key} ) );
                    }
                }
                do_action(
                    'tc_created_order_ticket_instance',
                    $owner_record_id,
                    $order_id,
                    $blocks
                );
            }
            $cart_contents = (array) get_post_meta( $order_id, 'tc_cart_contents', true );
            $cart_contents[$tc_ticket_type_id] = @$cart_contents[$tc_ticket_type_id] + $quantity;
            update_post_meta( $order_id, 'tc_cart_contents', array_filter( $cart_contents ) );
        }

        /**
         * Update Ticket Instances based on WC Order line changes
         * TODO: Optimize Codes
         *
         * @param $order_id
         * @param $items
         */
        function update_order_ticket_instances( $order_id, $items ) {
            if ( 'woocommerce_save_order_items' == sanitize_text_field( $_POST['action'] ) && isset( $_POST['security'] ) ) {
                $user_id = get_current_user_id();
                $order = wc_get_order( $order_id );
                $cart_contents = [];
                $cart_info = get_post_meta( $order_id, 'tc_cart_info', true );
                $prev_cart_contents = get_post_meta( $order_id, 'tc_cart_contents', true );
                foreach ( $order->get_items() as $item_id => $wc_item_object ) {
                    /**
                     * Use this hook to skip cart item loop.
                     * @since 1.6.2
                     */
                    if ( !apply_filters( 'tc_skip_cart_item_loop', false, $wc_item_object ) ) {
                        $variation_id = $wc_item_object->get_variation_id();
                        $product_id = $wc_item_object->get_product_id();
                        $event_id = get_post_meta( $product_id, '_event_name', true );
                        $ticket_type_id = ( !$variation_id ? $product_id : $variation_id );
                        $is_ticket_instance = get_post_meta( $product_id, '_tc_is_ticket', true );
                        $is_shop_order = ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ? true : false );
                        // New Cart Contents
                        $cart_contents[$ticket_type_id] = (int) $items['order_item_qty'][$item_id];
                        /*
                         * Prepare item counts for comparison
                         * Process the update when prev and latest count didn't match
                         */
                        $prev_count = (int) @$prev_cart_contents[$ticket_type_id];
                        $latest_count = (int) $items['order_item_qty'][$item_id];
                        if ( $prev_count != $latest_count && $is_ticket_instance && $is_shop_order ) {
                            // Calculate quantity difference
                            $count_difference = abs( $prev_count - $latest_count );
                            // Create Ticket Instances
                            if ( $latest_count && $latest_count > $prev_count ) {
                                // Update Order Ticket Instances
                                for ($x = 0; $x < $count_difference; $x++) {
                                    // Retrieve Ticket instances
                                    $tc_ticket_code_count = [];
                                    $tc_tickets_instances = Tickera\TC_Orders::get_tickets_ids( $order_id, 'publish', 'DESC' );
                                    // Collect Ticket Codes
                                    foreach ( $tc_tickets_instances as $ticket_instance_id ) {
                                        $tc_ticket_code = get_post_meta( $ticket_instance_id, 'ticket_code', true );
                                        $tc_ticket_code = explode( '-', $tc_ticket_code );
                                        $tc_ticket_code = ( @$tc_ticket_code[1] ? $tc_ticket_code[1] : '' );
                                        if ( $tc_ticket_code ) {
                                            $tc_ticket_code_count[] = $tc_ticket_code;
                                        }
                                    }
                                    // Identify available slot in ticket_code
                                    $available_slot = [];
                                    if ( $tc_ticket_code_count ) {
                                        sort( $tc_ticket_code_count );
                                        $temp = range( 1, max( $tc_ticket_code_count ) );
                                        $available_slot = array_diff( $temp, $tc_ticket_code_count );
                                    }
                                    // Insert post and postmeta
                                    $owner_record_id = @wp_insert_post( array(
                                        'post_author'  => ( $user_id ? $user_id : '' ),
                                        'post_parent'  => $order_id,
                                        'post_excerpt' => '',
                                        'post_content' => '',
                                        'post_status'  => 'publish',
                                        'post_title'   => '',
                                        'post_type'    => 'tc_tickets_instances',
                                    ), true );
                                    do_action( 'tc_after_owner_post_field_type_check' );
                                    $available_slot = ( $available_slot ? reset( $available_slot ) : end( $tc_ticket_code_count ) + 1 );
                                    $ticket_code = ( apply_filters( 'tc_use_only_digit_order_number', false ) == true ? apply_filters( 'tc_ticket_code', $order_id . '' . $available_slot, $ticket_type_id ) : apply_filters( 'tc_ticket_code', $order_id . '-' . $available_slot, $ticket_type_id ) );
                                    // Update Ticket Instance Postmeta Values
                                    $metas = [
                                        'ticket_type_id',
                                        'ticket_code',
                                        'event_id',
                                        'item_id'
                                    ];
                                    foreach ( $metas as $key ) {
                                        add_post_meta( $owner_record_id, $key, sanitize_text_field( ${$key} ) );
                                    }
                                    // New Cart Info
                                    foreach ( $cart_info['owner_data'] as $total_name => $total_values ) {
                                        if ( strpos( $total_name, 'ticket_type_id' ) !== false ) {
                                            $cart_info['owner_data'][$total_name][$ticket_type_id][] = $ticket_type_id;
                                        } else {
                                            $cart_info['owner_data'][$total_name][$ticket_type_id][] = '';
                                        }
                                    }
                                }
                            } else {
                                /**
                                 * Remove Tickera Ticket Instance if order line quantity is deducted
                                 * @var  $tc_tickets_instances
                                 */
                                $tc_tickets_instances = Tickera\TC_ORders::get_tickets_ids( $order_id, 'publish', 'DESC' );
                                foreach ( $tc_tickets_instances as $instance_id ) {
                                    $_item_id = get_post_meta( $instance_id, 'item_id', true );
                                    if ( $count_difference && $item_id == $_item_id ) {
                                        // Delete Post
                                        wp_delete_post( $instance_id );
                                        $count_difference--;
                                        // New Cart Info
                                        foreach ( $cart_info['owner_data'] as $total_name => $total_values ) {
                                            array_pop( $cart_info['owner_data'][$total_name][$ticket_type_id] );
                                            if ( !$cart_info['owner_data'][$total_name][$ticket_type_id] ) {
                                                unset($cart_info['owner_data'][$total_name][$ticket_type_id]);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                // Update Cart Contents
                update_post_meta( $order_id, 'tc_cart_contents', array_filter( $cart_contents ) );
                // Update Cart Info
                update_post_meta( $order_id, 'tc_cart_info', $cart_info );
            }
        }

        /**
         * Delete Ticket Instances when WC Order line item is deleted
         * TODO: Optimize Codes
         *
         * @param $item_id
         * @return bool
         * @throws Exception
         */
        function remove_order_ticket_instances( $item_id ) {
            global $wpdb;
            $item = new WC_Order_Item_Product($item_id);
            $order_id = (int) wc_get_order_id_by_order_item_id( $item_id );
            $order_id = ( $order_id ? $order_id : $item->get_order_id() );
            // Exit when method not executed by WC_Order_Item_Product class
            if ( !is_a( $item, 'WC_Order_Item_Product' ) ) {
                return false;
            }
            $cart_info = get_post_meta( $order_id, 'tc_cart_info', true );
            // Remove item before hook
            $wpdb->delete( $wpdb->prefix . 'woocommerce_order_items', array(
                'order_item_id' => $item_id,
            ) );
            $wpdb->delete( $wpdb->prefix . 'woocommerce_order_itemmeta', array(
                'order_item_id' => $item_id,
            ) );
            // Collect Ticket Instances
            $cart_contents = (array) get_post_meta( $order_id, 'tc_cart_contents', true );
            $tc_tickets_instances = Tickera\TC_Orders::get_tickets_ids( $order_id );
            foreach ( $tc_tickets_instances as $instance_id ) {
                $postmeta = get_post_meta( $instance_id );
                $ticket_type_id = ( isset( $postmeta['ticket_type_id'] ) ? $postmeta['ticket_type_id'][0] : '' );
                $ticket_instance_item_id = ( isset( $postmeta['item_id'] ) ? $postmeta['item_id'][0] : '' );
                if ( $ticket_instance_item_id == $item_id ) {
                    // Delete Post
                    wp_delete_post( $instance_id );
                    $cart_contents[$ticket_type_id] -= 1;
                    // New Cart Info
                    foreach ( $cart_info['owner_data'] as $total_name => $total_values ) {
                        array_pop( $cart_info['owner_data'][$total_name][$ticket_type_id] );
                        if ( !$cart_info['owner_data'][$total_name][$ticket_type_id] ) {
                            unset($cart_info['owner_data'][$total_name][$ticket_type_id]);
                        }
                    }
                }
            }
            // Update Cart Contents
            update_post_meta( $order_id, 'tc_cart_contents', array_filter( $cart_contents ) );
            // Update Cart Info
            update_post_meta( $order_id, 'tc_cart_info', $cart_info );
        }

        function update_order_review_fragments( $fragments ) {
            ob_start();
            $this->render_tc_owner_fields();
            $fragments['.tickera_owner_info'] = ob_get_clean();
            return $fragments;
        }

        function change_product_post_link_in_the_admin( $column, $product_id ) {
            if ( $column == 'name' ) {
                $is_ticket_meta = get_post_meta( $product_id, '_tc_is_ticket', true );
                $is_ticket = ( $is_ticket_meta == 'yes' ? true : false );
                if ( $is_ticket ) {
                    $event_id = get_post_meta( $product_id, '_event_name', true );
                    echo '<span class="tc_wc_product_title_event"><a href="' . admin_url( 'post.php?post=' . $event_id . '&action=edit' ) . '">' . get_the_title( $event_id ) . '</a></span>';
                }
            }
        }

        /**
         * Seating chart's price html
         *
         * @param $price
         * @param $id
         * @param bool $formatted
         * @param bool $semi_formatted
         * @return string
         */
        function modify_price_per_ticket(
            $price,
            $id,
            $formatted = true,
            $semi_formatted = false
        ) {
            /**
             * Seating map legend's sale price format.
             * @since 1.5.7
             */
            add_filter(
                'woocommerce_format_sale_price',
                function ( $price, $regular_price, $sale_price ) {
                    // Strikethrough pricing.
                    $price = '<del aria-hidden="true">' . wc_price( $regular_price ) . '</del> ';
                    // Add the sale price.
                    $price .= '<ins aria-hidden="true">' . wc_price( $sale_price ) . '</ins>';
                    return $price;
                },
                10,
                3
            );
            $id = TC_WooCommerce_Bridge::maybe_get_original_wpml_product_id( $id );
            $post_type = get_post_type( $id );
            if ( $post_type == 'product' || $post_type == 'product_variation' ) {
                $product = wc_get_product( $id );
                if ( $formatted ) {
                    $price = $product->get_price_html();
                } else {
                    if ( $semi_formatted ) {
                        $price = strip_tags( $product->get_price_html(), '<del>' );
                    } else {
                        $price = strip_tags( $product->get_price_html() );
                    }
                }
            }
            return $price;
        }

        function modify_seat_chart_checkout_url() {
            return wc_get_checkout_url();
        }

        function modify_seat_chart_add_to_cart_url() {
            return wc_get_cart_url();
        }

        function modify_seat_chart_in_cart_count() {
            return WC()->cart->get_cart_contents_count();
        }

        function modify_seat_chart_cart_subtotal() {
            return WC()->cart->get_cart_total();
        }

        function modify_tc_seat_charts_is_woo() {
            return true;
        }

        function admin_notices() {
            global $tc;
            if ( current_user_can( 'manage_options' ) ) {
                if ( isset( $tc->version ) && version_compare( $tc->version, $this->tc_version_required, '<' ) ) {
                    ?>
                    <div class="notice notice-error">
                        <p><?php 
                    printf(
                        __( '%s add-on requires at least %s version of %s plugin. Your current version of %s is %s. Please update it.', 'woocommerce-tickera-bridge' ),
                        $this->title,
                        $this->tc_version_required,
                        $tc->title,
                        $tc->title,
                        $tc->version
                    );
                    ?></p>
                    </div>
                    <?php 
                }
                /* if (!$this->is_3_0_version()) {
                   ?>
                   <div class="notice notice-error">
                   <p><?php printf(__('%s add-on requires at least 3.0 version of WooCommerce plugin. Please update it.', 'woocommerce-tickera-bridge'), $this->title); ?></p>
                   </div>
                   <?php
                   } */
            }
        }

        function tc_bridge_for_woocommerce_is_active() {
            return true;
        }

        /**
         * Update order on order status change
         *
         * @param $post_id
         * @param $old_status
         * @param $new_status
         */
        function check_tickets_action( $post_id, $old_status, $new_status ) {
            if ( 'cancelled' == $new_status ) {
                $this->trash_associated_tickets( $post_id, true );
            } else {
                $this->untrash_associated_tickets( $post_id );
            }
            $wc_order = new WC_Order($post_id);
            if ( $this->allowed_tickets_download( $wc_order ) ) {
                do_action( 'tc_wb_allowed_tickets_access', $wc_order );
            }
        }

        function check_order_deletion() {
            add_action(
                'delete_post',
                array(&$this, 'delete_associated_tickets'),
                10,
                1
            );
            add_action(
                'wp_trash_post',
                array(&$this, 'trash_associated_tickets'),
                10,
                1
            );
            add_action(
                'untrashed_post',
                array(&$this, 'untrash_associated_tickets'),
                10,
                1
            );
        }

        /**
         * When an order status is changed but not cancelled, untrash ticket instance and publish
         *
         * @param $post_id
         */
        function untrash_associated_tickets( $post_id ) {
            if ( in_array( get_post_type( $post_id ), self::get_woo_order_types() ) ) {
                $ticket_instances = Tickera\TC_Orders::get_tickets_ids( $post_id, 'trash' );
                foreach ( $ticket_instances as $ticket_instance_id ) {
                    wp_untrash_post( $ticket_instance_id );
                    wp_update_post( [
                        'ID'          => $ticket_instance_id,
                        'post_status' => 'publish',
                    ] );
                    delete_post_meta( $ticket_instance_id, '_cancelled_order', 'yes' );
                }
            } elseif ( 'tc_tickets_instances' == get_post_type( $post_id ) ) {
                wp_untrash_post( $post_id );
                wp_update_post( [
                    'ID'          => $post_id,
                    'post_status' => 'publish',
                ] );
                delete_post_meta( $post_id, '_cancelled_order', 'yes' );
            }
        }

        /**
         * When an order status is changed to cancelled, trash ticket instance
         *
         * @param $order_id
         * @param bool $cancelled_order
         */
        function trash_associated_tickets( $order_id, $cancelled_order = false ) {
            if ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ) {
                $ticket_instances = Tickera\TC_Orders::get_tickets_ids( $order_id, 'any' );
                foreach ( $ticket_instances as $ticket_instance_id ) {
                    $ticket_instance_instance = new Tickera\TC_Ticket_Instance($ticket_instance_id);
                    $ticket_instance_instance->delete_ticket_instance( false );
                    if ( $cancelled_order ) {
                        update_post_meta( $ticket_instance_id, '_cancelled_order', 'yes' );
                    } else {
                        delete_post_meta( $ticket_instance_id, '_cancelled_order', 'yes' );
                    }
                }
            }
        }

        /**
         * If an order is deleted, remove all associated ticket instances
         *
         * @param $order_id
         * @param bool $trash
         */
        function delete_associated_tickets( $order_id, $trash = true ) {
            if ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ) {
                $trash = ( $trash ? 'trash' : 'publish' );
                $ticket_instances = Tickera\TC_Orders::get_tickets_ids( $order_id, $trash );
                foreach ( $ticket_instances as $ticket_instance_id ) {
                    $ticket_instance_instance = new Tickera\TC_Ticket_Instance($ticket_instance_id);
                    $ticket_instance_instance->delete_ticket_instance( true );
                }
            }
        }

        /**
         * Load Localisation files (first translation file found will be loaded, others will be ignored)
         *
         * Frontend/global Locales found in:
         * 	- WP_LANG_DIR/woocommerce-tickera-bridge-LOCALE.mo
         * 	- WP_LANG_DIR/woocommerce-tickera-bridge/woocommerce-tickera-bridge-LOCALE.mo
         *  - woocommerce-tickera-bridge/languages/woocommerce-tickera-bridge-LOCALE.mo (which if not found falls back to:)
         */
        public function load_plugin_textdomain() {
            $locale = apply_filters( 'plugin_locale', get_locale(), 'woocommerce-tickera-bridge' );
            load_textdomain( 'woocommerce-tickera-bridge', WP_LANG_DIR . '/woocommerce-tickera-bridge-' . $locale . '.mo' );
            load_textdomain( 'woocommerce-tickera-bridge', WP_LANG_DIR . '/woocommerce-tickera-bridge/woocommerce-tickera-bridge-' . $locale . '.mo' );
            load_plugin_textdomain( 'woocommerce-tickera-bridge', false, plugin_basename( dirname( __FILE__ ) ) . "/languages" );
        }

        /**
         * @param type $url
         * @return type
         */
        function modify_ticket_type_admin_url( $url ) {
            return admin_url( 'edit.php?post_type=product' );
        }

        /**
         * Modify shortcode which is appended automatically on the event post single content page if "Show Tickets Automatically" is checked
         *
         * @param string $shortcode
         * @return string
         */
        function modify_event_shortcode( $shortcode ) {
            return '[tc_wb_event]';
        }

        /**
         * Check if order post type is valid and return wheter the order is found or not based on that
         *
         * @param boolean $value
         * @param int $order_id
         * @return bool
         */
        function modify_order_found( $value, $order_id ) {
            if ( in_array( get_post_type( $order_id ), ['tc_orders', 'shop_order', 'shop_order_placehold'] ) ) {
                $value = true;
            }
            return $value;
        }

        /**
         * Remove unneeded shortcode and add new one
         */
        function replace_shortcodes() {
            remove_shortcode( 'tc_event' );
            remove_shortcode( 'tc_ticket' );
            add_shortcode( 'tc_wb_event', array(&$this, 'tc_wb_event') );
        }

        /**
         * Shortcode for showing event tickets
         *
         * @global object $post
         * @param array $atts
         * @return mixed $content event tickets
         */
        function tc_wb_event( $atts ) {
            global $post;
            ob_start();
            $count_tickets = 0;
            $_tc_used_for_seatings_count = 0;
            extract( shortcode_atts( array(
                'id'                => $post->ID,
                'event_table_class' => 'event_tickets tickera',
                'display_type'      => 'table',
                'show_event_title'  => false,
                'show_price'        => false,
                'ticket_type_title' => __( 'Ticket Type', 'woocommerce-tickera-bridge' ),
                'price_title'       => __( 'Price', 'woocommerce-tickera-bridge' ),
                'cart_title'        => __( 'Cart', 'woocommerce-tickera-bridge' ),
                'quantity_title'    => __( 'Quantity', 'woocommerce-tickera-bridge' ),
                'quantity'          => false,
                'wrapper'           => '',
            ), $atts ) );
            $args = [
                'post_type'      => 'product',
                'post_status'    => 'publish',
                'posts_per_page' => -1,
                'orderby'        => 'menu_order',
                'order'          => 'ASC',
                'meta_query'     => [
                    'relation' => 'AND',
                    [
                        'key'     => '_tc_is_ticket',
                        'compare' => '=',
                        'value'   => 'yes',
                    ],
                    [
                        'key'     => '_event_name',
                        'compare' => '=',
                        'value'   => (int) $id,
                    ],
                ],
            ];
            $ticket_types = get_posts( $args );
            ?>
            <div class="tickera">
                <?php 
            if ( 'table' == $display_type ) {
                ?>
                    <table class="<?php 
                echo esc_attr( $event_table_class );
                ?>">
                        <tr>
                            <?php 
                do_action( 'tc_wb_event_col_title_before_ticket_title' );
                ?>
                            <th><?php 
                echo $ticket_type_title;
                ?></th>
                            <?php 
                do_action( 'tc_wb_event_col_title_before_ticket_price' );
                ?>
                            <th><?php 
                echo $price_title;
                ?></th>
                            <?php 
                if ( $quantity ) {
                    ?>
                                <?php 
                    do_action( 'tc_wb_event_col_title_before_quantity' );
                    ?>
                                <th><?php 
                    echo esc_html( $quantity_title );
                    ?></th>
                            <?php 
                }
                ?>
                            <?php 
                do_action( 'tc_wb_event_col_title_before_cart_title' );
                ?>
                            <th><?php 
                echo $cart_title;
                ?></th>
                        </tr>
                        <?php 
                foreach ( $ticket_types as $ticket_type ) {
                    $_tc_used_for_seatings = get_post_meta( $ticket_type->ID, '_tc_used_for_seatings', true );
                    $_tc_used_for_seatings = ( $_tc_used_for_seatings == 'yes' ? true : false );
                    $product = wc_get_product( $ticket_type->ID );
                    $wc_catalog_visibility = $product->get_catalog_visibility();
                    if ( !in_array( $wc_catalog_visibility, ['hidden', 'search'] ) && Tickera\TC_Ticket::is_sales_available( $ticket_type->ID ) ) {
                        $count_tickets++;
                        if ( !$_tc_used_for_seatings ) {
                            ?>
                                    <tr>
                                    <?php 
                            do_action( 'tc_wb_event_col_value_before_ticket_type', $ticket_type->ID );
                            ?>
                                    <td data-column="<?php 
                            esc_attr_e( 'Ticket Type', 'woocommerce-tickera-bridge' );
                            ?>"><?php 
                            echo apply_filters( 'wc_product_tickets_table_title', $ticket_type->post_title, $ticket_type->ID );
                            ?></td>
                                    <?php 
                            do_action( 'tc_wb_event_col_value_before_ticket_price', $ticket_type->ID );
                            ?>
                                    <td data-column="<?php 
                            esc_attr_e( 'Price', 'woocommerce-tickera-bridge' );
                            ?>"><?php 
                            echo apply_filters( 'wc_product_tickets_table_price', $product->get_price_html(), $ticket_type->ID );
                            ?></td>
                                    <?php 
                            if ( $quantity ) {
                                if ( !$product->is_type( 'variable' ) ) {
                                    do_action( 'tc_wb_event_col_value_before_quantity', $ticket_type->ID );
                                    $quantiy_selector_field = woocommerce_quantity_input( array(
                                        'min_value'   => $product->get_min_purchase_quantity(),
                                        'max_value'   => $product->get_max_purchase_quantity(),
                                        'input_value' => $product->get_min_purchase_quantity(),
                                        'classes'     => 'tc-wb-quantity-selector',
                                    ), $product, false );
                                    echo '<td data-column="' . esc_attr__( 'Quantity', 'woocommerce-tickera-bridge' ) . '">' . $quantiy_selector_field . '</td>';
                                } else {
                                    echo '<td data-column="' . esc_attr__( 'Quantity', 'woocommerce-tickera-bridge' ) . '"></td>';
                                }
                            }
                            ?>
                                    <?php 
                            do_action( 'tc_wb_event_col_value_before_cart_title', $ticket_type->ID );
                            ?>
                                    <td data-column="<?php 
                            esc_attr_e( 'Cart', 'woocommerce-tickera-bridge' );
                            ?>">
                                        <?php 
                            echo apply_filters( 'tc_wb_event_col_value_add_to_cart', do_shortcode( '[add_to_cart id="' . $ticket_type->ID . '" style="" show_price="false" class="tc-wb-add-to-cart"]' ), $ticket_type->ID );
                            ?>
                                    </td>
                                    </tr><?php 
                        } else {
                            $_tc_used_for_seatings_count++;
                        }
                    }
                }
                ?>
                    </table>
                <?php 
            } else {
                $event = new Tickera\TC_Event($id);
                ?>
                    <div class="tc-wb-event-dropdown-wrap">
                        <?php 
                if ( $show_event_title ) {
                    ?>
                            <h3><?php 
                    _e( $event->details->post_title, 'woocommerce-tickera-bridge' );
                    ?></h3>
                        <?php 
                }
                ?>
                        <div class="inner-wrap">
                            <select class="ticket-type-id">
                                <?php 
                foreach ( $ticket_types as $ticket_type ) {
                    $count_tickets++;
                    $_tc_used_for_seatings = get_post_meta( $ticket_type->ID, '_tc_used_for_seatings', true );
                    $_tc_used_for_seatings = ( $_tc_used_for_seatings == 'yes' ? true : false );
                    $product = wc_get_product( $ticket_type->ID );
                    $wc_catalog_visibility = $product->get_catalog_visibility();
                    if ( !in_array( $wc_catalog_visibility, ['hidden', 'search'] ) && true == Tickera\TC_Ticket::is_sales_available( (int) $ticket_type->ID ) ) {
                        if ( !$_tc_used_for_seatings ) {
                            ?>
                                            <option value="<?php 
                            echo (int) $ticket_type->ID;
                            ?>"><?php 
                            echo esc_html( $ticket_type->post_title ) . (( $show_price ? ' - ' . $product->get_price_html() : '' ));
                            ?></option><?php 
                        } else {
                            $_tc_used_for_seatings_count++;
                        }
                    }
                }
                ?>
                            </select>
                            <div class="actions">
                                <?php 
                foreach ( $ticket_types as $ticket_type ) {
                    $_tc_used_for_seatings = get_post_meta( $ticket_type->ID, '_tc_used_for_seatings', true );
                    $_tc_used_for_seatings = ( $_tc_used_for_seatings == 'yes' ? true : false );
                    $product = wc_get_product( $ticket_type->ID );
                    $wc_catalog_visibility = $product->get_catalog_visibility();
                    if ( !in_array( $wc_catalog_visibility, ['hidden', 'search'] ) && true == Tickera\TC_Ticket::is_sales_available( (int) $ticket_type->ID ) && !$_tc_used_for_seatings ) {
                        echo '<div class="add-to-cart" id="ticket-type-' . (int) $ticket_type->ID . '">';
                        if ( $quantity && !$product->is_type( 'variable' ) ) {
                            woocommerce_quantity_input( array(
                                'min_value'   => $product->get_min_purchase_quantity(),
                                'max_value'   => $product->get_max_purchase_quantity(),
                                'input_value' => $product->get_min_purchase_quantity(),
                                'classes'     => 'tc-wb-quantity-selector',
                            ), $product );
                        }
                        echo do_shortcode( '[add_to_cart id="' . $ticket_type->ID . '" style="" show_price="false" class="tc-wb-add-to-cart"]' );
                        echo '</div>';
                    }
                }
                ?>
                            </div>
                        </div>
                    </div>
                <?php 
            }
            ?>
            </div><!-- tickera -->
            <?php 
            $content = ob_get_clean();
            if ( current_user_can( 'manage_options' ) && $_tc_used_for_seatings_count > 0 ) {
                $content .= '<div class="tc_warning_ticket_types_needed">' . sprintf(
                    __( '%s ADMIN NOTICE:%s  %s ticket types are hidden because they are used for seatings and can be added to a cart that way only.', 'woocommerce-tickera-bridge' ),
                    '<strong>',
                    '</strong>',
                    '<u>' . $_tc_used_for_seatings_count . '</u>'
                ) . '</div>';
            }
            return $content;
        }

        /**
         * Render add to cart button in frontend.
         * Currently used in Event Tickets - Shortcode
         * @since 1.3.8
         */
        function render_add_to_cart_button() {
            if ( isset( $_POST['ticket_id'] ) ) {
                $ticket_id = (int) $_POST['ticket_id'];
                ob_start();
                echo do_shortcode( '[add_to_cart id="' . $ticket_id . '" style="" show_price="false" class="tc-wb-add-to-cart"]' );
                ob_end_flush();
                exit;
            }
        }

        /**
         * Update Tickera's shortcode builder list
         *
         * @param array $shortcodes
         * @return array $shortcodes modified shortcode list
         */
        function modify_shortcode_builder_list( $shortcodes ) {
            unset($shortcodes['tc_ticket']);
            unset($shortcodes['tc_ticket_group']);
            unset($shortcodes['tc_event']);
            unset($shortcodes['tc_event_group']);
            unset($shortcodes['event_tickets_sold']);
            unset($shortcodes['event_tickets_left']);
            unset($shortcodes['tickets_sold']);
            unset($shortcodes['tickets_left']);
            unset($shortcodes['tc_order_history']);
            $shortcodes['add_to_cart'] = __( 'Ticket / Add to cart button', 'woocommerce-tickera-bridge' );
            $shortcodes['tc_wb_event'] = __( 'Table with Event tickets', 'woocommerce-tickera-bridge' );
            return $shortcodes;
        }

        /**
         * Update Tickera's gutenberg blocks list
         *
         * @param $blocks
         * @return mixed
         *
         * @since 1.4.1
         */
        function modify_gutenberg_blocks_list( $blocks ) {
            unset($blocks['tc_ticket']);
            unset($blocks['tc_ticket_group']);
            unset($blocks['tc_event']);
            unset($blocks['tc_event_group']);
            $blocks['add_to_cart'] = 'tickera/woo-add-to-cart';
            $blocks['tc_wb_event'] = 'tickera/woo-event-add-to-cart';
            $blocks['add_to_cart_group'] = 'tickera/woo-add-to-cart-group';
            $blocks['tc_wb_event_group'] = 'tickera/woo-event-add-to-cart-group';
            return $blocks;
        }

        /**
         * Collection of Woocommerce order statuses to reserve seats.
         *
         * @param $status
         * @return string[]
         */
        function modify_seat_charts_get_reserved_seats_post_status( $statuses = [], $draft = false ) {
            $statuses = [
                'wc-completed',
                'completed',
                'wc-processing',
                'processing',
                'wc-pending',
                'pending',
                'wc-on-hold',
                'on-hold'
            ];
            if ( $draft ) {
                $statuses = array_merge( $statuses, ['wc-checkout-draft', 'checkout-draft'] );
            }
            return $statuses;
        }

        public static function maybe_get_original_wpml_product_id( $product_id ) {
            global $sitepress;
            if ( function_exists( 'icl_object_id' ) && is_object( $sitepress ) ) {
                $product_id = icl_object_id(
                    $product_id,
                    get_post_type( $product_id ),
                    true,
                    $sitepress->get_default_language()
                );
            }
            return $product_id;
        }

        /**
         * Allow / Disallow download of tickets (based on the order status and / or payment gateway)
         *
         * @global array $tc_general_settings
         * @param type $order
         * @return boolean
         */
        function allowed_tickets_download( $order ) {
            $has_ticket = false;
            $allowed_tickets_download = false;
            if ( is_object( $order ) && method_exists( $order, 'get_items' ) ) {
                $items = $order->get_items();
                if ( empty( $items ) ) {
                    $items = ( is_cart() || is_checkout() ? WC()->cart->get_cart() : array() );
                }
            } else {
                $items = ( is_cart() || is_checkout() ? WC()->cart->get_cart() : array() );
            }
            foreach ( $items as $item ) {
                /**
                 * Use this hook to skip cart item loop.
                 * @since 1.6.2
                 */
                if ( !apply_filters( 'tc_skip_cart_item_loop', false, $item ) ) {
                    $product_id = $item['product_id'];
                    $is_ticket_meta = get_post_meta( TC_WooCommerce_Bridge::maybe_get_original_wpml_product_id( $product_id ), '_tc_is_ticket', true );
                    if ( 'yes' == $is_ticket_meta ) {
                        $has_ticket = true;
                        break;
                    }
                }
            }
            if ( $has_ticket ) {
                if ( is_object( $order ) ) {
                    $order_id = ( method_exists( $order, 'get_id' ) ? $order->get_id() : $order->id );
                    // Make suret to initialize order class
                    $order = wc_get_order( $order_id );
                    $order_payment_method = $order->get_payment_method();
                    if ( 'cod' == $order_payment_method ) {
                        $tc_general_settings = $this->tc_general_settings;
                        $tc_woo_allow_cash_on_delivery_processing = ( isset( $tc_general_settings['tc_woo_allow_cash_on_delivery_processing'] ) ? $tc_general_settings['tc_woo_allow_cash_on_delivery_processing'] : 'no' );
                        if ( 'completed' == $order->get_status() || 'yes' == $tc_woo_allow_cash_on_delivery_processing ) {
                            $allowed_tickets_download = true;
                        }
                    } else {
                        /**
                         * Show tickets table on completed and processing orders
                         * @since 1.5.2
                         */
                        if ( in_array( $order->get_status(), ['completed', 'processing'] ) ) {
                            $allowed_tickets_download = true;
                        }
                    }
                }
            }
            return apply_filters( 'tc_allow_tickets_download', $allowed_tickets_download, $order );
        }

        /**
         * Redirect WooCommerce ticket single post to its event / associated event single post
         *
         * @global array $tc_general_settings
         * @global object $post
         */
        function tc_redirect_ticket_single_to_event() {
            global $post;
            $tc_general_settings = $this->tc_general_settings;
            $tc_woo_redirect_single_post_to_event = ( isset( $tc_general_settings['tc_woo_redirect_single_post_to_event'] ) ? $tc_general_settings['tc_woo_redirect_single_post_to_event'] : 'no' );
            if ( is_single() ) {
                if ( $tc_woo_redirect_single_post_to_event == 'yes' ) {
                    if ( get_post_type( $post->ID ) == 'product' ) {
                        $product = wc_get_product( $post->ID );
                        $is_ticket_meta = get_post_meta( $post->ID, '_tc_is_ticket', true );
                        $is_ticket = ( $is_ticket_meta == 'yes' ? true : false );
                        if ( $is_ticket && !$product->has_child() ) {
                            $event_id = get_post_meta( $post->ID, '_event_name', true );
                            wp_redirect( get_permalink( $event_id ) );
                            exit;
                        }
                    }
                }
            }
        }

        /**
         * Modify order status title
         *
         * @param string $status_title
         * @param int $order_id
         * @param string $status
         * @return string $status_title title of the WooCommerce order status
         */
        function modify_order_status_title( $status_title, $order_id, $status ) {
            $payment_statuses = array(
                'completed'  => __( 'Completed', 'woocommerce-tickera-bridge' ),
                'processing' => __( 'Processing', 'woocommerce-tickera-bridge' ),
                'cancelled'  => __( 'Cancelled', 'woocommerce-tickera-bridge' ),
                'on-hold'    => __( 'On Hold', 'woocommerce-tickera-bridge' ),
                'pending'    => __( 'Pending Payment', 'woocommerce-tickera-bridge' ),
            );
            if ( array_key_exists( $status, $payment_statuses ) ) {
                $status_title = $payment_statuses[$status];
            }
            return $status_title;
        }

        /**
         * Modify Payment Gateway Name (get one from WooCommerce)
         *
         * @param string $gateway_name
         * @param int $order_id
         * @return string $gateway_name WooCommerce payment method title
         */
        function modify_order_payment_gateway_name( $gateway_name, $order_id ) {
            if ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ) {
                $order = new \WC_Order($order_id);
                $gateway_name = $order->get_payment_method_title();
            }
            return $gateway_name;
        }

        function is_3_0_version( $version = '3.0' ) {
            global $woocommerce;
            if ( version_compare( $woocommerce->version, $version, ">=" ) ) {
                return true;
            }
            return false;
        }

        /**
         * Modify ticket type title for variations
         *
         * @param string $title
         * @param int $ticket_type_id
         * @return string
         */
        function modify_checkout_owner_info_ticket_title( $title, $ticket_type_id ) {
            if ( get_post_type( $ticket_type_id ) == 'product_variation' ) {
                $variation = wc_get_product( $ticket_type_id );
                //var_dump($variation);
                $title = $variation->get_title();
                if ( !$this->is_3_0_version() ) {
                    $variation_data = wc_get_formatted_variation( $variation->variation_data, true );
                } else {
                    // different args to reflect WooCommerce 3.0 changes
                    $variation_data = wc_get_formatted_variation( $variation, true, false );
                }
                if ( isset( $variation_data ) ) {
                    $title .= ' (' . $variation_data . ')';
                }
            }
            return $title;
        }

        /**
         * Modify payment status select box and add WooCommerce payment statuses
         *
         * @param array $payment_statuses
         * @return array $payment_statuses modified list of WooCommerce order payment statuses
         */
        function modify_csv_payment_statuses( $payment_statuses ) {
            $payment_statuses = array(
                'any'           => __( 'Any', 'woocommerce-tickera-bridge' ),
                'wc-completed'  => __( 'Completed', 'woocommerce-tickera-bridge' ),
                'wc-processing' => __( 'Processing', 'woocommerce-tickera-bridge' ),
                'wc-on-hold'    => __( 'On Hold', 'woocommerce-tickera-bridge' ),
                'wc-cancelled'  => __( 'Cancelled', 'woocommerce-tickera-bridge' ),
                'wc-pending'    => __( 'Pending Payment', 'woocommerce-tickera-bridge' ),
                'wc-refunded'   => __( 'Refunded', 'woocommerce-tickera-bridge' ),
            );
            return $payment_statuses;
        }

        function modify_export_order_number_column_value( $order_number, $order_id ) {
            if ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ) {
                $order_number = '#' . $order_id;
            }
            return $order_number;
        }

        /**
         * Modify ticket type id
         *
         * If the product is Variable / Variation, ticket type ID should be product's ID which is parent to the variation in order to get right custom form
         *
         * @param int $ticket_type_id
         * @return int
         */
        public static function modify_ticket_type_id( $ticket_type_id ) {
            if ( get_post_type( $ticket_type_id ) == 'product_variation' ) {
                $ticket_type_id = wp_get_post_parent_id( $ticket_type_id );
            }
            return TC_WooCommerce_Bridge::maybe_get_original_wpml_product_id( $ticket_type_id );
        }

        /**
         * Modify buyer info fields
         *
         * Remove standard buyer fields (First Name, Last Name and E-mail) since we already have them in WooCommerce
         *
         * @param array $fields
         * @return array
         */
        function modify_buyer_info_fields( $fields ) {
            return array();
        }

        /**
         * Modify order details
         * Don't show standard Tickera buyer fields, just extra custom fields (priority of 9 is required then)
         *
         * @global object $post
         * @param array $fields
         * @return array
         */
        function modify_order_fields( $fields ) {
            $hpos = tc_wb_hpos();
            if ( $hpos && in_array( $hpos->post_type, self::get_woo_order_types() ) ) {
                $fields = [];
            }
            return $fields;
        }

        /**
         * Show order details buyer custom fields on front-end
         *
         * @global object $post
         * @param object $order
         */
        function tc_woocommerce_admin_order_data_after_order_details( $order ) {
            $order_id = ( method_exists( $order, 'get_id' ) ? $order->get_id() : $order->id );
            $tc_cart_info = get_post_meta( $order_id, 'tc_cart_info' );
            if ( $tc_cart_info && is_array( $tc_cart_info ) && count( $tc_cart_info ) > 0 && !empty( $tc_cart_info[0]['buyer_data'] ) ) {
                if ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ) {
                    tickera_get_order_details_buyer_custom_fields( $order_id );
                }
            }
        }

        /**
         * Indicator to show custom fields on the WooCommerce order page in the admin
         *
         * @param boolean $value
         * @return boolean
         */
        function modify_show_custom_fields_as_order_columns( $value ) {
            return true;
        }

        /**
         * Remove tickets from shop page if needed (if it's set in Tickera > Settings > WooCommerce > Hide Tickets)
         *
         * @global array $tc_general_settings
         * @param mixed $query
         */
        function tc_custom_pre_get_posts_query( $query ) {
            $tc_general_settings = $this->tc_general_settings;
            if ( !$query->is_main_query() ) {
                return;
            }
            if ( !$query->is_post_type_archive() ) {
                return;
            }
            if ( !is_admin() && is_shop() ) {
                $tc_woo_hide_products = ( isset( $tc_general_settings['tc_woo_hide_products'] ) ? $tc_general_settings['tc_woo_hide_products'] : 'no' );
                if ( $tc_woo_hide_products == 'yes' ) {
                    $query->set( 'meta_query', array(array(
                        'key'     => '_tc_is_ticket',
                        'value'   => 'yes',
                        'compare' => 'NOT EXISTS',
                    )) );
                } else {
                    //hide products which are used for seating charts because they shouldn't be purchased in any other way
                    $query->set( 'meta_query', array(array(
                        'relation' => 'OR',
                        array(
                            'key'     => '_tc_used_for_seatings',
                            'compare' => 'NOT EXISTS',
                        ),
                        array(
                            'key'     => '_tc_used_for_seatings',
                            'value'   => 'yes',
                            'compare' => '!=',
                        ),
                    )) );
                }
            }
        }

        /**
         * Validate Tickera fields
         */
        function tc_validate_tickera_fields() {
            if ( isset( $_POST['tc_cart_required'] ) ) {
                $required_fields_error_count = 0;
                foreach ( $_POST as $key => $value ) {
                    if ( $key !== 'tc_cart_required' ) {
                        if ( in_array( $key, $_POST['tc_cart_required'] ) ) {
                            if ( !is_array( $value ) ) {
                                if ( trim( $value ) == '' ) {
                                    $required_fields_error_count++;
                                }
                            } else {
                                foreach ( $_POST[$key] as $val ) {
                                    if ( !is_array( $val ) ) {
                                        if ( '' == trim( $val ) ) {
                                            $required_fields_error_count++;
                                        }
                                    } else {
                                        foreach ( $val as $val_str ) {
                                            if ( trim( $val_str ) == '' ) {
                                                $required_fields_error_count++;
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                if ( $required_fields_error_count > 0 ) {
                    // wc_add_notice(__('All fields marked with * are required.', 'woocommerce-tickera-bridge'), 'error');
                    throw new Exception(__( 'All fields marked with * are required.', 'woocommerce-tickera-bridge' ));
                }
            }
            $owner_emails = ( isset( $_POST['owner_data_owner_email_post_meta'] ) ? $_POST['owner_data_owner_email_post_meta'] : [] );
            $verify_email = ( isset( $this->tc_general_settings['email_verification_buyer_owner'] ) ? $this->tc_general_settings['email_verification_buyer_owner'] : 'no' );
            $verification_emails = ( isset( $_POST['owner_data_owner_confirm_email_post_meta'] ) ? $_POST['owner_data_owner_confirm_email_post_meta'] : [] );
            // Validate Email ( Buyer/Owner Email and Confirmation Email )
            foreach ( $owner_emails as $ticket_type_id => $_owner_emails ) {
                foreach ( $_owner_emails as $key => $owner_email ) {
                    $verification_email = ( isset( $verification_emails[$ticket_type_id] ) ? $verification_emails[$ticket_type_id][$key] : '' );
                    if ( $owner_email && !filter_var( $owner_email, FILTER_VALIDATE_EMAIL ) ) {
                        // wc_add_notice( __( 'Please enter a valid attendee\'s email.', 'woocommerce-tickera-bridge' ), 'error' );
                        throw new Exception(__( 'Please enter a valid attendee\'s email.', 'woocommerce-tickera-bridge' ));
                        break 2;
                    } elseif ( 'yes' == $verify_email && $verification_email && !filter_var( $verification_email, FILTER_VALIDATE_EMAIL ) ) {
                        // wc_add_notice( __( 'Please enter a valid attendee\'s confirmation email.', 'woocommerce-tickera-bridge' ), 'error' );
                        throw new Exception(__( 'Please enter a valid attendee\'s confirmation email.', 'woocommerce-tickera-bridge' ));
                        break 2;
                    } elseif ( 'yes' == $verify_email && $owner_email && $verification_email && $owner_email != $verification_email ) {
                        // wc_add_notice( __( 'Attendee\'s confirmation email did not match.', 'woocommerce-tickera-bridge' ), 'error' );
                        throw new Exception(__( 'Attendee\'s confirmation email did not match.', 'woocommerce-tickera-bridge' ));
                        break 2;
                    }
                }
            }
        }

        /**
         * Change the name of the owner form template meta
         *
         * @param string $meta_name
         * @return string
         */
        function modify_custom_forms_owner_form_template_meta( $meta_name ) {
            return '_owner_form_template';
        }

        /**
         * Disable Tickera gateways network menu item
         *
         * @return boolean
         */
        function modify_add_network_admin_menu() {
            return false;
        }

        /**
         * Add additional content to the Woo emails
         *
         * @param object $order
         * @param mixed $sent_to_admin
         * @param mixed $plain_text
         * @param bool $email
         * @return bool|void
         * @global array $tc_general_settings
         */
        function tc_add_content_email_after_order_table(
            $order,
            $sent_to_admin,
            $plain_text,
            $email = false
        ) {
            if ( doing_action( 'woocommerce_email_after_order_table' ) && apply_filters( 'tc_bridge_for_woocommerce_content_order_table_is_after', true ) || doing_action( 'woocommerce_email_before_order_table' ) && !apply_filters( 'tc_bridge_for_woocommerce_content_order_table_is_after', true ) ) {
                /*
                 * Modify the Tickets Table placement in email.
                 * Filter hook: tc_bridge_for_woocommerce_content_order_table_is_after
                 * Return value: boolean
                 * Default: true
                 */
            } else {
                return false;
            }
            $allowed_email_templates_ids = apply_filters( 'tc_add_content_email_after_order_table_allowed_email_type_ids', array(
                'customer_invoice',
                'customer_processing_order',
                'customer_completed_order',
                'customer_on_hold_order',
                'customer_invoice',
                'new_order'
            ) );
            if ( $email && isset( $email->id ) && !in_array( $email->id, $allowed_email_templates_ids ) ) {
                return;
            }
            $tc_email_settings = get_option( 'tickera_email_setting', false );
            $send_woo_invoice_message = ( isset( $tc_email_settings['send_woo_invoice_message'] ) ? $tc_email_settings['send_woo_invoice_message'] : 'yes' );
            $send_processing_woo_order_message = ( isset( $tc_email_settings['send_processing_woo_order_message'] ) ? $tc_email_settings['send_processing_woo_order_message'] : 'yes' );
            $send_completed_woo_order_message = ( isset( $tc_email_settings['send_completed_woo_order_message'] ) ? $tc_email_settings['send_completed_woo_order_message'] : 'yes' );
            $send_admin_woo_order_placed_message = ( isset( $tc_email_settings['send_admin_woo_order_placed_message'] ) ? $tc_email_settings['send_admin_woo_order_placed_message'] : 'yes' );
            $send_admin_woo_order_completed_message = ( isset( $tc_email_settings['send_admin_woo_order_completed_message'] ) ? $tc_email_settings['send_admin_woo_order_completed_message'] : 'yes' );
            $order_id = ( method_exists( $order, 'get_id' ) ? $order->get_id() : $order->id );
            $tickets_table = tickera_get_tickets_table_email( $order_id );
            // Client Invoice
            if ( 'customer_invoice' == $email->id && $this->allowed_tickets_download( $order ) && !$sent_to_admin && 'yes' == $send_woo_invoice_message ) {
                $woo_invoice_message_content = ( isset( $tc_email_settings['woo_invoice_message_content'] ) ? $tc_email_settings['woo_invoice_message_content'] : '<strong>' . __( 'Tickets:', 'woocommerce-tickera-bridge' ) . '</strong><br />TICKETS_TABLE' );
                $woo_invoice_message_content = str_replace( 'TICKETS_TABLE', $tickets_table, $woo_invoice_message_content );
                $woo_invoice_message_content = stripslashes( $woo_invoice_message_content );
                echo ( '<br />' . $woo_invoice_message_content ? wpautop( $woo_invoice_message_content ) : '' );
            }
            // Client Processing
            if ( in_array( $email->id, ['customer_processing_order', 'new_order'] ) && $order->has_status( 'processing' ) && $this->allowed_tickets_download( $order ) && !$sent_to_admin && 'yes' == $send_processing_woo_order_message ) {
                $processing_woo_order_message_content = ( isset( $tc_email_settings['processing_woo_order_message_content'] ) ? $tc_email_settings['processing_woo_order_message_content'] : '<strong>' . __( 'Tickets:', 'woocommerce-tickera-bridge' ) . '</strong><br />TICKETS_TABLE' );
                $processing_woo_order_message_content = str_replace( 'TICKETS_TABLE', $tickets_table, $processing_woo_order_message_content );
                $processing_woo_order_message_content = stripslashes( $processing_woo_order_message_content );
                echo ( '<br />' . $processing_woo_order_message_content ? wpautop( $processing_woo_order_message_content ) : '' );
            }
            // Client Completed
            if ( in_array( $email->id, ['customer_completed_order', 'new_order'] ) && $order->has_status( 'completed' ) && $this->allowed_tickets_download( $order ) && !$sent_to_admin && 'yes' == $send_completed_woo_order_message ) {
                $completed_woo_order_message_content = ( isset( $tc_email_settings['completed_woo_order_message_content'] ) ? $tc_email_settings['completed_woo_order_message_content'] : '<strong>' . __( 'Tickets:', 'woocommerce-tickera-bridge' ) . '</strong><br />TICKETS_TABLE' );
                $completed_woo_order_message_content = str_replace( 'TICKETS_TABLE', $tickets_table, $completed_woo_order_message_content );
                $completed_woo_order_message_content = stripslashes( $completed_woo_order_message_content );
                echo ( '<br />' . $completed_woo_order_message_content ? wpautop( $completed_woo_order_message_content ) : '' );
            }
            // Admin Order Placed Email
            if ( 'new_order' == $email->id && in_array( $order->get_status(), ['processing', 'completed'] ) && $this->allowed_tickets_download( $order ) && $sent_to_admin && 'yes' == $send_admin_woo_order_placed_message ) {
                $admin_woo_order_placed_message_content = ( isset( $tc_email_settings['admin_woo_order_placed_message_content'] ) ? $tc_email_settings['admin_woo_order_placed_message_content'] : '<strong>' . __( 'Tickets:', 'woocommerce-tickera-bridge' ) . '</strong><br />TICKETS_TABLE' );
                $admin_woo_order_placed_message_content = str_replace( 'TICKETS_TABLE', $tickets_table, $admin_woo_order_placed_message_content );
                $admin_woo_order_placed_message_content = stripslashes( $admin_woo_order_placed_message_content );
                echo ( '<br />' . $admin_woo_order_placed_message_content ? wpautop( $admin_woo_order_placed_message_content ) : '' );
            }
        }

        /**
         * Add extra email fields / options
         *
         * @param array $fields
         * @return array additional settings fields for emails section
         */
        function modify_settings_email_fields( $fields ) {
            $woo_invoice_email_content = array(array(
                'field_name'    => 'send_woo_invoice_message',
                'field_title'   => __( 'Enable', 'woocommerce-tickera-bridge' ),
                'field_type'    => 'function',
                'function'      => 'tickera_yes_no_email',
                'default_value' => 'yes',
                'tooltip'       => __( 'Whether to append content to the WooCommerce invoice e-mails.', 'woocommerce-tickera-bridge' ),
                'section'       => 'woo_invoice_email_content',
            ), array(
                'field_name'        => 'woo_invoice_message_content',
                'field_title'       => __( 'Message Content', 'woocommerce-tickera-bridge' ),
                'field_type'        => 'function',
                'function'          => 'tickera_get_admin_order_message',
                'default_value'     => '<strong>' . __( 'Tickets', 'woocommerce-tickera-bridge' ) . ':</strong><br />TICKETS_TABLE',
                'field_description' => __( 'Content of the message. You can use following placeholder: TICKETS_TABLE. This placeholder shows ordered tickets with the download links.', 'woocommerce-tickera-bridge' ),
                'section'           => 'woo_invoice_email_content',
                'conditional'       => array(
                    'field_name' => 'send_woo_invoice_message',
                    'field_type' => 'radio',
                    'value'      => 'no',
                    'action'     => 'hide',
                ),
            ));
            $woo_processing_email_content = array(array(
                'field_name'    => 'send_processing_woo_order_message',
                'field_title'   => __( 'Enable', 'woocommerce-tickera-bridge' ),
                'field_type'    => 'function',
                'function'      => 'tickera_yes_no_email',
                'default_value' => 'yes',
                'tooltip'       => __( 'Whether to append content to the WooCommerce processing e-mails.', 'woocommerce-tickera-bridge' ),
                'section'       => 'woo_processing_email_content',
            ), array(
                'field_name'        => 'processing_woo_order_message_content',
                'field_title'       => __( 'Message Content', 'woocommerce-tickera-bridge' ),
                'field_type'        => 'function',
                'function'          => 'tickera_get_admin_order_message',
                'default_value'     => '<strong>' . __( 'Tickets', 'woocommerce-tickera-bridge' ) . ':</strong><br />TICKETS_TABLE',
                'field_description' => __( 'Content of the message. You can use following placeholder: TICKETS_TABLE. This placeholder shows ordered tickets with the download links.', 'woocommerce-tickera-bridge' ),
                'section'           => 'woo_processing_email_content',
                'conditional'       => array(
                    'field_name' => 'send_processing_woo_order_message',
                    'field_type' => 'radio',
                    'value'      => 'no',
                    'action'     => 'hide',
                ),
            ));
            $woo_completed_email_content = array(array(
                'field_name'    => 'send_completed_woo_order_message',
                'field_title'   => __( 'Enable', 'woocommerce-tickera-bridge' ),
                'field_type'    => 'function',
                'function'      => 'tickera_yes_no_email',
                'default_value' => 'yes',
                'tooltip'       => __( 'Whether to append content to the WooCommerce completed e-mails.', 'woocommerce-tickera-bridge' ),
                'section'       => 'woo_completed_email_content',
            ), array(
                'field_name'        => 'completed_woo_order_message_content',
                'field_title'       => __( 'Message Content', 'woocommerce-tickera-bridge' ),
                'field_type'        => 'function',
                'function'          => 'tickera_get_admin_order_message',
                'default_value'     => '<strong>' . __( 'Tickets', 'woocommerce-tickera-bridge' ) . ':</strong><br />TICKETS_TABLE',
                'field_description' => __( 'Content of the message. You can use following placeholder: TICKETS_TABLE. This placeholder shows ordered tickets with the download links.', 'woocommerce-tickera-bridge' ),
                'section'           => 'woo_completed_email_content',
                'conditional'       => array(
                    'field_name' => 'send_completed_woo_order_message',
                    'field_type' => 'radio',
                    'value'      => 'no',
                    'action'     => 'hide',
                ),
            ));
            $admin_order_placed_email_content = array(array(
                'field_name'    => 'send_admin_woo_order_placed_message',
                'field_title'   => __( 'Enable', 'woocommerce-tickera-bridge' ),
                'field_type'    => 'function',
                'function'      => 'tickera_yes_no_email',
                'default_value' => 'yes',
                'tooltip'       => __( 'Whether to send e-mail notification to admin when order is placed.', 'woocommerce-tickera-bridge' ),
                'section'       => 'admin_woo_order_placed_email_content',
            ), array(
                'field_name'        => 'admin_woo_order_placed_message_content',
                'field_title'       => __( 'Message Content', 'woocommerce-tickera-bridge' ),
                'field_type'        => 'function',
                'function'          => 'tickera_get_admin_order_message',
                'default_value'     => '<strong>' . __( 'Tickets', 'woocommerce-tickera-bridge' ) . ':</strong><br />TICKETS_TABLE',
                'field_description' => __( 'Content of the message. You can use following placeholder: TICKETS_TABLE. This placeholder shows ordered tickets with the download links.', 'woocommerce-tickera-bridge' ),
                'section'           => 'admin_woo_order_placed_email_content',
                'conditional'       => array(
                    'field_name' => 'send_admin_woo_order_placed_message',
                    'field_type' => 'radio',
                    'value'      => 'no',
                    'action'     => 'hide',
                ),
            ));
            $admin_woo_completed_email_content = array(
                array(
                    'field_name'    => 'send_admin_woo_order_completed_message',
                    'field_title'   => __( 'Enable', 'woocommerce-tickera-bridge' ),
                    'field_type'    => 'function',
                    'function'      => 'tickera_yes_no_email',
                    'default_value' => 'yes',
                    'tooltip'       => __( 'Whether to send e-mail notification to admin when an order has completed', 'woocommerce-tickera-bridge' ),
                    'section'       => 'admin_woo_completed_email_content',
                ),
                array(
                    'field_name'    => 'admin_woo_order_completed_subject',
                    'field_title'   => __( 'Subject', 'woocommerce-tickera-bridge' ),
                    'field_type'    => 'option',
                    'default_value' => __( 'Order Completed', 'woocommerce-tickera-bridge' ),
                    'tooltip'       => __( 'Subject of the e-mail', 'woocommerce-tickera-bridge' ),
                    'section'       => 'admin_woo_completed_email_content',
                    'conditional'   => array(
                        'field_name' => 'send_admin_woo_order_completed_message',
                        'field_type' => 'radio',
                        'value'      => 'no',
                        'action'     => 'hide',
                    ),
                ),
                array(
                    'field_name'    => 'admin_woo_order_completed_from_name',
                    'field_title'   => __( 'From Name', 'woocommerce-tickera-bridge' ),
                    'field_type'    => 'option',
                    'default_value' => get_option( 'blogname' ),
                    'tooltip'       => __( 'This name will appear as sent from name in the e-mail', 'woocommerce-tickera-bridge' ),
                    'section'       => 'admin_woo_completed_email_content',
                    'conditional'   => array(
                        'field_name' => 'send_admin_woo_order_completed_message',
                        'field_type' => 'radio',
                        'value'      => 'no',
                        'action'     => 'hide',
                    ),
                ),
                array(
                    'field_name'    => 'admin_woo_order_completed_to_email',
                    'field_title'   => __( 'To E-mail Address', 'woocommerce-tickera-bridge' ),
                    'field_type'    => 'option',
                    'default_value' => get_option( 'admin_email' ),
                    'tooltip'       => __( 'This is the e-mail address where the email will be sent. If you want to send it to multiple addresses at once, you can separate them by comma like this admin1@example.com,admin2@example.com' ),
                    'section'       => 'admin_woo_completed_email_content',
                    'conditional'   => array(
                        'field_name' => 'send_admin_woo_order_completed_message',
                        'field_type' => 'radio',
                        'value'      => 'no',
                        'action'     => 'hide',
                    ),
                ),
                array(
                    'field_name'        => 'admin_woo_order_completed_message_content',
                    'field_title'       => __( 'Message Content', 'woocommerce-tickera-bridge' ),
                    'field_type'        => 'function',
                    'function'          => 'tickera_get_admin_order_message',
                    'default_value'     => 'Hello, <br /><br />An order ORDER_ID totalling <strong>ORDER_TOTAL</strong> has been completed. <br /><br />You can check the order details here ORDER_DETAILS_URL',
                    'field_description' => __( 'Content of the message. You can use following placeholders (ORDER_ID, ORDER_TOTAL, ORDER_DETAILS_URL, BUYER_NAME, EVENT_NAME, EVENT_LOCATION, TICKETS_TABLE)', 'woocommerce-tickera-bridge' ),
                    'section'           => 'admin_woo_completed_email_content',
                    'conditional'       => array(
                        'field_name' => 'send_admin_woo_order_completed_message',
                        'field_type' => 'radio',
                        'value'      => 'no',
                        'action'     => 'hide',
                    ),
                )
            );
            $admin_woo_refunded_email_content = array(
                array(
                    'field_name'    => 'send_admin_woo_order_refunded_message',
                    'field_title'   => __( 'Enable', 'woocommerce-tickera-bridge' ),
                    'field_type'    => 'function',
                    'function'      => 'tickera_yes_no_email',
                    'default_value' => 'yes',
                    'tooltip'       => __( 'Whether to send e-mail notification to admin when an order has refunded', 'woocommerce-tickera-bridge' ),
                    'section'       => 'admin_woo_refunded_email_content',
                ),
                array(
                    'field_name'    => 'admin_woo_order_refunded_subject',
                    'field_title'   => __( 'Subject', 'woocommerce-tickera-bridge' ),
                    'field_type'    => 'option',
                    'default_value' => __( 'Order Refunded', 'woocommerce-tickera-bridge' ),
                    'tooltip'       => __( 'Subject of the e-mail', 'woocommerce-tickera-bridge' ),
                    'section'       => 'admin_woo_refunded_email_content',
                    'conditional'   => array(
                        'field_name' => 'send_admin_woo_order_refunded_message',
                        'field_type' => 'radio',
                        'value'      => 'no',
                        'action'     => 'hide',
                    ),
                ),
                array(
                    'field_name'    => 'admin_woo_order_refunded_from_name',
                    'field_title'   => __( 'From Name', 'woocommerce-tickera-bridge' ),
                    'field_type'    => 'option',
                    'default_value' => get_option( 'blogname' ),
                    'tooltip'       => __( 'This name will appear as sent from name in the e-mail', 'woocommerce-tickera-bridge' ),
                    'section'       => 'admin_woo_refunded_email_content',
                    'conditional'   => array(
                        'field_name' => 'send_admin_woo_order_refunded_message',
                        'field_type' => 'radio',
                        'value'      => 'no',
                        'action'     => 'hide',
                    ),
                ),
                array(
                    'field_name'    => 'admin_woo_order_refunded_to_email',
                    'field_title'   => __( 'To E-mail Address', 'woocommerce-tickera-bridge' ),
                    'field_type'    => 'option',
                    'default_value' => get_option( 'admin_email' ),
                    'tooltip'       => __( 'This is the e-mail address where the email will be sent. If you want to send it to multiple addresses at once, you can separate them by comma like this admin1@example.com,admin2@example.com' ),
                    'section'       => 'admin_woo_refunded_email_content',
                    'conditional'   => array(
                        'field_name' => 'send_admin_woo_order_refunded_message',
                        'field_type' => 'radio',
                        'value'      => 'no',
                        'action'     => 'hide',
                    ),
                ),
                array(
                    'field_name'        => 'admin_woo_order_refunded_message_content',
                    'field_title'       => __( 'Message Content', 'woocommerce-tickera-bridge' ),
                    'field_type'        => 'function',
                    'function'          => 'tickera_get_admin_order_message',
                    'default_value'     => 'Hello, <br /><br />An order ORDER_ID totalling <strong>ORDER_TOTAL</strong> has been refunded. <br /><br />You can check the order details here ORDER_DETAILS_URL',
                    'field_description' => __( 'Content of the message. You can use following placeholders (ORDER_ID, ORDER_TOTAL, ORDER_DETAILS_URL, BUYER_NAME, EVENT_NAME, EVENT_LOCATION)', 'woocommerce-tickera-bridge' ),
                    'section'           => 'admin_woo_refunded_email_content',
                    'conditional'       => array(
                        'field_name' => 'send_admin_woo_order_refunded_message',
                        'field_type' => 'radio',
                        'value'      => 'no',
                        'action'     => 'hide',
                    ),
                )
            );
            $fields = array_merge(
                $fields,
                array_values( $woo_processing_email_content ),
                array_values( $woo_completed_email_content ),
                array_values( $admin_order_placed_email_content ),
                array_values( $admin_woo_completed_email_content ),
                array_values( $admin_woo_refunded_email_content ),
                array_values( $woo_invoice_email_content )
            );
            return $fields;
        }

        /**
         * Remove email settings sections and add new ones
         *
         * @param array $sections
         * @return array
         */
        function modify_settings_email_sections( $sections ) {
            $array_fields_to_unset = array(
                'client_order_completed_email',
                'admin_order_refunded_email',
                'admin_order_completed_email',
                'admin_order_placed_email',
                'client_order_placed_email',
                'misc_email'
            );
            $miscellaneous = [];
            foreach ( $sections as $index => $section ) {
                if ( in_array( $section['name'], $array_fields_to_unset ) ) {
                    if ( 'misc_email' == $section['name'] ) {
                        $miscellaneous = $section;
                    }
                    unset($sections[$index]);
                }
            }
            $sections[] = array(
                'name'        => 'woo_invoice_email_content',
                'title'       => __( 'Client Invoice E-mail', 'woocommerce-tickera-bridge' ),
                'description' => __( 'E-mail content shown after order details in the WooCommerce email invoice.', 'woocommerce-tickera-bridge' ),
            );
            $sections[] = array(
                'name'        => 'woo_processing_email_content',
                'title'       => __( 'Client Order Processing E-mail', 'woocommerce-tickera-bridge' ),
                'description' => __( 'E-mail content shown after order details in the WooCommerce order processing e-mail.', 'woocommerce-tickera-bridge' ),
            );
            $sections[] = array(
                'name'        => 'woo_completed_email_content',
                'title'       => __( 'Client Order Completed E-mail', 'woocommerce-tickera-bridge' ),
                'description' => __( 'E-mail content shown after order details in the WooCommerce order completed e-mail.', 'woocommerce-tickera-bridge' ),
            );
            $sections[] = array(
                'name'        => 'admin_woo_order_placed_email_content',
                'title'       => __( 'Admin Order Placed E-mail', 'woocommerce-tickera-bridge' ),
                'description' => '',
            );
            $sections[] = array(
                'name'        => 'admin_woo_completed_email_content',
                'title'       => __( 'Admin Order Completed E-mail', 'woocommerce-tickera-bridge' ),
                'description' => '',
            );
            $sections[] = array(
                'name'        => 'admin_woo_refunded_email_content',
                'title'       => __( 'Admin Order Refunded E-mail', 'woocommerce-tickera-bridge' ),
                'description' => '',
            );
            // Position Miscellaneous Section at the very last.
            $sections[] = $miscellaneous;
            return $sections;
        }

        /**
         * Remove miscellaneous fields from Tickera general settings
         *
         * @param array $fields
         * @return array
         */
        function modify_settings_miscellaneous_fields( $fields ) {
            $array_fields_to_unset = array('use_order_details_pretty_links', 'delete_pending_orders');
            $index = 0;
            foreach ( $fields as $field ) {
                if ( in_array( $field['field_name'], $array_fields_to_unset ) ) {
                    unset($fields[$index]);
                }
                $index++;
            }
            return $fields;
        }

        /**
         * Remove unneeded general settings sections
         *
         * @param array $sections
         * @return array
         */
        function modify_settings_general_sections( $sections ) {
            $sections[] = array(
                'name'        => 'tc_woo_settings',
                'title'       => __( 'WooCommerce', 'woocommerce-tickera-bridge' ),
                'description' => '',
            );
            unset($sections[1]);
            //Remove Pages section from the general settings
            unset($sections[2]);
            //Remove Menu section from the general settings
            return $sections;
        }

        /**
         * Remove Tickera General Settings fields not needed for WooCommerce and add new ones
         *
         * @param array $fields
         * @return array
         */
        function modify_general_settings_store_fields( $fields ) {
            $array_fields_to_unset = apply_filters( 'woobridge_hidden_fields', array(
                'currencies',
                'currency_symbol',
                'currency_position',
                'price_format',
                'tax_rate',
                'tax_inclusive',
                'show_tax_rate',
                'tax_label',
                'tax_before_fees',
                'use_global_fees',
                'global_fee_type',
                'global_fee_scope',
                'global_fee_value',
                'show_fees',
                'fees_label',
                'force_login',
                'show_discount_field'
            ) );
            $index = 0;
            $fields[] = array(
                'field_name'    => 'tc_woo_hide_products',
                'field_title'   => __( 'Hide Tickets', 'woocommerce-tickera-bridge' ),
                'field_type'    => 'function',
                'function'      => 'tickera_yes_no',
                'default_value' => 'no',
                'tooltip'       => __( 'Hide WooCommerce products / tickets from the Store page on the front-end.', 'woocommerce-tickera-bridge' ),
                'section'       => 'tc_woo_settings',
            );
            $fields[] = array(
                'field_name'    => 'tc_woo_redirect_single_post_to_event',
                'field_title'   => __( 'Redirect product single post to an event', 'woocommerce-tickera-bridge' ),
                'field_type'    => 'function',
                'function'      => 'tickera_yes_no',
                'default_value' => 'no',
                'tooltip'       => sprintf( __( 'Whether to redirect ticket product single posts to associated event post. %sNOTE: Variable products won\'t be redirected.%s', 'woocommerce-tickera-bridge' ), '<strong>', '</strong>' ),
                'section'       => 'tc_woo_settings',
            );
            $fields[] = array(
                'field_name'    => 'tc_woo_allow_cash_on_delivery_processing',
                'field_title'   => __( 'Cash on Delivery Ticket Download', 'woocommerce-tickera-bridge' ),
                'field_type'    => 'function',
                'function'      => 'tickera_yes_no',
                'default_value' => 'no',
                'tooltip'       => sprintf( __( 'Allow ticket download for processing order status for "Cash on Delivery" gateway.', 'woocommerce-tickera-bridge' ), '<strong>', '</strong>' ),
                'section'       => 'tc_woo_settings',
            );
            $fields[] = array(
                'field_name'    => 'tc_woo_checkout_owner_fields_placement',
                'field_title'   => __( 'Attendees Information Fields Placement.', 'woocommerce-tickera-bridge' ),
                'field_type'    => 'select',
                'default_value' => 'woocommerce_checkout_after_customer_details',
                'values'        => [
                    'woocommerce_checkout_before_customer_details' => __( 'Before customer details', 'woocommerce-tickera-bridge' ),
                    'woocommerce_checkout_after_customer_details'  => __( 'After customer details', 'woocommerce-tickera-bridge' ),
                    'woocommerce_before_order_notes'               => __( 'Before order notes', 'woocommerce-tickera-bridge' ),
                    'woocommerce_after_order_notes'                => __( 'After order notes', 'woocommerce-tickera-bridge' ),
                    'woocommerce_checkout_before_order_review'     => __( 'Before order review', 'woocommerce-tickera-bridge' ),
                    'woocommerce_checkout_after_order_review'      => __( 'After order review', 'woocommerce-tickera-bridge' ),
                    'woocommerce_checkout_billing'                 => __( 'Within Billing Information', 'woocommerce-tickera-bridge' ),
                    'woocommerce_checkout_shipping'                => __( 'Within Shipping Information', 'woocommerce-tickera-bridge' ),
                ],
                'tooltip'       => sprintf( __( 'Attendees Information fields placement in checkout page. Applicable only with Woocommerce cart/checkout shortcodes.', 'woocommerce-tickera-bridge' ), '<strong>', '</strong>' ),
                'section'       => 'tc_woo_settings',
            );
            foreach ( $fields as $field ) {
                if ( in_array( $field['field_name'], $array_fields_to_unset ) ) {
                    unset($fields[$index]);
                }
                $index++;
            }
            return $fields;
        }

        /**
         * Modify event ticket types
         *
         * @param object $ticket_types
         * @param int $event_id
         * @param $show_variations
         * @param array $post_status
         * @return object
         */
        function modify_get_event_ticket_types(
            $ticket_types,
            $event_id,
            $show_variations,
            $post_status = ['publish', 'private']
        ) {
            $args = array(
                'post_type'      => array('product', 'tc_tickets'),
                'post_status'    => $post_status,
                'posts_per_page' => -1,
                'meta_query'     => array(
                    'relation' => 'OR',
                    array(
                        'key'     => '_event_name',
                        'compare' => '=',
                        'value'   => $event_id,
                    ),
                    array(
                        'key'     => 'event_name',
                        'compare' => '=',
                        'value'   => $event_id,
                    ),
                ),
                'orderby'        => 'menu_order',
                'order'          => 'ASC',
            );
            $ticket_types = get_posts( apply_filters( 'tc_wc_modify_get_event_ticket_types_args', $args ) );
            $ticket_type_index = 0;
            foreach ( $ticket_types as $ticket_type ) {
                if ( get_post_type( $ticket_type->ID ) == 'product' ) {
                    $variations_args = array(
                        'posts_per_page' => -1,
                        'post_type'      => 'product_variation',
                        'post_status'    => $post_status,
                        'post_parent'    => $ticket_type->ID,
                    );
                    $ticket_type_variations = get_posts( $variations_args );
                    if ( count( $ticket_type_variations ) ) {
                        // Product has variations
                        if ( apply_filters( 'woo_modify_get_event_ticket_types_show_variations', $show_variations ) == true ) {
                            $ticket_types = array_merge( $ticket_types, $ticket_type_variations );
                        }
                    }
                }
                $ticket_type_index++;
            }
            return $ticket_types;
        }

        /**
         * Modify check-in order date
         *
         * Get the date from WooCommerce order post_date
         *
         * @param string $date
         * @param int $order_id
         * @return string
         */
        function modify_checkin_order_date( $date, $order_id ) {
            if ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ) {
                $order = new Tickera\TC_Order($order_id);
                $date = strtotime( $order->details->post_date );
            }
            return $date;
        }

        /**
         * Modify buyer first name
         *
         * Get the one associated to Woo order
         *
         * @param string $buyer_first_name
         * @param int $order_id
         * @return string
         */
        function modify_checkin_buyer_first_name( $buyer_first_name, $order_id ) {
            if ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ) {
                $order = new \WC_Order($order_id);
                $buyer_first_name = $order->get_billing_first_name();
            }
            return $buyer_first_name;
        }

        /**
         * Modify buyer last name
         *
         * Get the one associated to Woo order
         *
         * @param string $buyer_last_name
         * @param int $order_id
         * @return string
         */
        function modify_checkin_buyer_last_name( $buyer_last_name, $order_id ) {
            if ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ) {
                $order = new \WC_Order($order_id);
                $buyer_last_name = $order->get_billing_last_name();
            }
            return $buyer_last_name;
        }

        /**
         * Modify full name
         *
         * Get the one associated to Woo order
         *
         * @param string $buyer_name
         * @param int $order_id
         * @return string
         */
        function modify_checkin_buyer_full_name( $buyer_name, $order_id ) {
            if ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ) {
                $order = new WC_Order($order_id);
                $first_name = $order->get_billing_first_name();
                $last_name = $order->get_billing_last_name();
                $buyer_name = $first_name . ' ' . $last_name;
            }
            return $buyer_name;
        }

        /**
         * Modify email
         *
         * Get the one associated to Woo order
         *
         * @param string $buyer_email
         * @param int $order_id
         * @return string
         */
        function modify_checkin_buyer_email( $buyer_email, $order_id ) {
            if ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ) {
                $order = new WC_Order($order_id);
                $buyer_email = $order->get_billing_email();
            }
            return $buyer_email;
        }

        /**
         * Add "Tickets" meta box to WooCommerce orders where order contains tickets
         *
         * @param $screen_id
         * @param $order
         */
        function add_meta_boxes( $screen_id, $order ) {
            if ( in_array( $screen_id, ['shop_order', 'woocommerce_page_wc-orders'] ) ) {
                $hpos = tc_wb_hpos();
                $post_id = ( $hpos->status ? $order->get_id() : $order->ID );
                $cart_contents = get_post_meta( $post_id, 'tc_cart_contents' );
                if ( $cart_contents && count( $cart_contents ) > 0 ) {
                    add_meta_box(
                        'tc-order-details-tickets',
                        __( 'Tickets', 'woocommerce-tickera-bridge' ),
                        'TC_WooCommerce_Bridge::tc_get_order_tickets',
                        $screen_id,
                        'normal',
                        'default'
                    );
                }
                if ( defined( 'tc_woo_has_manual_tickets' ) && tc_woo_has_manual_tickets == true ) {
                    add_meta_box(
                        'tc-order-details-tickets-edit',
                        __( 'Book Tickets', 'woocommerce-tickera-bridge' ),
                        'TC_WooCommerce_Bridge::tc_get_order_tickets_additional',
                        $screen_id,
                        'normal',
                        'default'
                    );
                }
            }
        }

        /**
         * Collection of Ticket Types
         * @return bool|false|float|string
         */
        function tc_get_ticket_instances() {
            $args = array(
                'posts_per_page' => -1,
                'order'          => 'ASC',
                'meta_key'       => '_tc_is_ticket',
                'meta_value'     => 'yes',
                'post_type'      => 'product',
                'post_status'    => 'publish',
            );
            $collection = [];
            foreach ( get_posts( $args ) as $key => $val ) {
                $collection[$key]['id'] = $val->ID;
                $collection[$key]['text'] = $val->post_title;
            }
            return json_encode( $collection );
        }

        /**
         * Retrieve ticket details from the current order
         * Called from Woo admin Order page
         * @param object $post
         */
        public static function tc_get_order_tickets( $post ) {
            $hpos = tc_wb_hpos();
            $post_id = ( $hpos->status ? $post->get_id() : $post->ID );
            ?>
            <form>
                <input type="hidden" name="hiddenField" />
            </form>
            <script type="text/javascript">
                jQuery(document).ready(function ($) {
                    var replaceWithInput = $('<input name="temp" class="tc_temp_value" type="text" />'),
                        connectWith = $('input[name="hiddenField"]');
                    $('td.first_name, td.last_name, td.owner_email').inlineEdit(replaceWithInput, connectWith);
                });
            </script>
            <?php 
            do_action( 'tc_wb_get_order_tickets_before', $post_id );
            // Get tickets table / list
            tickera_get_order_event( 'tc_cart_contents', $post_id );
            do_action( 'tc_wb_get_order_tickets_after', $post_id );
        }

        public static function get_additional_tickets_tabs() {
            $tickets_content = '';
            ob_start();
            $has_custom_forms = class_exists( 'TC_Custom_Fields' );
            ?>
            <label class="tc_book_tickets_label"><?php 
            _e( 'Event', 'woocommerce-tickera-bridge' );
            ?><br />
                <select name="event_new" class="event_new">
                    <option value=""></option>
                    <?php 
            $events = tickera_get_events_array();
            foreach ( $events as $event_id => $event_title ) {
                ?>
                        <option value="<?php 
                echo esc_attr( $event_id );
                ?>"><?php 
                echo $event_title;
                ?></option>
                        <?php 
            }
            ?>
                </select>
            </label>

            <script type="text/javascript">
                var ticket_types_list = new Array();
                var ticket_types_custom_forms = new Array();
                <?php 
            $args = array(
                'post_type'      => 'product',
                'post_status'    => 'publish',
                'posts_per_page' => -1,
                'meta_query'     => array(
                    'relation' => 'AND',
                    array(
                        'key'     => '_tc_is_ticket',
                        'compare' => '=',
                        'value'   => 'yes',
                    ),
                ),
            );
            $ticket_types = get_posts( $args );
            foreach ( $ticket_types as $ticket_type ) {
                $id = $ticket_type->ID;
                $event_id = get_post_meta( $id, '_event_name', true );
                $title = $ticket_type->post_title;
                $product = wc_get_product( $id );
                ?>

                if (typeof ticket_types_list['<?php 
                echo $event_id;
                ?>'] == "undefined") {
                    ticket_types_list['<?php 
                echo $event_id;
                ?>'] = new Array();
                }
                <?php 
                if ( $product->is_type( 'variable' ) ) {
                    $variations = $product->get_available_variations();
                    foreach ( $variations as $variation ) {
                        ?>if (typeof ticket_types_custom_forms['<?php 
                        echo $variation['variation_id'];
                        ?>'] == "undefined") {
                    ticket_types_custom_forms['<?php 
                        echo $variation['variation_id'];
                        ?>'] = '-1';
                }
                ticket_types_custom_forms['<?php 
                        echo $variation['variation_id'];
                        ?>'] = '<?php 
                        echo tc_get_order_details_custom_forms_fields_values_helper( $id );
                        ?>';
                ticket_types_list['<?php 
                        echo $event_id;
                        ?>']['<?php 
                        echo $variation['variation_id'];
                        ?>'] = '<?php 
                        echo get_the_title( $variation['variation_id'] );
                        ?>';
                <?php 
                    }
                } else {
                    ?>if (typeof ticket_types_custom_forms['<?php 
                    echo $id;
                    ?>'] == "undefined") {
                    ticket_types_custom_forms['<?php 
                    echo $id;
                    ?>'] = '-1';
                }
                ticket_types_custom_forms['<?php 
                    echo $id;
                    ?>'] = '<?php 
                    echo tc_get_order_details_custom_forms_fields_values_helper( $id );
                    ?>';
                ticket_types_list['<?php 
                    echo $event_id;
                    ?>']['<?php 
                    echo $id;
                    ?>'] = '<?php 
                    echo $title;
                    ?>';
                <?php 
                }
                ?>

                <?php 
            }
            ?>
            </script>

            <label class="tc_book_tickets_label"><?php 
            _e( 'Ticket Type', 'woocommerce-tickera-bridge' );
            ?><br />
                <span id="ticket_type_new_holder">
                    <select name="ticket_type_new" id="ticket_type_new"></select>
                </span>
            </label>

            <?php 
            $tickets_content = ob_get_clean();
            $seats_content = 'Seats content';
            $tabs = array(
                'tickets' => array(__( 'Book Tickets', 'woocommerce-tickera-bridge' ), $tickets_content),
            );
            return apply_filters( 'tc_additional_tickets_tabs', $tabs );
        }

        public static function tc_get_order_tickets_additional() {
            $tabs = TC_WooCommerce_Bridge::get_additional_tickets_tabs();
            $tabs_count = count( $tabs );
            ?>
            <div id="tc_tickets_additional_tabs" class="<?php 
            echo 'tc_tickets_additional_tabs_' . (int) $tabs_count;
            ?>">
                <ul>
                    <?php 
            foreach ( $tabs as $tab_key => $tab_info ) {
                ?>
                        <li class="tc_tickets_additional_tab_title"><a href="#<?php 
                echo esc_attr( $tab_key );
                ?>"><?php 
                echo $tab_info[0];
                ?></a></li>
                    <?php 
            }
            ?>
                </ul>
                <?php 
            foreach ( $tabs as $tab_key => $tab_info ) {
                ?>
                    <div id="<?php 
                echo esc_attr( $tab_key );
                ?>" class="tc_tickets_additional_tab_content">
                        <?php 
                echo $tab_info[1];
                ?>
                    </div>
                <?php 
            }
            ?>
            </div>
            <?php 
        }

        /**
         * Get Add New Ticket metabox
         */
        public static function tc_get_order_tickets_additional_old() {
            global $pagenow;
            if ( isset( $_GET['action'] ) && $_GET['action'] == 'edit' || in_array( $pagenow, array('post-new.php') ) ) {
                //$tc_orders = new TC_Orders();
                //$fields = $tc_orders->get_owner_info_fields();
                $has_custom_forms = class_exists( 'TC_Custom_Fields' );
                ?>
                <table class="order-details widefat shadow-table">
                    <tbody>
                    <tr>
                        <th><?php 
                _e( 'Event', 'woocommerce-tickera-bridge' );
                ?></th>
                        <th><?php 
                _e( 'Ticket Type', 'woocommerce-tickera-bridge' );
                ?></th>
                        <th><?php 
                _e( 'First Name', 'woocommerce-tickera-bridge' );
                ?></th>
                        <th><?php 
                _e( 'Last Name', 'woocommerce-tickera-bridge' );
                ?></th>
                        <th><?php 
                _e( 'Attendee E-mail', 'woocommerce-tickera-bridge' );
                ?></th>
                        <?php 
                if ( $has_custom_forms ) {
                    ?>
                            <th class="custom_fields_col"><?php 
                    _e( 'Custom Fields', 'woocommerce-tickera-bridge' );
                    ?></th>
                        <?php 
                }
                ?>
                    </tr>
                    <tr class="alternate">
                        <td class="event_id">
                            <select name="event_new" class="event_new">
                                <option value=""></option>
                                <?php 
                $events = tickera_get_events_array();
                foreach ( $events as $event_id => $event_title ) {
                    ?>
                                    <option value="<?php 
                    echo esc_attr( $event_id );
                    ?>"><?php 
                    echo $event_title;
                    ?></option>
                                    <?php 
                }
                ?>
                            </select>
                        </td>
                        <td class="ticket_type_id">
                            <script type="text/javascript">
                                var ticket_types_list = new Array();
                                var ticket_types_custom_forms = new Array();
                                <?php 
                $args = array(
                    'post_type'      => 'product',
                    'post_status'    => 'publish',
                    'posts_per_page' => -1,
                    'meta_query'     => array(
                        'relation' => 'AND',
                        array(
                            'key'     => '_tc_is_ticket',
                            'compare' => '=',
                            'value'   => 'yes',
                        ),
                    ),
                );
                $ticket_types = get_posts( $args );
                foreach ( $ticket_types as $ticket_type ) {
                    $id = $ticket_type->ID;
                    $event_id = get_post_meta( $id, '_event_name', true );
                    $title = $ticket_type->post_title;
                    $product = wc_get_product( $id );
                    ?>

                                if (typeof ticket_types_list['<?php 
                    echo $event_id;
                    ?>'] == "undefined") {
                                    ticket_types_list['<?php 
                    echo $event_id;
                    ?>'] = new Array();
                                }
                                <?php 
                    if ( $product->is_type( 'variable' ) ) {
                        $variations = $product->get_available_variations();
                        foreach ( $variations as $variation ) {
                            ?>if (typeof ticket_types_custom_forms['<?php 
                            echo $variation['variation_id'];
                            ?>'] == "undefined") {
                                    ticket_types_custom_forms['<?php 
                            echo $variation['variation_id'];
                            ?>'] = '-1';
                                }
                                ticket_types_custom_forms['<?php 
                            echo $variation['variation_id'];
                            ?>'] = '<?php 
                            echo tc_get_order_details_custom_forms_fields_values_helper( $id );
                            ?>';
                                ticket_types_list['<?php 
                            echo $event_id;
                            ?>']['<?php 
                            echo $variation['variation_id'];
                            ?>'] = '<?php 
                            echo get_the_title( $variation['variation_id'] );
                            ?>';
                                <?php 
                        }
                    } else {
                        ?>if (typeof ticket_types_custom_forms['<?php 
                        echo $id;
                        ?>'] == "undefined") {
                                    ticket_types_custom_forms['<?php 
                        echo $id;
                        ?>'] = '-1';
                                }
                                ticket_types_custom_forms['<?php 
                        echo $id;
                        ?>'] = '<?php 
                        echo tc_get_order_details_custom_forms_fields_values_helper( $id );
                        ?>';
                                ticket_types_list['<?php 
                        echo $event_id;
                        ?>']['<?php 
                        echo $id;
                        ?>'] = '<?php 
                        echo $title;
                        ?>';
                                <?php 
                    }
                    ?>

                                <?php 
                }
                ?>
                            </script>
                            <div id="ticket_type_new_holder">
                                <select name="ticket_type_new" id="ticket_type_new"></select>
                            </div>

                        </td>
                        <td class="first_name_new">
                            <input type="text" name="first_name_new" class="first_name_new" value="" />
                        </td>
                        <td class="last_name_new">
                            <input type="text" name="last_name_new" class="last_name_new" value="" />
                        </td>
                        <td class="owner_email_new">
                            <input type="text" name="owner_email_new" class="owner_email_new" value="" />
                        </td>
                        <?php 
                if ( $has_custom_forms ) {
                    ?>
                            <td class="custom_fields">
                                -
                            </td>
                        <?php 
                }
                ?>
                    </tr>
                    </tbody>
                </table>
                <?php 
                submit_button(
                    __( 'Add Ticket', 'woocommerce-tickera-bridge' ),
                    'primary',
                    'add-ticket-submit',
                    false
                );
                ?>
                <br clear="all" />
                <?php 
            }
        }

        /**
         * Modify ticket download URL (Used on the front-end )
         *
         * @param string $url
         * @param string $order_key
         * @param int $ticket_id
         * @return string
         */
        function tc_modify_download_ticket_url_front( $url, $order_key, $ticket_id ) {
            $ticket_instance = new Tickera\TC_Ticket_Instance($ticket_id);
            $order_id = $ticket_instance->details->post_parent;
            if ( get_post_type( $order_id ) == 'tc_orders' ) {
                return $url;
            }
            return str_replace( ' ', '', trailingslashit( site_url() ) . '?download_ticket=' . $ticket_id . '&order_key=' . $order_key . '&nonce=' . wp_hash( $ticket_id . $order_key ) );
        }

        /**
         * Add tickets table on the Woo order details page
         *
         * @param object $order
         */
        function tc_add_tickets_table_on_woo_order_details_page( $order ) {
            if ( is_int( $order ) ) {
                $order_id = (int) $order;
            } else {
                $order_id = ( method_exists( $order, 'get_id' ) ? (int) $order->get_id() : (int) $order->id );
            }
            $wc_order = new WC_Order($order_id);
            if ( is_object( $wc_order ) ) {
                if ( $this->allowed_tickets_download( $order ) ) {
                    $order_id = ( method_exists( $order, 'get_id' ) ? (int) $order->get_id() : (int) $order->id );
                    tickera_order_details_table_front( $order_id );
                }
            }
        }

        /**
         * Shows event shortcode on the events page in the admin
         *
         * @param type $shortcode
         * @param type $event_id
         * @return string
         */
        function tc_event_shortcode_column_modify( $shortcode, $event_id ) {
            return '[tc_wb_event id="' . $event_id . '"]';
        }

        /**
         * Modify Tickera admin menu items
         *
         * @param array $menu_items
         * @return array
         */
        function tc_modify_plugin_admin_menu_items( $menu_items ) {
            unset($menu_items['ticket_types']);
            unset($menu_items['discount_codes']);
            unset($menu_items['orders']);
            return $menu_items;
        }

        /**
         * @param $status
         * @param $order
         * @return string
         */
        function tc_modify_order_details_post_status( $status, $order ) {
            return wc_get_order( $order->details->ID )->get_status();
        }

        /**
         * Modify if order is paid based on Woo post status
         *
         * @param boolean $is_paid
         * @param int $post_id
         * @return boolean
         */
        function tc_modify_order_is_paid( $is_paid, $post_id ) {
            if ( in_array( get_post_type( $post_id ), self::get_woo_order_types() ) ) {
                $order = wc_get_order( $post_id );
                $post_status = $order->get_status();
                if ( in_array( $post_status, ['completed', 'processing'] ) ) {
                    return true;
                } else {
                    return false;
                }
            }
            return $is_paid;
        }

        function tc_modify_order_paid_statuses( $paid_statuses ) {
            $paid_statuses = ['wc-completed', 'wc-processing'];
            if ( tc_wb_hpos()->status ) {
                $paid_statuses[] = 'draft';
            }
            return $paid_statuses;
        }

        function tc_modify_order_post_type_name( $order_post_type ) {
            $order_post_type = ['tc_orders', 'shop_order', 'shop_order_placehold'];
            return $order_post_type;
        }

        /**
         * Modify Order admin url if needed
         *
         * @param string $url
         * @param int $order_id
         * @param string $order_title
         * @return string
         */
        function tc_modify_ticket_instance_order_admin_url( $url, $order_id, $order_title ) {
            if ( in_array( get_post_type( $order_id ), self::get_woo_order_types() ) ) {
                if ( !current_user_can( 'edit_shop_orders' ) ) {
                    $url = $order_id;
                } else {
                    $url = '<a href = "' . admin_url( 'post.php?post=' . (int) $order_id ) . '&action=edit">' . esc_attr( $order_id ) . '</a>';
                }
            } else {
                $url = $order_title;
            }
            return $url;
        }

        /**
         * Modify post meta key for getting available checkins number
         * @return string
         */
        function tc_modify_available_checkins_per_ticket_field_name() {
            return '_available_checkins_per_ticket';
        }

        /**
         * Modify post meta key to check if checkins time basis is enabled or disabled.
         * @since 1.3.9
         * @return string
         */
        function tc_modify_checkins_time_basis_field_name() {
            return '_checkins_time_basis';
        }

        /**
         * Modify post meta key for the allowed checkins per time basis.
         * @since 1.3.9
         * @return string
         */
        function tc_modify_allowed_checkins_per_time_basis_field_name() {
            return '_allowed_checkins_per_time_basis';
        }

        /**
         * Modify post meta key for checkins time basis type
         * @since 1.3.9
         * @return string
         */
        function tc_modify_checkins_time_basis_type_field_name() {
            return '_checkins_time_basis_type';
        }

        /**
         * Modify post meta key for checkins time calendar basis
         * @since 1.5.5
         * @return string
         */
        function tc_modify_checkins_time_calendar_basis_field_name() {
            return '_checkins_time_calendar_basis';
        }

        /**
         * Modify event meta name
         * @return string
         */
        function tc_modify_event_name_field_name() {
            return '_event_name';
        }

        /**
         * Modify ticket template meta name
         * @return string
         */
        function tc_modify_ticket_template_field_name() {
            return '_ticket_template';
        }

        /**
         * Modify allow ticket checkout meta name
         * @since 1.4.5
         * @return string
         */
        function tc_modify_allow_ticket_checkout() {
            return '_allow_ticket_checkout';
        }

        /**
         * Enqueue front-end styles and scripts
         */
        function front_header() {
            if ( apply_filters( 'tc_woo_use_front_header', true ) == true ) {
                wp_enqueue_style(
                    $this->name . '-front',
                    $this->plugin_url . 'assets/css/front.css',
                    array(),
                    $this->version
                );
                wp_enqueue_script(
                    $this->name . '-front',
                    $this->plugin_url . 'assets/js/front.js',
                    array('jquery'),
                    $this->version
                );
            }
        }

        /**
         * Enqueue admin styles and scripts
         */
        function admin_header() {
            global $post;
            $hpos = tc_wb_hpos();
            if ( $hpos->status ) {
                $post_type = $hpos->post_type;
            } else {
                $post_type = ( $post && isset( $post->post_type ) ? $post->post_type : '' );
            }
            if ( in_array( $post_type, [
                'product',
                'shop_order',
                'shop_order_placehold',
                'tc_events'
            ] ) ) {
                wp_enqueue_script(
                    $this->name . '-admin',
                    $this->plugin_url . 'assets/js/admin.js',
                    array(
                        'jquery',
                        'jquery-ui-core',
                        'jquery-ui-sortable',
                        'jquery-ui-draggable',
                        'jquery-ui-droppable',
                        'jquery-ui-accordion',
                        'wp-color-picker'
                    ),
                    $this->version,
                    false
                );
                wp_enqueue_style(
                    $this->name . '-admin',
                    $this->plugin_url . 'assets/css/admin.css',
                    array(),
                    $this->version
                );
            }
            if ( defined( 'tc_woo_has_manual_tickets' ) && tc_woo_has_manual_tickets == true ) {
                if ( in_array( $post_type, self::get_woo_order_types() ) ) {
                    wp_enqueue_script(
                        $this->name . '-admin-ticket',
                        $this->plugin_url . 'assets/js/admin-ticket.js',
                        array('jquery', 'jquery-ui-tabs'),
                        $this->version,
                        false
                    );
                    wp_localize_script( $this->name . '-admin-ticket', 'tc_woo_ajax', array(
                        'ajaxUrl' => admin_url( 'admin-ajax.php', ( is_ssl() ? 'https' : 'http' ) ),
                    ) );
                }
            }
        }

        /**
         * Create new order via WooCommerce Legacy Rest API.
         *
         * Send attendee order completed email only when order is completed.
         * Added "tc_wb_maybe_send_attendee_order_completed_email" action hook.
         * @since 1.4.4
         *
         * @global mixed $woocommerce
         * @param type $order_id
         * @param type $data
         */
        function tc_api_create_order( $order_id, $data ) {
            /*
             * Make sure to remove expired ticket instances.
             * This will usually occur when a payment has been canceled during checkout.
             */
            $this->delete_associated_tickets( $order_id, false );
            update_post_meta( (int) $order_id, '_tc_paid_date', time() );
            $cart_info = [
                'buyer_data' => [],
                'owner_data' => [],
            ];
            update_post_meta( (int) $order_id, 'tc_cart_info', $cart_info );
            $wc_order = new WC_Order($order_id);
            if ( $this->allowed_tickets_download( $wc_order ) ) {
                do_action( 'tc_wb_allowed_tickets_access', $wc_order );
                if ( 'completed' == $wc_order->get_status() ) {
                    do_action( 'tc_wb_maybe_send_attendee_order_completed_email', $wc_order );
                }
            }
            do_action( 'tc_woo_bridge_after_order_completed', $wc_order );
        }

        /**
         * Update recently created order with meta values.
         *
         * Send attendee order completed email only when order is completed.
         * Added "tc_wb_maybe_send_attendee_order_completed_email" action hook.
         * @param int $order_id
         * @param bool $pre_delete_tickets
         * @throws Exception
         * @since 1.4.4
         *
         * @global mixed $woocommerce
         */
        function tc_order_created( $order_id, $pre_delete_tickets = true ) {
            $_post = self::handle_post( $_POST );
            /*
             * Avoid redundant process on create order via API.
             * As of Woocommerce 8.0.2, woocommerce_api_create_order and woocommerce_new_order are triggered simultaneously.
             *
             * @since 1.4.4
             */
            if ( did_action( 'woocommerce_api_create_order' ) ) {
                return;
            }
            /*
             * Make sure to remove expired ticket instances.
             * Cancelled order duplicates tickets as it reprocess the order line items.
             *
             * Remove associated tickets on resume order. Order line items are being recreated on resume orders.
             * @since 1.6.4
             */
            if ( doing_action( 'woocommerce_resume_order' ) ) {
                $this->delete_associated_tickets( $order_id, false );
            }
            // Prepare buyer and owner data
            $buyer_data = [];
            $owner_data = [];
            foreach ( $_post as $field => $value ) {
                $value = ( is_array( $value ) ? tickera_sanitize_array( $value, false, true ) : sanitize_text_field( $value ) );
                if ( preg_match( '/buyer_data_/', $field ) ) {
                    $buyer_data[str_replace( 'buyer_data_', '', $field )] = $value;
                } elseif ( preg_match( '/owner_data_/', $field ) ) {
                    $owner_data[str_replace( 'owner_data_', '', $field )] = $value;
                }
            }
            $wc_order = wc_get_order( $order_id );
            if ( $this->allowed_tickets_download( $wc_order ) ) {
                do_action( 'tc_wb_allowed_tickets_access', $wc_order );
                if ( 'completed' == $wc_order->get_status() ) {
                    do_action( 'tc_wb_maybe_send_attendee_order_completed_email', $wc_order );
                }
            }
            $cart_info = [
                'buyer_data' => $buyer_data,
                'owner_data' => $owner_data,
            ];
            update_post_meta( (int) $order_id, 'tc_cart_info', $cart_info );
            update_post_meta( (int) $order_id, '_tc_paid_date', time() );
            do_action( 'tc_woo_bridge_after_order_completed', $wc_order );
        }

        /**
         * Save Tickera custom fields located on the Woo product page
         *
         * @param int $post_id
         */
        function tc_custom_settings_fields_save( $post_id ) {
            $post_id = (int) $post_id;
            // Check if product is a ticket
            $_tc_is_ticket = ( isset( $_POST['_tc_is_ticket'] ) ? 'yes' : 'no' );
            if ( 'yes' == $_tc_is_ticket ) {
                // Save is ticket value
                update_post_meta( $post_id, '_tc_is_ticket', $_tc_is_ticket );
                // Save related event
                $_event_name = ( empty( $_POST['_event_name'] ) ? '' : (int) $_POST['_event_name'] );
                update_post_meta( $post_id, '_event_name', (int) $_event_name );
                // Save choosen ticket template
                $_ticket_template = ( empty( $_POST['_ticket_template'] ) ? '' : (int) $_POST['_ticket_template'] );
                update_post_meta( $post_id, '_ticket_template', $_ticket_template );
                // Save available check-ins
                $_available_checkins_per_ticket = ( empty( $_POST['_available_checkins_per_ticket'] ) ? '' : (int) $_POST['_available_checkins_per_ticket'] );
                update_post_meta( $post_id, '_available_checkins_per_ticket', $_available_checkins_per_ticket );
                // Save Checkins on time basis
                $_checkins_time_basis = ( empty( $_POST['_checkins_time_basis'] ) ? 'no' : sanitize_key( $_POST['_checkins_time_basis'] ) );
                update_post_meta( $post_id, '_checkins_time_basis', $_checkins_time_basis );
                $_allowed_checkins_per_time_basis = ( empty( $_POST['_allowed_checkins_per_time_basis'] ) ? '' : (int) $_POST['_allowed_checkins_per_time_basis'] );
                update_post_meta( $post_id, '_allowed_checkins_per_time_basis', $_allowed_checkins_per_time_basis );
                $_checkins_time_basis_type = ( empty( $_POST['_checkins_time_basis_type'] ) ? 'hour' : sanitize_key( $_POST['_checkins_time_basis_type'] ) );
                update_post_meta( $post_id, '_checkins_time_basis_type', $_checkins_time_basis_type );
                $_checkins_time_calendar_basis = ( empty( $_POST['_checkins_time_calendar_basis'] ) ? 'no' : sanitize_key( $_POST['_checkins_time_calendar_basis'] ) );
                update_post_meta( $post_id, '_checkins_time_calendar_basis', $_checkins_time_calendar_basis );
                // Save availability dates check-ins
                $_ticket_checkin_availability = ( empty( $_POST['_ticket_checkin_availability'] ) ? 'open_ended' : $_POST['_ticket_checkin_availability'] );
                update_post_meta( $post_id, '_ticket_checkin_availability', $_ticket_checkin_availability );
                // Allow Ticket Checkout
                $_allow_ticket_checkout = ( isset( $_POST['_allow_ticket_checkout'] ) ? $_POST['_allow_ticket_checkout'] : 'no' );
                update_post_meta( $post_id, '_allow_ticket_checkout', $_allow_ticket_checkout );
                if ( 'range' == $_ticket_checkin_availability ) {
                    update_post_meta( $post_id, '_ticket_checkin_availability_from_date', $_POST['_ticket_checkin_availability_from_date'] );
                    update_post_meta( $post_id, '_ticket_checkin_availability_to_date', $_POST['_ticket_checkin_availability_to_date'] );
                } else {
                    delete_post_meta( $post_id, '_ticket_checkin_availability_from_date' );
                    delete_post_meta( $post_id, '_ticket_checkin_availability_to_date' );
                }
                if ( 'time_after_order' == $_ticket_checkin_availability ) {
                    update_post_meta( $post_id, '_time_after_order_days', $_POST['_time_after_order_days'] );
                    update_post_meta( $post_id, '_time_after_order_hours', $_POST['_time_after_order_hours'] );
                    update_post_meta( $post_id, '_time_after_order_minutes', $_POST['_time_after_order_minutes'] );
                } else {
                    delete_post_meta( $post_id, '_time_after_order_days' );
                    delete_post_meta( $post_id, '_time_after_order_hours' );
                    delete_post_meta( $post_id, '_time_after_order_minutes' );
                }
                if ( 'time_after_first_checkin' == $_ticket_checkin_availability ) {
                    update_post_meta( $post_id, '_time_after_first_checkin_days', $_POST['_time_after_first_checkin_days'] );
                    update_post_meta( $post_id, '_time_after_first_checkin_hours', $_POST['_time_after_first_checkin_hours'] );
                    update_post_meta( $post_id, '_time_after_first_checkin_minutes', $_POST['_time_after_first_checkin_minutes'] );
                } else {
                    delete_post_meta( $post_id, '_time_after_first_checkin_days' );
                    delete_post_meta( $post_id, '_time_after_first_checkin_hours' );
                    delete_post_meta( $post_id, '_time_after_first_checkin_minutes' );
                }
                if ( 'yes' == $_checkins_time_basis ) {
                    update_post_meta( $post_id, '_allowed_checkins_per_time_basis', $_allowed_checkins_per_time_basis );
                    update_post_meta( $post_id, '_checkins_time_basis_type', $_checkins_time_basis_type );
                    update_post_meta( $post_id, '_checkins_time_calendar_basis', $_checkins_time_calendar_basis );
                } else {
                    delete_post_meta( $post_id, '_allowed_checkins_per_time_basis' );
                    delete_post_meta( $post_id, '_checkins_time_basis_type' );
                    delete_post_meta( $post_id, '_checkins_time_calendar_basis' );
                }
                // Save selling availability dates
                $_ticket_selling_availability = ( empty( $_POST['_ticket_availability'] ) ? 'open_ended' : $_POST['_ticket_availability'] );
                update_post_meta( $post_id, '_ticket_availability', $_ticket_selling_availability );
                if ( 'range' == $_ticket_selling_availability ) {
                    update_post_meta( $post_id, '_ticket_availability_from_date', $_POST['_ticket_availability_from_date'] );
                    update_post_meta( $post_id, '_ticket_availability_to_date', $_POST['_ticket_availability_to_date'] );
                } else {
                    delete_post_meta( $post_id, '_ticket_availability_from_date' );
                    delete_post_meta( $post_id, '_ticket_availability_to_date' );
                }
                if ( function_exists( 'tc_custom_form_fields_owner_form_templates_array' ) ) {
                    $_owner_form_template = ( empty( $_POST['_owner_form_template'] ) ? '' : (int) $_POST['_owner_form_template'] );
                    update_post_meta( $post_id, '_owner_form_template', $_owner_form_template );
                }
            } else {
                delete_post_meta( $post_id, '_tc_is_ticket' );
                delete_post_meta( $post_id, '_event_name' );
                delete_post_meta( $post_id, '_ticket_template' );
                delete_post_meta( $post_id, '_owner_form_template' );
                delete_post_meta( $post_id, '_ticket_checkin_availability' );
                delete_post_meta( $post_id, '_ticket_checkin_availability_from_date' );
                delete_post_meta( $post_id, '_ticket_checkin_availability_to_date' );
                delete_post_meta( $post_id, '_ticket_availability' );
                delete_post_meta( $post_id, '_ticket_availability_from_date' );
                delete_post_meta( $post_id, '_ticket_availability_to_date' );
                delete_post_meta( $post_id, '_time_after_order_days' );
                delete_post_meta( $post_id, '_time_after_order_hours' );
                delete_post_meta( $post_id, '_time_after_order_minutes' );
                delete_post_meta( $post_id, '_allow_ticket_checkout' );
                delete_post_meta( $post_id, '_available_checkins_per_ticket' );
                delete_post_meta( $post_id, '_checkins_time_basis' );
                delete_post_meta( $post_id, '_allowed_checkins_per_time_basis' );
                delete_post_meta( $post_id, '_checkins_time_basis_type' );
                delete_post_meta( $post_id, '_checkins_time_calendar_basis' );
            }
        }

        function tc_custom_additional_meta_save_variations( $variation_id, $i ) {
            $_ticket_type_variation_override_checkins = ( isset( $_POST['_ticket_type_variation_override_checkins'] ) ? $_POST['_ticket_type_variation_override_checkins'] : array() );
            update_post_meta( $variation_id, '_ticket_type_variation_override_checkins', $_ticket_type_variation_override_checkins[$i] );
        }

        /**
         * Add custom Tickera fields to the admin product screen
         *
         * @global mixed $woocommerce
         * @global object $post
         */
        function tc_add_custom_settings() {
            global $post;
            $tc_general_settings = get_option( 'tickera_general_setting', false );
            /**
             * ========================================
             * Available dates / times for check-in
             * ========================================
             */
            $_ticket_checkin_availability = get_post_meta( $post->ID, '_ticket_checkin_availability', true );
            $_ticket_checkin_availability = ( !isset( $_ticket_checkin_availability ) || empty( $_ticket_checkin_availability ) ? 'open_ended' : $_ticket_checkin_availability );
            /**
             * ========================================
             * Available dates for tickets selling
             * ========================================
             */
            $_ticket_availability = get_post_meta( $post->ID, '_ticket_availability', true );
            $_ticket_availability = ( !isset( $_ticket_availability ) || empty( $_ticket_availability ) ? 'open_ended' : $_ticket_availability );
            /**
             * ========================================
             * Limit check-ins on time basis
             * ========================================
             */
            $_time_basis = get_post_meta( $post->ID, '_checkins_time_basis_type', true );
            $_time_basis = ( $_time_basis ? $_time_basis : 'hour' );
            $_calendar_basis = get_post_meta( $post->ID, '_checkins_time_calendar_basis', true );
            $_calendar_basis = ( $_calendar_basis ? $_calendar_basis : 'no' );
            echo '<div class = "options_group show_if_simple show_if_variable">';
            // Class to include to ensure the options are hidden in downloadable type products hide_if_downloadable
            woocommerce_wp_checkbox( [
                'id'          => '_tc_is_ticket',
                'label'       => __( 'Product is a Ticket', 'woocommerce-tickera-bridge' ),
                'desc_tip'    => 'true',
                'description' => __( 'Select if this product is an event ticket.', 'woocommerce-tickera-bridge' ),
            ] );
            echo '<div class="options_group show_if_tc_ticket">';
            do_action( 'tc_woo_show_if_tc_ticket_before' );
            woocommerce_wp_select( [
                'id'          => '_event_name',
                'label'       => __( 'Event', 'woocommerce-tickera-bridge' ),
                'desc_tip'    => 'true',
                'description' => __( 'Select an Event created in Tickera', 'woocommerce-tickera-bridge' ),
                'options'     => tickera_get_events_array(),
            ] );
            woocommerce_wp_select( [
                'id'          => '_ticket_template',
                'label'       => __( 'Ticket Template', 'woocommerce-tickera-bridge' ),
                'options'     => tickera_get_ticket_templates_array(),
                'desc_tip'    => 'true',
                'description' => __( 'Select how ticket will look.', 'woocommerce-tickera-bridge' ),
            ] );
            // Make sure that Tickera's Custom Forms add-on is active
            if ( function_exists( 'tc_custom_form_fields_owner_form_templates_array' ) ) {
                woocommerce_wp_select( [
                    'id'          => '_owner_form_template',
                    'label'       => __( 'Attendee Form', 'woocommerce-tickera-bridge' ),
                    'options'     => tc_custom_form_fields_owner_form_templates_array(),
                    'desc_tip'    => 'true',
                    'description' => __( 'Attendee form will be shown for each ticket in the cart for selected ticket types / ticket products. You can created multiple different attendee forms.', 'woocommerce-tickera-bridge' ),
                ] );
            }
            // Currently use as separator.
            echo '<div class="options_group"></div>';
            if ( method_exists( 'Tickera\\TC_Ticket', 'is_sales_available' ) ) {
                // Allow Ticket Checkout
                echo '<div class="options_group">';
                $use_global_ticket_checkouts = ( isset( $tc_general_settings['allow_global_ticket_checkout'] ) ? $tc_general_settings['allow_global_ticket_checkout'] : 'no' );
                if ( 'no' == $use_global_ticket_checkouts ) {
                    woocommerce_wp_checkbox( [
                        'id'          => '_allow_ticket_checkout',
                        'label'       => __( 'Allow ticket check-out', 'woocommerce-tickera-bridge' ),
                        'desc_tip'    => 'true',
                        'description' => __( 'If enabled, when attendee initially checks their ticket in, it will be recorded as check-in. Scanning it for the second time will mark the ticket as checked-out.<br/>Useful if you need to keep track of how many attendees are currently at the event but want to allow attendees to leave and re-enter the event', 'woocommerce-tickera-bridge' ),
                    ] );
                }
                echo '</div>';
                // Available dates for tickets selling
                echo '<div class="options_group">';
                woocommerce_wp_radio( [
                    'id'          => '_ticket_availability',
                    'label'       => __( 'Available dates for tickets selling', 'woocommerce-tickera-bridge' ),
                    'desc_tip'    => 'true',
                    'description' => __( 'Choose if you want to limit ticket sales availability for certain date range or leave it as open-ended.', 'woocommerce-tickera-bridge' ),
                    'options'     => [
                        'open_ended' => 'Open-ended',
                        'range'      => 'During selected date range',
                    ],
                    'value'       => $_ticket_availability,
                ] );
                echo '<div id="_ticket_availability_dates">';
                woocommerce_wp_text_input( [
                    'id'    => '_ticket_availability_from_date',
                    'class' => 'tc_date_field',
                    'label' => __( 'Selling allowed From', 'woocommerce-tickera-bridge' ),
                ] );
                woocommerce_wp_text_input( [
                    'id'    => '_ticket_availability_to_date',
                    'class' => 'tc_date_field',
                    'label' => __( 'Selling allowed until', 'woocommerce-tickera-bridge' ),
                ] );
                echo '</div></div>';
                // Available Check-ins
                echo '<div class="options_group">';
                woocommerce_wp_text_input( [
                    'id'          => '_available_checkins_per_ticket',
                    'label'       => __( 'Check-ins per ticket', 'woocommerce-tickera-bridge' ),
                    'placeholder' => __( 'Unlimited', 'woocommerce-tickera-bridge' ),
                    'desc_tip'    => 'true',
                    'description' => __( 'It is useful if the event last more than one day. For instance, if duration of your event is 5 day, you should choose 5 or more for Available Check-ins', 'woocommerce-tickera-bridge' ),
                    'type'        => 'number',
                ] );
                echo '</div>';
                // Limit check-ins on time basis
                echo '<div class="options_group">';
                woocommerce_wp_checkbox( [
                    'id'          => '_checkins_time_basis',
                    'label'       => __( 'Limit check-ins on time basis', 'woocommerce-tickera-bridge' ),
                    'desc_tip'    => 'true',
                    'description' => __( 'If enabled, when attendee initially checks their ticket in, it will be recorded as check-in. Scanning it for the second time will mark the ticket as checked-out.<br/>Useful if you need to keep track of how many attendees are currently at the event but want to allow attendees to leave and re-enter the event', 'woocommerce-tickera-bridge' ),
                ] );
                echo '<div id="_checkins_time_basis_fields">';
                woocommerce_wp_text_input( [
                    'id'          => '_allowed_checkins_per_time_basis',
                    'label'       => __( 'Allow', 'woocommerce-tickera-bridge' ),
                    'placeholder' => __( 'Unlimited', 'woocommerce-tickera-bridge' ),
                    'desc_tip'    => 'true',
                    'description' => __( '', 'woocommerce-tickera-bridge' ),
                    'type'        => 'number',
                ] );
                woocommerce_wp_select( [
                    'id'          => '_checkins_time_basis_type',
                    'label'       => __( 'Check-ins per', 'woocommerce-tickera-bridge' ),
                    'desc_tip'    => 'true',
                    'description' => __( '', 'woocommerce-tickera-bridge' ),
                    'options'     => [
                        'hour'  => __( 'Hour', 'woocommerce-tickera-bridge' ),
                        'day'   => __( 'Day', 'woocommerce-tickera-bridge' ),
                        'week'  => __( 'Week', 'woocommerce-tickera-bridge' ),
                        'month' => __( 'Month', 'woocommerce-tickera-bridge' ),
                    ],
                    'value'       => $_time_basis,
                ] );
                woocommerce_wp_select( [
                    'id'          => '_checkins_time_calendar_basis',
                    'label'       => '',
                    'desc_tip'    => 'true',
                    'description' => __( '', 'woocommerce-tickera-bridge' ),
                    'options'     => [
                        'no'  => __( '24 hours basis', 'woocommerce-tickera-bridge' ),
                        'yes' => __( 'Calendar day basis (resets at midnight)', 'woocommerce-tickera-bridge' ),
                    ],
                    'value'       => $_calendar_basis,
                ] );
                echo '</div></div>';
                // Available dates / times for check-in
                echo '<div class="options_group>">';
                woocommerce_wp_radio( [
                    'id'          => '_ticket_checkin_availability',
                    'label'       => __( 'Available dates / times for check-in', 'woocommerce-tickera-bridge' ),
                    'desc_tip'    => 'true',
                    'description' => __( 'Choose if you want to limit check-ins for certain date range or leave it as open-ended.', 'woocommerce-tickera-bridge' ),
                    'options'     => [
                        'open_ended'               => __( 'Open-ended', 'woocommerce-tickera-bridge' ),
                        'range'                    => __( 'During selected date range', 'woocommerce-tickera-bridge' ),
                        'time_after_order'         => __( 'Within the following time after order is placed', 'woocommerce-tickera-bridge' ),
                        'time_after_first_checkin' => __( 'Within the following time after first check-in', 'woocommerce-tickera-bridge' ),
                        'upon_event_starts'        => __( 'When the event starts', 'woocommerce-tickera-bridge' ),
                    ],
                    'value'       => $_ticket_checkin_availability,
                ] );
                // During selected date range
                echo '<div id="_ticket_checkin_availability_dates">';
                woocommerce_wp_text_input( [
                    'id'    => '_ticket_checkin_availability_from_date',
                    'class' => 'tc_date_field',
                    'label' => __( 'Check-in allowed from', 'woocommerce-tickera-bridge' ),
                ] );
                woocommerce_wp_text_input( [
                    'id'    => '_ticket_checkin_availability_to_date',
                    'class' => 'tc_date_field',
                    'label' => __( 'Check-in allowed until', 'woocommerce-tickera-bridge' ),
                ] );
                echo '</div>';
                // Within the following time after order is placed
                echo '<div id="_ticket_checkin_availability_after_order_time">';
                $days_options = [];
                for ($day = apply_filters( 'tc_ticket_checkin_availability_time_after_order_day_min', 0 ); $day <= apply_filters( 'tc_ticket_checkin_availability_time_after_order_day_max', 365 ); $day++) {
                    $days_options[$day] = $day;
                }
                woocommerce_wp_select( [
                    'id'          => '_time_after_order_days',
                    'label'       => __( 'Days', 'woocommerce-tickera-bridge' ),
                    'options'     => $days_options,
                    'desc_tip'    => 'true',
                    'description' => __( 'Days after order is placed', 'woocommerce-tickera-bridge' ),
                ] );
                $hours_options = [];
                for ($hour = apply_filters( 'tc_ticket_checkin_availability_time_after_order_hour_min', 0 ); $hour <= apply_filters( 'tc_ticket_checkin_availability_time_after_order_hour_max', 24 ); $hour++) {
                    $hours_options[$hour] = $hour;
                }
                woocommerce_wp_select( [
                    'id'          => '_time_after_order_hours',
                    'label'       => __( 'Hours', 'woocommerce-tickera-bridge' ),
                    'options'     => $hours_options,
                    'desc_tip'    => 'true',
                    'description' => __( 'Hours after order is placed', 'woocommerce-tickera-bridge' ),
                ] );
                $minutes_options = [];
                for ($minute = apply_filters( 'tc_ticket_checkin_availability_time_after_order_minute_min', 0 ); $minute <= apply_filters( 'tc_ticket_checkin_availability_time_after_order_minute_max', 60 ); $minute++) {
                    $minutes_options[$minute] = $minute;
                }
                woocommerce_wp_select( [
                    'id'          => '_time_after_order_minutes',
                    'label'       => __( 'Minutes', 'woocommerce-tickera-bridge' ),
                    'options'     => $minutes_options,
                    'desc_tip'    => 'true',
                    'description' => __( 'Minutes after order is placed', 'woocommerce-tickera-bridge' ),
                ] );
                echo '</div>';
                // Within the following time after first check-in
                echo '<div id="_ticket_checkin_availability_after_first_checkin">';
                $days_options = [];
                for ($day = apply_filters( 'tc_ticket_checkin_availability_time_after_first_checkin_day_min', 0 ); $day <= apply_filters( 'tc_ticket_checkin_availability_time_after_first_checkin_day_max', 365 ); $day++) {
                    $days_options[$day] = $day;
                }
                woocommerce_wp_select( [
                    'id'          => '_time_after_first_checkin_days',
                    'label'       => __( 'Days', 'woocommerce-tickera-bridge' ),
                    'options'     => $days_options,
                    'desc_tip'    => 'true',
                    'description' => __( 'Days after first check-in', 'woocommerce-tickera-bridge' ),
                ] );
                $hours_options = [];
                for ($hour = apply_filters( 'tc_ticket_checkin_availability_time_after_first_checkin_hour_min', 0 ); $hour <= apply_filters( 'tc_ticket_checkin_availability_time_after_first_checkin_hour_max', 24 ); $hour++) {
                    $hours_options[$hour] = $hour;
                }
                woocommerce_wp_select( [
                    'id'          => '_time_after_first_checkin_hours',
                    'label'       => __( 'Hours', 'woocommerce-tickera-bridge' ),
                    'options'     => $hours_options,
                    'desc_tip'    => 'true',
                    'description' => __( 'Hours after first check-in', 'woocommerce-tickera-bridge' ),
                ] );
                $minutes_options = [];
                for ($minute = apply_filters( 'tc_ticket_checkin_availability_time_after_first_checkin_minute_min', 0 ); $minute <= apply_filters( 'tc_ticket_checkin_availability_time_after_first_checkin_minute_max', 60 ); $minute++) {
                    $minutes_options[$minute] = $minute;
                }
                woocommerce_wp_select( [
                    'id'          => '_time_after_first_checkin_minutes',
                    'label'       => __( 'Minutes', 'woocommerce-tickera-bridge' ),
                    'options'     => $minutes_options,
                    'desc_tip'    => 'true',
                    'description' => __( 'Minutes after first check-in', 'woocommerce-tickera-bridge' ),
                ] );
                echo '</div></div>';
            }
            do_action( 'tc_woo_show_if_tc_ticket_after' );
            echo '</div></div>';
        }

        /**
         * Get WooCommerce items from cart and show ticket fields
         *
         * @param array $cart_contents
         * @return array
         * @global mixed $woocommerce
         */
        function tc_modify_cart_contents( $cart_contents ) {
            $cart_contents = [];
            $items = WC()->cart->get_cart();
            if ( !$items ) {
                // Collect cart contents from WC Cart Contents property
                $wc_cart = new WC_Cart();
                $items = $wc_cart->get_cart_contents();
            }
            foreach ( (array) $items as $item => $values ) {
                /**
                 * Use this hook to skip cart item loop.
                 * @since 1.6.2
                 */
                if ( !apply_filters( 'tc_skip_cart_item_loop', false, $values ) ) {
                    $is_ticket_meta = get_post_meta( TC_WooCommerce_Bridge::maybe_get_original_wpml_product_id( (int) $values['product_id'] ), '_tc_is_ticket', true );
                    $is_ticket = ( $is_ticket_meta == 'yes' ? true : false );
                    if ( $is_ticket ) {
                        $product_id = ( isset( $values['variation_id'] ) && is_numeric( $values['variation_id'] ) && $values['variation_id'] > 0 ? (int) $values['variation_id'] : (int) $values['product_id'] );
                        $cart_contents[$product_id] = ( array_key_exists( $product_id, $cart_contents ) ? (int) $cart_contents[$product_id] + (int) $values['quantity'] : (int) $values['quantity'] );
                    }
                }
            }
            return $cart_contents;
        }

        /**
         * Render Tickera buyer fields after billing information.
         * @since 1.5.5
         *
         * Display additional buyer fields only if cart has tickets.
         * @since 1.5.6
         */
        function render_tc_buyer_fields() {
            if ( tc_wb_cart_has_ticket() ) {
                echo do_shortcode( '[tc_additional_buyer_fields]' );
            }
        }

        /**
         * Backward compatibility for render_tc_owner_fields.
         * Keeping the function active. Third party plugins (e.g FunnelKit Builder) is utilizing the function.
         * @since 1.5.7
         */
        function add_standard_tc_fields_to_checkout() {
            self::render_tc_owner_fields();
        }

        /**
         * Render Tickera owner fields after order notes.
         *
         * Replace function name from add_standard_tc_fields_to_checkout to render_tc_owner_fields
         * @since 1.5.5
         */
        function render_tc_owner_fields() {
            if ( tc_wb_cart_has_ticket() ) {
                echo do_shortcode( '[tc_additional_owner_fields]' );
            } else {
                // We need this placehoder for one-page checkout
                echo '<div class="tickera_owner_info"></div>';
            }
        }

        /**
         * Remove Tickera Payment Gateways TAB
         *
         * @param array $tabs
         * @return array
         */
        function tc_settings_new_menus( $tabs ) {
            unset($tabs['gateways']);
            return $tabs;
        }

        /**
         * Initialize plugin variables
         */
        function init_vars() {
            // Initialize General Settings
            $this->tc_general_settings = get_option( 'tickera_general_setting', false );
            // Setup proper directories
            if ( defined( 'WP_PLUGIN_URL' ) && defined( 'WP_PLUGIN_DIR' ) && file_exists( WP_PLUGIN_DIR . '/' . $this->dir_name . '/' . basename( __FILE__ ) ) ) {
                $this->location = 'subfolder-plugins';
                $this->plugin_dir = WP_PLUGIN_DIR . '/' . $this->dir_name . '/';
                $this->plugin_url = plugins_url( '/', __FILE__ );
            } elseif ( defined( 'WP_PLUGIN_URL' ) && defined( 'WP_PLUGIN_DIR' ) && file_exists( WP_PLUGIN_DIR . '/' . basename( __FILE__ ) ) ) {
                $this->location = 'plugins';
                $this->plugin_dir = WP_PLUGIN_DIR . '/';
                $this->plugin_url = plugins_url( '/', __FILE__ );
            } elseif ( is_multisite() && defined( 'WPMU_PLUGIN_URL' ) && defined( 'WPMU_PLUGIN_DIR' ) && file_exists( WPMU_PLUGIN_DIR . '/' . basename( __FILE__ ) ) ) {
                $this->location = 'mu-plugins';
                $this->plugin_dir = WPMU_PLUGIN_DIR;
                $this->plugin_url = WPMU_PLUGIN_URL;
            } else {
                wp_die( sprintf( __( 'There was an issue determining where %s is installed. Please reinstall it.', 'woocommerce-tickera-bridge' ), $this->title ) );
            }
        }

        function load_ticket_template_elements() {
            $dir = $this->plugin_dir . 'includes/ticket-elements/';
            $ticket_template_elements = array();
            if ( !is_dir( $dir ) ) {
                return;
            }
            if ( !($dh = opendir( $dir )) ) {
                return;
            }
            while ( ($plugin = readdir( $dh )) !== false ) {
                if ( substr( $plugin, -4 ) == '.php' ) {
                    $ticket_template_elements[] = $dir . '/' . $plugin;
                }
            }
            closedir( $dh );
            sort( $ticket_template_elements );
            foreach ( $ticket_template_elements as $file ) {
                include $file;
            }
        }

        /**
         * Add billing info to CSV Export
         * @param $fields
         * @return mixed
         *
         * @since 1.4.3 Exclude fee and discount code
         */
        function tc_csv_admin_fields_woo_billing( $fields ) {
            $unset = ['col_ticket_fee', 'col_discount_code', 'col_order_notes'];
            foreach ( $unset as $field_name ) {
                unset($fields[$field_name]);
            }
            $fields['col_woo_company_name'] = __( 'Company Name', 'woocommerce-tickera-bridge' );
            $fields['col_woo_address_1'] = __( 'Address Line 1', 'woocommerce-tickera-bridge' );
            $fields['col_woo_address_2'] = __( 'Address Line 2', 'woocommerce-tickera-bridge' );
            $fields['col_woo_city'] = __( 'City', 'woocommerce-tickera-bridge' );
            $fields['col_woo_state'] = __( 'State', 'woocommerce-tickera-bridge' );
            $fields['col_woo_postcode'] = __( 'Postcode', 'woocommerce-tickera-bridge' );
            $fields['col_woo_country'] = __( 'Country', 'woocommerce-tickera-bridge' );
            $fields['col_woo_phone'] = __( 'Phone', 'woocommerce-tickera-bridge' );
            $fields['col_woo_coupon_code'] = __( 'Coupon Code', 'woocommerce-tickera-bridge' );
            // Added woo coupon code csv export
            $fields['col_woo_customer_notes'] = __( 'Customer Order Notes', 'woocommerce-tickera-bridge' );
            return $fields;
        }

        function tc_csv_array_woo_billing(
            $export_array,
            $order,
            $instance,
            $post
        ) {
            $woo_order = wc_get_order( $order->details->ID );
            if ( isset( $post['col_woo_company_name'] ) ) {
                $billing_company = $woo_order->get_billing_company();
                $new_export_array = array(
                    __( 'Company Name', 'woocommerce-tickera-bridge' ) => $billing_company,
                );
                $export_array = array_merge( $export_array, $new_export_array );
            }
            if ( isset( $post['col_woo_address_1'] ) ) {
                $billing_address_line_1 = $woo_order->get_billing_address_1();
                $new_export_array = array(
                    __( 'Address Line 1', 'woocommerce-tickera-bridge' ) => $billing_address_line_1,
                );
                $export_array = array_merge( $export_array, $new_export_array );
            }
            if ( isset( $post['col_woo_address_2'] ) ) {
                $billing_address_line_2 = $woo_order->get_billing_address_2();
                $new_export_array = array(
                    __( 'Address Line 2', 'woocommerce-tickera-bridge' ) => $billing_address_line_2,
                );
                $export_array = array_merge( $export_array, $new_export_array );
            }
            if ( isset( $post['col_woo_city'] ) ) {
                $billing_city = $woo_order->get_billing_city();
                $new_export_array = array(
                    __( 'City', 'woocommerce-tickera-bridge' ) => $billing_city,
                );
                $export_array = array_merge( $export_array, $new_export_array );
            }
            if ( isset( $post['col_woo_state'] ) ) {
                $billing_state = $woo_order->get_billing_state();
                $new_export_array = array(
                    __( 'State', 'woocommerce-tickera-bridge' ) => $billing_state,
                );
                $export_array = array_merge( $export_array, $new_export_array );
            }
            if ( isset( $post['col_woo_postcode'] ) ) {
                $billing_postcode = $woo_order->get_billing_postcode();
                $new_export_array = array(
                    __( 'Postcode', 'woocommerce-tickera-bridge' ) => $billing_postcode,
                );
                $export_array = array_merge( $export_array, $new_export_array );
            }
            if ( isset( $post['col_woo_country'] ) ) {
                $billing_country = $woo_order->get_billing_country();
                $new_export_array = array(
                    __( 'Country', 'woocommerce-tickera-bridge' ) => $billing_country,
                );
                $export_array = array_merge( $export_array, $new_export_array );
            }
            if ( isset( $post['col_woo_phone'] ) ) {
                $billing_phone = $woo_order->get_billing_phone();
                $new_export_array = array(
                    __( 'Phone', 'woocommerce-tickera-bridge' ) => $billing_phone,
                );
                $export_array = array_merge( $export_array, $new_export_array );
            }
            // Get coupon code
            if ( isset( $post['col_woo_coupon_code'] ) ) {
                $coupons = $woo_order->get_items( 'coupon' );
                $codes = [];
                if ( is_array( $coupons ) && $coupons ) {
                    foreach ( $coupons as $item_id => $item ) {
                        $codes[] = $item->get_code();
                    }
                }
                $export_array = array_merge( $export_array, [
                    __( 'Coupon Code', 'woocommerce-tickera-bridge' ) => implode( ', ', $codes ),
                ] );
            }
            // Get customer order notes
            if ( isset( $post['col_woo_customer_notes'] ) ) {
                $export_array = array_merge( $export_array, [
                    __( 'Customer Order Notes', 'woocommerce-tickera-bridge' ) => $woo_order->get_customer_note(),
                ] );
            }
            return $export_array;
        }

    }

    if ( !function_exists( 'is_plugin_active_for_network' ) ) {
        require_once ABSPATH . '/wp-admin/includes/plugin.php';
    }
    //    if ( ( woo_bridge_fs_is_parent_active_and_loaded() ) && woo_bridge_fs()->can_use_premium_code() ) {
    //        if ( is_multisite() && is_plugin_active_for_network( plugin_basename( __FILE__ ) ) ) {
    //
    //            add_action( 'tets_fs_loaded', 'tc_woocommerce_bridge_load', 20 );
    //            function tc_woocommerce_bridge_load() {
    //                global $tc_woocommerce_bridge;
    //                $tc_woocommerce_bridge = new TC_WooCommerce_Bridge();
    //            }
    //
    //        } else {
    //            $tc_woocommerce_bridge = new TC_WooCommerce_Bridge();
    //        }
    //    }
    $tc_woocommerce_bridge = new TC_WooCommerce_Bridge();
}
/**
 * Remove ticket types from menu when WooBridge is activated
 * @param array $args
 * @return boolean
 */
add_filter(
    'tc_ticket_type_post_type_args',
    function ( $args ) {
        $args['show_ui'] = false;
        return $args;
    },
    21,
    1
);
/**
 * Remove standalone orders from menu when WooBridge is activated
 * @param array $args
 * @return boolean
 */
add_filter(
    'tc_orders_post_type_args',
    function ( $args ) {
        $args['show_ui'] = false;
        return $args;
    },
    21,
    1
);
/**
 * Overwrite WooCommerce function for "Order Again" - we don't show the button if there is a seat in the order
 */
if ( !function_exists( 'woocommerce_order_again_button' ) ) {
    function woocommerce_order_again_button(  $order  ) {
        if ( is_int( $order ) ) {
            $order_id = $order;
        } else {
            $order_id = ( method_exists( $order, 'get_id' ) ? $order->get_id() : $order->id );
        }
        $ticket_instances = Tickera\TC_Orders::get_tickets_ids( $order_id, 'any' );
        foreach ( $ticket_instances as $ticket_instance_id ) {
            $tc_check_seat = get_post_meta( $ticket_instance_id, 'seat_id', true );
            if ( !empty( $tc_check_seat ) ) {
                return;
            }
        }
        wc_get_template( 'order/order-again.php', array(
            'order'           => $order,
            'wp_button_class' => ( wc_wp_theme_get_element_class_name( 'button' ) ? ' ' . wc_wp_theme_get_element_class_name( 'button' ) : '' ),
            'order_again_url' => wp_nonce_url( add_query_arg( 'order_again', $order->get_id(), wc_get_cart_url() ), 'woocommerce-order_again' ),
        ) );
    }

}
// Remove dashboard sale stats widget
define( 'TC_HIDE_STATS_WIDGET', true );