import { __ } from "@wordpress/i18n";
import { useCallback, useEffect, useState } from "@wordpress/element";
import { CheckboxControl } from '@woocommerce/blocks-checkout';
import { useSelect, useDispatch } from '@wordpress/data';
import { debounce } from "lodash";

const blockNamespace = 'tickera/checkout-fields';
var selectedValues = [],
    error;

export const CheckboxFieldControl = ( prop ) => {

    let field = prop.field,
        pageLoaded = prop.loaded,
        category = prop.category,
        item = prop.item,
        i = prop.i;

    prop = prop.data.prop;

    /*
     * setExtensionData will update the wc/store/checkout data store with the values supplied.
     * It can be used to pass data from the client to the server when submitting the checkout form.
     */
    const { setExtensionData } = prop.checkoutExtensionData;

    /*
     * Debounce the setExtensionData function to avoid
     * multiple calls to the API when rapidly changing options.
     */
    const debouncedSetExtensionData = useCallback(
        debounce((namespace, key, value) => {
            setExtensionData(namespace, key, value);
        }, 1000),
        [setExtensionData]
    );

    if ( typeof field.field_values !== 'undefined' ) {

        if ( typeof field.field_values === 'object' ) {
            var fieldValues = field.field_values;

        } else {
            var fieldValues = field.field_values.split( ',' );
        }

    } else {
        var fieldValues = [];
    }

    var checkboxes = [],
        contents = [];

    fieldValues.map( function( value, index ) {

        let label;
        if ( typeof value === 'object' ) {
            label = value.label;
            value = value.value;

        } else {
            value = value.trim();
            label = value;
        }

        if ( '' != value ) {

            let postMeta = ( typeof field.post_meta !== 'undefined' ) ? field.post_meta : true,
                key;

            if ( typeof category !== 'undefined' ) {

                key = ( 'owner' == category )
                    ? item.id + '_' + i + '_' + field.field_name + '_option_' + index + ( postMeta == true ? '_post_meta' : '' )
                    : 'buyer_data_' + field.field_name + '_option_' + index + ( postMeta == true ? '_post_meta' : '' );

            } else {

                if ( fieldValues.length > 1 ) {
                    key = field.field_name + '_option_' + index + ( postMeta == true ? '_post_meta' : '' );

                } else {
                    key = field.field_name + ( postMeta == true ? '_post_meta' : '' );
                }
            }

            /**
             * Handles fields validation
             * Applies on checkbox last option
             */
            const validationKey = ( 'owner' == category )
                ? item.id + '_' + i + '_' + field.field_name + '_post_meta'
                : 'buyer_data_' + field.field_name + '_post_meta';

            const validationErrorId = validationKey + '-validator';
            const { setValidationErrors, clearValidationError } = useDispatch( 'wc/store/validation' );
            const validationError = useSelect( ( select ) => {
                const store = select( 'wc/store/validation' );
                return store.getValidationError( validationErrorId );
            } );

            /**
             * Handles changes in fields value.
             * Exclude Label field.
             */
            let [ fieldValue, setFieldValue ] = useState( '' );

            useEffect( () => {

                // Store selected values for later comparison
                if ( typeof fieldValue !== 'undefined' && fieldValue ) {
                    selectedValues[ key ] = fieldValue;

                } else {
                    delete selectedValues[ key ]
                }

                if ( field.required ) {

                    // Set Validation Errors. Initially hidden until page has been fully loaded
                    let hideError = !pageLoaded ? true : false;
                    setValidationErrors( {
                        [ validationErrorId ]: {
                            message: tc_woobridge_block.please_enter_a_valid + ' ' + field.field_title.toLowerCase(),
                            hidden: hideError,
                        },
                    } );

                    // Clear error with the following conditions
                    if ( pageLoaded && Object.keys( selectedValues ).length ) {
                        clearValidationError( validationErrorId );
                    }
                }

                fieldValue = fieldValue ? value : '';

                // Update in tickera/checkout-fields extended data
                setExtensionData( blockNamespace, key, fieldValue );

                // Ensures setExtensionData is being called one at a time.
                debouncedSetExtensionData( blockNamespace, key, fieldValue );

            }, [ setExtensionData, fieldValue, setValidationErrors, clearValidationError ] );

            error = ( !validationError?.hidden ) ? validationError?.message : false;
            checkboxes.push( <CheckboxControl id={key} label={label} checked={ fieldValue } onChange={ setFieldValue }/> );
        }
    })

    contents.push(
        <div className={ 'tc-wb-block-component tc-wb-block-components-' + field.field_name + ( error ? ' has-error' : '' ) }>
            { field.field_title ? <h5>{ field.field_title }</h5> : '' }
            <div className='wc-block-components-checkbox'>
                { checkboxes }
            </div>
            { error ? <div className='wc-block-components-validation-error'><p>{ error }</p></div> : '' }
            { ( field.field_description ) ? <div className='wc-block-components-checkout-step__description description'>{ field.field_description }</div> : '' }
        </div>
    );

    return contents;
}
