<?php

use Automattic\WooCommerce\Blocks\Integrations\IntegrationInterface;

if ( ! class_exists( 'TC_CHRE_Blocks_Integration' ) ) {

    class TC_CHRE_Blocks_Integration implements IntegrationInterface {

        /**
         * Plugin's name
         * @var string
         */
        private $name;

        /** Plugin's base directory name
         * @var string
         */
        private $dir_name;

        /**
         * Plugin's version
         * @var string
         */
        private $version;

        /**
         * Plugin's General Settings
         * @var array
         */
        private $settings;

        /**
         * The name of the integration.
         * @return string
         */
        public function get_name() {
            return 'checkout-recaptcha';
        }

        /**
         * When called invokes any initialization/setup for the integration.
         */
        public function initialize() {
            $this->name = 'tc_chre';
            $this->dir_name = 'checkout-recaptcha';
            $this->version = 1.0;
            $this->settings = get_option( 'tickera_general_setting' );
            $this->register_scripts_styles();
            add_action( 'wp_ajax_chre_verify_user_response', array( $this, 'verify_user_response' ) );
        }

        function register_scripts_styles() {

            // Frontend Scripts
            $script_path = '../../assets/blocks/build/frontend.js';
            $script_url = plugins_url( $script_path, __FILE__ );
            $script_asset_path = trailingslashit( WP_PLUGIN_DIR ) . $this->dir_name . '/assets/blocks/build/frontend.asset.php';
            $script_asset = file_exists( $script_asset_path ) ? require $script_asset_path : [ 'dependencies' => [], 'version' => $this->get_file_version( $script_asset_path ) ];
            wp_register_script( $this->name . '-block-frontend', $script_url, $script_asset[ 'dependencies' ], $script_asset[ 'version' ], true );

            // Google Recaptcha
            $settings = TC_Checkout_reCAPTCHA::get_settings();
            if ( isset( $settings[ 'show_recaptcha' ] ) && $settings[ 'show_recaptcha' ] ) {
                $language = isset( $settings[ 'language' ] ) ? $settings[ 'language' ] : 'en';
                $script_url = "https://www.google.com/recaptcha/api.js?onload=onChreCallback&render=explicit&hl=" . $language;
                wp_enqueue_script( 'tc-checkout-recaptcha', $script_url, [], $this->version );

                $script_path = '../../assets/blocks/src/extra.js';
                $script_url = plugins_url( $script_path, __FILE__ );
                wp_enqueue_script( 'tc-checkout-recaptcha-extra', $script_url, [], $script_asset[ 'version' ], [ 'in_footer' => true ] );
                wp_localize_script( 'tc-checkout-recaptcha-extra', 'tcchre', [
                    'ajaxUrl' => apply_filters( 'tc_ajaxurl', admin_url( 'admin-ajax.php', ( is_ssl() ? 'https' : 'http' ) ) ),
                    'key' => self::encode( $settings[ 'site_key' ] )
                ] );
            }

            // Frontend Styles
            $style_path = '../../assets/blocks/build/style-frontend.css';
            $style_url = plugins_url( $style_path, __FILE__ );
            wp_enqueue_style( $this->name . '-block', $style_url, [], $this->get_file_version( $style_path ) );

            // Register main integration
            $script_path = '../../assets/blocks/build/index.js';
            $script_url = plugins_url( $script_path, __FILE__ );
            $script_asset_path = trailingslashit( WP_PLUGIN_DIR ) . $this->dir_name . '/assets/blocks/build/index.asset.php';
            $script_asset = file_exists( $script_asset_path ) ? require $script_asset_path : [ 'dependencies' => [], 'version' => $this->get_file_version( $script_path ), ];
            wp_register_script( $this->name . '-blocks-integration', $script_url, $script_asset[ 'dependencies' ], $script_asset[ 'version' ], true );
        }

        /**
         * Returns an array of script handles to enqueue in the frontend context.
         *
         * @return string[]
         */
        public function get_script_handles() {
            return [ $this->name . '-blocks-integration', $this->name . '-block-frontend' ];
        }

        /**
         * Returns an array of script handles to enqueue in the editor context.
         *
         * @return string[]
         */
        public function get_editor_script_handles() {
            return [ $this->name . '-blocks-integration', $this->name . '-block-editor', ];
        }

        /**
         * An array of key, value pairs of data made available to the block on the client side.
         *
         * @var array $errors
         * @return array
         */
        public function get_script_data() {

            $settings = TC_Checkout_reCAPTCHA::get_settings();

            return [
                'show_recaptcha' => isset( $settings[ 'show_recaptcha' ] ) ? $settings[ 'show_recaptcha' ] : '0',
                'site_key' => isset( $settings[ 'site_key' ] ) ? $settings[ 'site_key' ] : ''
            ];
        }

        /**
         * Get the file modified time as a cache buster if we're in dev mode.
         *
         * @param string $file Local path to the file.
         * @return string The cache buster value to use for the given file.
         */
        protected function get_file_version( $file ) {

            if ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG && file_exists( $file ) ) {
                return filemtime( $file );
            }

            return $this->version;
        }

        protected function encode( $string ) {
            if ( ! $string ) { return; }
            $dummy_char = round( strlen( $string ) / 4 );
            $string = substr_replace( $string, $string[ (int) $dummy_char ], round( strlen( $string ) / 3 ), 0 );
            return $string;
        }

        function verify_user_response() {

            $post = tickera_sanitize_array( $_POST, false, true );
            $settings = TC_Checkout_reCAPTCHA::get_settings();

            wp_send_json( $this->request( [
                'secret' => $settings[ 'secret_key' ],
                'response' => $post[ 'user_response' ]
            ] ) );
        }

        function request( $data ) {

            $ch = curl_init();
            curl_setopt( $ch, CURLOPT_URL, 'https://www.google.com/recaptcha/api/siteverify' );
            curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
            curl_setopt( $ch, CURLOPT_POST, true );
            curl_setopt( $ch, CURLOPT_POSTFIELDS, $data );
            $response = curl_exec( $ch );

            if ( curl_errno( $ch ) ) {
                $response = [];
                $response[ 'error' ] = curl_error( $ch );

            } else {
                $response = json_decode( $response, true );
            }

            curl_close( $ch );
            return $response;
        }
    }
}
