<?php

if ( ! defined( 'ABSPATH' ) )
    exit; // Exit if accessed directly

if ( ! class_exists( 'TC_Export_Resource' ) ) {

    class TC_Export_Resource extends TC_Export_Bridge
    {

        /**
         * Initializes the constructor by invoking the parent's init method.
         *
         * @return void
         */
        function __construct() {
            parent::init();
        }

        /**
         * Retrieves ticket type IDs for specified events.
         *
         * @param array $event_ids An array of event IDs. If an empty array or 'all' is provided, all events' ticket type IDs will be retrieved.
         * @return array An array of ticket type IDs, including any possible ticket variations.
         * @throws Exception
         */
        function get_events_ticket_type_ids( $event_ids = [] ) {

            if ( ! $event_ids )
                return [];

            if ( in_array( 'all', $event_ids ) ) {
                $events_search = new Tickera\TC_Events_Search( '', '', -1 );
                $event_ids = wp_list_pluck( $events_search->get_results(), 'ID' );
            }

            $ticket_type_ids = get_posts( [
                'post_type' => $this->ticket_post_types,
                'meta_key' => $this->event_meta_key,
                'meta_value' => $event_ids,
                'fields' => 'ids',
                'posts_per_page' => -1
            ]);

            // Maybe variation product exists
            return $this->maybe_include_ticket_variations( $ticket_type_ids );
        }

        /**
         * Validates and retrieves ticket type IDs based on given event IDs and ticket type IDs.
         *
         * @param array $event_ids List of event IDs to filter ticket types. Supports 'all' to include all event IDs.
         * @param array $ticket_type_ids List of ticket type IDs to validate. Supports 'all' to include all ticket types for the provided events.
         * @return array Filtered list of ticket type IDs, potentially including variations if applicable.
         */
        function validate_events_ticket_type_ids( $event_ids = [], $ticket_type_ids = [] ) {

            if ( ! $event_ids || ! $ticket_type_ids )
                return [];

            if ( in_array( 'all', $event_ids ) ) {
                $events_search = new Tickera\TC_Events_Search( '', '', -1 );
                $event_ids = wp_list_pluck( $events_search->get_results(), 'ID' );
            }

            if ( in_array( 'all', $ticket_type_ids ) ) {

                $ticket_type_ids = get_posts( [
                    'post_type' => $this->ticket_post_types,
                    'meta_key' => $this->event_meta_key,
                    'meta_value' => $event_ids,
                    'fields' => 'ids',
                    'posts_per_page' => -1
                ]);

            } else {

                $ticket_type_ids = get_posts( [
                    'post__in' => $ticket_type_ids,
                    'post_type' => $this->ticket_post_types,
                    'meta_key' => $this->event_meta_key,
                    'meta_value' => $event_ids,
                    'fields' => 'ids',
                    'posts_per_page' => -1
                ]);
            }

            // Maybe variation product exists
            return $this->maybe_include_ticket_variations( $ticket_type_ids );
        }

        /**
         * Includes ticket variations for a given list of ticket type IDs.
         *
         * @param array $ticket_type_ids List of ticket type IDs to check for variations.
         * @return array Combined list of original ticket type IDs and their variations.
         */
        function maybe_include_ticket_variations( $ticket_type_ids = [] ) {

            if ( ! $ticket_type_ids )
                return [];

            $ticket_variations = get_posts( [
                'post_parent__in' => $ticket_type_ids,
                'post_type' => 'product_variation',
                'fields' => 'ids',
                'posts_per_page' => -1
            ]);

            return array_merge( $ticket_type_ids, $ticket_variations );
        }

        /**
         * Retrieves API keys associated with events.
         *
         * This method fetches API keys from the database and maps them to event IDs.
         * It also handles "all" cases where API keys are applicable to all events.
         *
         * @return array An associative array where keys are event IDs and values
         *               are arrays of API keys mapped to those events, including
         *               all API keys applicable to "all" events.
         * @global wpdb $wpdb WordPress database abstraction object.
         */
        function get_events_api_keys() {

            global $wpdb;

            $results = $wpdb->get_results( "
                SELECT 
                    pm.meta_value, 
                    ( SELECT pm2.meta_value FROM {$wpdb->postmeta} pm2 WHERE p.ID=pm2.post_id AND pm2.meta_key='api_key' LIMIT 0,1 ) as api_key
                FROM {$wpdb->posts} p 
                LEFT JOIN {$wpdb->postmeta} pm ON p.ID=pm.post_id 
                WHERE p.post_type='tc_api_keys'
                AND pm.meta_key='event_name'"
            );

            $all_api_keys = [];
            $events_api_keys = [];

            if ( $results ) {

                foreach ( $results as $result ) {

                    $api_key = $result->api_key;
                    $event_ids = maybe_unserialize( $result->meta_value );

                    if ( is_array( $event_ids ) ) {

                        foreach ( $event_ids as $_event_ids ) {

                            if ( in_array( 'all', $_event_ids ) ) {
                                if ( $api_key ) $all_api_keys[] = $api_key;

                            } else {
                                foreach ( $_event_ids as $_event_id ) {
                                    if ( $api_key ) $events_api_keys[ $_event_id ][] = $api_key;
                                }
                            }
                        }

                    } elseif ( 'all' == $event_ids ) {
                        if ( $api_key ) $all_api_keys[] = $api_key;
                    }
                }
            }

            return array_map( function( $arr ) use ( $all_api_keys ) {
                return array_merge( $arr, $all_api_keys );
            }, $events_api_keys );
        }

        /**
         * Retrieves all event IDs or filters the provided event IDs.
         *
         * @param array $event_ids An array of event IDs. If 'all' is included in the array, retrieves all event IDs.
         * @return array An array of event IDs, either the filtered list or all event IDs if 'all' is specified.
         */
        function maybe_get_all_events( $event_ids = [] ) {

            if ( in_array( 'all', $event_ids ) ) {
                $events_search = new Tickera\TC_Events_Search( '', '', -1 );
                $event_ids = wp_list_pluck( $events_search->get_results(), 'ID' );
            }

            return $event_ids;
        }

        /**
         * Retrieves order details based on the provided order ID.
         *
         * @param int $order_id The ID of the order to retrieve.
         * @return object An object containing the order details. If the bridge is active, returns a custom object with
         *                order details such as title, status, date, and cart information. If the bridge is not active,
         *                returns a Tickera\TC_Order object.
         */
        function get_order( $order_id ) {

            if ( $this->is_bridge_active ) {

                $order = new StdClass;
                $order->details = new StdClass;
                $order->details->ID = $order_id;

                try {

                    $wc_order = new WC_Order( $order_id );
                    $order->details->post_title = $wc_order->get_order_number();
                    $order->details->post_status = $wc_order->get_status();
                    $order->details->post_date_gmt = $wc_order->get_date_created();

                    $cart_info = get_post_meta( $order_id, 'tc_cart_info', true );

                    if ( ! $cart_info || ! isset( $cart_info['buyer_data'] ) ) {
                        $cart_info = [
                            'buyer_data' => [
                                'first_name_post_meta' => $wc_order->get_billing_first_name(),
                                'last_name_post_meta' => $wc_order->get_billing_last_name(),
                                'email_post_meta' => $wc_order->get_billing_email()
                            ]
                        ];
                    }

                    $order->details->tc_cart_info = $cart_info;

                } Catch ( Exception $e ) {

                    $order->details->post_title = '';
                    $order->details->post_status = '';
                    $order->details->post_date_gmt = '';
                    $order->details->tc_cart_info = [];
                }

            } else {
                $order = new Tickera\TC_Order( $order_id );
            }

            return $order;
        }
    }
}