<?php
/**
 * Paymill - Payment Gateway
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( '\Tickera\Gateway\TC_Gateway_Paymill' ) ) {

    class TC_Gateway_Paymill extends TC_Gateway_API {

        var $plugin_name = 'paymill';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl = true;
        var $ipn_url;
        var $publishable_key, $private_key, $currency;
        var $currencies = array();
        var $permanently_active = false;
        var $skip_payment_screen = false;
        var $public_key;


        /**
         * Support for older payment gateway API
         */
        function on_creation() {
            $this->init();
        }

        function init() {
            global $tc;

            $this->admin_name = __( 'Paymill', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'Credit Card', 'tickera-event-ticketing-system' );

            $this->method_img_url = apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/paymill.png', $this->plugin_name );
            $this->admin_img_url = apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-paymill.png', $this->plugin_name );

            $this->public_key = $this->get_option( 'public_key' );
            $this->private_key = $this->get_option( 'private_key' );

            $this->force_ssl = $this->get_option( 'is_ssl', '0' ) == '1' ? true : false;
            $this->currency = $this->get_option( 'currency', 'EUR' );

            $currencies = array(
                "EUR" => __( 'EUR - Euro', 'tickera-event-ticketing-system' ),
                "CZK" => __( 'CZK - Czech Koruna', 'tickera-event-ticketing-system' ),
                "DKK" => __( 'DKK - Danish Krone', 'tickera-event-ticketing-system' ),
                "HUF" => __( 'HUF - Hungarian Forint', 'tickera-event-ticketing-system' ),
                "ISK" => __( 'ISK - Iceland Krona', 'tickera-event-ticketing-system' ),
                "ILS" => __( 'ILS - Israeli Shekel', 'tickera-event-ticketing-system' ),
                "LVL" => __( 'LVL - Latvian Lat', 'tickera-event-ticketing-system' ),
                "CHF" => __( 'CHF - Swiss Franc', 'tickera-event-ticketing-system' ),
                "LTL" => __( 'LTL - Lithuanian Litas', 'tickera-event-ticketing-system' ),
                "NOK" => __( 'NOK - Norwegian Krone', 'tickera-event-ticketing-system' ),
                "PLN" => __( 'PLN - Polish Zloty', 'tickera-event-ticketing-system' ),
                "SEK" => __( 'SEK - Swedish Krona', 'tickera-event-ticketing-system' ),
                "TRY" => __( 'TRY - Turkish Lira', 'tickera-event-ticketing-system' ),
                "GBP" => __( 'GBP - British Pound', 'tickera-event-ticketing-system' )
            );

            $this->currencies = $currencies;

            add_action( 'wp_enqueue_scripts', array( &$this, 'enqueue_scripts' ) );
        }

        function enqueue_scripts() {
            global $tc, $wp;


            if ( $this->is_active() && $this->is_payment_page() ) {

                wp_enqueue_script( 'js-paymill', 'https://bridge.paymill.com/', array( 'jquery' ) );
                wp_enqueue_script( 'paymill-token', $tc->plugin_url . '/includes/gateways/paymill/paymill_token.js', array( 'js-paymill', 'jquery' ) );
                wp_localize_script( 'paymill-token', 'paymill_token', array(
                        'public_key' => $this->public_key,
                        'invalid_cc_number' => __( 'Please enter a valid Credit Card Number.', 'tickera-event-ticketing-system' ),
                        'invalid_expiration' => __( 'Please choose a valid Expiration Date.', 'tickera-event-ticketing-system' ),
                        'invalid_cvc' => __( 'Please enter a valid Card CVC', 'tickera-event-ticketing-system' ),
                        'expired_card' => __( 'Card is no longer valid or has expired', 'tickera-event-ticketing-system' ),
                        'invalid_cardholder' => __( 'Invalid cardholder', 'tickera-event-ticketing-system' ),
                    )
                );
            }
        }

        function payment_form( $cart ) {
            global $tc;
            $content = '';
            $content .= '<div id="paymill_checkout_errors"></div>';
            $content .= '<table class="tc_cart_billing"><thead><tr><th colspan="2">' . esc_html__( 'Enter Your Credit Card Information:', 'tickera-event-ticketing-system' ) . '</th></tr></thead><tbody><tr><td>' . esc_html__( 'Cardholder Name:', 'tickera-event-ticketing-system' ) . '</td><td><input class="card-holdername tickera-input-field" type="text" value="' . esc_attr( $this->buyer_info( 'full_name' ) ) . '" /> </td></tr>';
            $content .= '<tr>';
            $content .= '<td>';
            $content .= esc_html__( 'Card Number', 'tickera-event-ticketing-system' );
            $content .= '</td>';
            $content .= '<td>';
            $content .= '<input type="text" autocomplete="off" class="card-number"/>';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td>';
            $content .= esc_html__( 'Expiration:', 'tickera-event-ticketing-system' );
            $content .= '</td>';
            $content .= '<td>';
            $content .= '<select class="card-expiry-month">';
            $content .= tickera_months_dropdown();
            $content .= '</select>';
            $content .= '<span> / </span>';
            $content .= '<select class="card-expiry-year">';
            $content .= tickera_years_dropdown( '', true );
            $content .= '</select>';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td>';
            $content .= esc_html__( 'CVC:', 'tickera-event-ticketing-system' );
            $content .= '</td>';
            $content .= '<td>';
            $content .= '<input type="text" size="4" autocomplete="off" class="card-cvc" />';
            $content .= '<input type="hidden" class="currency" value="' . $this->currency . '" />';
            $content .= '<input type="hidden" class="amount" value="' . $this->total() * 100 . '" />';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '</table>';
            $content .= '<span id="paymill_processing" style="display: none;float: right;"><img src="' . esc_url( $tc->plugin_url ) . 'images/loading.gif" /> ' . esc_html__( 'Processing...', 'tickera-event-ticketing-system' ) . '</span>';
            return $content;
        }

        /**
         * Generate Order Confirmation Page upon success checkout
         *
         * @param $order
         * @param string $cart_info
         * @return string
         */
        function order_confirmation_message( $order, $cart_info = '' ) {

            global $tc;
            $order = tickera_get_order_id_by_name( $order );
            $order = new \Tickera\TC_Order( $order->ID );
            $content = '';

            switch ( $order->details->post_status ) {

                case 'order_received':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: %s: Payment gateway name */ __( 'Your payment via Paymill for this order totaling <strong>%s</strong> is not yet complete.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    $content .= '<p>' . wp_kses_post( __( 'Current order status: <strong>Pending Payment</strong>', 'tickera-event-ticketing-system' ) ) . '</p>';
                    break;

                case 'order_fraud':
                    $content .= '<p>' . esc_html__( 'Your payment is under review. We will back to you soon.', 'tickera-event-ticketing-system' ) . '</p>';
                    break;

                case 'order_paid':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: %s: Payment gateway name */ __( 'Your payment via Paymill for this order totaling <strong>%s</strong> is complete.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    break;

                case 'order_cancelled':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: %s: Payment gateway name */ __( 'Your payment via Paymill for this order totaling <strong>%s</strong> is cancelled.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    break;

                case 'order_refunded':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: %s: Payment gateway name */ __( 'Your payment via Paymill for this order totaling <strong>%s</strong> is refunded.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    break;

            }

            $content = wp_kses_post( apply_filters( 'tc_order_confirmation_message_content', $content, $order, $this->plugin_name ) );
            $tc->remove_order_session_data();
            $tc->session->drop( 'paymillToken' );
            $tc->maybe_skip_confirmation_screen( $this, $order );
            return $content;
        }

        function process_payment( $cart ) {

            global $tc;
            tickera_final_cart_check( $cart );
            $this->save_cart_info();

            if ( isset( $_POST[ 'paymillToken' ] ) ) {
                $tc->session->set( 'paymillToken', sanitize_text_field( $_POST[ 'paymillToken' ] ) );
            }

            $session_paymill_token = $tc->session->get( 'paymillToken' );
            if ( is_null($session_paymill_token ) ) {
                $tc->session->set( 'tc_gateway_error', __( 'The Paymill Token was not generated correctly.', 'tickera-event-ticketing-system' ) );
                tickera_redirect( $tc->get_payment_slug( true ), true );
            }

            define( 'TC_PAYMILL_API_HOST', 'https://api.paymill.com/v2/' );
            define( 'TC_PAYMILL_API_KEY', $this->get_option( 'private_key' ) );

            $token = sanitize_text_field( $session_paymill_token );

            if ( $token ) {
                require "paymill/lib/Services/Paymill/Transactions.php";
                $transactionsObject = new Services_Paymill_Transactions( TC_PAYMILL_API_KEY, TC_PAYMILL_API_KEY );

                $order_id = $tc->generate_order_id();

                try {
                    $params = array(
                        'amount' => $this->total() * 100, //// I.e. 49 * 100 = 4900 Cents = 49 EUR
                        'currency' => strtolower( $this->currency ), // ISO 4217
                        'token' => $token,
                        'description' => $this->cart_items()
                    );

                    $charge = $transactionsObject->create( $params );
                    $charge = tickera_sanitize_array( $charge, false, true );

                    if ( $charge[ 'status' ] == 'closed' ) {

                        // Setup our payment details
                        $payment_info = array();
                        $payment_info[ 'method' ] = sprintf( /* translators: 1: Paymill Card type 2: Last four digits of account number 3: Expiry month */ __( '%1$s Card ending in %2$s - Expires %3$s', 'tickera-event-ticketing-system' ), ucfirst( $charge[ 'payment' ][ 'card_type' ] ), $charge[ 'payment' ][ 'last4' ], $charge[ 'payment' ][ 'expire_month' ] . '/' . $charge[ 'payment' ][ 'expire_year' ] );
                        $payment_info[ 'transaction_id' ] = $charge[ 'id' ];
                        $payment_info = $this->save_payment_info();

                        $paid = true;
                        $order = $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, $paid );
                        tickera_redirect( $tc->get_confirmation_slug( true, $order_id ), true );
                    }

                } catch ( Exception $e ) {
                    $tc->session->drop( 'paymillToken' );
                    $tc->session->set( 'tc_gateway_error', sprintf( /* translators: %s: Paymill processing card error. */ __( 'There was an error processing your card: "%s".', 'tickera-event-ticketing-system' ), $e->getMessage() ) );
                    tickera_redirect( $tc->get_payment_slug( true ), true );
                }
            }
        }

        function gateway_admin_settings( $settings, $visible ) {
            global $tc;
            ?>
            <div id="<?php echo esc_attr( $this->plugin_name ); ?>" class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3>
                    <span><?php echo esc_html( sprintf( /* translators: %s: Paymill Payment Gateway admin name */ __( '%s Settings', 'tickera-event-ticketing-system' ), esc_html( $this->admin_name ) ) ); ?></span>
                    <span class="description"><?php esc_html_e( 'Accept all major credit and debit cards directly on your site. Credit cards go directly to Paymill\'s secure environment, and never hit your servers so you can avoid most PCI requirements.', 'tickera-event-ticketing-system' ); ?></span>
                </h3>
                <div class="inside">

                    <?php
                    $fields = array(
                        'is_ssl' => array(
                            'title' => __( 'Force SSL', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                '0' => __( 'No', 'tickera-event-ticketing-system' ),
                                '1' => __( 'Yes', 'tickera-event-ticketing-system' )
                            ),
                            'default' => '0',
                        ),
                        'private_key' => array(
                            'title' => __( 'Private key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                            'description' => __( 'You must login to Paymill to <a target="_blank" href="https://app.paymill.com/en-gb/auth/login">get your API credentials</a>. You can enter your test keys, then live ones when ready.', 'tickera-event-ticketing-system' ),
                        ),
                        'public_key' => array(
                            'title' => __( 'Public key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                            'description' => '',
                            'default' => ''
                        ),
                        'currency' => array(
                            'title' => __( 'Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->currencies,
                            'default' => 'USD',
                        ),
                    );
                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name );
                    ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>
                </div>
            </div>
            <?php
        }
    }

    \Tickera\tickera_register_gateway_plugin( '\Tickera\Gateway\TC_Gateway_Paymill', 'paymill', __( 'Paymill', 'tickera-event-ticketing-system' ) );
}
