<?php
/**
 * Komoju - Payment Gateway
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( '\Tickera\Gateway\TC_Gateway_Komoju' ) ) {

    class TC_Gateway_Komoju extends TC_Gateway_API {

        var $plugin_name = 'komoju';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl = false;
        var $ipn_url;
        var $currencies = array();
        var $live_url;
        var $merchant_id = '';
        var $notify_url = '';
        var $success_url = '';
        var $fail_url = '';
        var $permanently_active = false;
        var $currency = '';
        var $skip_payment_screen = false;
        var $sandbox_url;
        var $locale;
        var $mode;
        var $account_id;
        var $secret_key;
        var $locales;
        var $payment_methods;

        /**
         * Support for older payment gateway API
         */
        function on_creation() {
            $this->init();
        }

        function init() {

            global $tc;

            $this->admin_name = __( 'Komoju', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'Komoju', 'tickera-event-ticketing-system' );

            $this->sandbox_url = 'https://sandbox.komoju.com';
            $this->live_url = 'https://komoju.com';

            $this->notify_url = $this->ipn_url;
            $this->success_url = '';

            $this->method_img_url = apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/komoju.png', $this->plugin_name );
            $this->admin_img_url = apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-komoju.png', $this->plugin_name );

            $this->locale = $this->get_option( 'locale', 'ja' );
            $this->currency = $this->get_option( 'currency', 'JPY' );

            $this->mode = $this->get_option( 'mode', 'sandbox' );

            $this->account_id = $this->get_option( 'account_id' );
            $this->secret_key = $this->get_option( 'secret_key' );

            $currencies = array(
                "JPY" => __( 'JPY - Japanese Yen', 'tickera-event-ticketing-system' ),
            );

            $this->locales = array(
                "ja" => __( 'Japanese', 'tickera-event-ticketing-system' ),
                "en" => __( 'English', 'tickera-event-ticketing-system' ),
            );

            $this->payment_methods = array(
                'bank_transfer' => __( '銀行振込 Bank transfer', 'tickera-event-ticketing-system' ),
                'credit_card' => __( 'コンビニ Credit Card (Visa, MasterCard, JCB, AMEX)', 'tickera-event-ticketing-system' ),
                'konbini' => __( 'クレジットカード Convenience Store Payment (Konbini)', 'tickera-event-ticketing-system' ),
                'pay_easy' => __( 'ペイジー PayEasy', 'tickera-event-ticketing-system' ),
                'web_money' => __( 'WebMoney', 'tickera-event-ticketing-system' ),
                'bit_cash' => __( 'BitCash', 'tickera-event-ticketing-system' )
            );

            $this->currencies = $currencies;
        }

        function payment_form( $cart ) {

            $saved_payment_option_values = (array) $this->get_option( 'payment_methods' );
            $content = '<table class="tc_cart_billing">';
            $content .= '<thead>';
            $content .= '<tr>';
            $content .= '<th colspan="2">' . esc_html__( 'Choose payment method:', 'tickera-event-ticketing-system' ) . '</th>';
            $content .= '</tr>';
            $content .= '</thead>';
            $content .= '<tbody>';
            $content .= '<tr>';
            $content .= '<td colspan="2">';

            $first = true;
            foreach ( $this->payment_methods as $payment_method => $payment_title ) {
                if ( in_array( $payment_method, $saved_payment_option_values ) ) {
                    $content .= '<input type="radio" name="komoju_payment_method" value="' . esc_attr( $payment_method ) . '" ' . ( $first ? 'checked' : '' ) . ' /> ' . esc_html( $payment_title ) . '<br />';
                    $first = false;
                }
            }

            $content .= '</td>';
            $content .= '</tr>';
            $content .= '</tbody></table>';
            return $content;
        }

        function process_payment( $cart ) {

            global $tc;

            tickera_final_cart_check( $cart );
            $this->save_cart_info();
            $payment_method = sanitize_text_field( $_POST[ 'komoju_payment_method' ] );
            $order_id = $tc->generate_order_id();

            $paid = false;
            $start_point = $this->mode == 'sandbox' ? $this->sandbox_url : $this->live_url;
            $secret_key = $this->secret_key;
            $endpoint = "/" . $this->locale . "/api/" . $this->account_id . "/transactions/" . $payment_method . "/new";

            $params = array(
                "transaction[amount]" => (float) $this->total(),
                "transaction[currency]" => $this->currency,
                "transaction[customer][given_name]" => $this->buyer_info( 'first_name' ),
                "transaction[customer][family_name]" => $this->buyer_info( 'last_name' ),
                "transaction[customer][given_name_kana]" => $this->buyer_info( 'first_name' ),
                "transaction[customer][family_name_kana]" => $this->buyer_info( 'last_name' ),
                "transaction[external_order_num]" => $order_id,
                "transaction[return_url]" => $tc->get_confirmation_slug( true, $order_id ),
                "transaction[cancel_url]" => $tc->get_cancel_url( $order_id ),
                "transaction[callback_url]" => $this->ipn_url,
                "transaction[tax]" => "0",
                "timestamp" => time(),
            );

            $qs_params = array();

            foreach ( $params as $key => $val ) {
                $qs_params[] = urlencode( $key ) . '=' . urlencode( $val );
            }

            sort( $qs_params );

            $query_string = implode( '&', $qs_params );
            $url = $endpoint . '?' . $query_string;
            $hmac = hash_hmac( 'sha256', $url, $secret_key );

            $payment_info = $this->save_payment_info();
            $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, $paid );
            tickera_redirect( $start_point . $url . '&hmac=' . $hmac, true );
        }

        function order_confirmation( $order, $payment_info = '', $cart_info = '' ) {

            global $tc;

            if ( isset( $_GET[ 'hmac' ] ) ) {

                $order_num = sanitize_text_field( $_GET[ 'transaction' ][ 'external_order_num' ] );
                $status = sanitize_text_field( $_GET[ 'transaction' ][ 'status' ] );
                $order = tickera_get_order_id_by_name( $order_num );

                if ( 'captured' == $status ) {
                    $tc->update_order_payment_status( $order->ID, true );
                }

                $tc->remove_order_session_data();
            }
        }

        function gateway_admin_settings( $settings, $visible ) {
            ?>
            <div id="<?php echo esc_attr( $this->plugin_name ); ?>" class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3>
                    <span><?php echo esc_html( sprintf( /* translators: %s: Komoju Payment Gateway admin name */ __( '%s Settings', 'tickera-event-ticketing-system' ), esc_html( $this->admin_name ) ) ); ?></span>
                    <span class="description"><?php esc_html_e( 'Accept payments in Japan with Komoju.', 'tickera-event-ticketing-system' ) ?></span>
                </h3>
                <div class="inside">
                    <?php
                    $fields = array(
                        'mode' => array(
                            'title' => __( 'Mode', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                'sandbox' => __( 'Sandbox / Test', 'tickera-event-ticketing-system' ),
                                'live' => __( 'Live', 'tickera-event-ticketing-system' )
                            ),
                            'default' => 'sandbox',
                        ),
                        'account_id' => array(
                            'title' => __( 'Account ID', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        /* 'public_key'		 => array(
                          'title'	 => __( 'Publishable Key', 'tickera-event-ticketing-system' ),
                          'type'	 => 'text',
                          ), */
                        'secret_key' => array(
                            'title' => __( 'Secret key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'payment_methods' => array(
                            'title' => __( 'Payment methods', 'tickera-event-ticketing-system' ),
                            'type' => 'checkboxes',
                            'options' => $this->payment_methods,
                        ),
                        'locale' => array(
                            'title' => __( 'Locale', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->locales,
                            'default' => 'ja',
                        ),
                        'currency' => array(
                            'title' => __( 'Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->currencies,
                            'default' => 'JPY',
                        ),
                    );

                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name );
                    ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>
                </div>
            </div>
            <?php
        }

        function ipn() {
            global $tc;

            if ( isset( $_GET[ 'hmac' ] ) ) {

                $order_num = sanitize_text_field( $_GET[ 'transaction' ][ 'external_order_num' ] );
                $status = sanitize_text_field( $_GET[ 'transaction' ][ 'status' ] );
                $order = tickera_get_order_id_by_name( $order_num );

                if ( $status == 'captured' ) {
                    $tc->update_order_payment_status( $order->ID, true );
                }
            }
        }
    }

    \Tickera\tickera_register_gateway_plugin( '\Tickera\Gateway\TC_Gateway_Komoju', 'komoju', __( 'Komoju', 'tickera-event-ticketing-system' ) );
}