<?php
/**
 * Authorize.net (AIM) - Payment Gateway
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( '\Tickera\Gateway\TC_Gateway_AuthorizeNet_AIM' ) ) {

    class TC_Gateway_AuthorizeNet_AIM extends TC_Gateway_API {

        const API_PRODUCTION = 'https://api.authorize.net/xml/v1/request.api';
        const API_TEST = 'https://apitest.authorize.net/xml/v1/request.api';

        var $plugin_name = 'authorizenet-aim';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $currencies = array();
        var $permanently_active = false;
        var $skip_payment_screen = false;
        var $API_Login_ID, $API_Transaction_Key, $API_Endpoint, $currency, $force_ssl, $ipn_url, $additional_fields;

        /**
         * Support for older payment gateway API
         */
        function on_creation() {
            $this->init();
        }

        /**
         * Initialize Variables
         */
        function init() {
            global $tc;

            $this->admin_name = __( 'Authorize.Net API', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'Authorize.net', 'tickera-event-ticketing-system' );

            $this->method_img_url = apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/authorize.png', $this->plugin_name );
            $this->admin_img_url = apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-authorize.png', $this->plugin_name );

            $this->API_Login_ID = $this->get_option( 'api_user' );
            $this->API_Transaction_Key = $this->get_option( 'api_key' );
            $this->currency = $this->get_option( 'currency', 'USD' );
            $this->additional_fields = $this->get_option( 'additional_fields', 'no' );
            $this->force_ssl = ( $this->get_option( 'mode', 'sandbox' ) == 'sandbox' ) ? false : true;
            $this->API_Endpoint = ( $this->force_ssl ) ? self::API_PRODUCTION : self::API_TEST;

            $currencies = array(
                'USD' => __( 'USD - U.S. Dollar', 'tickera-event-ticketing-system' ),
                'CAD' => __( 'CAD - Canadian Dollar', 'tickera-event-ticketing-system' ),
                'AUD' => __( 'AUD - Australian Dollar', 'tickera-event-ticketing-system' ),
                'NZD' => __( 'NZD - New Zealand Dollar', 'tickera-event-ticketing-system' ),
                'CHF' => __( 'CHF - Swiss Franc', 'tickera-event-ticketing-system' ),
                'DKK' => __( 'DKK - Danish Krone', 'tickera-event-ticketing-system' ),
                'EUR' => __( 'EUR - Euro', 'tickera-event-ticketing-system' ),
                'GBP' => __( 'GBP - Pound Sterling', 'tickera-event-ticketing-system' ),
                'NOK' => __( 'NOK - Norwegian Krone', 'tickera-event-ticketing-system' ),
                'PLN' => __( 'PLN - Poland Złoty', 'tickera-event-ticketing-system' ),
                'SEK' => __( 'SEK - Swedish Krona', 'tickera-event-ticketing-system' ),
            );

            $this->currencies = $currencies;
            add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        }

        /**
         * Load CSS and JS Files
         */
        function enqueue_scripts() {

            if ( $this->is_payment_page() && $this->is_active() ) {

                wp_register_script( 'js-authorizenet', plugins_url( '/authorizenet-aim/assets/js/authorizenet.js', __FILE__ ), array( 'jquery' ) );
                wp_localize_script( 'js-authorizenet', 'authorizenet_params', [ 'region_data' => json_decode( $this->get_region_data(), true ), ] );
                wp_enqueue_script( 'js-authorizenet' );

                wp_enqueue_style( 'css-authorizenet', plugins_url( '/authorizenet-aim/assets/css/authorizenet.css', __FILE__ ) );
            }
        }

        /**
         * Generate Payment Form
         * @param $cart
         * @return string|void
         */
        function payment_form( $cart ) {
            global $tc;

            $content = '<div class="overlay"><span class="loading" style="background-image: url(' . esc_url( plugins_url( 'authorizenet-aim/assets/icons/loading.gif', __FILE__ ) ) . ')"></span></div>';

            // IF: Additional fields is enabled
            if ( 'yes' == $this->additional_fields ) {
                $content .= $this->get_additional_fields();
            }

            $content .= '<table class="tc_cart_billing tbl_authorizenet">';
            $content .= '<thead>';
            $content .= '<tr>';
            $content .= '<th colspan="2">' . esc_html__( 'Pay with card:', 'tickera-event-ticketing-system' ) . '</th>';
            $content .= '</tr>';
            $content .= '</thead>';
            $content .= '<tbody>';
            $content .= '<tr>';
            $content .= '<td><label for="authorize-card-num">' . esc_html__( 'Credit Card Number:', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input name="authorize-card-num" id="authorize-card-num" class="credit_card_number input_field tc-numbers-only noautocomplete" type="text" minlength="4" maxlength="22"/></td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label>' . esc_html__( 'Expiration Date:', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td class="authorize-card-expiration">';
            $content .= '<div><select name="authorize-exp-month" id="authorize-exp-month">' . tickera_months_dropdown() . '</select><label for="authorize-exp-month">' . esc_html__( 'Month', 'tickera-event-ticketing-system' ) . '</label></div>';
            $content .= '<div><select name="authorize-exp-year" id="authorize-exp-year">' . tickera_years_dropdown( '', true ) . '</select><label for="authorize-exp-year">' . esc_html__( 'Year', 'tickera-event-ticketing-system' ) . '</label></div>';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="authorize-card-code">' . esc_html__( 'CCV:', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input id="authorize-card-code" name="authorize-card-code" class="input_field tc-numbers-only noautocomplete" type="text" minlength="3" maxlength="4"/></td>';
            $content .= '</tr>';
            $content .= '</tbody></table>';
            return $content;
        }

        /**
         * Generate HTML for additional fields
         * @return string
         */
        function get_additional_fields() {

            $content = '<table class="tbl_authorizenet">';
            $content .= '<thead>';
            $content .= '<tr>';
            $content .= '<th colspan="2">' . esc_html__( 'Billing Information:', 'tickera-event-ticketing-system' ) . '</th>';
            $content .= '</tr>';
            $content .= '</thead><tbody>';
            $content .= '<tr>';
            $content .= '<td><label>' . esc_html__( 'First Name', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td>' . esc_html( $this->buyer_info( 'first_name' ) ) . '</td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label>' . esc_html__( 'Last Name', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td>' . esc_html( $this->buyer_info( 'last_name' ) ) . '</td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label>' . esc_html__( 'Email address', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td>' . esc_html( $this->buyer_info( 'email' ) ) . '</td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="authorize-billing-address">' . esc_html__( 'Address', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input type="authorize-billing-address" id="authorize-billing-address" name="billing_address" class="input_field noautocomplete" type="text"/></td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="authorize-billing-city">' . esc_html__( 'City', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input type="authorize-billing-city" id="authorize-billing-city" name="billing_city" class="input_field noautocomplete" type="text"/></td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="authorize-billing-country">' . esc_html__( 'Country', 'tickera-event-ticketing-system' ) . '</label></td>';

            $content .= '<td><select type="authorize-billing-country" id="authorize-billing-country" name="billing_country" class="input_field noautocomplete authorizenet_billing_country" type="text"><option></option>';
            foreach ( json_decode( $this->get_country_data(), true ) as $country ) {
                $content .= '<option value="' . $country[ 'id' ] . '">' . $country[ 'text' ] . '</option>';
            }
            $content .= '</select></td>';

            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="authorize-billing-state">' . esc_html__( 'State', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><select type="authorize-billing-state" id="authorize-billing-state" name="billing_state" class="input_field noautocomplete authorizenet_billing_state" type="text"><option></option></select></td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="authorize-billing-postal_code">' . esc_html__( 'Zip Code', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input type="authorize-billing-postal_code" id="authorize-billing-postal-code" name="billing_postal_code" class="input_field noautocomplete" type="text"/></td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="authorize-billing-phone">' . esc_html__( 'Phone Number', 'tickera-event-ticketing-system' ) . '</td>';
            $content .= '<td><input type="authorize-billing-phone" id="authorize-billing-phone" name="billing_phone" class="input_field noautocomplete" type="text" onkeypress="return isNumeric(event)"/></td>';
            $content .= '</tr>';
            $content .= '</tbody></table>';
            return $content;
        }

        /**
         * Process Payment and Create Tickera Order
         * @param $cart
         * @return bool|void
         */
        function process_payment( $cart ) {

            global $tc;
            tickera_final_cart_check( $cart );
            $this->save_cart_info();

            $transaction = [];
            $transaction[ "createTransactionRequest" ][ "merchantAuthentication" ] = self::authorize_merchant(); // Authorize Merchant
            $transaction[ "createTransactionRequest" ][ "refId" ] = 'ref' . time(); // Set the transaction's refId
            $transaction[ "createTransactionRequest" ][ "transactionRequest" ] = self::request_transaction();

            $args[ 'user-agent' ] = $tc->title;
            $args[ 'body' ] = json_encode( $transaction );
            $args[ 'sslverify' ] = false;
            $args[ 'timeout' ] = 30;

            $response = wp_remote_post( $this->API_Endpoint, $args );
            $response = is_wp_error( $response ) ? $response->get_error_message() : wp_remote_retrieve_body( $response );
            $response = preg_replace( '/[\x00-\x1F\x80-\xFF]/', '', $response ); // Removing BOM

            // Request successfully delivered to server
            if ( $response && $this->is_json( $response ) ) {

                $responseDecoded = json_decode( $response, true );
                $responseDecoded = tickera_sanitize_array( $responseDecoded, false, true );

                $transactionResponse = $responseDecoded[ 'transactionResponse' ] ? $responseDecoded[ 'transactionResponse' ] : [];
                $responseCode = $transactionResponse[ 'responseCode' ] ? $transactionResponse[ 'responseCode' ] : 0;

                switch ( $responseCode ) {

                    case '1':

                        /**
                         * This transaction has been approved.
                         */
                        $order_id = $tc->generate_order_id();
                        self::tc_create_order( $order_id, $responseDecoded );
                        tickera_redirect( $tc->get_confirmation_slug( true, $order_id ), true );

                        break;

                    default:

                        /**
                         * Transaction failed. Payment is not accepted.
                         */
                        $errors = '';
                        $session_gateway_errors = [];

                        if ( isset( $transactionResponse[ 'errors' ] ) ) {
                            $errors = $transactionResponse[ 'errors' ];

                        } elseif ( isset( $responseDecoded[ 'messages' ] ) && isset( $responseDecoded[ 'messages' ][ 'message' ] ) ) {
                            $errors = $responseDecoded[ 'messages' ][ 'message' ];

                        } else {
                            $tc->session->set( 'tc_cart_errors', __( 'Sorry, something went wrong.', 'tickera-event-ticketing-system' ) );
                        }

                        $errors = array_filter( (array) $errors ); // Remove empty strings from array

                        foreach ( $errors as $key => $error ) {
                            if ( isset( $error[ 'errorText' ] ) || isset( $error[ 'text' ] ) ) {
                                $error_msg = apply_filters( 'tc_authorize_error_response', ( $error[ 'errorText' ] ? $error[ 'errorText' ] : $error[ 'text' ] ), ( isset( $error[ 'errorText' ] ) ? $error[ 'errorCode' ] : $error[ 'code' ] ) );
                                $session_gateway_errors[] = $error_msg;
                            }
                        }

                        $tc->session->set( 'tc_gateway_error', $session_gateway_errors );
                        tickera_redirect( $tc->get_payment_slug( true ), true );
                }

            } else {

                if ( $this->is_html( $response ) ) {
                    $response = htmlspecialchars( $response );
                }

                // Request failed
                if ( $response ) {
                    $tc->session->set( 'tc_gateway_error', $response . ' ' . __( 'Please try again', 'tickera-event-ticketing-system' ) );

                } else {
                    $tc->session->set( 'tc_cart_errors', __( 'Sorry, something went wrong.', 'tickera-event-ticketing-system' ) );
                }

                tickera_redirect( $tc->get_payment_slug( true ), true );
            }
        }

        /**
         * Authorize Merchant before request transactions
         * @return array
         */
        function authorize_merchant() {
            return array( "name" => $this->API_Login_ID, "transactionKey" => $this->API_Transaction_Key );
        }

        /**
         * Prepare Object for a transaction request
         * @return array
         */
        function request_transaction() {
            global $tc;
            return array(
                "transactionType" => "authCaptureTransaction",
                "amount" => $this->total(),
                "currencyCode" => $this->currency,
                "payment" => self::create_payment_data(),
                "order" => self::create_invoice(),
                "customer" => self::create_customer(),
                "billTo" => self::customer_bill_to_address(),
                "customerIP" => sanitize_text_field( $_SERVER[ 'REMOTE_ADDR' ] ),
            );
        }

        /**
         * Prepare Card Object
         * @return array
         */
        function create_payment_data() {
            $card_number = sanitize_text_field( $_POST[ 'authorize-card-num' ] );
            $expiration = sanitize_text_field( $_POST[ 'authorize-exp-year' ] ) . '-' . sanitize_text_field( $_POST[ 'authorize-exp-month' ] );
            $card_cvv = sanitize_text_field( $_POST[ 'authorize-card-code' ] );
            return array( "creditCard" => array( "cardNumber" => $card_number, "expirationDate" => $expiration, "cardCode" => $card_cvv ) );
        }

        /**
         * Prepare Billing To Object
         * @return array
         */
        function customer_bill_to_address() {
            $customerAddress = array( "firstName" => $this->buyer_info( "first_name" ), "lastName" => $this->buyer_info( "last_name" ) );
            if ( 'yes' == $this->additional_fields ) {
                $customerAddress[ "address" ] = sanitize_text_field( $_POST[ 'billing_address' ] );
                $customerAddress[ "city" ] = sanitize_text_field( $_POST[ 'billing_city' ] );
                $customerAddress[ "state" ] = sanitize_text_field( $_POST[ 'billing_state' ] );
                $customerAddress[ "zip" ] = sanitize_text_field( $_POST[ 'billing_postal_code' ] );
                $customerAddress[ "country" ] = sanitize_text_field( $_POST[ 'billing_country' ] );
                $customerAddress[ "phoneNumber" ] = sanitize_text_field( $_POST[ 'billing_phone' ] );
            }
            return $customerAddress;
        }

        /**
         * Prepare Order Invoice Object
         * @return array
         */
        function create_invoice() {
            global $tc;
            return array( "invoiceNumber" => sanitize_text_field( apply_filters( 'tc_authorize_invoice_name', $tc->generate_order_id() ) ), "description" => $this->cart_items() );
        }

        /**
         * Prepare Customer Object
         * @return array
         */
        function create_customer() {
            return array( "type" => "individual", "email" => $this->buyer_info( 'email' ) );
        }

        /**
         * Create Tickera Order if payment request is valid
         * @param $order_id
         * @param $response
         */
        function tc_create_order( $order_id, $response ) {

            global $tc;

            $transactionResponse = $response[ 'transactionResponse' ] ? $response[ 'transactionResponse' ] : [];

            // Create Tickera Order
            $payment_info = array();
            $payment_info[ 'method' ] = __( 'Credit Card', 'tickera-event-ticketing-system' );
            $payment_info[ 'transaction_id' ] = $transactionResponse[ 'transId' ] ? $transactionResponse[ 'transId' ] : 0;
            $payment_info = $this->save_payment_info( $payment_info );

            // Create Tickera Order
            $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, false );
            $order_id = tickera_get_order_id_by_name( $order_id )->ID;

            // Update Order Status to paid
            $tc->update_order_payment_status( $order_id, true );

            // Insert Order and Payment Response Notes
            if ( isset( $transactionResponse[ 'messages' ] ) ) {
                $notes = $transactionResponse[ 'messages' ];

            } elseif ( isset( $response[ 'messages' ] ) && isset( $response[ 'messages' ][ 'message' ] ) ) {
                $notes = $response[ 'messages' ][ 'message' ];

            } elseif ( isset( $transactionResponse[ 'errors' ] ) ) {
                $notes = $transactionResponse[ 'errors' ];

            } else {
                $notes = '';
            }

            $notes = array_filter( (array) $notes ); // Remove empty string from array

            foreach ( $notes as $note ) {
                if ( isset( $note[ 'description' ] ) || isset( $note[ 'text' ] ) ) {
                    $note = $note[ 'description' ] ? $note[ 'description' ] : $note[ 'text' ];
                    \Tickera\TC_Order::add_order_note( $order_id, $note );
                }
            }
        }

        function is_json( $string ) {
            json_decode( $string );
            return ( json_last_error() == JSON_ERROR_NONE );
        }

        function is_html( $string ) {
            return ( $string != strip_tags( $string ) ) ? true : false;
        }

        /**
         * Generate view for Admin Setting
         * @param $settings
         * @param $visible
         */
        function gateway_admin_settings( $settings, $visible ) {
            global $tc;
            ?>
            <div id="<?php echo esc_attr( $this->plugin_name ); ?>" class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3>
                    <span><?php echo esc_html( sprintf( /* translators: %s: AuthorizeNet Payment Gateway admin name */ __( '%s Settings', 'tickera-event-ticketing-system' ), esc_html( $this->admin_name ) ) ); ?></span>
                    <span class="description"><?php esc_html_e( 'Sell your tickets via Authorize.net. SSL certificate is required for live transactions.', 'tickera-event-ticketing-system' ) ?></span>
                </h3>
                <div class="inside">
                    <?php
                    $fields = array(
                        'mode' => array(
                            'title' => __( 'Mode', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                'sandbox' => __( 'Sandbox / Test', 'tickera-event-ticketing-system' ),
                                'live' => __( 'Live', 'tickera-event-ticketing-system' )
                            ),
                            'default' => 'sandbox',
                        ),
                        'api_user' => array(
                            'title' => __( 'Login ID', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'api_key' => array(
                            'title' => __( 'Transaction key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                            'description' => '',
                            'default' => ''
                        ),
                        'additional_fields' => array(
                            'title' => __( 'Show additional fields (required for EU merchants)', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'default' => 'no',
                            'options' => array(
                                'yes' => __( 'Yes', 'tickera-event-ticketing-system' ),
                                'no' => __( 'No', 'tickera-event-ticketing-system' )
                            ),
                            'description' => 'Fields added to checkout are billing information: Address, City, State, Zip Code, Country',
                            'default' => 'no'
                        ),
                        'currency' => array(
                            'title' => __( 'Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->currencies,
                            'default' => 'USD',
                        ),
                    );
                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name );
                    ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>

                </div>
            </div>
            <?php
        }
    }

    // Register payment gateway plugin
    \Tickera\tickera_register_gateway_plugin( '\Tickera\Gateway\TC_Gateway_AuthorizeNet_AIM', 'authorizenet-aim', __( 'Authorize.Net API', 'tickera-event-ticketing-system' ) );
}