import './style.scss';

import { useCallback, useEffect, useState } from "@wordpress/element";
import { useSelect, useDispatch } from '@wordpress/data';
import { debounce } from "lodash";

/**
 * Woocommerce Integration Data
 */
import { getSetting } from '@woocommerce/settings';
const checkoutData = getSetting( 'checkout-recaptcha_data' );
const blockNamespace = 'checkout-recaptcha';

var pageLoaded = false;
var $ = jQuery;

export const Block = ( { checkoutExtensionData, extensions } ) => {

	if ( document.readyState == 'complete' ) {
		pageLoaded = true;
	}

	let contents = [];

	if ( checkoutData.show_recaptcha ) {

		/*
		 * setExtensionData will update the wc/store/checkout data store with the values supplied.
		 * It can be used to pass data from the client to the server when submitting the checkout form.
		 */
		const { setExtensionData } = checkoutExtensionData;

		/*
         * Debounce the setExtensionData function to avoid
         * multiple calls to the API when rapidly changing options.
         */
		const debouncedSetExtensionData = useCallback(
			debounce((namespace, key, value) => {
				setExtensionData(namespace, key, value);
			}, 1000),
			[setExtensionData]
		);

		const key = 'chre_g_recaptcha';

		/**
		 * Handles fields validation
		 * Applies on checkbox last option
		 */
		const validationErrorId = key + '-validator';
		const { setValidationErrors, clearValidationError } = useDispatch( 'wc/store/validation' );
		const validationError = useSelect( ( select ) => {
			const store = select( 'wc/store/validation' );
			return store.getValidationError( validationErrorId );
		} );

		/**
		 * Handles changes in fields value.
		 * Exclude Label field.
		 */
		const [ fieldValue, setFieldValue ] = useState('');

		useEffect( () => {

			// Update in checkout-owner-fields extended data
			setExtensionData( blockNamespace, key, fieldValue );

			// Ensures setExtensionData is being called one at a time.
			debouncedSetExtensionData( blockNamespace, key, fieldValue );

			// Initialize error to ensure checkout will not proceed until successful recaptcha
			setValidationErrors( {
				[ validationErrorId ]: {
					message: '',
					hidden: true,
				},
			} );

			if ( pageLoaded ) {

				$.post( tcchre.ajaxUrl, {
					action: 'chre_verify_user_response',
					user_response: fieldValue,

				}, function ( response ) {

					if ( typeof response !== 'undefined' ) {

						if ( response.success ) {
							clearValidationError( validationErrorId );

						} else {

							setValidationErrors( {
								[ validationErrorId ]: {
									message: '',
									hidden: false,
								},
							} );
						}
					}
				});
			}

		}, [ setExtensionData, fieldValue, setValidationErrors, clearValidationError ] );

		let error = ( ( typeof validationError !== 'undefined' ) && ! validationError?.hidden ) ? true : false;

		contents.push(
			<div className={ "wp-block-checkout-recaptcha" + ( error ? ' has-error' : '' ) }>
				<div id="tcchre-g-recaptcha"></div>
				<input type="text" id="chre_response" onChange={ ( e ) => setFieldValue( e.target.value ) }/>
			</div>
		);
	}

	return contents;
};
