<?php
/**
 * Stripe Elements 3D Secure - Payment Gateway
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

use TCStripe\Event;
use TCStripe\Customer;
use TCStripe\Exception\ApiErrorException;

use TCStripe\Stripe;
use TCStripe\StripeClient;
use TCStripe\PaymentIntent;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'Tickera\Gateway\TC_Gateway_Stripe_Elements_3DS' ) ) {

    class TC_Gateway_Stripe_Elements_3DS extends TC_Gateway_API {

        var $plugin_name = 'stripe-elements-3d-secure';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl;
        var $ipn_url;
        var $publishable_key, $private_key, $currency;
        var $currencies = array();
        var $permanently_active = false;
        var $skip_payment_screen = false;
        var $send_receipt = 0;
        var $manually_capture_payments;
        var $enable_webhook;
        var $zero_decimal_currencies;

        /**
         * Support for older payment gateway API
         */
        public function on_creation() {
            $this->init();
            $this->init_stripe();
            add_action( 'tc_save_tc_gateway_settings', array( $this, 'save_settings' ) );
        }

        /**
         * Register/Remove Stripe Webhook
         *
         * @return void
         * @since 3.5.5.6
         */
        function save_settings() {

            try {

                $webhook_exists = false;
                $manual_capture = $this->get_option( 'manually_capture_payments', 0 );
                $enabled_webhook = $this->get_option( 'enable_webhook', 0 );

                // Retrieve all stripe webhooks and check if Tickera Stripe Exists
                $stripe_webhook = new StripeClient( $this->private_key );
                $stripe_webhook_obj = $stripe_webhook->webhookEndpoints->all();
                $stripe_webhook_obj = tickera_sanitize_array( $stripe_webhook_obj, false, true );

                foreach ( $stripe_webhook_obj[ 'data' ] as $key => $value ) {
                    if ( $value[ 'url' ] == $this->ipn_url ) {

                        // Remove webhook
                        if ( ! $enabled_webhook && ! $manual_capture ) {
                            $stripe_webhook->webhookEndpoints->delete( $value[ 'id' ] );
                        }

                        $webhook_exists = true;
                        break;
                    }
                }

                // Create webhook
                if ( ( $enabled_webhook || $manual_capture ) && ! $webhook_exists ) {
                    $stripe_webhook->webhookEndpoints->create([
                        'url' => $this->ipn_url,
                        'description' => 'Tickera: Charge Captured Event',
                        'enabled_events' => [ 'charge.captured', 'charge.refunded', 'payment_intent.succeeded', 'payment_intent.payment_failed' ]
                    ]);
                }

            } Catch ( \Exception $e ) {
                // Nothing to do here for now
            }
        }

        /**
         * Initialize Stripe Files
         */
        public function init_stripe() {
            global $tc;
            try {
                require_once( $tc->plugin_dir . 'includes/gateways/stripe/init.php' );
                Stripe::setApiKey( $this->private_key );

            } Catch( \Exception $e ) {
                $tc->session->set( 'tc_gateway_error', sanitize_text_field( $e->getMessage() ) . ' ' . sprintf( /* translators: 1: A link to Tickera cart page */ __( '<a href="%s">Please try again</a>.', 'tickera-event-ticketing-system' ), esc_url( $tc->get_cart_slug( true ) ) ) );
            }
        }

        /**
         * Initialize Variables
         */
        public function init() {
            global $tc;

            // Register Ajax Actions
            add_action( 'wp_ajax_process_payment', array( &$this, 'process_payment_ajax' ) );
            add_action( 'wp_ajax_nopriv_process_payment', array( &$this, 'process_payment_ajax' ) );
            add_action( 'wp_ajax_order_confirmation', array( &$this, 'order_confirmation_ajax' ) );
            add_action( 'wp_ajax_nopriv_order_confirmation', array( &$this, 'order_confirmation_ajax' ) );

            $this->admin_name = __( 'Stripe Elements 3D Secure', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'Stripe Elements 3D Secure', 'tickera-event-ticketing-system' );

            $this->method_img_url = apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/stripe.png', $this->plugin_name );
            $this->admin_img_url = apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-stripe-elements-3ds.png', $this->plugin_name );
            $this->publishable_key = $this->get_option( 'publishable_key' );
            $this->private_key = $this->get_option( 'private_key' );
            $this->force_ssl = $this->get_option( 'is_ssl', '0' ) === '1';
            $this->currency = $this->get_option( 'currency', 'USD' );
            $this->zero_decimal_currencies = array( 'MGA', 'BIF', 'CLP', 'PYG', 'DJF', 'RWF', 'GNF', 'UGX', 'JPY', 'VND', 'VUV', 'XAF', 'KMF', 'KRW', 'XOF', 'XPF' );
            $this->send_receipt = $this->get_option( 'send_receipt', '0' );
            $this->manually_capture_payments = $this->get_option( 'manually_capture_payments', 0 );
            $this->enable_webhook = $this->get_option( 'enable_webhook', 0 );

            $currencies = array(
                'AED' => __( 'AED - United Arab Emirates Dirham', 'tickera-event-ticketing-system' ),
                'AFN' => __( 'AFN - Afghan Afghani', 'tickera-event-ticketing-system' ),
                'ALL' => __( 'ALL - Albanian Lek', 'tickera-event-ticketing-system' ),
                'AMD' => __( 'AMD - Armenian Dram', 'tickera-event-ticketing-system' ),
                'ANG' => __( 'ANG - Netherlands Antillean Gulden', 'tickera-event-ticketing-system' ),
                'AOA' => __( 'AOA - Angolan Kwanza', 'tickera-event-ticketing-system' ),
                'ARS' => __( 'ARS - Argentine Peso', 'tickera-event-ticketing-system' ),
                'AUD' => __( 'AUD - Australian Dollar', 'tickera-event-ticketing-system' ),
                'AWG' => __( 'AWG - Aruban Florin', 'tickera-event-ticketing-system' ),
                'AZN' => __( 'AZN - Azerbaijani Manat', 'tickera-event-ticketing-system' ),
                'BAM' => __( 'BAM - Bosnia & Herzegovina Convertible Mark', 'tickera-event-ticketing-system' ),
                'BBD' => __( 'BBD - Barbadian Dollar', 'tickera-event-ticketing-system' ),
                'BDT' => __( 'BDT - Bangladeshi Taka', 'tickera-event-ticketing-system' ),
                'BGN' => __( 'BGN - Bulgarian Lev', 'tickera-event-ticketing-system' ),
                'BIF' => __( 'BIF - Burundian Franc', 'tickera-event-ticketing-system' ),
                'BMD' => __( 'BMD - Bermudian Dollar', 'tickera-event-ticketing-system' ),
                'BND' => __( 'BND - Brunei Dollar', 'tickera-event-ticketing-system' ),
                'BOB' => __( 'BOB - Bolivian Boliviano', 'tickera-event-ticketing-system' ),
                'BRL' => __( 'BRL - Brazilian Real', 'tickera-event-ticketing-system' ),
                'BSD' => __( 'BSD - Bahamian Dollar', 'tickera-event-ticketing-system' ),
                'BWP' => __( 'BWP - Botswana Pula', 'tickera-event-ticketing-system' ),
                'BZD' => __( 'BZD - Belize Dollar', 'tickera-event-ticketing-system' ),
                'CAD' => __( 'CAD - Canadian Dollar', 'tickera-event-ticketing-system' ),
                'CDF' => __( 'CDF - Congolese Franc', 'tickera-event-ticketing-system' ),
                'CHF' => __( 'CHF - Swiss Franc', 'tickera-event-ticketing-system' ),
                'CLP' => __( 'CLP - Chilean Peso', 'tickera-event-ticketing-system' ),
                'CNY' => __( 'CNY - Chinese Renminbi Yuan', 'tickera-event-ticketing-system' ),
                'COP' => __( 'COP - Colombian Peso', 'tickera-event-ticketing-system' ),
                'CRC' => __( 'CRC - Costa Rican Colon', 'tickera-event-ticketing-system' ),
                'CVE' => __( 'CVE - Cape Verdean Escudo', 'tickera-event-ticketing-system' ),
                'CZK' => __( 'CZK - Czech Koruna', 'tickera-event-ticketing-system' ),
                'DJF' => __( 'DJF - Djiboutian Franc', 'tickera-event-ticketing-system' ),
                'DKK' => __( 'DKK - Danish Krone', 'tickera-event-ticketing-system' ),
                'DOP' => __( 'DOP - Dominican Peso', 'tickera-event-ticketing-system' ),
                'DZD' => __( 'DZD - Algerian Dinar', 'tickera-event-ticketing-system' ),
                'EEK' => __( 'EEK - Estonian Kroon', 'tickera-event-ticketing-system' ),
                'EGP' => __( 'EGP - Egyptian Pound', 'tickera-event-ticketing-system' ),
                'ETB' => __( 'ETB - Ethiopian Birr', 'tickera-event-ticketing-system' ),
                'EUR' => __( 'EUR - Euro', 'tickera-event-ticketing-system' ),
                'FJD' => __( 'FJD - Fijian Dollar', 'tickera-event-ticketing-system' ),
                'FKP' => __( 'FKP - Falkland Islands Pound', 'tickera-event-ticketing-system' ),
                'GBP' => __( 'GBP - British Pound', 'tickera-event-ticketing-system' ),
                'GEL' => __( 'GEL - Georgian Lari', 'tickera-event-ticketing-system' ),
                'GIP' => __( 'GIP - Gibraltar Pound', 'tickera-event-ticketing-system' ),
                'GMD' => __( 'GMD - Gambian Dalasi', 'tickera-event-ticketing-system' ),
                'GNF' => __( 'GNF - Guinean Franc', 'tickera-event-ticketing-system' ),
                'GTQ' => __( 'GTQ - Guatemalan Quetzal', 'tickera-event-ticketing-system' ),
                'GYD' => __( 'GYD - Guyanese Dollar', 'tickera-event-ticketing-system' ),
                'HKD' => __( 'HKD - Hong Kong Dollar', 'tickera-event-ticketing-system' ),
                'HNL' => __( 'HNL - Honduran Lempira', 'tickera-event-ticketing-system' ),
                'HRK' => __( 'HRK - Croatian Kuna', 'tickera-event-ticketing-system' ),
                'HTG' => __( 'HTG - Haitian Gourde', 'tickera-event-ticketing-system' ),
                'HUF' => __( 'HUF - Hungarian Forint', 'tickera-event-ticketing-system' ),
                'IDR' => __( 'IDR - Indonesian Rupiah', 'tickera-event-ticketing-system' ),
                'ILS' => __( 'ILS - Israeli New Sheqel', 'tickera-event-ticketing-system' ),
                'INR' => __( 'INR - Indian Rupee', 'tickera-event-ticketing-system' ),
                'ISK' => __( 'ISK - Icelandic Krona', 'tickera-event-ticketing-system' ),
                'JMD' => __( 'JMD - Jamaican Dollar', 'tickera-event-ticketing-system' ),
                'JPY' => __( 'JPY - Japanese Yen', 'tickera-event-ticketing-system' ),
                'KES' => __( 'KES - Kenyan Shilling', 'tickera-event-ticketing-system' ),
                'KGS' => __( 'KGS - Kyrgyzstani Som', 'tickera-event-ticketing-system' ),
                'KHR' => __( 'KHR - Cambodian Riel', 'tickera-event-ticketing-system' ),
                'KMF' => __( 'KMF - Comorian Franc', 'tickera-event-ticketing-system' ),
                'KRW' => __( 'KRW - South Korean Won', 'tickera-event-ticketing-system' ),
                'KYD' => __( 'KYD - Cayman Islands Dollar', 'tickera-event-ticketing-system' ),
                'KZT' => __( 'KZT - Kazakhstani Tenge', 'tickera-event-ticketing-system' ),
                'LAK' => __( 'LAK - Lao Kip', 'tickera-event-ticketing-system' ),
                'LBP' => __( 'LBP - Lebanese Pound', 'tickera-event-ticketing-system' ),
                'LKR' => __( 'LKR - Sri Lankan Rupee', 'tickera-event-ticketing-system' ),
                'LRD' => __( 'LRD - Liberian Dollar', 'tickera-event-ticketing-system' ),
                'LSL' => __( 'LSL - Lesotho Loti', 'tickera-event-ticketing-system' ),
                'LTL' => __( 'LTL - Lithuanian Litas', 'tickera-event-ticketing-system' ),
                'LVL' => __( 'LVL - Latvian Lats', 'tickera-event-ticketing-system' ),
                'MAD' => __( 'MAD - Moroccan Dirham', 'tickera-event-ticketing-system' ),
                'MDL' => __( 'MDL - Moldovan Leu', 'tickera-event-ticketing-system' ),
                'MGA' => __( 'MGA - Malagasy Ariary', 'tickera-event-ticketing-system' ),
                'MKD' => __( 'MKD - Macedonian Denar', 'tickera-event-ticketing-system' ),
                'MNT' => __( 'MNT - Mongolian TÃ¶grÃ¶g', 'tickera-event-ticketing-system' ),
                'MOP' => __( 'MOP - Macanese Pataca', 'tickera-event-ticketing-system' ),
                'MRO' => __( 'MRO - Mauritanian Ouguiya', 'tickera-event-ticketing-system' ),
                'MUR' => __( 'MUR - Mauritian Rupee', 'tickera-event-ticketing-system' ),
                'MVR' => __( 'MVR - Maldivian Rufiyaa', 'tickera-event-ticketing-system' ),
                'MWK' => __( 'MWK - Malawian Kwacha', 'tickera-event-ticketing-system' ),
                'MXN' => __( 'MXN - Mexican Peso', 'tickera-event-ticketing-system' ),
                'MYR' => __( 'MYR - Malaysian Ringgit', 'tickera-event-ticketing-system' ),
                'MZN' => __( 'MZN - Mozambican Metical', 'tickera-event-ticketing-system' ),
                'NAD' => __( 'NAD - Namibian Dollar', 'tickera-event-ticketing-system' ),
                'NGN' => __( 'NGN - Nigerian Naira', 'tickera-event-ticketing-system' ),
                'NIO' => __( 'NIO - Nicaraguan Cordoba', 'tickera-event-ticketing-system' ),
                'NOK' => __( 'NOK - Norwegian Krone', 'tickera-event-ticketing-system' ),
                'NPR' => __( 'NPR - Nepalese Rupee', 'tickera-event-ticketing-system' ),
                'NZD' => __( 'NZD - New Zealand Dollar', 'tickera-event-ticketing-system' ),
                'PAB' => __( 'PAB - Panamanian Balboa', 'tickera-event-ticketing-system' ),
                'PEN' => __( 'PEN - Peruvian Nuevo Sol', 'tickera-event-ticketing-system' ),
                'PGK' => __( 'PGK - Papua New Guinean Kina', 'tickera-event-ticketing-system' ),
                'PHP' => __( 'PHP - Philippine Peso', 'tickera-event-ticketing-system' ),
                'PKR' => __( 'PKR - Pakistani Rupee', 'tickera-event-ticketing-system' ),
                'PLN' => __( 'PLN - Polish Zloty', 'tickera-event-ticketing-system' ),
                'PYG' => __( 'PYG - Paraguayan GuaranÃ­', 'tickera-event-ticketing-system' ),
                'QAR' => __( 'QAR - Qatari Riyal', 'tickera-event-ticketing-system' ),
                'RON' => __( 'RON - Romanian Leu', 'tickera-event-ticketing-system' ),
                'RSD' => __( 'RSD - Serbian Dinar', 'tickera-event-ticketing-system' ),
                'RUB' => __( 'RUB - Russian Ruble', 'tickera-event-ticketing-system' ),
                'RWF' => __( 'RWF - Rwandan Franc', 'tickera-event-ticketing-system' ),
                'SAR' => __( 'SAR - Saudi Riyal', 'tickera-event-ticketing-system' ),
                'SBD' => __( 'SBD - Solomon Islands Dollar', 'tickera-event-ticketing-system' ),
                'SCR' => __( 'SCR - Seychellois Rupee', 'tickera-event-ticketing-system' ),
                'SEK' => __( 'SEK - Swedish Krona', 'tickera-event-ticketing-system' ),
                'SGD' => __( 'SGD - Singapore Dollar', 'tickera-event-ticketing-system' ),
                'SHP' => __( 'SHP - Saint Helenian Pound', 'tickera-event-ticketing-system' ),
                'SLL' => __( 'SLL - Sierra Leonean Leone', 'tickera-event-ticketing-system' ),
                'SOS' => __( 'SOS - Somali Shilling', 'tickera-event-ticketing-system' ),
                'SRD' => __( 'SRD - Surinamese Dollar', 'tickera-event-ticketing-system' ),
                'STD' => __( 'STD - SÃ£o TomÃ© and PrÃ­ncipe Dobra', 'tickera-event-ticketing-system' ),
                'SVC' => __( 'SVC - Salvadoran Colon', 'tickera-event-ticketing-system' ),
                'SZL' => __( 'SZL - Swazi Lilangeni', 'tickera-event-ticketing-system' ),
                'THB' => __( 'THB - Thai Baht', 'tickera-event-ticketing-system' ),
                'TJS' => __( 'TJS - Tajikistani Somoni', 'tickera-event-ticketing-system' ),
                'TOP' => __( 'TOP - Tonga Pa\'anga', 'tickera-event-ticketing-system' ),
                'TRY' => __( 'TRY - Turkish Lira', 'tickera-event-ticketing-system' ),
                'TTD' => __( 'TTD - Trinidad and Tobago Dollar', 'tickera-event-ticketing-system' ),
                'TWD' => __( 'TWD - New Taiwan Dollar', 'tickera-event-ticketing-system' ),
                'TZS' => __( 'TZS - Tanzanian Shilling', 'tickera-event-ticketing-system' ),
                'UAH' => __( 'UAH - Ukrainian Hryvnia', 'tickera-event-ticketing-system' ),
                'UGX' => __( 'UGX - Ugandan Shilling', 'tickera-event-ticketing-system' ),
                'USD' => __( 'USD - United States Dollar', 'tickera-event-ticketing-system' ),
                'UYI' => __( 'UYI - Uruguayan Peso', 'tickera-event-ticketing-system' ),
                'UZS' => __( 'UZS - Uzbekistani Som', 'tickera-event-ticketing-system' ),
                'VEF' => __( 'VEF - Venezuelan Bolivar', 'tickera-event-ticketing-system' ),
                'VND' => __( 'VND - Vietnamese Dong ', 'tickera-event-ticketing-system' ),
                'VUV' => __( 'VUV - Vanuatu Vatu', 'tickera-event-ticketing-system' ),
                'WST' => __( 'WST - Samoan Tala', 'tickera-event-ticketing-system' ),
                'XAF' => __( 'XAF - Central African Cfa Franc', 'tickera-event-ticketing-system' ),
                'XCD' => __( 'XCD - East Caribbean Dollar', 'tickera-event-ticketing-system' ),
                'XOF' => __( 'XOF - West African Cfa Franc', 'tickera-event-ticketing-system' ),
                'XPF' => __( 'XPF - Cfp Franc', 'tickera-event-ticketing-system' ),
                'YER' => __( 'YER - Yemeni Rial', 'tickera-event-ticketing-system' ),
                'ZAR' => __( 'ZAR - South African Rand', 'tickera-event-ticketing-system' ),
                'ZMW' => __( 'ZMW - Zambian Kwacha', 'tickera-event-ticketing-system' ),
            );

            $this->currencies = $currencies;
        }

        /**
         * Load CSS and JS Files
         */
        public function enqueue_scripts() {
            if ( $this->is_payment_page() && $this->is_active() ) {
                wp_enqueue_script( 'js-stripe-elements', 'https://js.stripe.com/v3/', array( 'jquery' ) );
                wp_enqueue_style( 'css-stripe', plugins_url( 'stripe/assets/css/stripe.css', __FILE__ ) );
                wp_enqueue_script( 'js-stripe-client', plugins_url( '/stripe/assets/js/client.js', __FILE__ ), array( 'jquery' ) );
                wp_localize_script( 'js-stripe-client', 'stripe_client', array(
                    'publishable_key' => $this->publishable_key,
                    'styles' => apply_filters( 'tc_stripe_checkout_form_styles', [
                        'base' => [
                            'color' => '#32325d',
                            'fontSmoothing' => 'antialiased',
                            'fontSize' => '1em',
                            '::placeholder' => [
                                'color' => '#aab7c4'
                            ]
                        ],
                        'invalid' => [
                            'color' => '#fa755a',
                            'iconColor' => '#fa755a'
                        ]
                    ])
                ) );
            }
        }

        /**
         *  Generate Payment Form
         * @param $cart
         * @return string|void
         */
        public function payment_form( $cart ) {
            $content = '';
            $content .= '<div id="stripe-inner">';
            $content .= '<div id="card-errors" role="alert"></div>';
            $content .= '<div class="form-row">';
            $content .= '<div id="card-element"></div>';
            $content .= '</div>';
            $content .= '<img id="stripe-loading" title="' . esc_attr__( 'Loading...', 'tickera-event-ticketing-system' ) . '" src="' . esc_url( plugins_url( '/stripe/assets/images/loading.gif', __FILE__ ) ) . '"/><input type="button" id="stripe-submit" class="tickera-button" value="' . esc_attr__( 'Submit Payment', 'tickera-event-ticketing-system' ) . '"/>';
            $content .= '</div>';
            $this->enqueue_scripts();
            return $content;
        }

        /**
         * Initialize transaction in Stripe Gateway.
         * This will serve as a payment validation.
         */
        public function process_payment_ajax() {

            if ( isset( $_POST[ 'nonce' ] ) && wp_verify_nonce( sanitize_key( $_POST['nonce'] ), 'tc_ajax_nonce' ) ) {

                global $tc;
                self::stripe_final_cart_check();

                /*
                 * Execute if "PaymentIntent" is not yet created.
                 * This will initialize a transaction in Stripe Gateway.
                 */
                $session_stripe_payment_gateway = $tc->session->get( 'stripe_payment_gateway' );
                $session_stripe_payment_gateway = $session_stripe_payment_gateway ? $session_stripe_payment_gateway : [];

                if ( ! isset( $session_stripe_payment_gateway[ 'payment_intent' ] ) ) {

                    try {

                        $payment_intent = self::stripe_generate_payment_intent();

                        /*
                         * Create Tickera Order
                         * Update Payment Intent Metadata Order ID and Customer
                         */
                        $order_id = self::stripe_create_order( $payment_intent );
                        $order_title = get_the_title( $order_id );
                        self::stripe_payment_intent_update( $payment_intent, [ 'metadata' => [ 'order_id' => $order_title ] ] );

                        // Set Sessions
                        self::set_sessions( [
                            'payment_intent' => $payment_intent[ 'id' ],
                            'order_id' => $order_id,
                            'first_name' => $this->buyer_info( 'first_name' ),
                            'last_name' => $this->buyer_info( 'last_name' ),
                            'email' => $this->buyer_info( 'email' )
                        ] );
                        self::stripe_customer_update( $payment_intent );

                        $stripe_client = array(
                            'client_secret' => $payment_intent[ 'client_secret' ],
                            'customer_name' => $this->buyer_info( 'first_name' ) . ' ' . $this->buyer_info( 'last_name' ),
                            'email' => $this->buyer_info( 'email' )
                        );

                    } Catch ( \Exception $e ) {
                        $stripe_client = [ 'error' => $e->getMessage() ];
                    }

                } else {

                    /*
                     * Execute if "PaymentIntent" has been created.
                     * Update Tickera Order and Payment Intent
                     * Triggers if the first attempt to submit payment fails.
                     */
                    try {
                        $session_payment_intent = $session_stripe_payment_gateway[ 'payment_intent' ];
                        $payment_intent = tickera_sanitize_array( PaymentIntent::retrieve( sanitize_text_field( $session_payment_intent ) ), false, true );

                        $session_payment_first_name = $session_stripe_payment_gateway[ 'first_name' ];
                        $session_payment_last_name = $session_stripe_payment_gateway[ 'last_name' ];
                        $session_payment_email = $session_stripe_payment_gateway[ 'email' ];

                        $stripe_client = array(
                            'client_secret' => $payment_intent[ 'client_secret' ],
                            'customer_name' => sanitize_text_field( $session_payment_first_name ) . ' ' . sanitize_text_field( $session_payment_last_name ),
                            'email' => sanitize_text_field( $session_payment_email )
                        );

                        // Payment Intent has been cancelled in Stripe Gateway. Cancel existing order and create another one.
                        if ( $payment_intent && isset( $payment_intent[ 'status' ] ) && 'canceled' == $payment_intent[ 'status' ] ) {
                            $tc->update_order_status( $session_stripe_payment_gateway[ 'order_id' ], 'order_cancelled' );
                            self::unset_sessions();
                        }

                    } catch ( \Exception $e ) {
                        $stripe_client = [ 'error' => $e->getMessage() ];
                    }
                }

                wp_send_json( $stripe_client );
            }
        }

        /**
         * Generate Stripe Payment Intent
         *
         * @return array
         * @throws ApiErrorException
         */
        function stripe_generate_payment_intent() {

            $payment_intent_args = [
                'amount' => $this->maybe_fix_total( $this->total() ),
                'currency' => $this->currency,
                'description' => $this->cart_items(),
                'payment_method_types' => [ 'card' ],
                'capture_method' => ( $this->manually_capture_payments ) ? 'manual' : 'automatic',
                'metadata' => apply_filters( 'tc_stripe_checkout_metadata', [], $this->cart_info() )
            ];

            if ( $this->send_receipt ) {
                $payment_intent_args[ 'receipt_email' ] = $this->buyer_info( 'email' );
            }

            $payment_intent = PaymentIntent::create( $payment_intent_args );
            $payment_intent = json_encode( $payment_intent );

            return tickera_sanitize_array( json_decode( $payment_intent, true ), false, true );
        }

        /**
         * Update Stripe Payment Intent Data
         *
         * @param $payment_intent
         * @param $arguments
         * @throws ApiErrorException
         */
        function stripe_payment_intent_update( $payment_intent, $arguments ) {

            PaymentIntent::update(
                $payment_intent[ 'id' ],
                $arguments
            );
        }

        /**
         * Create customer if doesn't exists
         *
         * @param $payment_intent
         * @throws ApiErrorException
         */
        function stripe_customer_update( $payment_intent ) {

            global $tc;
            $session_stripe_payment_gateway = $tc->session->get( 'stripe_payment_gateway' );
            $session_payment_first_name = $session_stripe_payment_gateway[ 'first_name' ];
            $session_payment_last_name = $session_stripe_payment_gateway[ 'last_name' ];
            $session_payment_email = $session_stripe_payment_gateway[ 'email' ];

            $customer = Customer::all( [ 'email' => sanitize_text_field( $session_payment_email ), 'limit' => 1 ] );
            $customer_data = $customer[ 'data' ];

            if ( ! $customer_data ) {

                $customer_data = Customer::create( [
                    'name' => sanitize_text_field( $session_payment_first_name ) . ' ' . sanitize_text_field( $session_payment_last_name ),
                    'email' => sanitize_text_field( $session_payment_email )
                ] );
            }

            $customer_data = is_array( $customer_data ) ? $customer_data[ 0 ] : $customer_data;
            self::stripe_payment_intent_update( $payment_intent, [ 'customer' => $customer_data ] );
        }

        /**
         * Create Tickera Order and Ticket Instances.
         *
         * @param $payment_intent
         * @return
         */
        function stripe_create_order( $payment_intent ) {

            global $tc;

            // Populate payment and cart info
            $payment_info[ 'method' ] = __( 'Credit Card', 'tickera-event-ticketing-system' );
            $payment_info = $this->save_payment_info( $payment_info );
            $this->save_cart_info();

            // Create order post
            $order = $tc->generate_order_id();
            $tc->create_order( $order, $this->cart_contents(), $this->cart_info(), $payment_info, false );
            $order_id = tickera_get_order_id_by_name( $order )->ID;

            // Add order note: Payment Intent
            $live_or_test = ( ! $this->force_ssl ) ? '/test' : '';

            $note = sprintf( /* translators: 1: Either /test or ot 2: Payment Intent */ __( '<a href="https://dashboard.stripe.com%1$s/payments/%2$s" target="_blank">View Payment Intent</a>', 'tickera-event-ticketing-system' ), esc_attr( $live_or_test ), esc_attr( $payment_intent[ 'id' ] ) );
            \Tickera\TC_Order::add_order_note( $order_id, $note );

            // Save order metas
            update_post_meta( $order_id, 'stripe_payment_intent', $payment_intent[ 'id' ] ? sanitize_text_field( $payment_intent[ 'id' ] ) : 'N/A' );

            // Initially set order to cancelled in order to avoid unnecessary committed stocks if the customer failed to send payment
            wp_update_post( [ 'ID' => $order_id, 'post_status' => 'order_cancelled' ] );

            return $order_id;
        }

        /**
         * Confirm Payment Intent.
         * Update Order Status base on Payment Intent Result
         */
        public function order_confirmation_ajax() {

            global $tc;
            $redirect = false;
            $stripe_result = tickera_sanitize_array( $_POST[ 'payment_result' ], false, true );

            $session_stripe_payment_gateway = $tc->session->get( 'stripe_payment_gateway' );
            $order_id = (int) $session_stripe_payment_gateway[ 'order_id' ];
            $order_title = get_the_title( $order_id );

            if ( isset( $_POST[ 'nonce' ] ) && wp_verify_nonce( sanitize_key( $_POST['nonce'] ), 'tc_ajax_nonce' ) ) {

                if ( isset( $stripe_result[ 'error' ] ) ) {

                    // Update Order Status based on Payment Intent Error
                    \Tickera\TC_Order::add_order_note( $order_id, $stripe_result[ 'error' ][ 'message' ] );

                    if ( isset( $stripe_result[ 'error' ] )
                        && isset( $stripe_result[ 'error' ][ 'payment_intent' ] )
                        && isset( $stripe_result[ 'error' ][ 'payment_intent' ][ 'status' ] ) ) {

                        switch ( $stripe_result[ 'error' ][ 'payment_intent' ][ 'status' ] ) {

                            case 'succeeded':
                                $tc->update_order_payment_status( $order_id, true );
                                self::unset_sessions();
                                $redirect = $tc->get_confirmation_slug( true, $order_title );
                                break;
                        }
                    }

                } elseif ( 'succeeded' == $stripe_result[ 'status' ] ) {

                    /*
                     * Payment Intent has been successfully processed
                     * Update order status as paid
                     */
                    $tc->update_order_payment_status( $order_id, true );
                    $redirect = $tc->get_confirmation_slug( true, $order_title );
                    \Tickera\TC_Order::add_order_note( $order_id, 'Payment Intent: ' . $stripe_result[ 'status' ] );
                    self::unset_sessions();

                } elseif ( 'requires_capture' == $stripe_result[ 'status' ] ) {

                    /*
                     * Payment will require manual capture in stripe account
                     * Mark order as received
                     */
                    wp_update_post( [ 'ID' => $order_id, 'post_status' => 'order_received' ] );
                    $redirect = $tc->get_confirmation_slug( true, $order_title );
                    \Tickera\TC_Order::add_order_note( $order_id, 'Payment Intent: ' . $stripe_result[ 'status' ] );
                    self::unset_sessions();

                }

            } else {
                \Tickera\TC_Order::add_order_note( $order_id, __( 'Invalid transaction: Nonce did not matched.', 'tickera-event-ticketing-system' ) );
            }

            wp_send_json( $redirect );
        }

        /**
         * Final Cart Check
         * Create a different order if cart content was changed
         */
        function stripe_final_cart_check() {

            global $tc;
            tickera_final_cart_check( $this->cart_contents() );
            $session_stripe_payment_gateway = $tc->session->get( 'stripe_payment_gateway' );
            if ( !is_null( $session_stripe_payment_gateway ) ) {

                $order_id = (int) $session_stripe_payment_gateway[ 'order_id' ];
                $tc_cart_contents = get_post_meta( $order_id, 'tc_cart_contents', true );

                if ( array_sum( $this->cart_contents() ) != array_sum( $tc_cart_contents ) )
                    self::unset_sessions();
            }
        }

        /**
         * Unset Sessions
         */
        public static function unset_sessions() {
            global $tc;
            $tc->session->drop( 'stripe_payment_gateway' );
        }

        /**
         * Set Sessions
         *
         * @param $session_names
         */
        function set_sessions( $session_names ) {

            global $tc;
            $session_stripe_payment_gateway = $tc->session->get( 'stripe_payment_gateway' );

            foreach ( $session_names as $session_name => $session_value ) {
                $session_stripe_payment_gateway[ $session_name ] = $session_value;
            }

            $tc->session->set( 'stripe_payment_gateway', $session_stripe_payment_gateway );
        }

        /**
         * Validate Stripe Event.
         * If the event is not found, it is an indication that the event is invalid.
         *
         * @param $event_id
         * @return Event
         *
         * @since 3.5.1.7
         */
        function tc_validate_stripe_event( $event_id ) {

            try {

                $event_obj = Event::retrieve( $event_id );
                $event_obj = tickera_sanitize_array( $event_obj, false, true );

                // Set as default
                $event_obj[ 'tc_validated' ] = false;

                // Note: payment_intent.payment_failed - No action needed
                $enabled_events = [ 'charge.captured', 'charge.refunded', 'payment_intent.succeeded' ];

                if ( $event_obj ) {

                    if ( in_array( $event_obj[ 'type' ], $enabled_events ) ) {

                        // Retrieve Payment Intent ID
                        if ( isset( $event_obj[ 'data' ][ 'object' ][ 'payment_intent' ] ) && $event_obj[ 'data' ][ 'object' ][ 'payment_intent' ] ) {
                            $payment_intent_id = $event_obj[ 'data' ][ 'object' ][ 'payment_intent' ];

                        } else {
                            $payment_intent_id = $event_obj[ 'data' ][ 'object' ][ 'id' ];
                        }

                        $payment_intent = PaymentIntent::retrieve( $payment_intent_id );
                        $payment_intent = tickera_sanitize_array( $payment_intent, false, true );

                        $event_obj[ 'payment_intent' ] = $payment_intent;
                        $meta_order_id = $payment_intent[ 'metadata' ][ 'order_id' ]; // It could be post ID or custom title

                        $order_id = (int) @tickera_get_order_id_by_name( $meta_order_id )->ID;
                        $order_id = ( $order_id && 'tc_orders' == get_post_type( $order_id ) ) ? $order_id : null;

                        if ( $order_id ) {
                            $event_obj[ 'payment_intent' ][ 'metadata' ][ 'order_id' ] = $order_id;
                            $event_obj[ 'tc_validated' ] = true;
                        }
                    }
                }

                return $event_obj;

            } Catch ( \Exception $e ) {
                return [ 'tc_validate' => false ];
            }
        }

        /**
         * Capture Charge via Stripe account and mark order as paid
         *
         * Replacement for Webhook Rest API
         *
         * @since 3.5.1.7
         */
        function ipn() {

            global $tc;
            self::on_creation();
            self::init_stripe();

            $payload = @file_get_contents('php://input');

            // Convert to array
            $payload = json_decode( $payload, true );

            // Sanitization
            $request_body = tickera_sanitize_array( $payload, false, true );

            $event_id = $request_body[ 'id' ];
            $event_obj = self::tc_validate_stripe_event( $event_id );

            if ( $event_obj[ 'tc_validated' ] ) {

                $order_statuses = [ 'charge.refunded' => 'order_refunded' ];

                $payment_intent = $event_obj[ 'payment_intent' ];
                $order_id = $payment_intent[ 'metadata' ][ 'order_id' ];

                /*
                 * Check if the cancelled/refund reason is fraud.
                 * Update order status to fraud
                 */
                if ( 'charge.refunded' == $event_obj[ 'type' ] ) {

                    $is_fraud = false;
                    $refund_obj = @$event_obj[ 'data' ][ 'object' ][ 'refunds' ][ 'data' ];

                    foreach ( $refund_obj as $key => $val ) {
                        if ( 'fraudulent' == $val[ 'reason' ] ) {
                            $is_fraud = true;
                            break;
                        }
                    }

                    if ( $is_fraud )
                        $order_statuses[ 'charge.refunded' ] = 'order_fraud';
                }

                if ( 'charge.captured' == $event_obj[ 'type' ] || 'payment_intent.succeeded' == $event_obj[ 'type' ] ) {
                    $tc->update_order_payment_status( $order_id, true );
                    $tc->session->drop( 'stripe_payment_gateway' );

                } else {
                    wp_update_post( [ 'ID' => $order_id, 'post_status' => sanitize_key( $order_statuses[ $event_obj[ 'type' ] ] ) ] );
                }

                \Tickera\TC_Order::add_order_note( $order_id, $event_obj[ 'type' ] );
            }
        }

        /**
         * Initialize Admin Settings for Stripe Elements
         *
         * @param $settings
         * @param $visible
         */
        public function gateway_admin_settings( $settings, $visible ) {
            global $tc; ?>
            <div id="<?php echo esc_attr( $this->plugin_name ); ?>" class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3>
                    <span><?php echo esc_html( sprintf( /* translators: %s: Stripe Payment Gateway admin name */ __( '%s Settings', 'tickera-event-ticketing-system' ), esc_html( $this->admin_name ) ) ); ?></span>
                    <span class="description"><?php esc_html_e( "Sell your tickets via Stripe and accept Visa, MasterCard, American Express, Discover, JCB, and Diners Club cards", 'tickera-event-ticketing-system' ) ?></span>
                </h3>
                <div class="inside">

                    <?php
                    $fields = array(
                        'is_ssl' => array(
                            'title' => __( 'Mode', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                '0' => __( 'Sandbox / Test', 'tickera-event-ticketing-system' ),
                                '1' => __( 'Live', 'tickera-event-ticketing-system' )
                            ),
                            'default' => '0',
                        ),

                        'publishable_key' => array(
                            'title' => __( 'Publishable API key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),

                        'private_key' => array(
                            'title' => __( 'Secret API key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                            'description' => __( 'You must log in to your Stripe merchant account to <a target="_blank" href="https://manage.stripe.com/#account/apikeys">get your API credentials</a>.', 'tickera-event-ticketing-system' ),
                        ),

                        'currency' => array(
                            'title' => __( 'Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->currencies,
                            'default' => 'AUD',
                        ),

                        'send_receipt' => array(
                            'title' => __( 'Send Receipt', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                '1' => __( 'Yes', 'tickera-event-ticketing-system' ),
                                '0' => __( 'No', 'tickera-event-ticketing-system' )
                            ),
                            'default' => 0,
                            'description' => __( 'Allow Stripe to automatically send a receipt to the customer after their payment has been made.', 'tickera-event-ticketing-system' )
                        ),

                        'manually_capture_payments' => array(
                            'title' => __( 'Capture payments manually', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                '1' => __( 'Yes', 'tickera-event-ticketing-system' ),
                                '0' => __( 'No', 'tickera-event-ticketing-system' )
                            ),
                            'default' => 0,
                            'description' => __( 'Manually capture payments in Stripe dashboard.', 'tickera-event-ticketing-system' )
                        ),

                        'enable_webhook' => array(
                            'title' => __( 'Enable webhook', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                '1' => __( 'Yes', 'tickera-event-ticketing-system' ),
                                '0' => __( 'No', 'tickera-event-ticketing-system' )
                            ),
                            'default' => 0,
                            'description' => __( 'Allow Stripe webhook to stay connected with your website. This will automatically update order statuses based on Stripe\'s end payment intent status', 'tickera-event-ticketing-system' )
                        ),

                        // If this option doesn't work on the actual site, it means that there's a problem within the site server configuration. It might be due to a PHP compatibility or the OS itself.
                        'curl_persistent_connection' => array(
                            'title' => __( 'Enable Curl Persistent Connection', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                '1' => __( 'Yes', 'tickera-event-ticketing-system' ),
                                '0' => __( 'No', 'tickera-event-ticketing-system' )
                            ),
                            'default' => 1,
                            'description' => __( 'Allow your site to persistently connect to Stripe to maximize checkout performance.', 'tickera-event-ticketing-system' )
                        )
                    );
                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name );
                    ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>
                </div>
            </div>
            <?php
        }

        /**
         * Validate Decimal Numbers
         * @param $val
         * @return bool
         */
        public function is_decimal( $val ) {
            return is_numeric( $val ) && floor( $val ) != $val;
        }

        /**
         * Calculate Totals
         * @param $total
         * @return float|int
         */
        public function maybe_fix_total( $total ) {
            if ( in_array( $this->currency, $this->zero_decimal_currencies ) && ! $this->is_decimal( $total ) ) {
                return $total;
            } else {
                return $total * 100;
            }
        }
    }

    \Tickera\tickera_register_gateway_plugin( 'Tickera\Gateway\TC_Gateway_Stripe_Elements_3DS', 'stripe-elements-3d-secure', __( 'Stripe Elements 3D Secure', 'tickera-event-ticketing-system' ) );
}