<?php
/**
 * Simplify Commerce - Payment Gateway
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'Tickera\Gateway\TC_Gateway_Simplify' ) ) {

    class TC_Gateway_Simplify extends TC_Gateway_API {

        var $plugin_name = 'simplify_commerce';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl;
        var $ipn_url;
        var $currency;
        var $currencies = array();
        var $permanently_active = false;
        var $skip_payment_screen = false;
        var $sandbox_public_key;
        var $sandbox_private_key;
        var $live_public_key;
        var $live_private_key;
        var $public_key;
        var $private_key;

        /**
         * Support for older payment gateway API
         */
        function on_creation() {
            $this->init();
        }

        function init() {
            global $tc;
            $settings = get_option( 'tickera_settings' );

            $this->admin_name = __( 'Simplify Commerce', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'Simplify Commerce', 'tickera-event-ticketing-system' );

            $this->method_img_url = apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/simplify.png', $this->plugin_name );
            $this->admin_img_url = apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-simplify.png', $this->plugin_name );

            $this->public_key = $this->get_option( 'public_key' );
            $this->private_key = $this->get_option( 'private_key' );

            $this->currency = $this->get_option( 'currency', 'USD' );

            add_action( 'wp_enqueue_scripts', array( &$this, 'enqueue_scripts' ) );

            $this->currencies = array(
                "USD" => __( 'USD - United States Dollar', 'tickera-event-ticketing-system' ),
            );
        }

        function enqueue_scripts() {
            global $tc;

            if ( $this->is_active() && $this->is_payment_page() ) {

                wp_enqueue_script(
                    'js-simplify-commerce', 'https://www.simplify.com/commerce/v1/simplify.js', array( 'jquery' )
                );

                wp_enqueue_script(
                    'simplify-commerce-token', $tc->plugin_url . '/includes/gateways/simplify/simplify_commerce_token.js', array( 'js-simplify-commerce', 'jquery' )
                );

                wp_localize_script(
                    'simplify-commerce-token', 'SimplifyGateway', array(
                        'public_key' => $this->get_option( 'public_key' ),
                    )
                );
            }
        }

        function payment_form( $cart ) {
            global $tc;
            $content = '';
            $content .= '<div id="simplify_commerce_checkout_errors"></div>';
            $content .= '<table class="tc_cart_billing"><thead><tr><th colspan="2">' . esc_html__( 'Enter Your Credit Card Information:', 'tickera-event-ticketing-system' ) . '</th></tr></thead><tbody><tr><td align="right">' . esc_html__( 'Cardholder Name:', 'tickera-event-ticketing-system' ) . '</td><td><input id="sc_cc_name" type="text" value="' . esc_attr( $this->buyer_info( 'full_name' ) ) . '" /></td></tr>';
            $content .= '<tr>';
            $content .= '<td align="right">';
            $content .= esc_html__( 'Card Number', 'tickera-event-ticketing-system' );
            $content .= '</td>';
            $content .= '<td>';
            $content .= '<input type="text" autocomplete="off" id="sc_cc_number"/>';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td align="right">';
            $content .= esc_html__( 'Expiration:', 'tickera-event-ticketing-system' );
            $content .= '</td>';
            $content .= '<td>';
            $content .= '<select id="sc_cc_month">';
            $content .= tickera_months_dropdown();
            $content .= '</select>';
            $content .= '<span> / </span>';
            $content .= '<select id="sc_cc_year">';
            $content .= tickera_years_dropdown( '', false );
            $content .= '</select>';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td align="right">';
            $content .= esc_html__( 'CVC:', 'tickera-event-ticketing-system' );
            $content .= '</td>';
            $content .= '<td>';
            $content .= '<input id="sc_cc_cvc" type="text" maxlength="4" autocomplete="off" value=""/>';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '</table>';
            $content .= '<span id="simplify_commerce_processing" style="display:none; float:right;"><img src="' . esc_url( $tc->plugin_url ) . 'images/loading.gif" /> ' . esc_html__( 'Processing...', 'tickera-event-ticketing-system' ) . '</span>';
            return $content;
        }

        /**
         * Generate Order Confirmation Page upon success checkout
         * @param $order
         * @param string $cart_info
         * @return string
         */
        function order_confirmation_message( $order, $cart_info = '' ) {

            global $tc;
            $order = tickera_get_order_id_by_name( $order );
            $order = new \Tickera\TC_Order( $order->ID );
            $content = '';

            switch ( $order->details->post_status ) {

                case 'order_received':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: %s: Simplify Payment order total amount */ __( 'Your payment via Simplify Commerce for this order totaling <strong>%s</strong> is not yet complete.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    $content .= '<p>' . wp_kses_post( __( 'Current order status: <strong>Pending Payment</strong>', 'tickera-event-ticketing-system' ) ) . '</p>';
                    break;

                case 'order_fraud':
                    $content .= '<p>' . esc_html__( 'Your payment is under review. We will back to you soon.', 'tickera-event-ticketing-system' ) . '</p>';
                    break;

                case 'order_paid':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: %s: Simplify Payment order total amount */ __( 'Your payment via Simplify Commerce for this order totaling <strong>%s</strong> is complete.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    break;

                case 'order_cancelled':
                    $content .= '<p>' . wp_kses_post(sprintf( /* translators: %s: Simplify Payment order total amount */ __( 'Your payment via Simplify for this order totaling <strong>%s</strong> is cancelled.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    break;

                case 'order_refunded':
                    $content .= '<p>' . wp_kses_post(sprintf( /* translators: %s: Simplify Payment order total amount */ __( 'Your payment via Simplify for this order totaling <strong>%s</strong> is refunded.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    break;

            }

            $content = wp_kses_post( apply_filters( 'tc_order_confirmation_message_content', $content, $order, $this->plugin_name ) );
            $tc->remove_order_session_data();
            $tc->session->drop( 'simplifyToken' );
            $tc->maybe_skip_confirmation_screen( $this, $order );
            return $content;
        }

        function process_payment( $cart ) {

            global $tc;
            tickera_final_cart_check( $cart );
            $this->save_cart_info();

            if ( isset( $_POST[ 'simplify_payment_form' ] ) && $_POST[ 'simplify_payment_form' ] == 'not_available' ) {
                $tc->session->set( 'tc_gateway_error', __( 'The Simplify Commerce is not available at the moment. Please try another method or contact the admnistrator', 'tickera-event-ticketing-system' ) );
                tickera_redirect( $tc->get_payment_slug( true ), true );
            }

            if ( ! isset( $_POST[ 'simplifyToken' ] ) ) {
                $tc->session->set( 'tc_gateway_error', __( 'The Simplify Commerce Token was not generated correctly. Please go back and try again.', 'tickera-event-ticketing-system' ) );
                tickera_redirect( $tc->get_payment_slug( true ), true );
            }

            $tc->session->set( 'simplifyToken', sanitize_text_field( $_POST[ 'simplifyToken' ] ) );
            require_once( $tc->plugin_dir . "/includes/gateways/simplify/Simplify.php" );

            // Generate a tickera order id
            $order_id = $tc->generate_order_id();

            Simplify::$publicKey = $this->public_key;
            Simplify::$privateKey = $this->private_key;

            try {

                $session_simplify_token = $tc->session->get( 'simplifyToken' );
                $payment = Simplify_Payment::createPayment(
                    array(
                        'amount' => $this->total() * 100,
                        'token' => sanitize_text_field( $session_simplify_token ),
                        'description' => $this->cart_items(),
                        'reference' => $order_id,
                        'currency' => $this->currency
                    )
                );

                if ( $payment->paymentStatus == 'APPROVED' ) {

                    $payment_info = array();
                    $payment_info[ 'transaction_id' ] = $payment->id;
                    $payment_info = $this->save_payment_info();
                    $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, true );

                    tickera_redirect( $tc->get_confirmation_slug( true, $order_id ), false, false );
                    tickera_js_redirect( $tc->get_confirmation_slug( true ), false );
                    exit;

                } else if ( $payment->paymentStatus == 'DECLINED' ) {
                    // Run if the card is declined etc.
                    $tc->session->set( 'tc_gateway_error', apply_filters( 'tc_simplify_declined_card', __( 'We\'re very sorry but the card you entered was declined ', 'tickera-event-ticketing-system' ) ) );
                    tickera_redirect( $tc->get_payment_slug( true ), true );
                }
            } catch ( Simplify_ApiException $e ) {
                $tc->session->drop( 'simplifyToken' );
                $tc->session->set( 'tc_gateway_error', sprintf( /* translators: %s: Simplify Payment card processing error */ __( 'There was an error processing your card - "%s".', 'tickera-event-ticketing-system' ), $e->getMessage() ) );
                tickera_redirect( $tc->get_payment_slug( true ), true );
            }

            return false;
        }

        function gateway_admin_settings( $settings, $visible ) {
            global $tc;
            ?>
            <div id="<?php echo esc_attr( $this->plugin_name ); ?>" class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3>
                    <span><?php echo esc_html( sprintf( /* translators: %s: Simplify Payment Gateway admin name */ __( '%s Settings', 'tickera-event-ticketing-system' ), esc_html( $this->admin_name ) ) ); ?></span>
                    <span class="description"><?php esc_html_e( "Sell your tickets using Simplify Commerce", 'tickera-event-ticketing-system' ) ?></span>
                </h3>
                <div class="inside">

                    <?php
                    $fields = array(
                        'public_key' => array(
                            'title' => __( 'Public API key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'private_key' => array(
                            'title' => __( 'Private API key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'currency' => array(
                            'title' => __( 'Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->currencies,
                            'default' => 'USD',
                        ),
                    );
                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name );
                    ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>
                </div>
            </div>
            <?php
        }
    }

    \Tickera\tickera_register_gateway_plugin( 'Tickera\Gateway\TC_Gateway_Simplify', 'simplify_commerce', __( 'Simplify Commerce', 'tickera-event-ticketing-system' ) );
}