<?php
/**
 * PayPal PRO - Payment Gateway
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'Tickera\Gateway\TC_Gateway_PayPal_Pro' ) ) {

    class TC_Gateway_PayPal_Pro extends TC_Gateway_API {

        var $plugin_name = 'paypal_pro';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl;
        var $ipn_url;
        var $currency;
        var $currencies = array();
        var $api_version = '85.0';
        var $api_endpoint = '';
        var $sandbox = false;
        var $api_username = '';
        var $api_password = '';
        var $api_signature = '';
        var $permanently_active = false;
        var $skip_payment_screen = false;

        /**
         * Support for older payment gateway API
         */
        function on_creation() {
            $this->init();
        }

        function init() {
            global $tc;

            $this->admin_name = __( 'PayPal PRO', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'Credit Card', 'tickera-event-ticketing-system' );

            $this->method_img_url = sanitize_text_field( apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/paypal-pro.png', $this->plugin_name ) );
            $this->admin_img_url = sanitize_text_field( apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-paypal-pro.png', $this->plugin_name ) );

            $this->sandbox = $this->get_option( 'is_ssl', '0' );
            $this->api_endpoint = $this->sandbox == '0' ? 'https://api-3t.sandbox.paypal.com/nvp' : 'https://api-3t.paypal.com/nvp';
            $this->api_username = $this->get_option( 'api_username' );
            $this->api_password = $this->get_option( 'api_password' );
            $this->api_signature = $this->get_option( 'api_signature' );

            $this->force_ssl = $this->get_option( 'is_ssl', '0' ) == '1' ? true : false;
            $this->currency = $this->get_option( 'currency', 'USD' );

            $currencies = array(
                "AUD" => __( 'AUD - Australian Dollar', 'tickera-event-ticketing-system' ),
                "CAD" => __( 'CAD - Canadian Dollar', 'tickera-event-ticketing-system' ),
                "CZK" => __( 'CZK - Czech Koruna', 'tickera-event-ticketing-system' ),
                "DKK" => __( 'DKK - Danish Krone', 'tickera-event-ticketing-system' ),
                "EUR" => __( 'EUR - Euro', 'tickera-event-ticketing-system' ),
                "HKD" => __( 'HKD - Hong Kong Dollar', 'tickera-event-ticketing-system' ),
                "HUF" => __( 'HUF - Hungarian Forint', 'tickera-event-ticketing-system' ),
                "JPY" => __( 'JPY - Japanese Yen', 'tickera-event-ticketing-system' ),
                "NOK" => __( 'NOK - Norwegian Krone', 'tickera-event-ticketing-system' ),
                "NZD" => __( 'NZD - New Zealand Dollar', 'tickera-event-ticketing-system' ),
                "PLN" => __( 'PLN - Polish Zloty', 'tickera-event-ticketing-system' ),
                "GBP" => __( 'GBP - British Pound', 'tickera-event-ticketing-system' ),
                "SGD" => __( 'SGD - Singapore Dollar', 'tickera-event-ticketing-system' ),
                "SEK" => __( 'SEK - Swedish Krona', 'tickera-event-ticketing-system' ),
                "CHF" => __( 'CHF - Swiss Franc', 'tickera-event-ticketing-system' ),
                "USD" => __( 'USD - U.S. Dollar', 'tickera-event-ticketing-system' ),
            );

            $this->currencies = $currencies;

            add_action( 'wp_enqueue_scripts', array( &$this, 'enqueue_scripts' ) );
        }

        function enqueue_scripts() {
            global $tc, $wp;

            if ( $this->is_active() && $this->is_payment_page() ) {
                wp_enqueue_script( 'tc-paypal-pro', $tc->plugin_url . '/includes/gateways/paypal-pro/paypal-pro.js', array( 'jquery' ) );
            }
        }

        function payment_form( $cart ) {

            global $tc;
            $content = '';
            $content .= '<div id="paypal_checkout_errors"></div>';
            $content .= '<table class="tc_cart_billing" cellpadding="10">
        <thead><tr>
          <th colspan="2">' . esc_html__( 'Enter Your Credit Card Information:', 'tickera-event-ticketing-system' ) . '</th>
        </tr></thead>
        <tbody>
          <tr>
          <td>' . esc_html__( 'Cardholder First Name:', 'tickera-event-ticketing-system' ) . '</td>
          <td><input class="card-holdername tickera-input-field" name="FIRSTNAME" type="text" value="' . esc_attr( $this->buyer_info( 'first_name' ) ) . '" /> </td>
          </tr>';

            $content .= '<tr>
          <td>' . esc_html__( 'Cardholder Last Name:', 'tickera-event-ticketing-system' ) . '</td>
          <td><input class="card-holdername tickera-input-field" name="LASTNAME" type="text" value="' . esc_attr( $this->buyer_info( 'last_name' ) ) . '" /> </td>
          </tr>';

            $content .= '<tr>
          <td>' . esc_html__( 'Street:', 'tickera-event-ticketing-system' ) . '</td>
          <td><input class="card-street tickera-input-field" name="STREET" type="text" value="" /> </td>
          </tr>';

            $content .= '<tr>
          <td>' . esc_html__( 'City:', 'tickera-event-ticketing-system' ) . '</td>
          <td><input class="card-city tickera-input-field" name="CITY" type="text" value="' . esc_attr( tickera_get_client_city() ) . '" /> </td>
          </tr>';

            $content .= '<tr>
          <td>' . esc_html__( 'State or province:', 'tickera-event-ticketing-system' ) . '</td>
          <td><input class="card-state tickera-input-field" name="STATE" type="text" value="" /> </td>
          </tr>';

            $content .= '<tr>
          <td>' . esc_html__( 'Country:', 'tickera-event-ticketing-system' ) . '</td>
          <td>' . tickera_countries( '', 'COUNTRYCODE' ) . '</td>
          </tr>';

            $content .= '<tr>
          <td>' . esc_html__( 'ZIP Code:', 'tickera-event-ticketing-system' ) . '</td>
          <td><input class="card-state tickera-input-field" name="ZIP" type="text" value="' . esc_attr( tickera_get_client_zip() ) . '" /> </td>
          </tr>';

            $content .= '<tr>';
            $content .= '<td>';
            $content .= esc_html__( 'Card Number', 'tickera-event-ticketing-system' );
            $content .= '</td>';
            $content .= '<td>';
            $content .= '<input type="text" name="ACCT" autocomplete="off" class="card-number tickera-input-field"/>';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td>';
            $content .= esc_html__( 'Expiration:', 'tickera-event-ticketing-system' );
            $content .= '</td>';
            $content .= '<td>';
            $content .= '<select class="card-expiry-month" name="CARD_MONTH">';
            $content .= tickera_months_dropdown();
            $content .= '</select>';
            $content .= '<span> / </span>';
            $content .= '<select class="card-expiry-year" name="CARD_YEAR">';
            $content .= tickera_years_dropdown( '', true );
            $content .= '</select>';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td>';
            $content .= esc_html__( 'CVC:', 'tickera-event-ticketing-system' );
            $content .= '</td>';
            $content .= '<td>';
            $content .= '<input type="text" size="4" autocomplete="off" name="CCV2" class="card-cvc tickera-input-field" />';
            $content .= '<input type="hidden" name="CURRENCYCODE" value="' . esc_attr( $this->currency ) . '" />';
            $content .= '<input type="hidden" class="AMT" value="' . (float) $this->total() . '" />';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '</table>';
            $content .= '<span id="paypal_processing" style="display: none;float: right;"><img src="' . esc_url( $tc->plugin_url ) . 'images/loading.gif" /> ' . esc_html__( 'Processing...', 'tickera-event-ticketing-system' ) . '</span>';
            return $content;
        }

        function process_payment( $cart ) {

            global $tc;
            tickera_final_cart_check( $cart );
            $this->save_cart_info();
            $order_id = $tc->generate_order_id();

            $request_params = array
            (
                'METHOD' => 'DoDirectPayment',
                'USER' => $this->api_username,
                'PWD' => $this->api_password,
                'SIGNATURE' => $this->api_signature,
                'VERSION' => $this->api_version,
                'PAYMENTACTION' => 'Sale',
                'IPADDRESS' => sanitize_text_field( $_SERVER[ 'REMOTE_ADDR' ] ),
                'ACCT' => sanitize_text_field( $_POST[ 'ACCT' ] ),
                'EXPDATE' => sanitize_text_field( $_POST[ 'CARD_MONTH' ] ) . sanitize_text_field( $_POST[ 'CARD_YEAR' ] ),
                'CVV2' => sanitize_text_field( $_POST[ 'CCV2' ] ),
                'FIRSTNAME' => sanitize_text_field( $_POST[ 'FIRSTNAME' ] ),
                'LASTNAME' => sanitize_text_field( $_POST[ 'LASTNAME' ] ),
                'STREET' => sanitize_text_field( $_POST[ 'STREET' ] ),
                'CITY' => sanitize_text_field( $_POST[ 'CITY' ] ),
                'STATE' => sanitize_text_field( $_POST[ 'STATE' ] ),
                'COUNTRYCODE' => sanitize_text_field( $_POST[ 'COUNTRYCODE' ] ),
                'ZIP' => sanitize_text_field( $_POST[ 'ZIP' ] ),
                'AMT' => (float) $this->total(),
                'CURRENCYCODE' => sanitize_text_field( $_POST[ 'CURRENCYCODE' ] ),
                'DESC' => $this->cart_items(),
                'BUTTONSOURCE' => 'Tickera_SP'
            );

            $nvp_string = '';

            foreach ( $request_params as $var => $val ) {
                $nvp_string .= '&' . $var . '=' . urlencode( $val );
            }

            $response = wp_remote_post( $this->api_endpoint, array(
                'timeout' => 120,
                'httpversion' => '1.1',
                'body' => $request_params,
                'user-agent' => $tc->title,
                'sslverify' => false,
            ) );

            if ( is_wp_error( $response ) ) {
                $error_message = sanitize_text_field( $response->get_error_message() );
                $tc->session->set( 'tc_gateway_error', __( "Something went wrong:", 'tickera-event-ticketing-system' ) . sanitize_text_field( $error_message ) );
                tickera_redirect( $tc->get_payment_slug( true ), true );

            } else {
                $nvp_response = $this->NVPToArray( $response[ 'body' ] );

                if ( $nvp_response[ 'ACK' ] == 'Success' || $nvp_response[ 'ACK' ] == 'SuccessWithWarning' ) {
                    //setup our payment details

                    $payment_info = array();
                    $payment_info[ 'method' ] = __( 'Credit Card', 'tickera-event-ticketing-system' );
                    $payment_info[ 'transaction_id' ] = sanitize_text_field( $nvp_response[ 'TRANSACTIONID' ] );
                    $payment_info = $this->save_payment_info();

                    $paid = true;
                    $order = $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, $paid );
                    tickera_redirect( $tc->get_confirmation_slug( true, $order_id ), true );

                } else {
                    $tc->session->set( 'tc_gateway_error', $nvp_response[ 'L_LONGMESSAGE0' ] );
                    tickera_redirect( $tc->get_payment_slug( true ), true );
                }
            }
        }

        function gateway_admin_settings( $settings, $visible ) {
            global $tc;
            ?>
            <div id="<?php echo esc_attr( $this->plugin_name ); ?>" class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3><span><?php
                        echo esc_html( sprintf(
                        /* translators: %s: PayPal PRO */
                            __( '%s Settings', 'tickera-event-ticketing-system' ),
                            esc_html( $this->admin_name )
                        ) );
                        ?></span>
                    <span class="description"><?php esc_html_e( "Accept credit cards on your website via PayPal PRO. Currently, this payment method is available in US, UK and Canada.", 'tickera-event-ticketing-system' ); ?></span>
                </h3>
                <div class="inside">

                    <?php
                    $fields = array(
                        'is_ssl' => array(
                            'title' => __( 'Mode', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                '0' => __( 'Sandbox / Test', 'tickera-event-ticketing-system' ),
                                '1' => __( 'Live', 'tickera-event-ticketing-system' )
                            ),
                            'default' => '0',
                        ),
                        'api_username' => array(
                            'title' => __( 'API username', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'api_password' => array(
                            'title' => __( 'API password', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'api_signature' => array(
                            'title' => __( 'API signature', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                            'description' => '',
                        ),
                        'currency' => array(
                            'title' => __( 'Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->currencies,
                            'default' => 'USD',
                        ),
                    );
                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name );
                    ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>
                </div>
            </div>
            <?php
        }

        function NVPToArray( $NVPString ) {
            $proArray = array();
            while ( strlen( $NVPString ) ) {
                // name
                $keypos = strpos( $NVPString, '=' );
                $keyval = substr( $NVPString, 0, $keypos );
                // value
                $valuepos = strpos( $NVPString, '&' ) ? strpos( $NVPString, '&' ) : strlen( $NVPString );
                $valval = substr( $NVPString, $keypos + 1, $valuepos - $keypos - 1 );
                // decoding the respose
                $proArray[ $keyval ] = urldecode( $valval );
                $NVPString = substr( $NVPString, $valuepos + 1, strlen( $NVPString ) );
            }
            return $proArray;
        }
    }

    \Tickera\tickera_register_gateway_plugin( 'Tickera\Gateway\TC_Gateway_PayPal_Pro', 'paypal_pro', __( 'PayPal PRO', 'tickera-event-ticketing-system' ) );
}