<?php
/**
 * Paygate - Payment Gateway
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'Tickera\Gateway\TC_Gateway_Paygate' ) ) {

    class TC_Gateway_Paygate extends TC_Gateway_API {

        var $plugin_name = 'paygate';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl = false;
        var $ipn_url;
        var $live_url;
        var $merchant_id = '';
        var $notify_url = '';
        var $fail_url = '';
        var $permanently_active = false;
        var $skip_payment_screen = true;
        var $encryption_key = 'secret';

        /**
         * @var int
         * @since 3.5.2.0
         */
        var $payweb_version = 3;
        var $country_currency = [];
        var $init_errors = [];

        /**
         * Support for older payment gateway API
         */
        function on_creation() {
            $this->init();
        }

        function init() {

            global $tc;

            $this->admin_name = __( 'Paygate', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'Paygate', 'tickera-event-ticketing-system' );
            $this->notify_url = $this->ipn_url;

            $this->method_img_url = apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/paygate.png', $this->plugin_name );
            $this->admin_img_url = apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-paygate.png', $this->plugin_name );

            $this->merchant_id = $this->get_option( 'merchant_id', '10011072130' );
            $this->encryption_key = $this->get_option( 'encryption_key', 'secret' );
            $this->payweb_version = $this->get_option( 'payweb_version', 3 );

            $this->country_currency = [ 'AFG' => 'AFA', 'ALB' => 'ALL', 'DZA' => 'DZD', 'ASM' => 'USD', 'AND' => 'EUR', 'AGO' => 'AOA', 'AIA' => 'XCD', 'ATA' => 'NOK', 'ATG' => 'XCD', 'ARG' => 'ARS', 'ARM' => 'AMD', 'ABW' => 'AWG', 'AUS' => 'AUD', 'AUT' => 'EUR', 'AZE' => 'AZM', 'BHS' => 'BSD', 'BHR' => 'BHD', 'BGD' => 'BDT', 'BRB' => 'BBD', 'BLR' => 'BYR', 'BEL' => 'EUR', 'BLZ' => 'BZD', 'BEN' => 'XOF', 'BMU' => 'BMD', 'BTN' => 'INR', 'BOL' => 'BOB', 'BIH' => 'BAM', 'BWA' => 'BWP', 'BVT' => 'NOK', 'BRA' => 'BRL', 'IOT' => 'USD', 'VGB' => 'USD', 'BRN' => 'BND', 'BGR' => 'BGN', 'BFA' => 'XOF', 'BDI' => 'BIF', 'KHM' => 'KHR', 'CMR' => 'XAF', 'CAN' => 'CAD', 'CPV' => 'CVE', 'CYM' => 'KYD', 'CAF' => 'XAF', 'TCD' => 'XAF', 'CHL' => 'CLP', 'CHN' => 'CNY', 'CXR' => 'AUD', 'CCK' => 'AUD', 'COL' => 'COP', 'COM' => 'KMF', 'COG' => 'XAF', 'COK' => 'NZD', 'CRI' => 'CRC', 'CIV' => 'XOF', 'HRV' => 'HRK', 'CUB' => 'CUP', 'CYP' => 'CYP', 'CZE' => 'CZK', 'COD' => 'CDF', 'DNK' => 'DKK', 'DJI' => 'DJF', 'DMA' => 'XCD', 'DOM' => 'DOP', 'TMP' => 'TPE', 'ECU' => 'ECS', 'EGY' => 'EGP', 'SLV' => 'USD', 'GNQ' => 'XAF', 'ERI' => 'ERN', 'EST' => 'EEK', 'ETH' => 'ETB', 'FRO' => 'DKK', 'FLK' => 'FKP', 'FJI' => 'FJD', 'FIN' => 'EUR', 'FRA' => 'EUR', 'FXX' => 'EUR', 'GUF' => 'EUR', 'PYF' => 'XPF', 'ATF' => 'EUR', 'GAB' => 'XAF', 'GMB' => 'GMD', 'GEO' => 'GEL', 'DEU' => 'DEM', 'GHA' => 'GHC', 'GIB' => 'GIP', 'GRC' => 'EUR', 'GRL' => 'DKK', 'GRD' => 'XCD', 'GLP' => 'EUR', 'GUM' => 'USD', 'GTM' => 'GTQ', 'GIN' => 'GNF', 'GNB' => 'GWP', 'GUY' => 'GYD', 'HTI' => 'HTG', 'HMD' => 'AUD', 'VAT' => 'EUR', 'HND' => 'HNL', 'HKG' => 'HKD', 'HUN' => 'HUF', 'ISL' => 'ISK', 'IND' => 'INR', 'IDN' => 'IDR', 'IRN' => 'IRR', 'IRQ' => 'IQD', 'IRL' => 'EUR', 'ISR' => 'ILS', 'ITA' => 'EUR', 'JAM' => 'JMD', 'JPN' => 'JPY', 'JOR' => 'JOD', 'KAZ' => 'KZT', 'KEN' => 'KES', 'KIR' => 'AUD', 'PRK' => 'KPW', 'KOR' => 'KRW', 'KWT' => 'KWD', 'KGZ' => 'KGS', 'LAO' => 'LAK', 'LVA' => 'LVL', 'LBN' => 'LBP', 'LSO' => 'ZAR', 'LBR' => 'LRD', 'LBY' => 'LYD', 'LIE' => 'CHF', 'LTU' => 'LTL', 'LUX' => 'EUR', 'MAC' => 'MOP', 'MKD' => 'MKD', 'MDG' => 'MGF', 'MWI' => 'MWK', 'MYS' => 'MYR', 'MDV' => 'MVR', 'MLI' => 'XOF', 'MLT' => 'MTL', 'MHL' => 'USD', 'MTQ' => 'EUR', 'MRT' => 'MRO', 'MUS' => 'MUR', 'MYT' => 'EUR', 'MEX' => 'MXN', 'FSM' => 'USD', 'MDA' => 'MDL', 'MCO' => 'EUR', 'MNG' => 'MNT', 'MSR' => 'XCD', 'MAR' => 'MAD', 'MOZ' => 'MZM', 'MMR' => 'MMK', 'NAM' => 'NAD', 'NRU' => 'AUD', 'NPL' => 'NPR', 'NLD' => 'EUR', 'ANT' => 'ANG', 'NCL' => 'XPF', 'NZL' => 'NZD', 'NIC' => 'NIO', 'NER' => 'XOF', 'NGA' => 'NGN', 'NIU' => 'NZD', 'NFK' => 'AUD', 'MNP' => 'USD', 'NOR' => 'NOK', 'OMN' => 'OMR', 'PAK' => 'PKR', 'PLW' => 'USD', 'PAN' => 'PAB', 'PNG' => 'PGK', 'PRY' => 'PYG', 'PER' => 'PEN', 'PHL' => 'PHP', 'PCN' => 'NZD', 'POL' => 'PLN', 'PRT' => 'EUR', 'PRI' => 'USD', 'QAT' => 'QAR', 'REU' => 'EUR', 'ROM' => 'ROL', 'RUS' => 'RUB', 'RWA' => 'RWF', 'WSM' => 'WST', 'SMR' => 'EUR', 'STP' => 'STD', 'SAU' => 'SAR', 'SEN' => 'XOF', 'SYC' => 'SCR', 'SLE' => 'SLL', 'SGP' => 'SGD', 'SVK' => 'SKK', 'SVN' => 'SIT', 'SGS' => 'GBP', 'SLB' => 'SBD', 'SOM' => 'SOS', 'ZAF' => 'ZAR', 'ESP' => 'EUR', 'LKA' => 'LKR', 'SHN' => 'SHP', 'KNA' => 'XCD', 'LCA' => 'XCD', 'SPM' => 'EUR', 'VCT' => 'XCD', 'SDN' => 'SDD', 'SUR' => 'SRG', 'SJM' => 'NOK', 'SWZ' => 'SZL', 'SWE' => 'SEK', 'CHE' => 'CHF', 'SYR' => 'SYP', 'TWN' => 'TWD', 'TJK' => 'TJS', 'TZA' => 'TZS', 'THA' => 'THB', 'TGO' => 'XOF', 'TKL' => 'NZD', 'TON' => 'TOP', 'TTO' => 'TTD', 'TUN' => 'TND', 'TUR' => 'TRL', 'TKM' => 'TMM', 'TCA' => 'USD', 'TUV' => 'AUD', 'UMI' => 'USD', 'VIR' => 'USD', 'UGA' => 'UGX', 'UKR' => 'UAH', 'ARE' => 'AED', 'GBR' => 'GBP', 'USA' => 'USD', 'URY' => 'UYU', 'UZB' => 'UZS', 'VUT' => 'VUV', 'VEN' => 'VEB', 'VNM' => 'VND', 'WLF' => 'XPF', 'ESH' => 'MAD', 'YEM' => 'YER', 'YUG' => 'YUM', 'ZMB' => 'ZMK', 'ZWE' => 'ZWD' ];
            $this->init_errors = [
                'CNTRY_INVALID' => 'Invalid Country',
                'DATA_AMT_NUM' => 'Amount is not a number',
                'DATA_AMT_ZERO' => 'Amount value is zero',
                'DATA_CHK' => 'Checksum calculated incorrectly',
                'DATA_CREF' => 'No transaction reference',
                'DATA_DTTM' => 'Transaction date invalid',
                'DATA_INS' => 'Error creating record for transaction request',
                'DATA_PAY_REQ_ID' => 'Pay request ID missing or invalid',
                'DATA_PM' => 'Pay Method or Pay Method Detail fields invalid',
                'DATA_PW' => 'Not all required fields have been posted to PayWeb',
                'DATA_REGION' => 'No Country or Locale',
                'DATA_URL' => 'No return url',
                'INVALID_VAULT' => 'Vault value invalid',
                'INVALID_VAULT_ID' => 'Vault ID invalid',
                'INV_EMAIL' => 'Invalid Email address',
                'LOCALE_INVALID' => 'Invalid Locale',
                'ND_INV_PGID' => 'Invalid PayGate ID',
                'NOT_LIVE_PM' => 'No available payment methods',
                'NO_TRANS_DATA' => 'No transaction data found',
                'PAYVAULT_NOT_EN' => 'PayVault not enabled',
                'PGID_NOT_EN' => 'PayGate ID not enabled, there are no available payment methods or there are no available currencies',
                'TXN_CAN' => 'Transaction has already been cancelled',
                'TXN_CMP' => 'Transaction has already been completed',
                'TXN_PRC' => 'Transaction is older than 30 minutes or there has been an error processing it',
                'VAULT_NOT_ACCEPTED' => 'Card types enabled on terminal not available for vaulting',
            ];
        }

        function process_payment( $cart ) {

            switch ( $this->payweb_version ) {
                case 2:
                    self::process_payweb_two( $cart );
                    break;

                case 3:
                    self::process_payweb_three( $cart );
                    break;
            }
        }

        /**
         * Process PayWeb2 Workflow
         *
         * @param $cart
         * @since 3.5.2.0
         */
        function process_payweb_two( $cart ) {

            global $tc;
            tickera_final_cart_check( $cart );
            $this->save_cart_info();
            $order_id = $tc->generate_order_id();

            $live_url = 'https://www.paygate.co.za/paywebv2/process.trans';
            $currency = $this->get_option( 'currency', 'ZAR' );

            $paid = false;
            $payment_info = $this->save_payment_info();

            $fields = array(
                'PAYGATE_ID' => $this->merchant_id,
                'REFERENCE' => $order_id,
                'AMOUNT' => $this->total() * 100,
                'CURRENCY' => $currency,
                'RETURN_URL' => esc_url( $tc->get_confirmation_slug( true, $order_id ) ),
                'TRANSACTION_DATE' => date( 'Y-m-d H:m:s' ),
                'EMAIL' => $this->buyer_info( 'email' ),
            );

            $checksum_source = $fields[ 'PAYGATE_ID' ] . "|" . $fields[ 'REFERENCE' ] . "|" . $fields[ 'AMOUNT' ] . "|" . $fields[ 'CURRENCY' ] . "|" . $fields[ 'RETURN_URL' ] . "|" . $fields[ 'TRANSACTION_DATE' ] . "|" . $fields[ 'EMAIL' ] . "|" . $this->encryption_key;
            $CHECKSUM = md5( $checksum_source );
            $fields[ 'CHECKSUM' ] = $CHECKSUM;

            $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, $paid );
            @header( 'Content-Type: text/html' );
            ?>
            <form action="<?php echo esc_attr( $live_url ); ?>" method="post" name="paygate">
                <?php foreach ( $fields as $field_key => $field_val ) { ?>
                    <input name="<?php echo esc_attr( $field_key ); ?>" type="hidden" value="<?php echo esc_attr( $field_val ); ?>"/>
                <?php } ?>
            </form>
            <script>document.forms[ 'paygate' ].submit();</script>
            <?php
            die;
        }

        /**
         * Process PayWeb3 Workflow
         *
         * @param $cart
         * @since 3.5.2.0
         */
        function process_payweb_three( $cart ) {

            global $tc;
            tickera_final_cart_check( $cart );
            $this->save_cart_info();
            $order_id = $tc->generate_order_id();

            $init_url = 'https://secure.paygate.co.za/payweb3/initiate.trans';
            $process_url = 'https://secure.paygate.co.za/payweb3/process.trans';

            $country = $this->get_option( 'country_payweb_three', 'ZAF' );
            $locale = $this->get_option( 'locale_payweb_three', 'en-za' );

            $paid = false;
            $payment_info = $this->save_payment_info();

            $fields = array(
                'PAYGATE_ID' => $this->merchant_id,
                'REFERENCE' => $order_id,
                'AMOUNT' => $this->total() * 100,
                'CURRENCY' => isset( $this->country_currency[ $country ] ) ? $this->country_currency[ $country ] : 'ZAF',
                'RETURN_URL' => esc_url( $tc->get_confirmation_slug( true, $order_id ) ),
                'TRANSACTION_DATE' => date( 'Y-m-d H:m:s' ),
                'LOCALE' => $locale,
                'COUNTRY' => $country,
                'EMAIL' => $this->buyer_info( 'email' ),
                'NOTIFY_URL' => $this->ipn_url,
            );

            $fields[ 'CHECKSUM' ] = md5( implode( '', $fields ) . $this->encryption_key );
            $response = self::api( $init_url, $fields );

            if ( isset( $response[ 'ERROR' ] ) ) {
                $error = ( isset( $this->init_errors[ $response[ 'ERROR' ] ] ) ) ? sprintf( /* translators: %s: Paygate response error */ __( 'Something went wrong, %s. Please try again.'), $this->init_errors[ sanitize_text_field( $response[ 'ERROR' ] ) ] ) : __( 'Something went wrong. Please try again.', 'tickera-event-ticketing-system' );
                $tc->session->set( 'tc_gateway_error', $error );
                tickera_redirect( $tc->get_payment_slug( true ), true );

            } else {
                $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, $paid );
                $order = tickera_get_order_id_by_name( $order_id );
                add_post_meta( (int) $order->ID, 'payweb_request_id', sanitize_key( $response[ 'PAY_REQUEST_ID' ] ) );

                @header( 'Content-Type: text/html' ); ?>
                <form action="<?php echo esc_attr( $process_url ); ?>" method="post" name="paygate">
                    <input type="hidden" name="PAY_REQUEST_ID" value="<?php echo esc_attr( $response[ 'PAY_REQUEST_ID' ] ); ?>">
                    <input type="hidden" name="CHECKSUM" value="<?php echo esc_attr( $response[ 'CHECKSUM' ] ); ?>">
                </form>
                <script>document.forms[ 'paygate' ].submit();</script>
                <?php
                die;
            }
        }

        function update_order() {

            global $tc;
            switch( $this->payweb_version ) {

                case 2:
                    if ( isset( $_POST[ 'REFERENCE' ] ) ) {
                        $key = sanitize_text_field( $_POST[ 'REFERENCE' ] ); // Order id
                        $order = tickera_get_order_id_by_name( $key );
                        if ( '1' == $_POST[ 'TRANSACTION_STATUS' ] ) {
                            $tc->update_order_payment_status( $order->ID, true );
                        }
                    }
                    break;

                case 3:

                    if ( $_POST && isset( $_POST[ 'PAY_REQUEST_ID' ] ) ) {

                        $order_id = get_posts( [
                            'post_type' => 'tc_orders',
                            'post_status' => 'any',
                            'meta_key' => 'payweb_request_id',
                            'meta_value' =>  sanitize_key( $_POST[ 'PAY_REQUEST_ID' ] ),
                            'fields' => 'ids',
                            'posts_per_page' => 1
                        ]);

                        $order_id = reset( $order_id );

                        switch ( (int) $_POST[ 'TRANSACTION_STATUS' ] ) {

                            case '1':
                                $tc->update_order_payment_status( $order_id, true );
                                break;

                            case '2':
                            case '3':
                            case '4':
                                $tc->update_order_status( $order_id, 'order_cancelled' );
                                break;
                        }
                    }
                    break;
            }
        }

        function order_confirmation( $order, $payment_info = '', $cart_info = '' ) {
            self::update_order();
        }

        function gateway_admin_settings( $settings, $visible ) { ?>
            <div id="<?php echo esc_attr( $this->plugin_name ); ?>"
                 class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3>
                    <span><?php echo esc_html( sprintf( /* translators: %s: PayGate Payment Gateway admin name */ __( '%s Settings', 'tickera-event-ticketing-system' ), esc_html( $this->admin_name ) ) ); ?></span>
                    <span class="description"><?php echo wp_kses_post( __( 'Sell your tickets via <a target="_blank" href="https://www.paygate.co.za/">PayGate</a>', 'tickera-event-ticketing-system' ) ); ?></span>
                </h3>
                <div class="inside"><?php
                    $fields = array(
                        'merchant_id' => array(
                            'title' => __( 'Merchant ID', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                            'default' => '10011013800'
                        ),
                        'encryption_key' => array(
                            'title' => __( 'Encryption key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                            'default' => 'secret'
                        ),
                        'payweb_version' => array(
                            'field_name' => 'payweb_version',
                            'title' => __( 'PayWeb version', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => [
                                2 => __( 'PayWeb2', 'tickera-event-ticketing-system' ),
                                3 => __( 'PayWeb3', 'tickera-event-ticketing-system' )
                            ],
                            'default' => 3
                        ),
                        'currency' => array(
                            'title' => __( 'Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => [
                                "GBP" => __( 'GBP - British Pound', 'tickera-event-ticketing-system' ),
                                "USD" => __( 'USD - U.S. Dollar', 'tickera-event-ticketing-system' ),
                                "ZAR" => __( 'ZAR - South Africa', 'tickera-event-ticketing-system' ),
                            ],
                            'default' => 'ZAR',
                            'conditional' => array(
                                'field_name' => 'payweb_version',
                                'field_type' => 'select',
                                'value' => 2,
                                'action' => 'show'
                            )
                        ),
                        'country_payweb_three' => array(
                            'title' => __( 'Country/Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => [
                                'AFG' => __( 'AFG – Afghanistan (AFA)', 'tickera-event-ticketing-system' ),
                                'ALB' => __( 'ALB – Albania (ALL)', 'tickera-event-ticketing-system' ),
                                'DZA' => __( 'DZA – Algeria (DZD)', 'tickera-event-ticketing-system' ),
                                'ASM' => __( 'ASM – American Samoa (USD)', 'tickera-event-ticketing-system' ),
                                'AND' => __( 'AND – Andorra (EUR)', 'tickera-event-ticketing-system' ),
                                'AGO' => __( 'AGO – Angola (AOA)', 'tickera-event-ticketing-system' ),
                                'AIA' => __( 'AIA – Anguilla (XCD)', 'tickera-event-ticketing-system' ),
                                'ATA' => __( 'ATA – Antarctica (NOK)', 'tickera-event-ticketing-system' ),
                                'ATG' => __( 'ATG – Antigua and Barbuda (XCD)', 'tickera-event-ticketing-system' ),
                                'ARG' => __( 'ARG – Argentina (ARS)', 'tickera-event-ticketing-system' ),
                                'ARM' => __( 'ARM – Armenia (AMD)', 'tickera-event-ticketing-system' ),
                                'ABW' => __( 'ABW – Aruba (AWG)', 'tickera-event-ticketing-system' ),
                                'AUS' => __( 'AUS – Australia (AUD)', 'tickera-event-ticketing-system' ),
                                'AUT' => __( 'AUT – Austria (EUR)', 'tickera-event-ticketing-system' ),
                                'AZE' => __( 'AZE – Azerbaijan (AZM)', 'tickera-event-ticketing-system' ),
                                'BHS' => __( 'BHS – Bahamas (BSD)', 'tickera-event-ticketing-system' ),
                                'BHR' => __( 'BHR – Bahrain (BHD)', 'tickera-event-ticketing-system' ),
                                'BGD' => __( 'BGD – Bangladesh (BDT)', 'tickera-event-ticketing-system' ),
                                'BRB' => __( 'BRB – Barbados (BBD)', 'tickera-event-ticketing-system' ),
                                'BLR' => __( 'BLR – Belarus (BYR)', 'tickera-event-ticketing-system' ),
                                'BEL' => __( 'BEL – Belgium (EUR)', 'tickera-event-ticketing-system' ),
                                'BLZ' => __( 'BLZ – Belize (BZD)', 'tickera-event-ticketing-system' ),
                                'BEN' => __( 'BEN – Benin (XOF)', 'tickera-event-ticketing-system' ),
                                'BMU' => __( 'BMU – Bermuda (BMD)', 'tickera-event-ticketing-system' ),
                                'BTN' => __( 'BTN – Bhutan (INR)', 'tickera-event-ticketing-system' ),
                                'BOL' => __( 'BOL – Bolivia (BOB)', 'tickera-event-ticketing-system' ),
                                'BIH' => __( 'BIH – Bosnia and Herzegovina (BAM)', 'tickera-event-ticketing-system' ),
                                'BWA' => __( 'BWA – Botswana (BWP)', 'tickera-event-ticketing-system' ),
                                'BVT' => __( 'BVT – Bouvet Is. (NOK)', 'tickera-event-ticketing-system' ),
                                'BRA' => __( 'BRA – Brazil (BRL)', 'tickera-event-ticketing-system' ),
                                'IOT' => __( 'IOT – British Indian Ocean Territory (USD)', 'tickera-event-ticketing-system' ),
                                'VGB' => __( 'VGB – British Virgin Is. (USD)', 'tickera-event-ticketing-system' ),
                                'BRN' => __( 'BRN – Brunei Darussalam (BND)', 'tickera-event-ticketing-system' ),
                                'BGR' => __( 'BGR – Bulgaria (BGN)', 'tickera-event-ticketing-system' ),
                                'BFA' => __( 'BFA – Burkina Faso (XOF)', 'tickera-event-ticketing-system' ),
                                'BDI' => __( 'BDI – Burundi (BIF)', 'tickera-event-ticketing-system' ),
                                'KHM' => __( 'KHM – Cambodia (KHR)', 'tickera-event-ticketing-system' ),
                                'CMR' => __( 'CMR – Cameroon United Republic of (XAF)', 'tickera-event-ticketing-system' ),
                                'CAN' => __( 'CAN – Canada (CAD)', 'tickera-event-ticketing-system' ),
                                'CPV' => __( 'CPV – Cape Verde Is. (CVE)', 'tickera-event-ticketing-system' ),
                                'CYM' => __( 'CYM – Cayman Is. (KYD)', 'tickera-event-ticketing-system' ),
                                'CAF' => __( 'CAF – Central African Republic (XAF)', 'tickera-event-ticketing-system' ),
                                'TCD' => __( 'TCD – Chad (XAF)', 'tickera-event-ticketing-system' ),
                                'CHL' => __( 'CHL – Chile (CLP)', 'tickera-event-ticketing-system' ),
                                'CHN' => __( 'CHN – China (CNY)', 'tickera-event-ticketing-system' ),
                                'CXR' => __( 'CXR – Christmas Is. (AUD)', 'tickera-event-ticketing-system' ),
                                'CCK' => __( 'CCK – Cocos (Keeling) Is. (AUD)', 'tickera-event-ticketing-system' ),
                                'COL' => __( 'COL – Colombia (COP)', 'tickera-event-ticketing-system' ),
                                'COM' => __( 'COM – Comoros (KMF)', 'tickera-event-ticketing-system' ),
                                'COG' => __( 'COG – Congo (XAF)', 'tickera-event-ticketing-system' ),
                                'COK' => __( 'COK – Cook Is. (NZD)', 'tickera-event-ticketing-system' ),
                                'CRI' => __( 'CRI – Costa Rica (CRC)', 'tickera-event-ticketing-system' ),
                                'CIV' => __( 'CIV – Côte d’Ivoire (Ivory Coast) (XOF)', 'tickera-event-ticketing-system' ),
                                'HRV' => __( 'HRV – Croatia (HRK)', 'tickera-event-ticketing-system' ),
                                'CUB' => __( 'CUB – Cuba (CUP)', 'tickera-event-ticketing-system' ),
                                'CYP' => __( 'CYP – Cyprus (CYP)', 'tickera-event-ticketing-system' ),
                                'CZE' => __( 'CZE – Czech Republic (CZK)', 'tickera-event-ticketing-system' ),
                                'COD' => __( 'COD – Democratic Republic of the Congo (formerly Zaire) (CDF)', 'tickera-event-ticketing-system' ),
                                'DNK' => __( 'DNK – Denmark (DKK)', 'tickera-event-ticketing-system' ),
                                'DJI' => __( 'DJI – Djibouti (DJF)', 'tickera-event-ticketing-system' ),
                                'DMA' => __( 'DMA – Dominica (XCD)', 'tickera-event-ticketing-system' ),
                                'DOM' => __( 'DOM – Dominican Rep. (DOP)', 'tickera-event-ticketing-system' ),
                                'TMP' => __( 'TMP – East Timor (TPE)', 'tickera-event-ticketing-system' ),
                                'ECU' => __( 'ECU – Ecuador (ECS)', 'tickera-event-ticketing-system' ),
                                'EGY' => __( 'EGY – Egypt (EGP)', 'tickera-event-ticketing-system' ),
                                'SLV' => __( 'SLV – El Salvador (USD)', 'tickera-event-ticketing-system' ),
                                'GNQ' => __( 'GNQ – Equatorial Guinea (XAF)', 'tickera-event-ticketing-system' ),
                                'ERI' => __( 'ERI – Eritrea (ERN)', 'tickera-event-ticketing-system' ),
                                'EST' => __( 'EST – Estonia (EEK)', 'tickera-event-ticketing-system' ),
                                'ETH' => __( 'ETH – Ethiopia (ETB)', 'tickera-event-ticketing-system' ),
                                'FRO' => __( 'FRO – Faeroe Is. (DKK)', 'tickera-event-ticketing-system' ),
                                'FLK' => __( 'FLK – Falkland Is. (Malvinas) (FKP)', 'tickera-event-ticketing-system' ),
                                'FJI' => __( 'FJI – Fiji (FJD)', 'tickera-event-ticketing-system' ),
                                'FIN' => __( 'FIN – Finland (EUR)', 'tickera-event-ticketing-system' ),
                                'FRA' => __( 'FRA – France (EUR)', 'tickera-event-ticketing-system' ),
                                'FXX' => __( 'FXX – France Metropolitan (EUR)', 'tickera-event-ticketing-system' ),
                                'GUF' => __( 'GUF – French Guiana (EUR)', 'tickera-event-ticketing-system' ),
                                'PYF' => __( 'PYF – French Polynesia (XPF)', 'tickera-event-ticketing-system' ),
                                'ATF' => __( 'ATF – French Southern Territory (EUR)', 'tickera-event-ticketing-system' ),
                                'GAB' => __( 'GAB – Gabon (XAF)', 'tickera-event-ticketing-system' ),
                                'GMB' => __( 'GMB – Gambia (GMD)', 'tickera-event-ticketing-system' ),
                                'GEO' => __( 'GEO – Georgia (GEL)', 'tickera-event-ticketing-system' ),
                                'DEU' => __( 'DEU – Germany (DEM)', 'tickera-event-ticketing-system' ),
                                'GHA' => __( 'GHA – Ghana (GHC)', 'tickera-event-ticketing-system' ),
                                'GIB' => __( 'GIB – Gibraltar (GIP)', 'tickera-event-ticketing-system' ),
                                'GRC' => __( 'GRC – Greece (EUR)', 'tickera-event-ticketing-system' ),
                                'GRL' => __( 'GRL – Greenland (DKK)', 'tickera-event-ticketing-system' ),
                                'GRD' => __( 'GRD – Grenada (XCD)', 'tickera-event-ticketing-system' ),
                                'GLP' => __( 'GLP – Guadeloupe (EUR)', 'tickera-event-ticketing-system' ),
                                'GUM' => __( 'GUM – Guam (USD)', 'tickera-event-ticketing-system' ),
                                'GTM' => __( 'GTM – Guatemala (GTQ)', 'tickera-event-ticketing-system' ),
                                'GIN' => __( 'GIN – Guinea (GNF)', 'tickera-event-ticketing-system' ),
                                'GNB' => __( 'GNB – Guinea—Bissau (GWP)', 'tickera-event-ticketing-system' ),
                                'GUY' => __( 'GUY – Guyana (GYD)', 'tickera-event-ticketing-system' ),
                                'HTI' => __( 'HTI – Haiti (HTG)', 'tickera-event-ticketing-system' ),
                                'HMD' => __( 'HMD – Heard and McDonald Is. (AUD)', 'tickera-event-ticketing-system' ),
                                'VAT' => __( 'VAT – Holy See (Vatican City State) (EUR)', 'tickera-event-ticketing-system' ),
                                'HND' => __( 'HND – Honduras (HNL)', 'tickera-event-ticketing-system' ),
                                'HKG' => __( 'HKG – Hong Kong China (HKD)', 'tickera-event-ticketing-system' ),
                                'HUN' => __( 'HUN – Hungary (HUF)', 'tickera-event-ticketing-system' ),
                                'ISL' => __( 'ISL – Iceland (ISK)', 'tickera-event-ticketing-system' ),
                                'IND' => __( 'IND – India (INR)', 'tickera-event-ticketing-system' ),
                                'IDN' => __( 'IDN – Indonesia (IDR)', 'tickera-event-ticketing-system' ),
                                'IRN' => __( 'IRN – Iran Islamic Republic of (IRR)', 'tickera-event-ticketing-system' ),
                                'IRQ' => __( 'IRQ – Iraq (IQD)', 'tickera-event-ticketing-system' ),
                                'IRL' => __( 'IRL – Ireland Republic of (EUR)', 'tickera-event-ticketing-system' ),
                                'ISR' => __( 'ISR – Israel (ILS)', 'tickera-event-ticketing-system' ),
                                'ITA' => __( 'ITA – Italy (EUR)', 'tickera-event-ticketing-system' ),
                                'JAM' => __( 'JAM – Jamaica (JMD)', 'tickera-event-ticketing-system' ),
                                'JPN' => __( 'JPN – Japan (JPY)', 'tickera-event-ticketing-system' ),
                                'JOR' => __( 'JOR – Jordan (JOD)', 'tickera-event-ticketing-system' ),
                                'KAZ' => __( 'KAZ – Kazakhstan (KZT)', 'tickera-event-ticketing-system' ),
                                'KEN' => __( 'KEN – Kenya (KES)', 'tickera-event-ticketing-system' ),
                                'KIR' => __( 'KIR – Kiribati (AUD)', 'tickera-event-ticketing-system' ),
                                'PRK' => __( 'PRK – Korea Democratic People’s Republic of (North Korea) (KPW)', 'tickera-event-ticketing-system' ),
                                'KOR' => __( 'KOR – Korea Republic of (KRW)', 'tickera-event-ticketing-system' ),
                                'KWT' => __( 'KWT – Kuwait (KWD)', 'tickera-event-ticketing-system' ),
                                'KGZ' => __( 'KGZ – Kyrgyzstan (KGS)', 'tickera-event-ticketing-system' ),
                                'LAO' => __( 'LAO – Lao People’s Democratic Republic (LAK)', 'tickera-event-ticketing-system' ),
                                'LVA' => __( 'LVA – Latvia (LVL)', 'tickera-event-ticketing-system' ),
                                'LBN' => __( 'LBN – Lebanon (LBP)', 'tickera-event-ticketing-system' ),
                                'LSO' => __( 'LSO – Lesotho (ZAR)', 'tickera-event-ticketing-system' ),
                                'LBR' => __( 'LBR – Liberia (LRD)', 'tickera-event-ticketing-system' ),
                                'LBY' => __( 'LBY – Libyan Arab Jamahiriya (LYD)', 'tickera-event-ticketing-system' ),
                                'LIE' => __( 'LIE – Liechtenstein (CHF)', 'tickera-event-ticketing-system' ),
                                'LTU' => __( 'LTU – Lithuania (LTL)', 'tickera-event-ticketing-system' ),
                                'LUX' => __( 'LUX – Luxembourg (EUR)', 'tickera-event-ticketing-system' ),
                                'MAC' => __( 'MAC – Macau China (MOP)', 'tickera-event-ticketing-system' ),
                                'MKD' => __( 'MKD – Macedonia the Former Yugoslav Republic of (MKD)', 'tickera-event-ticketing-system' ),
                                'MDG' => __( 'MDG – Madagascar (MGF)', 'tickera-event-ticketing-system' ),
                                'MWI' => __( 'MWI – Malawi (MWK)', 'tickera-event-ticketing-system' ),
                                'MYS' => __( 'MYS – Malaysia (MYR)', 'tickera-event-ticketing-system' ),
                                'MDV' => __( 'MDV – Maldives (MVR)', 'tickera-event-ticketing-system' ),
                                'MLI' => __( 'MLI – Mali (XOF)', 'tickera-event-ticketing-system' ),
                                'MLT' => __( 'MLT – Malta (MTL)', 'tickera-event-ticketing-system' ),
                                'MHL' => __( 'MHL – Marshall Islands (USD)', 'tickera-event-ticketing-system' ),
                                'MTQ' => __( 'MTQ – Martinique (EUR)', 'tickera-event-ticketing-system' ),
                                'MRT' => __( 'MRT – Mauritania (MRO)', 'tickera-event-ticketing-system' ),
                                'MUS' => __( 'MUS – Mauritius (MUR)', 'tickera-event-ticketing-system' ),
                                'MYT' => __( 'MYT – Mayotte (EUR)', 'tickera-event-ticketing-system' ),
                                'MEX' => __( 'MEX – Mexico (MXN)', 'tickera-event-ticketing-system' ),
                                'FSM' => __( 'FSM – Micronesia (USD)', 'tickera-event-ticketing-system' ),
                                'MDA' => __( 'MDA – Moldova Republic of (MDL)', 'tickera-event-ticketing-system' ),
                                'MCO' => __( 'MCO – Monaco (EUR)', 'tickera-event-ticketing-system' ),
                                'MNG' => __( 'MNG – Mongolia (MNT)', 'tickera-event-ticketing-system' ),
                                'MSR' => __( 'MSR – Montserrat (XCD)', 'tickera-event-ticketing-system' ),
                                'MAR' => __( 'MAR – Morocco (MAD)', 'tickera-event-ticketing-system' ),
                                'MOZ' => __( 'MOZ – Mozambique (MZM)', 'tickera-event-ticketing-system' ),
                                'MMR' => __( 'MMR – Myanmar (MMK)', 'tickera-event-ticketing-system' ),
                                'NAM' => __( 'NAM – Namibia (NAD)', 'tickera-event-ticketing-system' ),
                                'NRU' => __( 'NRU – Nauru (AUD)', 'tickera-event-ticketing-system' ),
                                'NPL' => __( 'NPL – Nepal (NPR)', 'tickera-event-ticketing-system' ),
                                'NLD' => __( 'NLD – Netherlands (EUR)', 'tickera-event-ticketing-system' ),
                                'ANT' => __( 'ANT – Netherlands Antilles (ANG)', 'tickera-event-ticketing-system' ),
                                'NCL' => __( 'NCL – New Caledonia (XPF)', 'tickera-event-ticketing-system' ),
                                'NZL' => __( 'NZL – New Zealand (NZD)', 'tickera-event-ticketing-system' ),
                                'NIC' => __( 'NIC – Nicaragua (NIO)', 'tickera-event-ticketing-system' ),
                                'NER' => __( 'NER – Niger (XOF)', 'tickera-event-ticketing-system' ),
                                'NGA' => __( 'NGA – Nigeria (NGN)', 'tickera-event-ticketing-system' ),
                                'NIU' => __( 'NIU – Niue (NZD)', 'tickera-event-ticketing-system' ),
                                'NFK' => __( 'NFK – Norfolk Is. (AUD)', 'tickera-event-ticketing-system' ),
                                'MNP' => __( 'MNP – Northern Mariana Islands (USD)', 'tickera-event-ticketing-system' ),
                                'NOR' => __( 'NOR – Norway (NOK)', 'tickera-event-ticketing-system' ),
                                'OMN' => __( 'OMN – Oman (OMR)', 'tickera-event-ticketing-system' ),
                                'PAK' => __( 'PAK – Pakistan (PKR)', 'tickera-event-ticketing-system' ),
                                'PLW' => __( 'PLW – Palau (USD)', 'tickera-event-ticketing-system' ),
                                'PAN' => __( 'PAN – Panama (PAB)', 'tickera-event-ticketing-system' ),
                                'PNG' => __( 'PNG – Papua New Guinea (PGK)', 'tickera-event-ticketing-system' ),
                                'PRY' => __( 'PRY – Paraguay (PYG)', 'tickera-event-ticketing-system' ),
                                'PER' => __( 'PER – Peru (PEN)', 'tickera-event-ticketing-system' ),
                                'PHL' => __( 'PHL – Philippines (PHP)', 'tickera-event-ticketing-system' ),
                                'PCN' => __( 'PCN – Pitcairn (NZD)', 'tickera-event-ticketing-system' ),
                                'POL' => __( 'POL – Poland (PLN)', 'tickera-event-ticketing-system' ),
                                'PRT' => __( 'PRT – Portugal (EUR)', 'tickera-event-ticketing-system' ),
                                'PRI' => __( 'PRI – Puerto Rico (USD)', 'tickera-event-ticketing-system' ),
                                'QAT' => __( 'QAT – Qatar (QAR)', 'tickera-event-ticketing-system' ),
                                'REU' => __( 'REU – Reunion (EUR)', 'tickera-event-ticketing-system' ),
                                'ROM' => __( 'ROM – Romania (ROL)', 'tickera-event-ticketing-system' ),
                                'RUS' => __( 'RUS – Russian Federation (RUB)', 'tickera-event-ticketing-system' ),
                                'RWA' => __( 'RWA – Rwanda (RWF)', 'tickera-event-ticketing-system' ),
                                'WSM' => __( 'WSM – Samoa (WST)', 'tickera-event-ticketing-system' ),
                                'SMR' => __( 'SMR – San Marino (EUR)', 'tickera-event-ticketing-system' ),
                                'STP' => __( 'STP – Sao Tome and Principe (STD)', 'tickera-event-ticketing-system' ),
                                'SAU' => __( 'SAU – Saudi Arabia (SAR)', 'tickera-event-ticketing-system' ),
                                'SEN' => __( 'SEN – Senegal (XOF)', 'tickera-event-ticketing-system' ),
                                'SYC' => __( 'SYC – Seychelles (SCR)', 'tickera-event-ticketing-system' ),
                                'SLE' => __( 'SLE – Sierra Leone (SLL)', 'tickera-event-ticketing-system' ),
                                'SGP' => __( 'SGP – Singapore (SGD)', 'tickera-event-ticketing-system' ),
                                'SVK' => __( 'SVK – Slovakia (SKK)', 'tickera-event-ticketing-system' ),
                                'SVN' => __( 'SVN – Slovenia (SIT)', 'tickera-event-ticketing-system' ),
                                'SGS' => __( 'SGS – So. Georgia and So. Sandwich Is. (GBP)', 'tickera-event-ticketing-system' ),
                                'SLB' => __( 'SLB – Solomon Is. (SBD)', 'tickera-event-ticketing-system' ),
                                'SOM' => __( 'SOM – Somalia (SOS)', 'tickera-event-ticketing-system' ),
                                'ZAF' => __( 'ZAF – South Africa (ZAR)', 'tickera-event-ticketing-system' ),
                                'ESP' => __( 'ESP – Spain (EUR)', 'tickera-event-ticketing-system' ),
                                'LKA' => __( 'LKA – Sri Lanka (LKR)', 'tickera-event-ticketing-system' ),
                                'SHN' => __( 'SHN – St. Helena (SHP)', 'tickera-event-ticketing-system' ),
                                'KNA' => __( 'KNA – St. Kitts-Nevis (XCD)', 'tickera-event-ticketing-system' ),
                                'LCA' => __( 'LCA – St. Lucia (XCD)', 'tickera-event-ticketing-system' ),
                                'SPM' => __( 'SPM – St. Pierre and Miquelon (EUR)', 'tickera-event-ticketing-system' ),
                                'VCT' => __( 'VCT – St. Vincent and The Grenadines (XCD)', 'tickera-event-ticketing-system' ),
                                'SDN' => __( 'SDN – Sudan (SDD)', 'tickera-event-ticketing-system' ),
                                'SUR' => __( 'SUR – Suriname (SRG)', 'tickera-event-ticketing-system' ),
                                'SJM' => __( 'SJM – Svalbard and Jan Mayen Is. (NOK)', 'tickera-event-ticketing-system' ),
                                'SWZ' => __( 'SWZ – Swaziland (SZL)', 'tickera-event-ticketing-system' ),
                                'SWE' => __( 'SWE – Sweden (SEK)', 'tickera-event-ticketing-system' ),
                                'CHE' => __( 'CHE – Switzerland (CHF)', 'tickera-event-ticketing-system' ),
                                'SYR' => __( 'SYR – Syrian Arab Rep. (SYP)', 'tickera-event-ticketing-system' ),
                                'TWN' => __( 'TWN – Taiwan (TWD)', 'tickera-event-ticketing-system' ),
                                'TJK' => __( 'TJK – Tajikistan (TJS)', 'tickera-event-ticketing-system' ),
                                'TZA' => __( 'TZA – Tanzania United Republic of (TZS)', 'tickera-event-ticketing-system' ),
                                'THA' => __( 'THA – Thailand (THB)', 'tickera-event-ticketing-system' ),
                                'TGO' => __( 'TGO – Togo (XOF)', 'tickera-event-ticketing-system' ),
                                'TKL' => __( 'TKL – Tokelau (NZD)', 'tickera-event-ticketing-system' ),
                                'TON' => __( 'TON – Tonga (TOP)', 'tickera-event-ticketing-system' ),
                                'TTO' => __( 'TTO – Trinidad and Tobago (TTD)', 'tickera-event-ticketing-system' ),
                                'TUN' => __( 'TUN – Tunisia (TND)', 'tickera-event-ticketing-system' ),
                                'TUR' => __( 'TUR – Turkey (TRL)', 'tickera-event-ticketing-system' ),
                                'TKM' => __( 'TKM – Turkmenistan (TMM)', 'tickera-event-ticketing-system' ),
                                'TCA' => __( 'TCA – Turks and Caicos Is. (USD)', 'tickera-event-ticketing-system' ),
                                'TUV' => __( 'TUV – Tuvalu (AUD)', 'tickera-event-ticketing-system' ),
                                'UMI' => __( 'UMI – U.S. Minor Outlying Islands (USD)', 'tickera-event-ticketing-system' ),
                                'VIR' => __( 'VIR – U.S. Virgin Is. (USD)', 'tickera-event-ticketing-system' ),
                                'UGA' => __( 'UGA – Uganda (UGX)', 'tickera-event-ticketing-system' ),
                                'UKR' => __( 'UKR – Ukraine (UAH)', 'tickera-event-ticketing-system' ),
                                'ARE' => __( 'ARE – United Arab Emirates (AED)', 'tickera-event-ticketing-system' ),
                                'GBR' => __( 'GBR – United Kingdom (GBP)', 'tickera-event-ticketing-system' ),
                                'USA' => __( 'USA – United States (USD)', 'tickera-event-ticketing-system' ),
                                'URY' => __( 'URY – Uruguay (UYU)', 'tickera-event-ticketing-system' ),
                                'UZB' => __( 'UZB – Uzbekistan (UZS)', 'tickera-event-ticketing-system' ),
                                'VUT' => __( 'VUT – Vanuatu (VUV)', 'tickera-event-ticketing-system' ),
                                'VEN' => __( 'VEN – Venezuela (VEB)', 'tickera-event-ticketing-system' ),
                                'VNM' => __( 'VNM – Vietnam (VND)', 'tickera-event-ticketing-system' ),
                                'WLF' => __( 'WLF – Wallis and Futuna Is. (XPF)', 'tickera-event-ticketing-system' ),
                                'ESH' => __( 'ESH – Western Sahara (MAD)', 'tickera-event-ticketing-system' ),
                                'YEM' => __( 'YEM – Yemen (YER)', 'tickera-event-ticketing-system' ),
                                'YUG' => __( 'YUG – Yugoslavia (YUM)', 'tickera-event-ticketing-system' ),
                                'ZMB' => __( 'ZMB – Zambia (ZMK)', 'tickera-event-ticketing-system' ),
                                'ZWE' => __( 'ZWE – Zimbabwe (ZWD)', 'tickera-event-ticketing-system' ),
                            ],
                            'default' => 'ZAF',
                            'conditional' => array(
                                'field_name' => 'payweb_version',
                                'field_type' => 'select',
                                'value' => 3,
                                'action' => 'show'
                            )
                        ),
                        'locale_payweb_three' => array(
                            'title' => __( 'Locale', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => [
                                'af' => __( 'Afrikaans', 'tickera-event-ticketing-system' ),
                                'sq' => __( 'Albanian', 'tickera-event-ticketing-system' ),
                                'ar-dz' => __( 'Arabic (Algeria)', 'tickera-event-ticketing-system' ),
                                'ar-bh' => __( 'Arabic (Bahrain)', 'tickera-event-ticketing-system' ),
                                'ar-eg' => __( 'Arabic (Egypt)', 'tickera-event-ticketing-system' ),
                                'ar-iq' => __( 'Arabic (Iraq)', 'tickera-event-ticketing-system' ),
                                'ar-jo' => __( 'Arabic (Jordan)', 'tickera-event-ticketing-system' ),
                                'ar-kw' => __( 'Arabic (Kuwait)', 'tickera-event-ticketing-system' ),
                                'ar-lb' => __( 'Arabic (Lebanon)', 'tickera-event-ticketing-system' ),
                                'ar-ly' => __( 'Arabic (Libya)', 'tickera-event-ticketing-system' ),
                                'ar-ma' => __( 'Arabic (Morocco)', 'tickera-event-ticketing-system' ),
                                'ar-om' => __( 'Arabic (Oman)', 'tickera-event-ticketing-system' ),
                                'ar-qa' => __( 'Arabic (Qatar)', 'tickera-event-ticketing-system' ),
                                'ar-sa' => __( 'Arabic (Saudi Arabia)', 'tickera-event-ticketing-system' ),
                                'ar-sy' => __( 'Arabic (Syria)', 'tickera-event-ticketing-system' ),
                                'ar-tn' => __( 'Arabic (Tunisia)', 'tickera-event-ticketing-system' ),
                                'ar-ae' => __( 'Arabic (U.A.E.)', 'tickera-event-ticketing-system' ),
                                'ar-ye' => __( 'Arabic (Yemen)', 'tickera-event-ticketing-system' ),
                                'eu' => __( 'Basque', 'tickera-event-ticketing-system' ),
                                'br' => __( 'Belarusian', 'tickera-event-ticketing-system' ),
                                'bg' => __( 'Bulgarian', 'tickera-event-ticketing-system' ),
                                'ca' => __( 'Catalan', 'tickera-event-ticketing-system' ),
                                'zh-hk' => __( 'Chinese (Hong Kong SAR)', 'tickera-event-ticketing-system' ),
                                'zh-cn' => __( 'Chinese (PRC)', 'tickera-event-ticketing-system' ),
                                'zh-sg' => __( 'Chinese (Singapore)', 'tickera-event-ticketing-system' ),
                                'zh-tw' => __( 'Chinese (Taiwan)', 'tickera-event-ticketing-system' ),
                                'hr' => __( 'Croatian', 'tickera-event-ticketing-system' ),
                                'cs' => __( 'Czech', 'tickera-event-ticketing-system' ),
                                'da' => __( 'Danish', 'tickera-event-ticketing-system' ),
                                'nl' => __( 'Dutch (Standard)', 'tickera-event-ticketing-system' ),
                                'nl-be' => __( 'Dutch (Belgium)', 'tickera-event-ticketing-system' ),
                                'en' => __( 'English', 'tickera-event-ticketing-system' ),
                                'en-au' => __( 'English (Australia)', 'tickera-event-ticketing-system' ),
                                'en-bz' => __( 'English (Belize)', 'tickera-event-ticketing-system' ),
                                'en-ca' => __( 'English (Canada)', 'tickera-event-ticketing-system' ),
                                'en-cb' => __( 'English (Caribbean)', 'tickera-event-ticketing-system' ),
                                'en-ie' => __( 'English (Ireland)', 'tickera-event-ticketing-system' ),
                                'en-jm' => __( 'English (Jamaica)', 'tickera-event-ticketing-system' ),
                                'en-nz' => __( 'English (New Zealand)', 'tickera-event-ticketing-system' ),
                                'en-za' => __( 'English (South Africa)', 'tickera-event-ticketing-system' ),
                                'en-tt' => __( 'English (Trinidad)', 'tickera-event-ticketing-system' ),
                                'en-gb' => __( 'English (United Kingdom)', 'tickera-event-ticketing-system' ),
                                'en-us' => __( 'English (United States)', 'tickera-event-ticketing-system' ),
                                'et' => __( 'Estonian', 'tickera-event-ticketing-system' ),
                                'fo' => __( 'Faeroese', 'tickera-event-ticketing-system' ),
                                'fa' => __( 'Farsi', 'tickera-event-ticketing-system' ),
                                'fi' => __( 'Finnish', 'tickera-event-ticketing-system' ),
                                'fr' => __( 'French (Standard)', 'tickera-event-ticketing-system' ),
                                'fr-be' => __( 'French (Belgium)', 'tickera-event-ticketing-system' ),
                                'fr-ca' => __( 'French (Canada)', 'tickera-event-ticketing-system' ),
                                'fr-lu' => __( 'French (Luxembourg)', 'tickera-event-ticketing-system' ),
                                'fr-ch' => __( 'French (Switzerland)', 'tickera-event-ticketing-system' ),
                                'gd' => __( 'Gaelic (Scotland)', 'tickera-event-ticketing-system' ),
                                'de' => __( 'German (Standard)', 'tickera-event-ticketing-system' ),
                                'de-at' => __( 'German (Austria)', 'tickera-event-ticketing-system' ),
                                'de-li' => __( 'German (Liechtenstein)', 'tickera-event-ticketing-system' ),
                                'de-lu' => __( 'German (Luxembourg)', 'tickera-event-ticketing-system' ),
                                'de-ch' => __( 'German (Switzerland)', 'tickera-event-ticketing-system' ),
                                'el' => __( 'Greek', 'tickera-event-ticketing-system' ),
                                'he' => __( 'Hebrew', 'tickera-event-ticketing-system' ),
                                'hi' => __( 'Hindi', 'tickera-event-ticketing-system' ),
                                'hu' => __( 'Hungarian', 'tickera-event-ticketing-system' ),
                                'is' => __( 'Icelandic', 'tickera-event-ticketing-system' ),
                                'id' => __( 'Indonesian', 'tickera-event-ticketing-system' ),
                                'ga' => __( 'Irish', 'tickera-event-ticketing-system' ),
                                'it' => __( 'Italian (Standard)', 'tickera-event-ticketing-system' ),
                                'it-ch' => __( 'Italian (Switzerland)', 'tickera-event-ticketing-system' ),
                                'ko' => __( 'Korean', 'tickera-event-ticketing-system' ),
                                'lv' => __( 'Latvian', 'tickera-event-ticketing-system' ),
                                'lt' => __( 'Lithuanian', 'tickera-event-ticketing-system' ),
                                'mk' => __( 'Macedonian (FYROM)', 'tickera-event-ticketing-system' ),
                                'ms' => __( 'Malaysian', 'tickera-event-ticketing-system' ),
                                'mt' => __( 'Maltese', 'tickera-event-ticketing-system' ),
                                'no' => __( 'Norwegian', 'tickera-event-ticketing-system' ),
                                'pl' => __( 'Polish', 'tickera-event-ticketing-system' ),
                                'pt-br' => __( 'Portuguese (Brazil)', 'tickera-event-ticketing-system' ),
                                'pt' => __( 'Portuguese (Portugal)', 'tickera-event-ticketing-system' ),
                                'rm' => __( 'Rhaeto-Romanic', 'tickera-event-ticketing-system' ),
                                'ro' => __( 'Romanian', 'tickera-event-ticketing-system' ),
                                'ro-mo' => __( 'Romanian (Republic of Moldova)', 'tickera-event-ticketing-system' ),
                                'ru' => __( 'Russian', 'tickera-event-ticketing-system' ),
                                'ru-mo' => __( 'Russian (Republic of Moldova)', 'tickera-event-ticketing-system' ),
                                'sz' => __( 'Sami (Lappish)', 'tickera-event-ticketing-system' ),
                                'sr' => __( 'Serbian', 'tickera-event-ticketing-system' ),
                                'sk' => __( 'Slovak', 'tickera-event-ticketing-system' ),
                                'sl' => __( 'Slovenian', 'tickera-event-ticketing-system' ),
                                'sb' => __( 'Sorbian', 'tickera-event-ticketing-system' ),
                                'es' => __( 'Spanish (Spain)', 'tickera-event-ticketing-system' ),
                                'es-mx' => __( 'Spanish (Mexico)', 'tickera-event-ticketing-system' ),
                                'es-gt' => __( 'Spanish (Guatemala)', 'tickera-event-ticketing-system' ),
                                'es-cr' => __( 'Spanish (Costa Rica)', 'tickera-event-ticketing-system' ),
                                'es-pa' => __( 'Spanish (Panama)', 'tickera-event-ticketing-system' ),
                                'es-do' => __( 'Spanish (Dominican Republic)', 'tickera-event-ticketing-system' ),
                                'es-ve' => __( 'Spanish (Venezuela)', 'tickera-event-ticketing-system' ),
                                'es-co' => __( 'Spanish (Colombia)', 'tickera-event-ticketing-system' ),
                                'es-pe' => __( 'Spanish (Peru)', 'tickera-event-ticketing-system' ),
                                'es-ar' => __( 'Spanish (Argentina)', 'tickera-event-ticketing-system' ),
                                'es-ec' => __( 'Spanish (Ecuador)', 'tickera-event-ticketing-system' ),
                                'es-cl' => __( 'Spanish (Chile)', 'tickera-event-ticketing-system' ),
                                'es-uy' => __( 'Spanish (Uruguay)', 'tickera-event-ticketing-system' ),
                                'es-py' => __( 'Spanish (Paraguay)', 'tickera-event-ticketing-system' ),
                                'es-bo' => __( 'Spanish (Bolivia)', 'tickera-event-ticketing-system' ),
                                'es-sv' => __( 'Spanish (El Salvador)', 'tickera-event-ticketing-system' ),
                                'es-hn' => __( 'Spanish (Honduras)', 'tickera-event-ticketing-system' ),
                                'es-ni' => __( 'Spanish (Nicaragua)', 'tickera-event-ticketing-system' ),
                                'es-pr' => __( 'Spanish (Puerto Rico)', 'tickera-event-ticketing-system' ),
                                'sx' => __( 'Sutu', 'tickera-event-ticketing-system' ),
                                'sv' => __( 'Swedish', 'tickera-event-ticketing-system' ),
                                'sv-fi' => __( 'Swedish (Finland)', 'tickera-event-ticketing-system' ),
                                'th' => __( 'Thai', 'tickera-event-ticketing-system' ),
                                'ts' => __( 'Tsonga', 'tickera-event-ticketing-system' ),
                                'tn' => __( 'Tswana', 'tickera-event-ticketing-system' ),
                                'tr' => __( 'Turkish', 'tickera-event-ticketing-system' ),
                                'uk' => __( 'Ukrainian', 'tickera-event-ticketing-system' ),
                                'ur' => __( 'Urdu', 'tickera-event-ticketing-system' ),
                                've' => __( 'Venda', 'tickera-event-ticketing-system' ),
                                'vi' => __( 'Vietnamese xh Xhosa', 'tickera-event-ticketing-system' ),
                                'ji' => __( 'Yiddish', 'tickera-event-ticketing-system' ),
                                'zu' => __( 'Zulu', 'tickera-event-ticketing-system' ),
                            ],
                            'default' => 'en-za',
                            'conditional' => array(
                                'field_name' => 'payweb_version',
                                'field_type' => 'select',
                                'value' => 3,
                                'action' => 'show'
                            )
                        )
                    );
                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name ); ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>
                </div>
            </div>
            <?php
        }

        function api( $url, $fields ) {
            $ch = curl_init();
            curl_setopt( $ch, CURLOPT_URL, $url );
            curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
            curl_setopt( $ch, CURLOPT_NOBODY, false );
            curl_setopt( $ch, CURLOPT_REFERER, sanitize_text_field( $_SERVER[ 'HTTP_HOST' ] ) );
            curl_setopt( $ch, CURLOPT_POST, true );
            curl_setopt( $ch, CURLOPT_POSTFIELDS, http_build_query( $fields ) );
            parse_str( curl_exec( $ch ), $response );
            curl_close( $ch );
            return $response;
        }

        function ipn() {
            $this->update_order();
        }
    }

    \Tickera\tickera_register_gateway_plugin( 'Tickera\Gateway\TC_Gateway_Paygate', 'paygate', __( 'Paygate', 'tickera-event-ticketing-system' ) );
}