<?php
/**
 * Braintree 3D Secure 2 - Payment Gateway
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'Tickera\Gateway\TC_Gateway_Braintree_3ds2' ) ) {

    class TC_Gateway_Braintree_3ds2 extends TC_Gateway_API {

        var $plugin_name = 'braintree_3ds2';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl;
        var $ipn_url;
        var $merchant_key = '';
        var $public_key;
        var $private_key;
        var $cse_key;
        var $environment;
        var $clientToken;
        var $currency;
        var $totalAmount;
        var $currencies = array();
        var $permanently_active = false;
        var $skip_payment_screen = false;

        /**
         * Support for older payment gateway API
         */
        function on_creation() {
            $this->init();
        }

        /**
         * Initialize Variables
         */
        function init() {

            global $tc;

            require_once( $tc->plugin_dir . "includes/gateways/braintree/lib/Braintree.php" );

            // Register API Route
            add_action( 'rest_api_init', function () {
                register_rest_route( 'tc-braintree-3ds2/v1', '/callback/', array(
                    'methods' => 'POST',
                    'callback' => array( $this, 'process_payment' ),
                    'permission_callback' => '__return_true'
                ) );
            } );

            $this->admin_name = __( 'Braintree 3DS2', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'Braintree 3D Secure 2', 'tickera-event-ticketing-system' );

            $this->method_img_url = apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/braintree.png', $this->plugin_name );
            $this->admin_img_url = apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-braintree-3ds2.png', $this->plugin_name );

            $this->merchant_key = $this->get_option( 'merchant_key' );
            $this->public_key = $this->get_option( 'public_key' );
            $this->private_key = $this->get_option( 'private_key' );
            $this->cse_key = $this->get_option( 'cse_key' );
            $this->force_ssl = $this->get_option( 'is_ssl', '0' );
            $this->environment = ( $this->force_ssl == '1' ? 'production' : 'sandbox' );
            $this->currency = $this->get_option( 'currency', 'USD' );

            $currencies = array(
                "AFA" => __( 'AFA - Afghanistan Afghani', 'tickera-event-ticketing-system' ),
                "ALL" => __( 'ALL - Albanian Lek', 'tickera-event-ticketing-system' ),
                "DZD" => __( 'DZD - Algerian dinar', 'tickera-event-ticketing-system' ),
                "ARS" => __( 'ARS - Argentine Peso', 'tickera-event-ticketing-system' ),
                "AMD" => __( 'AMD - Armenian dram', 'tickera-event-ticketing-system' ),
                "AWG" => __( 'AWG - Aruban Guilder', 'tickera-event-ticketing-system' ),
                "AUD" => __( 'AUD - Australian Dollar', 'tickera-event-ticketing-system' ),
                "AZN" => __( 'AZN - Azerbaijani an Manat', 'tickera-event-ticketing-system' ),
                "BSD" => __( 'BSD - Bahamian Dollar', 'tickera-event-ticketing-system' ),
                "BHD" => __( 'BHD - Bahraini Dinar', 'tickera-event-ticketing-system' ),
                "BDT" => __( 'BDT - Bangladeshi Taka', 'tickera-event-ticketing-system' ),
                "BBD" => __( 'BBD - Barbados Dollar', 'tickera-event-ticketing-system' ),
                "BYR" => __( 'BYR - Belarussian ruble', 'tickera-event-ticketing-system' ),
                "BZD" => __( 'BZD - Belizean dollar', 'tickera-event-ticketing-system' ),
                "BMD" => __( 'BMD - Bermudian Dollar', 'tickera-event-ticketing-system' ),
                "BOB" => __( 'BOB - Bolivian Boliviano', 'tickera-event-ticketing-system' ),
                "BWP" => __( 'BWP - Botswana Pula', 'tickera-event-ticketing-system' ),
                "BRL" => __( 'BRL - Brazilian Real', 'tickera-event-ticketing-system' ),
                "BND" => __( 'BND - Brunei Dollar', 'tickera-event-ticketing-system' ),
                "BGN" => __( 'BGN - Bulgarian Lev', 'tickera-event-ticketing-system' ),
                "BIF" => __( 'BIF - Burundi Franc', 'tickera-event-ticketing-system' ),
                "KHR" => __( 'KHR - Cambodian Riel', 'tickera-event-ticketing-system' ),
                "CAD" => __( 'CAD - Canadian Dollar', 'tickera-event-ticketing-system' ),
                "CVE" => __( 'CVE - Cape Verde Escudo', 'tickera-event-ticketing-system' ),
                "KYD" => __( 'KYD - Cayman Islands Dollar', 'tickera-event-ticketing-system' ),
                "XAF" => __( 'XAF - Central African Republic Franc BCEAO', 'tickera-event-ticketing-system' ),
                "XPF" => __( 'XPF - CFP Franc', 'tickera-event-ticketing-system' ),
                "CLP" => __( 'CLP - Chilean Peso', 'tickera-event-ticketing-system' ),
                "CNY" => __( 'CNY - Chinese Yuan Renminbi', 'tickera-event-ticketing-system' ),
                "COP" => __( 'COP - Colombian Peso', 'tickera-event-ticketing-system' ),
                "KMF" => __( 'KMF - Comoroan franc', 'tickera-event-ticketing-system' ),
                "BAM" => __( 'BAM - Convertible Marks', 'tickera-event-ticketing-system' ),
                "CRC" => __( 'CRC - Costa Rican Colon', 'tickera-event-ticketing-system' ),
                "HRK" => __( 'HRK - Croatian Kuna', 'tickera-event-ticketing-system' ),
                "CUP" => __( 'CUP - Cuban Peso', 'tickera-event-ticketing-system' ),
                "CYP" => __( 'CYP - Cyprus Pound', 'tickera-event-ticketing-system' ),
                "CZK" => __( 'CZK - Czech Republic Koruna', 'tickera-event-ticketing-system' ),
                "DKK" => __( 'DKK - Danish Krone', 'tickera-event-ticketing-system' ),
                "DJF" => __( 'DJF - Djiboutian franc', 'tickera-event-ticketing-system' ),
                "DOP" => __( 'DOP - Dominican Peso', 'tickera-event-ticketing-system' ),
                "XCD" => __( 'XCD - East Caribbean Dollar', 'tickera-event-ticketing-system' ),
                "ECS" => __( 'ECS - Ecuador', 'tickera-event-ticketing-system' ),
                "EGP" => __( 'EGP - Egyptian Pound', 'tickera-event-ticketing-system' ),
                "SVC" => __( 'SVC - El Salvador Colon', 'tickera-event-ticketing-system' ),
                "ERN" => __( 'ERN - Eritrea Nakfa', 'tickera-event-ticketing-system' ),
                "EEK" => __( 'EEK - Estonian Kroon', 'tickera-event-ticketing-system' ),
                "ETB" => __( 'ETB - Ethiopian Birr', 'tickera-event-ticketing-system' ),
                "EUR" => __( 'EUR - European Union Euro', 'tickera-event-ticketing-system' ),
                "FKP" => __( 'FKP - Falkland Islands Pound', 'tickera-event-ticketing-system' ),
                "FJD" => __( 'FJD - Fiji Dollar', 'tickera-event-ticketing-system' ),
                "CDF" => __( 'CDF - Franc Congolais', 'tickera-event-ticketing-system' ),
                "GMD" => __( 'GMD - Gambian Delasi', 'tickera-event-ticketing-system' ),
                "GEL" => __( 'GEL - Georgian Lari', 'tickera-event-ticketing-system' ),
                "GHS" => __( 'GHS - Ghanan Cedi', 'tickera-event-ticketing-system' ),
                "GIP" => __( 'GIP - Gibraltar Pound', 'tickera-event-ticketing-system' ),
                "GTQ" => __( 'GTQ - Guatemala Quetzal', 'tickera-event-ticketing-system' ),
                "GNF" => __( 'GNF - Guinea Franc', 'tickera-event-ticketing-system' ),
                "GWP" => __( 'GWP - Guinea-Bissau Peso', 'tickera-event-ticketing-system' ),
                "GYD" => __( 'GYD - Guyanese dollar', 'tickera-event-ticketing-system' ),
                "HTG" => __( 'HTG - Haitian Gourde', 'tickera-event-ticketing-system' ),
                "HNL" => __( 'HNL - Honduras Lempira', 'tickera-event-ticketing-system' ),
                "HKD" => __( 'HKD - Hong Kong Dollar', 'tickera-event-ticketing-system' ),
                "HUF" => __( 'HUF - Hungarian Forint', 'tickera-event-ticketing-system' ),
                "ISK" => __( 'ISK - Iceland Krona', 'tickera-event-ticketing-system' ),
                "INR" => __( 'INR - Indian Rupee', 'tickera-event-ticketing-system' ),
                "IDR" => __( 'IDR - Indonesian Rupiah', 'tickera-event-ticketing-system' ),
                "IRR" => __( 'IRR - Iranian Rial', 'tickera-event-ticketing-system' ),
                "IQD" => __( 'IQD - Iraqi Dinar', 'tickera-event-ticketing-system' ),
                "ILS" => __( 'ILS - Israeli shekel', 'tickera-event-ticketing-system' ),
                "JMD" => __( 'JMD - Jamaican Dollar', 'tickera-event-ticketing-system' ),
                "JPY" => __( 'JPY - Japanese Yen', 'tickera-event-ticketing-system' ),
                "JOD" => __( 'JOD - Jordanian Dinar', 'tickera-event-ticketing-system' ),
                "KZT" => __( 'KZT - Kazakhstan Tenge', 'tickera-event-ticketing-system' ),
                "KES" => __( 'KES - Kenyan Shilling', 'tickera-event-ticketing-system' ),
                "KWD" => __( 'KWD - Kuwaiti Dinar', 'tickera-event-ticketing-system' ),
                "AOA" => __( 'AOA - Kwanza', 'tickera-event-ticketing-system' ),
                "KGS" => __( 'KGS - Kyrgyzstan Som', 'tickera-event-ticketing-system' ),
                "KIP" => __( 'KIP - Laos Kip', 'tickera-event-ticketing-system' ),
                "LAK" => __( 'LAK - Laosian kip', 'tickera-event-ticketing-system' ),
                "LVL" => __( 'LVL - Latvia Lat', 'tickera-event-ticketing-system' ),
                "LBP" => __( 'LBP - Lebanese Pound', 'tickera-event-ticketing-system' ),
                "LRD" => __( 'LRD - Liberian Dollar', 'tickera-event-ticketing-system' ),
                "LYD" => __( 'LYD - Libyan Dinar', 'tickera-event-ticketing-system' ),
                "LTL" => __( 'LTL - Lithuania Litas', 'tickera-event-ticketing-system' ),
                "LSL" => __( 'LSL - Loti', 'tickera-event-ticketing-system' ),
                "MOP" => __( 'MOP - Macanese Pataca', 'tickera-event-ticketing-system' ),
                "MOP" => __( 'MOP - Macao', 'tickera-event-ticketing-system' ),
                "MKD" => __( 'MKD - Macedonian Denar', 'tickera-event-ticketing-system' ),
                "MGF" => __( 'MGF - Madagascar Malagasy Franc', 'tickera-event-ticketing-system' ),
                "MGA" => __( 'MGA - Malagasy Ariary', 'tickera-event-ticketing-system' ),
                "MWK" => __( 'MWK - Malawi Kwacha', 'tickera-event-ticketing-system' ),
                "MYR" => __( 'MYR - Malaysia Ringgit', 'tickera-event-ticketing-system' ),
                "MVR" => __( 'MVR - Maldiveres Rufiyaa', 'tickera-event-ticketing-system' ),
                "MTL" => __( 'MTL - Maltese Lira', 'tickera-event-ticketing-system' ),
                "MRO" => __( 'MRO - Mauritanian Ouguiya', 'tickera-event-ticketing-system' ),
                "MUR" => __( 'MUR - Mauritius Rupee', 'tickera-event-ticketing-system' ),
                "MXN" => __( 'MXN - Mexican Peso', 'tickera-event-ticketing-system' ),
                "MDL" => __( 'MDL - Moldova Leu', 'tickera-event-ticketing-system' ),
                "MNT" => __( 'MNT - Mongolia Tugrik', 'tickera-event-ticketing-system' ),
                "MAD" => __( 'MAD - Moroccan Dirham', 'tickera-event-ticketing-system' ),
                "MZM" => __( 'MZM - Mozambique Metical', 'tickera-event-ticketing-system' ),
                "MMK" => __( 'MMK - Myanmar Kyat', 'tickera-event-ticketing-system' ),
                "NAD" => __( 'NAD - Namibia Dollar', 'tickera-event-ticketing-system' ),
                "NPR" => __( 'NPR - Nepalese Rupee', 'tickera-event-ticketing-system' ),
                "ANG" => __( 'ANG - Netherlands Antillean Guilder', 'tickera-event-ticketing-system' ),
                "PGK" => __( 'PGK - New Guinea kina', 'tickera-event-ticketing-system' ),
                "TWD" => __( 'TWD - New Taiwan Dollar', 'tickera-event-ticketing-system' ),
                "TRY" => __( 'TRY - New Turkish Lira', 'tickera-event-ticketing-system' ),
                "NZD" => __( 'NZD - New Zealand Dollar', 'tickera-event-ticketing-system' ),
                "NIO" => __( 'NIO - Nicaraguan Cordoba', 'tickera-event-ticketing-system' ),
                "NGN" => __( 'NGN - Nigeria Naira', 'tickera-event-ticketing-system' ),
                "KPW" => __( 'KPW - North Korea Won', 'tickera-event-ticketing-system' ),
                "NOK" => __( 'NOK - Norway Krone', 'tickera-event-ticketing-system' ),
                "PKR" => __( 'PKR - Pakistan Rupee', 'tickera-event-ticketing-system' ),
                "PAB" => __( 'PAB - Panama Balboa', 'tickera-event-ticketing-system' ),
                "PYG" => __( 'PYG - Paraguayan guarani', 'tickera-event-ticketing-system' ),
                "PEN" => __( 'PEN - Peru Nuevo Sol', 'tickera-event-ticketing-system' ),
                "PHP" => __( 'PHP - Philippine Peso', 'tickera-event-ticketing-system' ),
                "PLN" => __( 'PLN - Poland Zloty', 'tickera-event-ticketing-system' ),
                "QAR" => __( 'QAR - Qatari Rial', 'tickera-event-ticketing-system' ),
                "OMR" => __( 'OMR - Rial Omani', 'tickera-event-ticketing-system' ),
                "RON" => __( 'RON - Romanian leu', 'tickera-event-ticketing-system' ),
                "RUB" => __( 'RUB - Russian Ruble', 'tickera-event-ticketing-system' ),
                "RWF" => __( 'RWF - Rwanda Franc', 'tickera-event-ticketing-system' ),
                "WST" => __( 'WST - Samoan Tala', 'tickera-event-ticketing-system' ),
                "STD" => __( 'STD - Sao Tome &amp;amp; Principe Dobra', 'tickera-event-ticketing-system' ),
                "SAR" => __( 'SAR - Saudi Arabian riyal', 'tickera-event-ticketing-system' ),
                "RSD" => __( 'RSD - Serbian Dinar', 'tickera-event-ticketing-system' ),
                "SCR" => __( 'SCR - Seychelles Rupee', 'tickera-event-ticketing-system' ),
                "SLL" => __( 'SLL - Sierra Leone Leone', 'tickera-event-ticketing-system' ),
                "SGD" => __( 'SGD - Singapore Dollar', 'tickera-event-ticketing-system' ),
                "SKK" => __( 'SKK - Slovak Koruna Euro', 'tickera-event-ticketing-system' ),
                "SIT" => __( 'SIT - Slovenian Tolar', 'tickera-event-ticketing-system' ),
                "SBD" => __( 'SBD - Solomon Islands Dollar', 'tickera-event-ticketing-system' ),
                "SOS" => __( 'SOS - Somalia Shilling', 'tickera-event-ticketing-system' ),
                "ZAR" => __( 'ZAR - South Africa Rand', 'tickera-event-ticketing-system' ),
                "KRW" => __( 'KRW - South Korean Won', 'tickera-event-ticketing-system' ),
                "LKR" => __( 'LKR - Sri Lanka Rupee', 'tickera-event-ticketing-system' ),
                "SHP" => __( 'SHP - St. Helena Pound', 'tickera-event-ticketing-system' ),
                "SDD" => __( 'SDD - Sudanese Dollar', 'tickera-event-ticketing-system' ),
                "SRD" => __( 'SRD - Suriname Dollar', 'tickera-event-ticketing-system' ),
                "SZL" => __( 'SZL - Swaziland Lilangeni', 'tickera-event-ticketing-system' ),
                "SEK" => __( 'SEK - Sweden Krona', 'tickera-event-ticketing-system' ),
                "CHF" => __( 'CHF - Switzerland Franc', 'tickera-event-ticketing-system' ),
                "SYP" => __( 'SYP - Syrian Arab Republic Pound', 'tickera-event-ticketing-system' ),
                "TJS" => __( 'TJS - Tajikistani Somoni', 'tickera-event-ticketing-system' ),
                "TZS" => __( 'TZS - Tanzanian Shilling', 'tickera-event-ticketing-system' ),
                "THB" => __( 'THB - Thailand Baht', 'tickera-event-ticketing-system' ),
                "TOP" => __( 'TOP - Tonga Pa&#x27;anga', 'tickera-event-ticketing-system' ),
                "TTD" => __( 'TTD - Trinidad and Tobago Dollar', 'tickera-event-ticketing-system' ),
                "TMM" => __( 'TMM - Turkmenistan Manat', 'tickera-event-ticketing-system' ),
                "TND" => __( 'TND - Tunisian Dinar', 'tickera-event-ticketing-system' ),
                "UGX" => __( 'UGX - Uganda Shilling', 'tickera-event-ticketing-system' ),
                "UAH" => __( 'UAH - Ukraine Hryvnia', 'tickera-event-ticketing-system' ),
                "AED" => __( 'AED - United Arab Emirates Dirham', 'tickera-event-ticketing-system' ),
                "GBP" => __( 'GBP - United Kingdom Sterling Pound', 'tickera-event-ticketing-system' ),
                "USD" => __( 'USD - United States Dollar', 'tickera-event-ticketing-system' ),
                "UYU" => __( 'UYU - Uruguayo Peso', 'tickera-event-ticketing-system' ),
                "UZS" => __( 'UZS - Uzbekistan Som', 'tickera-event-ticketing-system' ),
                "VUV" => __( 'VUV - Vanuatu Vatu', 'tickera-event-ticketing-system' ),
                "VEF" => __( 'VEF - Venezuela Bolivar Fuerte', 'tickera-event-ticketing-system' ),
                "VND" => __( 'VND - Vietnam Dong', 'tickera-event-ticketing-system' ),
                "XOF" => __( 'XOF - West African CFA Franc BCEAO', 'tickera-event-ticketing-system' ),
                "YER" => __( 'YER - Yemeni Rial', 'tickera-event-ticketing-system' ),
                "ZMK" => __( 'ZMK - Zambian Kwacha', 'tickera-event-ticketing-system' ),
                "ZWD" => __( 'ZWD - Zimbabwean dollar', 'tickera-event-ticketing-system' ),
            );

            $this->currencies = $currencies;

            add_action( 'wp_enqueue_scripts', array( &$this, 'enqueue_scripts' ) );
        }

        /**
         * Load CSS and JS Files
         */
        function enqueue_scripts() {
            if ( $this->is_payment_page() && $this->is_active() ) {
                wp_register_script( 'js-dropin', 'https://js.braintreegateway.com/web/dropin/1.44.1/js/dropin.min.js', array( 'jquery' ) );
                wp_register_script( 'js-braintree-3ds2', plugins_url( '/braintree/braintree.js', __FILE__ ), array( 'jquery' ) );
                wp_enqueue_style( 'css-braintree-3ds2', plugins_url( 'braintree/assets/css/braintree.css', __FILE__ ) );
            }
        }

        /**
         * Generate Payment Form
         * @param $cart
         * @return string|void
         */
        function payment_form( $cart ) {
            global $tc;
            $content = '';
            $content .= '<div id="errors"></div>';
            $content .= '<table id="tbl_braintree" class="tc_cart_billing"><thead><tr><th colspan="2">' . esc_html__( 'Billing Information', 'tickera-event-ticketing-system' ) . '</th></tr></thead>';
            $content .= '<tr>';
            $content .= '<td><label for="billing-first-name">' . esc_html__( 'First Name', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input type="billing-first-name" class="form-control" id="billing-first-name" value="' . esc_attr( $this->buyer_info( 'first_name' ) ) . '"><span id="help-billing-first-name" class="help-block"></span>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="billing-last-name">' . esc_html__( 'Last Name', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input type="billing-last-name" class="form-control" id="billing-last-name" value="' . esc_attr( $this->buyer_info( 'last_name' ) ) . '"><span id="help-billing-last-name" class="help-block"></span></td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="billing-email">' . esc_html__( 'Email address', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input type="billing-email" class="form-control" id="billing-email" value="' . esc_attr( $this->buyer_info( 'email' ) ) . '"><span id="help-billing-email" class="help-block"></span></td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="billing-street-address">' . esc_html__( 'Address Line 1', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input type="billing-street-address" class="form-control" id="billing-street-address"><span id="help-billing-street-address" class="help-block"></span></td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="billing-extended-address">' . esc_html__( 'Address Line 2', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input type="billing-extended-address" class="form-control" id="billing-extended-address"><span id="help-billing-extended-address" class="help-block"></span></td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="billing-country-code">' . esc_html__( 'Country Code', 'tickera-event-ticketing-system' ) . '</label></td><td>';
            $content .= '<select type="billing-country-code" id="billing-country-code" class="form-control"><option></option></select>';
            $content .= '<span id="help-billing-country-code" class="help-block"></span>';
            $content .= '</td></tr>';
            $content .= '<tr>';
            $content .= '<td><label for="billing-region">' . esc_html__( 'Region', 'tickera-event-ticketing-system' ) . '</label></td><td>';
            $content .= '<select type="billing-region" id="billing-region" class="form-control"><option></option></select>';
            $content .= '<span id="help-billing-region" class="help-block"></span>';
            $content .= '</td></tr>';
            $content .= '<tr>';
            $content .= '<td><label for="billing-city">' . esc_html__( 'City', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input type="billing-city" class="form-control" id="billing-city"><span id="help-billing-city" class="help-block"></span></td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="billing-postal-code">' . esc_html__( 'Postal Code', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input type="billing-postal-code" class="form-control" id="billing-postal-code"><span id="help-billing-postal-code" class="help-block"></span></td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td><label for="billing-phone">' . esc_html__( 'Phone Number', 'tickera-event-ticketing-system' ) . '</label></td>';
            $content .= '<td><input type="billing-phone" class="form-control" id="billing-phone" onkeypress="return isNumeric(event)"><span id="help-billing-phone" class="help-block"></span></td>';
            $content .= '</tr>';
            $content .= '</table>';
            $content .= '<div class="input-group pay-group bt-drop-in-container">';
            $content .= '<div class="row">';
            $content .= '<div class="col-md-12"><div id="drop-in"></div></div>';
            $content .= '</div>';
            $content .= '<div class="row">';
            $content .= '<input disabled id="pay-btn-3ds2" class="btn btn-success" type="submit" value="' . esc_attr__( 'Loading...', 'tickera-event-ticketing-system' ) . '" style="display:none;">';
            $content .= '</div>';
            $content .= '</div>';
            $content .= '<div id="braintree_overlay" style="display:none;"><img title="' . esc_attr__( 'Loading...', 'tickera-event-ticketing-system' ) . '" src="' . esc_url( plugins_url( 'braintree/assets/images/loading.gif', __FILE__ ) ) . '"/></div>';

            // Check Authentication
            try {
                $gateway = $this->braintree_gateway();
                $this->clientToken = $gateway->clientToken()->generate();
            } Catch ( Exception $e ) {
                $tc->session->set( 'tc_gateway_error', sprintf( /* translators: %s: Braintree error message */ __( 'Error: "%s".', 'tickera-event-ticketing-system' ), $e->getMessage() ) );
            }

            // Define Country and Region data
            $country_data = $this->get_country_data();
            $region_data = $this->get_region_data();

            // Pass data to braintree.js script
            $session_cart_info = $tc->session->get( 'cart_info' );
            $total = (float) $session_cart_info[ 'total' ];
            $cart_total = ( intval( $total ) || floatval( $total ) ) ? $total : null;
            $formData = array(
                'country_data' => json_decode( $country_data, true ),
                'region_data' => json_decode( $region_data, true ),
                'token' => $this->clientToken,
                'amount' => $cart_total,
                'callback' => get_site_url() . '/wp-json/tc-braintree-3ds2/v1/callback/',
                'billing_error' => __( 'Field cannot be blank.', 'tickera-event-ticketing-system' ),
                'process_error' => __( 'Invalid transaction: Liability did not shift. Please contact your payment provider for more details', 'tickera-event-ticketing-system' ),
                'verification_success' => __( 'verification success:', 'tickera-event-ticketing-system' ),
                'tokenization_error' => __( 'Tokenization error: ', 'tickera-event-ticketing-system' ),
                'liability_shifted' => __( 'Liability shifted: ', 'tickera-event-ticketing-system' ),
                'component_error' => __( 'component error:', 'tickera-event-ticketing-system' ),
                'processing' => __( 'Processing...', 'tickera-event-ticketing-system' ),
                'pay_now' => __( 'Pay Now', 'tickera-event-ticketing-system' )
            );
            $jsonData = json_encode( $formData );
            $params = array( $jsonData );

            // Load script when payment method is selected
            wp_enqueue_script( 'js-dropin' );
            wp_localize_script( 'js-braintree-3ds2', 'braintree_params', $params );
            wp_enqueue_script( 'js-braintree-3ds2' );

            return $content;
        }

        /**
         * Process Payment and Create Tickera Order
         * @param $cart
         * @return bool|void
         */
        function process_payment( $cart ) {
            global $tc;

            tickera_final_cart_check( $cart );
            $payment_method_nonce = sanitize_text_field( $_POST[ 'nonce' ] );
            $gateway = $this->braintree_gateway();
            $this->save_cart_info();
            $order_id = $tc->generate_order_id();

            $session_cart_info = $tc->session->get( 'cart_info' );
            $total = (float) $session_cart_info[ 'total' ];
            $result = $gateway->transaction()->sale( [
                'amount' => ( intval( $total ) || floatval( $total ) ) ? floatval( $total ) : null, // Set as null to return a Transaction Error
                'orderId' => $order_id,
                'customer' => [
                    'firstName' => $this->buyer_info( 'first_name' ),
                    'lastName' => $this->buyer_info( 'last_name' ),
                    'email' => $this->buyer_info( 'email' )
                ],
                'paymentMethodNonce' => $payment_method_nonce,
                'options' => [ 'submitForSettlement' => (bool) apply_filters( 'tc_braintree_settle_payment', true ) ],
                'channel' => 'Tickera_SP'
            ] );

            $result = tickera_sanitize_array( $result, false, true );

            if ( isset( $result[ 'success' ] ) && $result[ 'success' ] ) {

                $payment_info = array();
                $payment_info[ 'method' ] = __( 'Credit Card', 'tickera-event-ticketing-system' );
                $payment_info[ 'transaction_id' ] = $result[ 'transaction' ][ 'id' ];
                $payment_info = $this->save_payment_info( $payment_info );
                $paid = true;
                $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, $paid );
                return $tc->get_confirmation_slug( true, $order_id );

            } else {
                $tc->session->set( 'tc_gateway_error', sprintf( /* translators: %s: Braintree processing transaction error message */ __( 'Error processing transaction: "%s".', 'tickera-event-ticketing-system' ), sanitize_text_field( $result[ 'message' ] ) ) );
                return false;
            }
        }

        /**
         * Initialize Gateway Connection
         */
        function braintree_gateway() {

            $gateway = new \Braintree\Gateway( [
                'environment' => $this->environment,
                'merchantId' => $this->merchant_key,
                'publicKey' => $this->public_key,
                'privateKey' => $this->private_key
            ] );
            return $gateway;
        }

        /**
         * Generate view for Admin Setting
         * @param $settings
         * @param $visible
         */
        function gateway_admin_settings( $settings, $visible ) {
            global $tc;
            ?>
            <div id="<?php echo esc_attr( $this->plugin_name ); ?>" class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3>
                    <span><?php echo esc_html( sprintf( /* translators: %s: Braintree Payment Gateway admin name */ __( '%s Settings', 'tickera-event-ticketing-system' ), esc_html( $this->admin_name ) ) ); ?></span>
                    <span class="description"><?php esc_html_e( 'Sell your tickets via Braintree 3DS and accept credit and debit cards (Visa, MasterCard, AmEx, Discover, JCB, Maestro and UnionPay)', 'tickera-event-ticketing-system' ) ?></span>
                </h3>
                <div class="inside">
                    <?php
                    $fields = array(
                        'is_ssl' => array(
                            'title' => __( 'Mode', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                '0' => __( 'Sandbox / Test', 'tickera-event-ticketing-system' ),
                                '1' => __( 'Live (Force SSL)', 'tickera-event-ticketing-system' )
                            ),
                            'default' => '0',
                        ),
                        'merchant_key' => array(
                            'title' => __( 'Merchant key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'private_key' => array(
                            'title' => __( 'Private key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'public_key' => array(
                            'title' => __( 'Public key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'cse_key' => array(
                            'title' => __( 'CSE key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'currency' => array(
                            'title' => __( 'Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->currencies,
                            'default' => 'USD',
                        ),
                    );

                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name );
                    ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>
                </div>
            </div>
            <?php
        }
    }

    \Tickera\tickera_register_gateway_plugin( 'Tickera\Gateway\TC_Gateway_Braintree_3ds2', 'braintree_3ds2', __( 'Braintree 3DS2', 'tickera-event-ticketing-system' ) );
}