import { useCallback, useEffect, useState } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { debounce } from "lodash";

const blockNamespace = 'tickera/checkout-fields';

export const SelectFieldControl = ( prop ) => {

    let field = prop.field,
        pageLoaded = prop.loaded,
        category = prop.category,
        item = prop.item,
        i = prop.i;

    prop = prop.data.prop;

    /*
     * setExtensionData will update the wc/store/checkout data store with the values supplied.
     * It can be used to pass data from the client to the server when submitting the checkout form.
     */
    const { setExtensionData } = prop.checkoutExtensionData;

    /*
     * Debounce the setExtensionData function to avoid
     * multiple calls to the API when rapidly changing options.
     */
    const debouncedSetExtensionData = useCallback(
        debounce((namespace, key, value) => {
            setExtensionData(namespace, key, value);
        }, 1000),
        [setExtensionData]
    );

    const key = ( 'owner' == category )
        ? item.id + '_' + i + '_' + field.field_name + '_post_meta'
        : 'buyer_data_' + field.field_name + '_post_meta';

    /**
     * Handles fields validation
     */
    const validationErrorId = key + '-validator';
    const { setValidationErrors, clearValidationError } = useDispatch( 'wc/store/validation' );
    const validationError = useSelect((select) => {
        const store = select('wc/store/validation');
        return store.getValidationError( validationErrorId );
    });

    /**
     * Handles changes in fields value.
     * Exclude Label field.
     */
    const [ fieldValue, setFieldValue ] = useState( field.field_default_value );

    useEffect(() => {

        if ( field.required ) {

            // Set Validation Errors. Initially hidden until page has been fully loaded
            let hideError = ! pageLoaded ? true : false;
            setValidationErrors( {
                [ validationErrorId ]: {
                    message: tc_woobridge_block.please_enter_a_valid + ' ' + field.field_title.toLowerCase(),
                    hidden: hideError,
                },
            } );

            // Clear error with the following conditions
            if ( pageLoaded && fieldValue.trim() ) {
                clearValidationError( validationErrorId );
            }
        }

        // Update in tickera/checkout-fields extended data
        setExtensionData( blockNamespace, key, fieldValue );

        // Ensures setExtensionData is being called one at a time.
        debouncedSetExtensionData( blockNamespace, key, fieldValue );

    }, [ setExtensionData, fieldValue, setValidationErrors, clearValidationError ] );

    let fieldValues = ( typeof field.field_values !== 'undefined' ) ? field.field_values.split( ',' ) : [],
        error = ( ! validationError?.hidden ) ? validationError?.message : false;

    return (
        <div className={ 'tc-wb-block-component tc-wb-block-component-' + field.field_name + ( error ? ' has-error' : '' )}>
            <div className={ 'wc-blocks-components-select tc-wb-block-combobox is-active' }>
                <div className={ 'wc-blocks-components-select__container' }>
                    <label htmlFor={ field.field_name } className="wc-blocks-components-select__label">{ ( field.field_placeholder ) ? field.field_placeholder : field.field_title }</label>
                    <select size="1" className="wc-blocks-components-select__select" id={ field.field_name } aria-invalid="false" autoComplete="off" onChange={ ( e ) => setFieldValue( e.target.value ) }>
                        { ( typeof field.field_default_value !== 'undefined' && '' !== field.field_default_value.trim() ) ? '' : <option value=''></option> };
                        {
                            ( ( field, value ) => {
                                let options = [];
                                fieldValues.forEach( function( optionValue, index ) {
                                    options.push( <option value={optionValue} selected={ value === optionValue || ( '' === optionValue && value === field.field_default_value ) }>{optionValue}</option> );
                                })
                                return options;
                            } )( field, fieldValue )
                        }
                    </select>
                </div>
            </div>
            { error ? <div className='wc-block-components-validation-error'><p>{ error }</p></div> : '' }
            { ( field.field_description ) ? <div className='wc-block-components-checkout-step__description description'>{ field.field_description }</div> : '' }
        </div>
    );
}
