<?php
/**
 * Sureforms Signature Markup Class file.
 *
 * @package sureforms pro and business.
 * @since 1.6.0
 */

namespace SRFM_Pro\Inc\Pro\Signature;

use SRFM\Inc\Helper;
use SRFM_Pro\Inc\Fields\Base;
use SRFM_Pro\Inc\Helper as Pro_Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Sureforms Signature Markup Class.
 *
 * @since 1.6.0
 */
class Signature_Markup extends Base {
	/**
	 * Pen color for the signature.
	 *
	 * @var string
	 * @since 1.6.0
	 */
	protected $pen_color;

	/**
	 * Signature file size limit.
	 *
	 * @var int
	 * @since 1.6.0
	 */
	protected $signature_file_size_limit;

	/**
	 * Initialize the properties based on block attributes.
	 *
	 * @param array<mixed> $attributes Block attributes.
	 * @since 1.6.0
	 */
	public function __construct( $attributes ) {
		$this->set_properties( $attributes );
		$this->set_error_msg( 'srfm_signature_block_required_text', $attributes );
		$this->set_input_label( __( 'Signature', 'sureforms-pro' ) );
		$this->slug = 'signature';
		$this->set_unique_slug();
		$this->set_field_name( $this->unique_slug );
		$this->set_markup_properties( $this->input_label . '-hidden', true );
		$this->set_aria_described_by();

		$this->pen_color                 = $attributes['penColor'] ?? '#434343';
		$this->signature_file_size_limit = $attributes['signatureFileSizeLimit'] ?? 1;
	}

	/**
	 * Render the sureforms Signature field
	 *
	 * @since 1.6.0
	 * @return string|bool
	 */
	public function markup() {
		// Random ID to fix issues when 2 same forms are embedded on the same page.
		$random_id = $this->unique_slug . '-' . wp_rand( 1000, 9999 );

		ob_start(); ?>
		<div data-block-id="<?php echo esc_attr( $this->block_id ); ?>" class="srfm-block-single srfm-block srfm-<?php echo esc_attr( $this->slug ); ?>-block <?php echo esc_attr( $this->block_width ); ?><?php echo esc_attr( $this->classname ); ?> <?php echo esc_attr( $this->conditional_class ); ?>">
		<?php echo wp_kses_post( '<label id="srfm-label-' . esc_attr( $this->block_id ) . '" for="' . $random_id . '-hidden" class="srfm-block-label">' . wp_kses_post( $this->label ) . ( $this->required ? '<span class="srfm-required" aria-hidden="true"> *</span>' : '' ) . '</label>' ); ?>
			<?php echo wp_kses_post( $this->help_markup ); ?>
			<div class="srfm-block-wrap">
				<input type="file" id="<?php echo esc_attr( $random_id ); ?>-file" class="srfm-signature-upload" accept="image/*" data-max-file-size="<?php echo esc_attr( Helper::get_string_value( $this->signature_file_size_limit ) ); ?>" aria-hidden="true">
				<input tabindex="0" type="text" name="<?php echo esc_attr( $this->field_name ); ?>" class="srfm-input-<?php echo esc_attr( $this->slug ); ?>-hidden screen-reader-text" id="<?php echo esc_attr( $random_id ); ?>-hidden" data-required="<?php echo esc_attr( $this->data_require_attr ); ?>" aria-required="<?php echo esc_attr( $this->data_require_attr ); ?>" <?php echo wp_kses_post( $this->placeholder_attr ); ?> aria-describedby="<?php echo esc_attr( $this->aria_described_by ); ?>" autocomplete="off" readonly>
				<canvas class="srfm-input-common srfm-input-<?php echo esc_attr( $this->slug ); ?>" id="<?php echo esc_attr( $random_id ); ?>"
				data-required="<?php echo esc_attr( $this->data_require_attr ); ?>" data-pen-color=<?php echo wp_kses_post( $this->pen_color ); ?> <?php echo wp_kses_post( $this->placeholder_attr ); ?>><?php echo esc_html( $this->default ); ?></canvas>
				<div class="srfm-signature-buttons">
					<span id="<?php echo esc_attr( $random_id ); ?>-clear" tabindex="0" role="button" aria-label="<?php esc_attr_e( 'Clear Signature', 'sureforms-pro' ); ?>">
						<?php
							echo Pro_Helper::fetch_pro_svg( //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Ignored to render svg
								'clear',
								'srfm-' . esc_attr( $this->slug ) . '-icon srfm-input-icon',
								' aria-hidden="true"',
							);
						?>
					</span>
					<label for="<?php echo esc_attr( $random_id ); ?>-file" id="<?php echo esc_attr( $random_id ); ?>-upload" tabindex="0" role="button" aria-label="<?php esc_attr_e( 'Upload Signature', 'sureforms-pro' ); ?>">
						<?php
							echo Pro_Helper::fetch_pro_svg( //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Ignored to render svg
								'signatureUpload',
								'srfm-' . esc_attr( $this->slug ) . '-icon srfm-input-icon',
							);
						?>
					</label>
				</div>
			</div>
			<div class="srfm-error-wrap">
				<?php echo wp_kses_post( $this->error_msg_markup ); ?>
			</div>
		</div>
		<?php
		$markup = ob_get_clean();

		return apply_filters(
			'srfm_block_field_markup',
			$markup,
			[
				'slug'       => $this->slug,
				'is_editing' => $this->is_editing,
				'field_name' => $this->field_name,
				'attributes' => $this->attributes,
			]
		);
	}
}
