<?php
/**
 * Class to add markup for the fields in the edit entry mode.
 *
 * @since 1.3.0
 * @package sureforms-pro
 */

namespace SRFM_Pro\Inc\Extensions;

use SRFM\Inc\Helper;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Class to add markup for the fields in the edit entry mode.
 *
 * @since s.x.x
 */
class Edit_Entry_Fields_Markup {
	/**
	 * Initialize the class.
	 *
	 * @since 1.3.0
	 * @return void
	 */
	public static function init() {
		add_filter( 'srfm_block_field_markup', [ self::class, 'render_field_markup' ], 10, 2 );
	}

	/**
	 * Render the field markup.
	 *
	 * @param string              $markup The field markup.
	 * @param array<string,mixed> $args The field arguments.
	 * @since 1.3.0
	 * @return string|false
	 */
	public static function render_field_markup( $markup, $args ) {
		if ( true !== $args['is_editing'] ) {
			return $markup;
		}

		switch ( $args['slug'] ) {
			case 'email':
				return self::email( $args );
			case 'gdpr':
				return self::gdpr( $args );
			case 'dropdown':
				return self::dropdown( $args );
			case 'checkbox':
				return self::checkbox( $args );
			case 'multi-choice':
				return self::multi_choice( $args );

			// Pro fields.
			case 'password':
			case 'hidden':
				return ''; // We don't need to render hidden field in edit mode.
			case 'upload':
				return self::upload( $args );
			case 'rating':
				return self::rating( $args );
			case 'slider':
				return self::slider( $args );
			case 'signature':
				return self::signature( $args );
			default:
				return $markup;
		}
	}

	/**
	 * Render the Email field markup.
	 *
	 * @param array<string,mixed> $args The field arguments.
	 * @since 1.3.0
	 * @return string|false
	 */
	protected static function email( $args ) {
		/**
		 * The field attributes.
		 *
		 * @var array<string,string> $attributes The field attributes.
		 */
		$attributes = $args['attributes'];
		$slug       = Helper::get_string_value( $args['slug'] );
		$default    = Helper::get_string_value( $attributes['defaultValue'] ?? '' );

		ob_start();
		?>
		<div data-block-id="<?php echo esc_attr( $attributes['block_id'] ); ?>" class="srfm-block-single srfm-block srfm-<?php echo esc_attr( $slug ); ?>-block-wrap">
			<div class="srfm-<?php echo esc_attr( $slug ); ?>-block srf-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] ); ?>-block">
				<legend class="srfm-block-legend">
					<?php echo wp_kses_post( Helper::generate_common_form_markup( $attributes['formId'], 'label', $attributes['label'], $slug, $attributes['block_id'] ) ); ?>
				</legend>
				<div class="srfm-block-wrap">
					<input class="srfm-input-common srfm-input-<?php echo esc_attr( $slug ); ?>" type="email" name="<?php echo esc_attr( Helper::get_string_value( $args['field_name'] ) ); ?>" value="<?php echo esc_attr( $default ); ?>"/>
				</div>
			</div>
		</div>
		<script>
			(function() {
				// Validate email address.
				const emailInput = document.querySelector('input[name="<?php echo esc_attr( Helper::get_string_value( $args['field_name'] ) ); ?>"]');
				const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;

				emailInput.addEventListener('input', () => {
					if (!emailRegex.test(emailInput.value)) {
						emailInput.setCustomValidity("<?php esc_html_e( 'Please enter a valid email address.', 'sureforms-pro' ); ?>");
					} else {
						emailInput.setCustomValidity("");
					}
				});
			}());
		</script>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render the GDPR field markup.
	 *
	 * @param array<string,mixed> $args The field arguments.
	 * @since 1.3.0
	 * @return string|false
	 */
	protected static function gdpr( $args ) {
		/**
		 * The field attributes.
		 *
		 * @var array<string,string> $attributes The field attributes.
		 */
		$attributes = $args['attributes'];
		$slug       = Helper::get_string_value( $args['slug'] );
		$field_name = Helper::get_string_value( $args['field_name'] );
		$default    = Helper::get_string_value( $attributes['defaultValue'] ?? '' );

		$label_random_id = 'srfm-' . $slug . '-' . wp_rand();
		ob_start();
		?>
			<div data-block-id="<?php echo esc_attr( $attributes['block_id'] ); ?>" class="srfm-block-single srfm-block srfm-<?php echo esc_attr( $slug ); ?>-block srf-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] ); ?>-block">
				<div class="srfm-block-wrap">
					<input
						class="srfm-input-common srfm-input-<?php echo esc_attr( $slug ); ?>"
						id="<?php echo esc_attr( $label_random_id ); ?>"
						name="srfm-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] ); ?><?php echo esc_attr( $field_name ); ?>"
						type="checkbox" <?php checked( 'on', $default ); ?>
					/>
					<label class="srfm-cbx" for="<?php echo esc_attr( $label_random_id ); ?>">
						<span class="srfm-span-wrap srfm-block-label"><?php echo wp_kses_post( $attributes['label'] ); ?>
					</label>
				</div>
			</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render the checkbox field markup.
	 *
	 * @param array<string,mixed> $args The field arguments.
	 * @since 1.3.0
	 * @return string|false
	 */
	protected static function checkbox( $args ) {
		/**
		 * The field attributes.
		 *
		 * @var array<string,string> $attributes The field attributes.
		 */
		$attributes = $args['attributes'];
		$slug       = Helper::get_string_value( $args['slug'] );
		$field_name = Helper::get_string_value( $args['field_name'] );
		$default    = Helper::get_string_value( $attributes['defaultValue'] ?? '' );

		$label_random_id = 'srfm-' . $slug . '-' . wp_rand();

		ob_start();
		?>
			<div data-block-id="<?php echo esc_attr( $attributes['block_id'] ); ?>" class="srfm-block-single srfm-block srfm-<?php echo esc_attr( $slug ); ?>-block srf-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] ); ?>-block">
				<div class="srfm-block-wrap">
					<input type="hidden" name="srfm-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] ); ?><?php echo esc_attr( $field_name ); ?>" value="" />
					<input
						class="srfm-input-common srfm-input-<?php echo esc_attr( $slug ); ?>"
						id="<?php echo esc_attr( $label_random_id ); ?>"
						name="srfm-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] ); ?><?php echo esc_attr( $field_name ); ?>"
						type="checkbox"
						<?php checked( 'on', $default ); ?>
					/>
					<label class="srfm-cbx" for="<?php echo esc_attr( $label_random_id ); ?>">
						<span class="srfm-span-wrap srfm-block-label"><?php echo wp_kses_post( $attributes['label'] ); ?></span>
					</label>
				</div>
			</div>
		<?php

		return ob_get_clean();
	}

	/**
	 * Render the dropdown field markup.
	 *
	 * @param array<string,mixed> $args The field arguments.
	 * @since 1.3.0
	 * @return string|false
	 */
	protected static function dropdown( $args ) {
		/**
		 * The field attributes.
		 *
		 * @var array<string,string> $attributes The field attributes.
		 */
		$attributes = $args['attributes'];
		$slug       = Helper::get_string_value( $args['slug'] );
		$field_name = Helper::get_string_value( $args['field_name'] );
		$default    = Helper::get_string_value( $attributes['defaultValue'] ?? '' );
		$options    = Helper::get_array_value( $attributes['options'] );

		$multi_select_attr = ! empty( $attributes['multiSelect'] ) ? 'true' : 'false';

		$name = "srfm-{$slug}-{$attributes['block_id']}{$field_name}";

		if ( 'true' === $multi_select_attr ) {
			$name = "srfm-{$slug}-{$attributes['block_id']}{$field_name}[]";
		}
		ob_start();
		?>
			<div data-block-id="<?php echo esc_attr( $attributes['block_id'] ); ?>" class="srfm-block-single srfm-block srfm-<?php echo esc_attr( $slug ); ?>-block srf-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] ); ?>-block">
				<fieldset>
					<legend class="srfm-block-legend">
						<?php echo wp_kses_post( Helper::generate_common_form_markup( $attributes['formId'], 'label', $attributes['label'], $slug, $attributes['block_id'] ) ); ?>
					</legend>
					<div class="srfm-block-wrap srfm-dropdown-common-wrap">
					<?php
					if ( is_array( $options ) ) {
						?>
						<select
							<?php echo 'true' === $multi_select_attr ? 'multiple' : ''; ?>
							class="srfm-dropdown-common srfm-<?php echo esc_attr( $slug ); ?>-input"
							name="<?php echo esc_attr( $name ); ?>">
							<?php
							if ( 'true' !== $multi_select_attr ) {
								?>
								<option class="srfm-dropdown-placeholder" value="" selected><?php echo esc_html( Helper::get_string_value( $attributes['placeholder'] ) ); ?></option>
								<?php
							}
							?>
							<?php foreach ( $options as $option ) { ?>
								<?php
									$label = $option['label'] ?? '';

								if ( 'true' === $multi_select_attr ) {
									$values = explode( ' | ', $default );
									?>
										<option <?php selected( in_array( $label, $values, true ), true ); ?> value="<?php echo esc_attr( $label ); ?>"><?php echo esc_html( $label ); ?></option>
										<?php
								} else {
									?>
										<option <?php selected( $default, $label ); ?> value="<?php echo esc_attr( $label ); ?>"><?php echo esc_html( $label ); ?></option>
										<?php
								}
							}
							?>
						</select>
					<?php } ?>
					</div>
				</fieldset>
			</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render the multi choice field markup.
	 *
	 * @param array<string,mixed> $args The field arguments.
	 * @since 1.3.0
	 * @return string|false
	 */
	protected static function multi_choice( $args ) {
		/**
		 * The field attributes.
		 *
		 * @var array<string,string> $attributes The field attributes.
		 */
		$attributes = $args['attributes'];
		$slug       = Helper::get_string_value( $args['slug'] );
		$default    = Helper::get_string_value( $attributes['defaultValue'] ?? '' );

		$field_name       = $attributes['fieldName'] ?? 'srfm-input-' . $slug . '-' . Helper::get_string_value( $attributes['block_id'] ) . Helper::get_string_value( $args['field_name'] ); // Just in-case if no options are selected and we don't get the field name.
		$single_selection = $attributes['singleSelection'] ?? false;
		$type_attr        = $single_selection ? 'radio' : 'checkbox';
		$options          = $attributes['options'];

		if ( ! $single_selection ) {
			$field_name .= '[]';
		}

		// Convert default string into an array of known options.
		$default_options = explode( ' | ', $default );

		ob_start();
		?>
		<div data-block-id="<?php echo esc_attr( $attributes['block_id'] ); ?>" class="srfm-block-single srfm-block srfm-<?php echo esc_attr( $slug ); ?>-block srf-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] ); ?>-block">
			<fieldset>
				<legend class="srfm-block-legend">
				<?php echo wp_kses_post( Helper::generate_common_form_markup( $attributes['formId'], 'label', $attributes['label'], $slug, $attributes['block_id'] ) ); ?>
				</legend>
					<?php if ( is_array( $options ) ) { ?>
						<div class="srfm-block-wrap">
							<?php foreach ( $options as $i => $option ) { ?>
								<div class="srfm-<?php echo esc_attr( $slug ); ?>-single">
									<label for="srfm-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] . '-' . $i ); ?>">
										<input value="<?php echo esc_attr( $option['optionTitle'] ); ?>" <?php checked( true, in_array( $option['optionTitle'], $default_options, true ) ); ?> type="<?php echo esc_attr( $type_attr ); ?>" id="srfm-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] . '-' . $i ); ?>" class="srfm-input-<?php echo esc_attr( $slug ); ?>-single" name="<?php echo esc_attr( $field_name ); ?>" />
										<span><?php echo isset( $option['optionTitle'] ) ? esc_html( $option['optionTitle'] ) : ''; ?></span>
									</label>
								</div>
							<?php } ?>
						</div>
					<?php } ?>
			</fieldset>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render the upload field markup.
	 *
	 * @param array<string,mixed> $args The field arguments.
	 * @since 1.3.0
	 * @return string|false
	 */
	protected static function upload( $args ) {
		/**
		 * The field attributes.
		 *
		 * @var array<string,string> $attributes The field attributes.
		 */
		$attributes = $args['attributes'];
		$slug       = Helper::get_string_value( $args['slug'] );
		$field_name = $attributes['fieldName'] ?? '';

		$files = array_filter(
			Helper::get_array_value( $attributes['defaultValue'] ?? [] ),
			static function( $file ) {
				return file_exists( Helper::convert_fileurl_to_filepath( urldecode( Helper::get_string_value( $file ) ) ) );
			}
		);
		ob_start();
		?>
		<div data-block-id="<?php echo esc_attr( $attributes['block_id'] ); ?>" class="srfm-block-single srfm-block srfm-<?php echo esc_attr( $slug ); ?>-block srf-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] ); ?>-block">
			<?php echo wp_kses_post( Helper::generate_common_form_markup( $attributes['formId'], 'label', $attributes['label'], $slug, $attributes['block_id'] ) ); ?>
			<div class="srfm-block-wrap">
				<div class="srfm-<?php echo esc_attr( $slug ); ?>-wrap">
					<?php
					if ( ! empty( $files ) && is_array( $files ) ) {
						foreach ( $files as $file ) {
							$file_url = urldecode( strval( $file ) );
							$fileinfo = pathinfo( $file_url );
							?>
							<div class="srfm-uploaded-file">
								<p>
									<a href="<?php echo esc_url( $file_url ); ?>" target="_blank"><?php echo esc_html( $fileinfo['basename'] ); ?></a>
								</p>
								<span class="file-actions">
									<input type="hidden" name="<?php echo esc_attr( $field_name ); ?>[]" value="<?php echo esc_url( $file_url ); ?>">
									<a target="_blank" href="<?php echo esc_url( $file_url ); ?>" download="<?php echo esc_attr( $fileinfo['basename'] ); ?>"><?php esc_html_e( 'Download', 'sureforms-pro' ); ?></a>
									|
									<a href="javascript:void(0);" data-fileurl-hash="<?php echo esc_attr( md5( $file_url ) ); ?>" data-fileurl="<?php echo esc_url( $file_url ); ?>" class="srfm-file-delete-btn"><?php esc_html_e( 'Delete', 'sureforms-pro' ); ?></a>
								</span>
							</div>
							<?php
						}
					} else {
						?>
						<p class="srfm-uploads-no-files"><?php esc_html_e( 'No files available.', 'sureforms-pro' ); ?></p>
						<?php
					}
					?>
				</div>
			</div>
		</div>
		<script>
			(function(){
				const container = document.querySelector( '.srf-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] ); ?>-block' );
				const uploadWrap = container.querySelector( '.srfm-upload-wrap' );
				const fileDeleteBtns = container.querySelectorAll( '.file-actions .srfm-file-delete-btn' );

				fileDeleteBtns.forEach((fileDeleteBtn) => {
					fileDeleteBtn.addEventListener('click', async function(e) {
						e.preventDefault();

						if ( ! confirm( '<?php echo esc_js( __( 'Are you sure you want to delete this file?', 'sureforms-pro' ) ); ?>' ) ) {
							return;
						}
						this.innerText = '<?php echo esc_js( __( 'Deleting...', 'sureforms-pro' ) ); ?>';

						const filePreviewEl = document.querySelector( '.file-card[data-fileurl-hash="' + this.getAttribute( 'data-fileurl-hash' ) + '"]' );

						const wrapper = this.parentElement.parentElement;
						const formData = new FormData();

						formData.append( 'file', this.getAttribute( 'data-fileurl' ) );
						formData.append( 'entryID', <?php echo esc_js( strval( $attributes['entryID'] ) ); ?> );

						await fetch( srfm_pro_entries.ajaxURLs.deleteFile, {
							method: 'POST',
							body: formData,
						} )
						.then( ( res ) => res.json() )
						.then( () => {
							filePreviewEl.remove();
							wrapper.remove();
							if ( uploadWrap.childElementCount === 0 ) {
								uploadWrap.innerHTML = '<p class="srfm-uploads-no-files"><?php echo esc_js( __( 'No files available.', 'sureforms-pro' ) ); ?></p>';
							}
						} );
					});
				});
			}());
		</script>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render the signature field markup.
	 *
	 * @param array<string,mixed> $args The field arguments.
	 * @since 1.6.0
	 * @return string|false
	 */
	protected static function signature( $args ) {
		/**
		 * The field attributes.
		 *
		 * @var array<string,string> $attributes The field attributes.
		 */
		$attributes = $args['attributes'];
		$slug       = Helper::get_string_value( $args['slug'] );

		$file_url   = Helper::get_string_value( $attributes['defaultValue'] ?? '' );
		$field_name = $attributes['fieldName'] ?? '';

		ob_start();
		?>
			<div data-block-id="<?php echo esc_attr( $attributes['block_id'] ); ?>" class="srfm-block-single srfm-block srfm-<?php echo esc_attr( $slug ); ?>-block srf-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] ); ?>-block">
				<input type="hidden" name="<?php echo esc_attr( $field_name ); ?>" value="<?php echo esc_url( $file_url ); ?>">
				<?php esc_html_e( 'Signature cannot be edited.', 'sureforms-pro' ); ?>
			</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render the rating field markup.
	 *
	 * @param array<string,mixed> $args The field arguments.
	 * @since 1.3.0
	 * @return string|false
	 */
	protected static function rating( $args ) {
		$slug       = Helper::get_string_value( $args['slug'] );
		$attributes = Helper::get_array_value( $args['attributes'] );
		$field_name = Helper::get_string_value( $args['field_name'] );

		$default = $attributes['defaultValue'] ?? '';

		$input_id    = "srfm-rating-{$attributes['block_id']}";
		$input_name  = "{$input_id}{$field_name}";
		$icon_shape  = $attributes['iconShape'] ?? '';
		$icon_labels = [];

		switch ( $icon_shape ) {
			case 'star':
				$icon_labels = array_map(
					// translators: %d represents the number of stars.
					static fn( $i ) => sprintf( _n( '%d star', '%d stars', $i, 'sureforms-pro' ), $i ),
					range( 1, 5 )
				);
				break;
			case 'smiley':
				$icon_labels = [
					__( 'Poor Smiley', 'sureforms-pro' ),
					__( 'Fair Smiley', 'sureforms-pro' ),
					__( 'Good Smiley', 'sureforms-pro' ),
					__( 'Very Good Smiley', 'sureforms-pro' ),
					__( 'Excellent Smiley', 'sureforms-pro' ),
				];
				break;
			case 'thumb':
				$icon_labels = [
					__( 'Thumbs Up', 'sureforms-pro' ),
					__( 'Thumbs Down', 'sureforms-pro' ),
				];
				break;
		}

		$wrapper_class = "srfm-block-single srfm-block srfm-rating-block srf-rating-{$attributes['block_id']}-block";

		ob_start();
		?>
		<div data-block-id="<?php echo esc_attr( $attributes['block_id'] ); ?>" class="<?php echo esc_attr( $wrapper_class ); ?>">
		<?php echo wp_kses_post( Helper::generate_common_form_markup( $attributes['formId'], 'label', $attributes['label'], $slug, $attributes['block_id'] ) ); ?>
			<div class="srfm-block-wrap">
				<?php
				foreach ( $icon_labels as $key => $label ) {
					?>
					<label>
						<input <?php checked( absint( $default ), $key + 1 ); ?> type="radio" value="<?php echo absint( $key ) + 1; ?>" name="<?php echo esc_attr( $input_name ); ?>">
						<span><?php echo esc_html( $label ); ?></span>
					</label>
					<?php
				}
				?>
			</div>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render the slider field markup.
	 *
	 * @param array<string,mixed> $args The field arguments.
	 * @since 1.3.0
	 * @return string|false
	 */
	protected static function slider( $args ) {
		$slug        = Helper::get_string_value( $args['slug'] );
		$attributes  = Helper::get_array_value( $args['attributes'] );
		$field_name  = Helper::get_string_value( $args['field_name'] );
		$unique_slug = Helper::get_string_value( $args['unique_slug'] );
		$step        = Helper::get_integer_value( $attributes['step'] ) ?? 1;
		$min         = Helper::get_integer_value( $attributes['min'] ) ?? 1;
		$max         = Helper::get_integer_value( $attributes['max'] ) ?? 1;
		$default     = ! empty( $attributes['defaultValue'] ) ? $attributes['defaultValue'] : '';
		$slider_type = $attributes['type'] ?? '';

		/**
		 * The field options.
		 *
		 * @var array<string,array<string>> $options The field options.
		 */
		$options = Helper::get_array_value( $attributes['options'] );

		ob_start();
		?>
		<div data-block-id="<?php echo esc_attr( $attributes['block_id'] ); ?>" class="srfm-block-single srfm-block srfm-<?php echo esc_attr( $slug ); ?>-block srfm-<?php echo esc_attr( $slug ); ?>-<?php echo esc_attr( $attributes['block_id'] ); ?>-block">
		<?php echo wp_kses_post( Helper::generate_common_form_markup( $attributes['formId'], 'label', $attributes['label'], $slug, $attributes['block_id'] ) ); ?>
			<div class="srfm-block-wrap">
				<?php if ( 'number' === $slider_type ) { ?>
					<input
						class="srfm-input-<?php echo esc_attr( $slug ); ?>"
						name="<?php echo esc_attr( $field_name ); ?>"
						id="<?php echo esc_attr( $unique_slug ); ?>"
						type="range"
						tabindex="0"
						value="<?php echo esc_attr( $default ? Helper::get_string_value( $default ) : Helper::get_string_value( $min ) ); ?>"
						max="<?php echo esc_attr( Helper::get_string_value( $max ) ); ?>"
						min="<?php echo esc_attr( Helper::get_string_value( $min ) ); ?>"
						step="<?php echo esc_attr( Helper::get_string_value( $step ) ); ?>"
					>
				<?php } else { ?>
					<div class="srfm-text-slider">
				<?php } ?>
				<?php if ( 'text' === $slider_type ) { ?>
					<div class="srfm-text-slider-options">
						<?php foreach ( $options as $index => $option ) { ?>
							<label class="srfm-text-slider-option" style="--index:<?php echo esc_attr( $index ); ?>;">
								<input type="radio" name="<?php echo esc_attr( $field_name ); ?>" value="<?php echo esc_attr( $option['label'] ); ?>" <?php echo $option['label'] === $default ? 'checked' : ''; ?>>
								<span><?php echo esc_html( $option['label'] ); ?></span>
							</label>
						<?php } ?>
					</div>
					<span aria-live="assertive" aria-atomic="true" id="srfm-slider-a11y-text" class="screen-reader-text"></span>
				</div>
				<?php } ?>

				<?php
				if ( 'number' === $slider_type ) {
					?>
					<span id="<?php echo esc_attr( $unique_slug ); ?>-slider-value">
						<?php echo esc_attr( $default ? Helper::get_string_value( $default ) : Helper::get_string_value( $min ) ); ?>
					</span>
					<?php
				}
				?>
			</div>
		</div>
		<?php
		if ( 'number' === $slider_type ) {
			?>
			<script>
				(function() {
					window.addEventListener('load', function() {
						var defaultValSpan = document.getElementById('<?php echo esc_attr( $unique_slug ); ?>-slider-value');
						document.getElementById('<?php echo esc_attr( $unique_slug ); ?>')
						.addEventListener('input', function() {
							defaultValSpan.innerText = this.value;
						});
					});
				}());
			</script>
			<?php
		}
		return ob_get_clean();
	}
}
