<?php
/**
 * Sureforms Repeater Markup Class file.
 *
 * This file contains the Repeater_Markup class which handles the rendering and functionality
 * of repeater fields in forms. It allows users to dynamically add/remove sets of form fields
 * and manages their display and data handling.
 *
 * @package sureforms-pro
 * @since 1.11.0
 */

namespace SRFM_Pro\Inc\Business\Repeater;

use SRFM_Pro\Inc\Fields\Base;
use SRFM_Pro\Inc\Helper as Pro_Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Sureforms Repeater Markup Class.
 *
 * Handles the rendering and functionality of repeater fields in forms.
 * Extends the Base field class to provide repeater-specific functionality.
 *
 * @since 1.11.0
 */
class Repeater_Markup extends Base {
	/**
	 * Minimum number of items.
	 *
	 * Defines the minimum number of repeater items that must be present.
	 *
	 * @var int
	 * @since 1.11.0
	 */
	protected $min_items;

	/**
	 * Maximum number of items.
	 *
	 * Defines the maximum number of repeater items that can be added.
	 *
	 * @var int
	 * @since 1.11.0
	 */
	protected $max_items;

	/**
	 * Inner block content (template).
	 *
	 * The template content that will be repeated for each repeater item.
	 *
	 * @var string
	 * @since 1.11.0
	 */
	protected $inner_content;

	/**
	 * Initialize the properties based on block attributes.
	 *
	 * Sets up the repeater field properties including labels, content, and limits
	 * based on the provided block attributes.
	 *
	 * @param array<mixed> $attributes Block attributes.
	 * @param string       $content Inner block content.
	 * @since 1.11.0
	 */
	public function __construct( $attributes, $content = '' ) {
		$this->set_properties( $attributes );
		$this->set_error_msg( 'srfm_repeater_block_required_text', $attributes );
		$this->set_input_label( __( 'Repeater Field', 'sureforms-pro' ) );
		$this->slug = 'repeater';
		$this->set_unique_slug();
		$this->set_field_name( $this->unique_slug );
		$this->set_markup_properties( $this->input_label, false );
		$this->set_aria_described_by();

		$this->min_items     = $attributes['minItems'] ?? 1;
		$this->max_items     = $attributes['maxItems'] ?? 100;
		$this->inner_content = $content;
	}

	/**
	 * Render the sureforms Repeater field like address field
	 *
	 * Generates the HTML markup for the repeater field including containers,
	 * templates, and controls for adding/removing items.
	 *
	 * @since 1.11.0
	 * @return string|bool
	 */
	public function markup() {
		// Inner content is the address field.
		if ( empty( $this->inner_content ) ) {
			return '';
		}

		$entry_data = null;
		// If block will render in the entry page that time we need to add the entry data so we can populate the repeater block by the client side.
		// We put the entry data if block is is_editing is true. and entry_data is not null.
		// We will put the data by the wp_json_encode.
		if ( $this->is_editing && ! empty( $this->attributes['entry_data'] ) ) {
			$entry_data = wp_json_encode( $this->attributes['entry_data'] );
		}

		// Random ID to fix issues when 2 same forms are embedded on the same page.
		$random_id = $this->unique_slug . '-' . wp_rand( 1000, 9999 );

		ob_start(); ?>
		<div data-block-id="<?php echo esc_attr( $this->block_id ); ?>" class="srfm-block-single srfm-block srfm-<?php echo esc_attr( $this->slug ); ?>-block <?php echo esc_attr( $this->block_width ); ?><?php echo esc_attr( $this->classname ); ?> <?php echo esc_attr( $this->conditional_class ); ?>">
			<div class="srfm-block-wrap">
				<div class="srfm-repeater-container" 
					id="<?php echo esc_attr( $random_id ); ?>"
					data-min-items="<?php echo intval( $this->min_items ); ?>"
					data-max-items="<?php echo intval( $this->max_items ); ?>"
					data-field-name="<?php echo esc_attr( $this->field_name ); ?>"
					<?php if ( ! empty( $entry_data ) ) { ?>
						data-entry-data="<?php echo esc_attr( $entry_data ); ?>"
					<?php } ?>
				>
					<div class="srfm-repeater-items">
						<?php $this->render_repeater_item( 0 ); ?>
					</div>
					<!-- If isEditing is true then don't add the template.  -->
					<?php if ( ! $this->is_editing ) { ?>
						<div class="srfm-repeater-template" style="display: none;">
							<?php $this->render_repeater_item( '__INDEX__' ); ?>
						</div>
					<?php } ?>
				</div>
			</div>
		</div>
		<?php
		$markup = ob_get_clean();

		return apply_filters(
			'srfm_block_field_markup',
			$markup,
			[
				'slug'       => $this->slug,
				'is_editing' => $this->is_editing,
				'field_name' => $this->field_name,
				'attributes' => $this->attributes,
			]
		);
	}

	/**
	 * Render a single repeater item.
	 *
	 * Generates the HTML markup for an individual repeater item including
	 * the header, controls, and content area.
	 *
	 * @param int|string $index The index of the repeater item.
	 * @since 1.11.0
	 * @return void
	 */
	private function render_repeater_item( $index = 0 ) {
		$icon_circle_plus   = Pro_Helper::fetch_pro_svg( 'circlePlus', 'srfm-' . esc_attr( $this->slug ) . '-icon', 'aria-hidden="true"' );
		$icon_circle_minus  = Pro_Helper::fetch_pro_svg( 'circleMinus', 'srfm-' . esc_attr( $this->slug ) . '-icon', 'aria-hidden="true"' );
		$index              = strval( $index );
		$group_label_id     = 'srfm-label-' . $this->block_id;
		$adding_item_text   = __( 'Repeater item added.', 'sureforms-pro' );
		$removing_item_text = __( 'Repeater item removed.', 'sureforms-pro' );
		?>
		<div class="srfm-repeater-item" data-index="<?php echo esc_attr( $index ); ?>" role="group" aria-labelledby="<?php echo esc_attr( $group_label_id ); ?>" aria-describedby="repeater-description-1">
			<div class="srfm-repeater-item-header">
				<div class="srfm-repeater-item-header-left">
					<div class="srfm-repeater-label-container">
						<label id="<?php echo esc_attr( $group_label_id ); ?>" class="srfm-block-label"><?php echo wp_kses_post( $this->label ); ?></label>
						<span class="srfm-repeater-index">#1</span>
					</div>
					<?php echo wp_kses_post( $this->help_markup ); ?>
				</div>
				<?php if ( ! $this->is_editing ) { ?>
					<div class="srfm-repeater-item-controls">
						<?php if ( $this->max_items > $this->min_items ) { ?>
						<button type="button" class="srfm-repeater-add-btn" title="<?php echo esc_html__( 'Add', 'sureforms-pro' ); ?>" data-add-item-text="<?php echo esc_attr( $adding_item_text ); ?>">
							<?php echo $icon_circle_plus; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Ignored to render svg ?>
						</button>
						<button type="button" class="srfm-repeater-remove-btn" title="<?php echo esc_html__( 'Remove', 'sureforms-pro' ); ?>" data-remove-item-text="<?php echo esc_attr( $removing_item_text ); ?>">
							<?php echo $icon_circle_minus; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Ignored to render svg ?>
						</button>
						<?php } ?>
					</div>
				<?php } ?>
			</div>
			<div class="srfm-repeater-item-content">
				<?php echo $this->modify_field_names_for_index( $this->inner_content, $index ); //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Ignored to render html ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Modify field names in content to include the repeater index.
	 *
	 * Updates the field names in the content to include the repeater index,
	 * ensuring proper data structure when the form is submitted.
	 *
	 * @param string     $content The inner block content.
	 * @param int|string $index The index to be added to field names.
	 * @since 1.11.0
	 * @return string Modified content with updated field names.
	 */
	private function modify_field_names_for_index( $content, $index ) {
		// Pattern to match field names in the content.
		$pattern = '/name="([^"]+)"/';

		$content = preg_replace_callback(
			$pattern,
			function( $matches ) use ( $index ) {
				$original_name = $matches[1];
				// Prepend the repeater field name and index.
				$new_name = $this->field_name . '[' . $index . '][' . $original_name . ']';
				return 'name="' . esc_attr( $new_name ) . '"';
			},
			$content
		);

		return is_string( $content ) ? $content : '';
	}
}
