<?php

defined( 'ABSPATH' ) || exit;

use WooCommerce\PayPalCommerce\WcGateway\Gateway\PayPalGateway;
use WooCommerce\PayPalCommerce\WcGateway\Gateway\CreditCardGateway;

/**
 * Compatibility class for WooCommerce PayPal Payments.
 *
 * @extends SUMOSubs_WC_PayPal_Payments_Integration
 */
class SUMOSubs_WC_PayPal_Payments_Integration {

    /**
     * Init SUMOSubs_WC_PayPal_Payments_Integration
     */
    public static function init() {
        self::include_files();
        self::define_class_alias();

        // Register module for paypal payments plugin.
        add_filter( 'woocommerce_paypal_payments_modules', __CLASS__ . '::add_module', 10, 1 );
        add_filter( 'sumosubscriptions_available_payment_gateways', __CLASS__ . '::add_subscription_supports' );
    }

    /**
     * Include required files for gateway integration
     */
    private static function include_files() {
        $pp_version = self::get_plugin_version();
        if ( empty( $pp_version ) ) {
            return;
        }

        $files_map = array(
            'trait-sumosubs-wc-paypal-payments-trial-handler.php'   => 'SUMOSubs_WC_PayPal_Payments_Trial_Handler',
            'class-sumosubs-wc-paypal-payments-helper.php'          => 'SUMOSubs_WC_PayPal_Payments_Helper',
            'class-sumosubs-wc-paypal-payments-renewal-handler.php' => 'SUMOSubs_WC_PayPal_Payments_Renewal_Handler',
            'class-sumosubs-wc-paypal-payments-module.php'          => 'SUMOSubs_WC_PayPal_Payments_Module',
        );

        // Conditionally load legacy files to grant backward compatibility.
        $legacy_dir = SUMO_SUBSCRIPTIONS_PLUGIN_DIR . 'includes/gateways/paypal/paypal-payments/module/src/legacy';
        foreach ( scandir( $legacy_dir ) as $legacy_version_dir ) {
            if ( in_array( $legacy_version_dir, array( '.', '..' ), true ) || ! version_compare( $pp_version, $legacy_version_dir, '<=' ) ) {
                continue;
            }

            foreach ( scandir( $legacy_dir . DIRECTORY_SEPARATOR . $legacy_version_dir ) as $file ) {
                if ( in_array( $file, array( '.', '..' ), true ) ) {
                    continue;
                }

                // Check class exists to avoid require duplicates.
                if ( empty( $files_map[ $file ] ) || ! class_exists( $files_map[ $file ] ) ) {
                    require_once $legacy_dir . DIRECTORY_SEPARATOR . $legacy_version_dir . DIRECTORY_SEPARATOR . $file;
                }
            }
        }

        foreach ( $files_map as $file => $class_name ) {
            if ( ! class_exists( $class_name ) ) {
                require_once SUMO_SUBSCRIPTIONS_PLUGIN_DIR . "includes/gateways/paypal/paypal-payments/module/src/{$file}";
            }
        }
    }

    /**
     * Define class aliases
     *
     * @return void
     */
    protected static function define_class_alias() {
        class_alias( 'SUMOSubs_WC_PayPal_Payments_Trial_Handler', 'WooCommerce\PayPalCommerce\Subscription\FreeTrialHandlerTrait' );
        class_alias( 'SUMOSubs_WC_PayPal_Payments_Trial_Handler', 'WooCommerce\PayPalCommerce\WcSubscriptions\FreeTrialHandlerTrait' );
    }

    /**
     * Add module to the WooCommerce PayPal Payments modules list
     *
     * @param array $modules Array of available modules.
     * @return array
     */
    public static function add_module( $modules ) {
        // Double check class exists. 
        if ( class_exists( 'SUMOSubs_WC_PayPal_Payments_Module', false ) ) {
            return array_merge(
                    $modules,
                    array(
                        ( require 'module/module.php' )(),
                    )
            );
        }

        return $modules;
    }

    /**
     * Get WooCommerce PayPal Payments plugin version.
     *
     * @return string|false
     */
    protected static function get_plugin_version() {
        $plugin_metadata = array_filter(
                get_plugins(),
                function( $plugin_init ) {
                    return false !== strpos( $plugin_init, 'woocommerce-paypal-payments.php' ) && is_plugin_active( $plugin_init );
                },
                ARRAY_FILTER_USE_KEY
        );

        if ( empty( $plugin_metadata ) ) {
            return false;
        }

        $plugin_metadata = array_shift( $plugin_metadata );
        return $plugin_metadata[ 'Version' ] ?? '1.0.0';
    }

    /**
     * Add gateway to support subscriptions.
     * 
     * @param array $subscription_gateways
     * @return array
     */
    public static function add_subscription_supports( $subscription_gateways ) {
        $subscription_gateways[] = PayPalGateway::ID;
        $subscription_gateways[] = CreditCardGateway::ID;
        return $subscription_gateways;
    }
}
