<?php

defined( 'ABSPATH' ) || exit;

include_once 'deprecated/schedulers/class-sumosubs-legacy-scheduler.php';
include_once 'deprecated/schedulers/class-sumosubs-legacy-background-process.php';
include_once 'deprecated/schedulers/class-sumosubs-legacy-schedulers.php';

SUMOSubs_Legacy_Schedulers::init();

/**
 * Scheduler for subscription events that uses the Action Scheduler
 * 
 * @class SUMOSubs_Action_Scheduler
 * @package Class
 */
class SUMOSubs_Action_Scheduler {

    /**
     * An internal cache of action hooks.
     *
     * @var array An array of $action_hook
     */
    protected static $action_hooks = array(
        'sumosubs_scheduled_subscription_activation',
        'sumosubs_scheduled_subscription_reactivation',
        'sumosubs_scheduled_subscription_due_order_creation',
        'sumosubs_scheduled_subscription_due_payment_automatic',
        'sumosubs_scheduled_subscription_due_payment_retry',
        'sumosubs_scheduled_subscription_reminder',
        'sumosubs_scheduled_subscription_overdue',
        'sumosubs_scheduled_subscription_suspend',
        'sumosubs_scheduled_subscription_cancel',
        'sumosubs_scheduled_subscription_expiration',
    );

    /**
     * Action scheduler group.
     *
     * @var string
     */
    const GROUP_NAME = 'sumosubscriptions';

    /**
     * Get the args to set on the scheduled action.
     *
     * @param int $subscription_id
     */
    public static function get_action_args( $subscription_id ) {
        $action_args = array( 'subscription_id' => absint( $subscription_id ) );

        /**
         * Get subscription scheduled action args.
         * 
         * @since 16.4.0
         */
        return apply_filters( 'sumosubscriptions_scheduled_action_args', $action_args, $subscription_id );
    }

    /**
     * Maybe schedule the legacy only if exists.
     * 
     * @param int $subscription_id
     * @param int $timestamp
     * @param string $event_name
     * @param array $action_args
     * @return bool
     */
    public static function maybe_schedule_legacy( $subscription_id, $timestamp, $event_name, $action_args = array() ) {
        $legacy_scheduler = new SUMOSubs_Legacy_Scheduler( $subscription_id );

        if ( $legacy_scheduler->event_id > 0 ) {
            // When schedules are available for this subscription, leave it. Legacy scheduler will be used in this case so return it. 
            if ( ! empty( $legacy_scheduler->cron_events ) ) {
                $legacy_scheduler->set_cron_event( $timestamp, $event_name, $action_args );
                return true;
            }

            // Force delete the scheduled entry when no schedules are available, so that new scheduler will be used hereafter for this subscription.
            wp_delete_post( $legacy_scheduler->event_id, true );
        }

        return false;
    }

    /**
     * Schedule subscription activation.
     *
     * @param int $subscription_id
     * @param int $start_time
     * @return bool 
     */
    public static function schedule_subscription_activation( $subscription_id, $start_time ) {
        $timestamp = absint( $start_time );

        if ( ! $timestamp ) {
            return false;
        }

        $action_args = self::get_action_args( $subscription_id );

        /**
         * Need to schedule ?
         * 
         * @since 16.4.0
         */
        if ( ! apply_filters( 'sumosubscriptions_schedule_subscription_activation', true, $action_args ) ) {
            return false;
        }

        // BKWD COMPAT
        if ( self::maybe_schedule_legacy( $subscription_id, $timestamp, 'start_subscription', $action_args ) ) {
            return true;
        }

        $action_args = array( 'args' => $action_args );

        if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_activation', $action_args, self::GROUP_NAME ) !== $timestamp ) {
            as_unschedule_all_actions( 'sumosubs_scheduled_subscription_activation', $action_args, self::GROUP_NAME );
            as_schedule_single_action( $timestamp, 'sumosubs_scheduled_subscription_activation', $action_args, self::GROUP_NAME );
        }

        return true;
    }

    /**
     * Schedule due order creation.
     *
     * @param int $subscription_id
     * @param string $due_date
     * @return bool 
     */
    public static function schedule_due_order_creation( $subscription_id, $due_date ) {
        $subscription_period = get_post_meta( $subscription_id, 'sumo_subscr_plan', true );
        $trial_period        = get_post_meta( $subscription_id, 'sumo_trial_plan', true );
        $subscription_status = get_post_meta( $subscription_id, 'sumo_get_status', true );
        $subscription_cycle  = sumo_get_subscription_cycle( 'Trial' === $subscription_status ? $trial_period : $subscription_period );
        $no_of_days_before   = absint( SUMOSubs_Admin_Options::get_option( 'renewal_order_creation_schedule' ) ) * 86400;

        if ( $subscription_cycle < $no_of_days_before ) {
            $no_of_days_before = $subscription_cycle;
        }

        if ( $no_of_days_before <= 0 ) {
            $no_of_days_before = 0;
        }

        $timestamp = sumo_get_subscription_timestamp( "{$due_date} -{$no_of_days_before} seconds" );
        $timestamp = absint( $timestamp );

        if ( ! $timestamp ) {
            return false;
        }

        $action_args                  = self::get_action_args( $subscription_id );
        $action_args[ 'next_due_on' ] = $due_date;

        /**
         * Need to schedule ?
         * 
         * @since 16.4.0
         */
        if ( ! apply_filters( 'sumosubscriptions_schedule_due_order_creation', true, $action_args ) ) {
            return false;
        }

        // BKWD COMPAT
        if ( self::maybe_schedule_legacy( $subscription_id, $timestamp, 'create_renewal_order', $action_args ) ) {
            return true;
        }

        $action_args = array( 'args' => $action_args );

        if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_due_order_creation', $action_args, self::GROUP_NAME ) !== $timestamp ) {
            as_unschedule_all_actions( 'sumosubs_scheduled_subscription_due_order_creation', $action_args, self::GROUP_NAME );
            as_schedule_single_action( $timestamp, 'sumosubs_scheduled_subscription_due_order_creation', $action_args, self::GROUP_NAME );
        }

        return true;
    }

    /**
     * Schedule due payment for automatic subscription.
     *
     * @param int $subscription_id
     * @param string $due_date
     * @param int $due_order_id
     * @return bool 
     */
    public static function schedule_due_payment_automatic( $subscription_id, $due_date, $due_order_id ) {
        $timestamp = sumo_get_subscription_timestamp( $due_date );
        $timestamp = absint( $timestamp );

        if ( ! $timestamp ) {
            return false;
        }

        $action_args                           = self::get_action_args( $subscription_id );
        $action_args[ 'renewal_order_id' ]     = absint( $due_order_id );
        $action_args[ 'next_eligible_status' ] = sumosubs_get_next_eligible_subscription_failed_status( $subscription_id );

        switch ( $action_args[ 'next_eligible_status' ] ) {
            case 'Overdue':
                $action_args[ 'payment_charging_days' ]       = sumosubs_get_overdue_days();
                $action_args[ 'payment_retry_times_per_day' ] = sumosubs_get_payment_retry_times_per_day_in( 'Overdue' );
                break;
            case 'Suspended':
                $action_args[ 'payment_charging_days' ]       = sumosubs_get_suspend_days();
                $action_args[ 'payment_retry_times_per_day' ] = sumosubs_get_payment_retry_times_per_day_in( 'Suspended' );
                break;
            default:
                $action_args[ 'payment_charging_days' ]       = 0;
                $action_args[ 'payment_retry_times_per_day' ] = 0;
                break;
        }

        /**
         * Need to schedule ?
         * 
         * @since 16.4.0
         */
        if ( ! apply_filters( 'sumosubscriptions_schedule_due_payment_automatic', true, $action_args ) ) {
            return false;
        }

        // BKWD COMPAT
        if ( self::maybe_schedule_legacy( $subscription_id, $timestamp, 'automatic_pay', $action_args ) ) {
            return true;
        }

        $action_args = array( 'args' => $action_args );

        if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_due_payment_automatic', $action_args, self::GROUP_NAME ) !== $timestamp ) {
            as_unschedule_all_actions( 'sumosubs_scheduled_subscription_due_payment_automatic', $action_args, self::GROUP_NAME );
            as_schedule_single_action( $timestamp, 'sumosubs_scheduled_subscription_due_payment_automatic', $action_args, self::GROUP_NAME );
        }

        return true;
    }

    /**
     * Schedule subscription overdue.
     *
     * @param int $subscription_id
     * @param string $due_date
     * @param int $due_order_id
     * @param int $payment_charging_days
     * @return bool 
     */
    public static function schedule_subscription_overdue( $subscription_id, $due_date, $due_order_id, $payment_charging_days = 0 ) {
        $timestamp = sumo_get_subscription_timestamp( $due_date );
        $timestamp = absint( $timestamp );

        if ( ! $timestamp ) {
            return false;
        }

        $action_args                            = self::get_action_args( $subscription_id );
        $action_args[ 'renewal_order_id' ]      = absint( $due_order_id );
        $action_args[ 'next_due_on' ]           = sumo_get_subscription_date( $due_date );
        $action_args[ 'payment_charging_days' ] = $payment_charging_days;

        /**
         * Need to schedule ?
         * 
         * @since 16.4.0
         */
        if ( ! apply_filters( 'sumosubscriptions_schedule_subscription_overdue', true, $action_args ) ) {
            return false;
        }

        // BKWD COMPAT
        if ( self::maybe_schedule_legacy( $subscription_id, $timestamp, 'notify_overdue', $action_args ) ) {
            return true;
        }

        $action_args = array( 'args' => $action_args );

        if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_overdue', $action_args, self::GROUP_NAME ) !== $timestamp ) {
            as_unschedule_all_actions( 'sumosubs_scheduled_subscription_overdue', $action_args, self::GROUP_NAME );
            as_schedule_single_action( $timestamp, 'sumosubs_scheduled_subscription_overdue', $action_args, self::GROUP_NAME );
        }

        return true;
    }

    /**
     * Schedule subscription suspend.
     *
     * @param int $subscription_id
     * @param string $due_date
     * @param int $due_order_id
     * @param int $payment_charging_days
     * @return bool 
     */
    public static function schedule_subscription_suspend( $subscription_id, $due_date, $due_order_id, $payment_charging_days = 0 ) {
        if ( ! empty( $due_date ) ) {
            $timestamp = sumo_get_subscription_timestamp( $due_date );
        } else if ( $payment_charging_days > 0 ) {
            $timestamp = sumo_get_subscription_timestamp() + ( $payment_charging_days * 86400 );
        } else {
            $timestamp = 0;
        }

        $timestamp = absint( $timestamp );

        if ( ! $timestamp ) {
            return false;
        }

        $action_args                            = self::get_action_args( $subscription_id );
        $action_args[ 'renewal_order_id' ]      = absint( $due_order_id );
        $action_args[ 'payment_charging_days' ] = sumosubs_get_suspend_days();

        /**
         * Need to schedule ?
         * 
         * @since 16.4.0
         */
        if ( ! apply_filters( 'sumosubscriptions_schedule_subscription_suspend', true, $action_args ) ) {
            return false;
        }

        // BKWD COMPAT
        if ( self::maybe_schedule_legacy( $subscription_id, $timestamp, 'notify_suspend', $action_args ) ) {
            return true;
        }

        $action_args = array( 'args' => $action_args );

        if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_suspend', $action_args, self::GROUP_NAME ) !== $timestamp ) {
            as_unschedule_all_actions( 'sumosubs_scheduled_subscription_suspend', $action_args, self::GROUP_NAME );
            as_schedule_single_action( $timestamp, 'sumosubs_scheduled_subscription_suspend', $action_args, self::GROUP_NAME );
        }

        return true;
    }

    /**
     * Schedule subscription cancel.
     *
     * @param int $subscription_id
     * @param string $due_date
     * @param int $due_order_id
     * @param int $payment_charging_days
     * @return bool 
     */
    public static function schedule_subscription_cancel( $subscription_id, $due_date, $due_order_id, $payment_charging_days = 0 ) {
        if ( ! empty( $due_date ) ) {
            $timestamp = sumo_get_subscription_timestamp( $due_date );
        } else if ( $payment_charging_days > 0 ) {
            $timestamp = sumo_get_subscription_timestamp() + ( $payment_charging_days * 86400 );
        } else {
            $timestamp = 0;
        }

        $timestamp = absint( $timestamp );

        if ( ! $timestamp ) {
            return false;
        }

        $action_args                       = self::get_action_args( $subscription_id );
        $action_args[ 'renewal_order_id' ] = absint( $due_order_id );

        /**
         * Need to schedule ?
         * 
         * @since 16.4.0
         */
        if ( ! apply_filters( 'sumosubscriptions_schedule_subscription_cancel', true, $action_args ) ) {
            return false;
        }

        // BKWD COMPAT
        if ( self::maybe_schedule_legacy( $subscription_id, $timestamp, 'notify_cancel', $action_args ) ) {
            return true;
        }

        $action_args = array( 'args' => $action_args );

        if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_cancel', $action_args, self::GROUP_NAME ) !== $timestamp ) {
            as_unschedule_all_actions( 'sumosubs_scheduled_subscription_cancel', $action_args, self::GROUP_NAME );
            as_schedule_single_action( $timestamp, 'sumosubs_scheduled_subscription_cancel', $action_args, self::GROUP_NAME );
        }

        return true;
    }

    /**
     * Schedule subscription expiration.
     *
     * @param int $subscription_id
     * @param string $expiration_date
     * @return bool 
     */
    public static function schedule_subscription_expiration( $subscription_id, $expiration_date ) {
        $timestamp = sumo_get_subscription_timestamp( $expiration_date );
        $timestamp = absint( $timestamp );

        if ( ! $timestamp ) {
            return false;
        }

        $action_args                = self::get_action_args( $subscription_id );
        $action_args[ 'expiry_on' ] = $expiration_date;

        /**
         * Need to schedule ?
         * 
         * @since 16.4.0
         */
        if ( ! apply_filters( 'sumosubscriptions_schedule_subscription_expiration', true, $action_args ) ) {
            return false;
        }

        // BKWD COMPAT
        if ( self::maybe_schedule_legacy( $subscription_id, $timestamp, 'notify_expire', $action_args ) ) {
            return true;
        }

        $action_args = array( 'args' => $action_args );

        if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_expiration', $action_args, self::GROUP_NAME ) !== $timestamp ) {
            as_unschedule_all_actions( 'sumosubs_scheduled_subscription_expiration', $action_args, self::GROUP_NAME );
            as_schedule_single_action( $timestamp, 'sumosubs_scheduled_subscription_expiration', $action_args, self::GROUP_NAME );
        }

        return true;
    }

    /**
     * Schedule subscription reactivation.
     *
     * @param int $subscription_id
     * @param string $reactivation_date
     * @return bool 
     */
    public static function schedule_subscription_reactivation( $subscription_id, $reactivation_date ) {
        $timestamp = sumo_get_subscription_timestamp( $reactivation_date );
        $timestamp = absint( $timestamp );

        if ( ! $timestamp ) {
            return false;
        }

        $action_args = self::get_action_args( $subscription_id );

        /**
         * Need to schedule ?
         * 
         * @since 16.4.0
         */
        if ( ! apply_filters( 'sumosubscriptions_schedule_subscription_reactivation', true, $action_args ) ) {
            return false;
        }

        // BKWD COMPAT
        if ( self::maybe_schedule_legacy( $subscription_id, $timestamp, 'automatic_resume', $action_args ) ) {
            return true;
        }

        $action_args = array( 'args' => $action_args );

        if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_reactivation', $action_args, self::GROUP_NAME ) !== $timestamp ) {
            as_unschedule_all_actions( 'sumosubs_scheduled_subscription_reactivation', $action_args, self::GROUP_NAME );
            as_schedule_single_action( $timestamp, 'sumosubs_scheduled_subscription_reactivation', $action_args, self::GROUP_NAME );
        }

        return true;
    }

    /**
     * Schedule subscription reminders.
     *
     * @param int $subscription_id
     * @param string $remind_date
     * @param int $due_order_id 
     * @param string $template_name
     * @return bool
     */
    public static function schedule_subscription_reminders( $subscription_id, $remind_date, $due_order_id, $template_name = 'subscription_invoice' ) {
        $timestamp = sumo_get_subscription_timestamp( $remind_date );
        $timestamp = absint( $timestamp );

        if ( ! $timestamp ) {
            return false;
        }

        $action_args                       = self::get_action_args( $subscription_id );
        $action_args[ 'renewal_order_id' ] = absint( $due_order_id );
        $action_args[ 'mail_template_id' ] = $template_name;

        /**
         * Need to schedule ?
         * 
         * @since 16.4.0
         */
        if ( ! apply_filters( 'sumosubscriptions_schedule_subscription_reminders', true, $action_args ) ) {
            return false;
        }

        $reminder_intervals = sumosubs_get_reminder_intervals( $subscription_id, $template_name );
        $current_time       = sumo_get_subscription_timestamp();
        $days_to_notify     = ceil( ( $timestamp - $current_time ) / 86400 );
        $legacy_job_name    = 'subscription_expiry_reminder' === $template_name ? 'notify_expiry_reminder' : 'notify_invoice_reminder';
        $scheduled          = false;

        //Schedule by times per day
        if ( isset( $reminder_intervals[ 'times-per-day' ] ) ) {
            for ( $c = 0; $c < $days_to_notify; $c ++ ) {
                for ( $r = 1; $r <= $reminder_intervals[ 'times-per-day' ]; $r ++ ) {

                    $time             = sumo_get_subscription_timestamp( "+$c days" );
                    $day_start_time   = sumo_get_subscription_timestamp( gmdate( 'Y-m-d 00:00:00', $time ) );
                    $day_end_time     = sumo_get_subscription_timestamp( gmdate( 'Y-m-d 23:59:59', $time ) );
                    $one_day_interval = $day_end_time - $time;

                    if ( $c > 0 ) {
                        $one_day_interval = $day_end_time - $day_start_time;
                        $time             = $day_start_time;
                    }

                    $notification_time = $one_day_interval ? $time + ( $one_day_interval / $r ) : 0;

                    if ( $notification_time >= $current_time ) {
                        // BKWD COMPAT
                        if ( self::maybe_schedule_legacy( $subscription_id, $notification_time, $legacy_job_name, $action_args ) ) {
                            $scheduled = true;
                        } else {
                            $_action_args = array( 'args' => $action_args );

                            if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_reminder', $_action_args, self::GROUP_NAME ) !== $notification_time ) {
                                as_schedule_single_action( $notification_time, 'sumosubs_scheduled_subscription_reminder', $_action_args, self::GROUP_NAME );
                            }

                            $scheduled = true;
                        }
                    }
                }
            }
        } else if ( isset( $reminder_intervals[ 'no-of-days' ] ) ) {
            $subscription_plan = sumo_get_subscription_plan( $subscription_id );
            $payment_type      = sumo_get_payment_type( $subscription_id );

            //Schedule by comma separated days
            if (
                    isset( $subscription_plan[ 'send_payment_reminder_email' ][ $payment_type ] ) &&
                    'yes' === $subscription_plan[ 'send_payment_reminder_email' ][ $payment_type ]
            ) {
                $notifications = array();

                foreach ( $reminder_intervals[ 'no-of-days' ] as $notify_day ) {
                    if ( $notify_day && $days_to_notify >= $notify_day ) {
                        $notifications[] = absint( $timestamp - ( 86400 * $notify_day ) );
                    }
                }

                $notifications = array_unique( $notifications );
                if ( $notifications ) {
                    foreach ( $notifications as $notification_time ) {
                        if ( $notification_time >= sumo_get_subscription_timestamp( 0, 0, true ) ) {
                            // BKWD COMPAT
                            if ( self::maybe_schedule_legacy( $subscription_id, $notification_time, $legacy_job_name, $action_args ) ) {
                                $scheduled = true;
                            } else {
                                $_action_args = array( 'args' => $action_args );

                                if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_reminder', $_action_args, self::GROUP_NAME ) !== $notification_time ) {
                                    as_schedule_single_action( $notification_time, 'sumosubs_scheduled_subscription_reminder', $_action_args, self::GROUP_NAME );
                                }

                                $scheduled = true;
                            }
                        }
                    }
                }
            } else {
                $scheduled = true;
            }
        }

        if ( ! $scheduled ) {
            // BKWD COMPAT
            if ( self::maybe_schedule_legacy( $subscription_id, $current_time, $legacy_job_name, $action_args ) ) {
                $scheduled = true;
            } else {
                $action_args = array( 'args' => $action_args );

                if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_reminder', $action_args, self::GROUP_NAME ) !== $current_time ) {
                    as_schedule_single_action( $current_time, 'sumosubs_scheduled_subscription_reminder', $action_args, self::GROUP_NAME );
                }

                $scheduled = true;
            }
        }

        return $scheduled;
    }

    /**
     * Schedule due payment retry for failed automatic subscription.
     *
     * @param int $subscription_id
     * @param int $due_order_id
     * @param string $next_eligible_status
     * @param int $payment_charging_days 
     * @param int $retry_times_per_day 
     * @return bool
     */
    public static function schedule_due_payment_retry( $subscription_id, $due_order_id, $next_eligible_status, $payment_charging_days = 0, $retry_times_per_day = 0 ) {
        if ( 0 === $payment_charging_days ) {
            return false;
        }

        $action_args                           = self::get_action_args( $subscription_id );
        $action_args[ 'renewal_order_id' ]     = absint( $due_order_id );
        $action_args[ 'next_eligible_status' ] = $next_eligible_status;

        /**
         * Need to schedule ?
         * 
         * @since 16.4.0
         */
        if ( ! apply_filters( 'sumosubscriptions_schedule_due_payment_retry', true, $action_args ) ) {
            return false;
        }

        $subscription_status = get_post_meta( $subscription_id, 'sumo_get_status', true );
        $legacy_job_name     = 'retry_automatic_pay_in_' . strtolower( $subscription_status );

        if ( 0 === $retry_times_per_day ) {
            $timestamp = sumo_get_subscription_timestamp() + ( $payment_charging_days * 86400 );
            $timestamp = absint( $timestamp );

            if ( ! $timestamp ) {
                return false;
            }

            // BKWD COMPAT
            if ( self::maybe_schedule_legacy( $subscription_id, $timestamp, $legacy_job_name, $action_args ) ) {
                return true;
            }

            $action_args = array( 'args' => $action_args );

            if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_due_payment_retry', $action_args, self::GROUP_NAME ) !== $timestamp ) {
                as_schedule_single_action( $timestamp, 'sumosubs_scheduled_subscription_due_payment_retry', $action_args, self::GROUP_NAME );
            }

            return true;
        }

        /** Automatic Payment retries 
         *  Charge multiple times based upon the retry count set by Admin
         *  Ex. 
         *  1) if $payment_charging_days === 1 && $retry_times_per_day === 2 then retry 2 times for 1 day
         *  2) if $payment_charging_days === 2 && $retry_times_per_day === 2 then retry 2 times for each day
         */
        $scheduled = false;
        for ( $c = 0; $c < $payment_charging_days; $c ++ ) {
            for ( $r = 1; $r <= $retry_times_per_day; $r ++ ) {

                $time             = sumo_get_subscription_timestamp( "+$c days" );
                $day_start_time   = sumo_get_subscription_timestamp( gmdate( 'Y-m-d 00:00:00', $time ) );
                $day_end_time     = sumo_get_subscription_timestamp( gmdate( 'Y-m-d 23:59:59', $time ) );
                $one_day_interval = $day_end_time - $time;

                if ( $c > 0 ) {
                    $one_day_interval = $day_end_time - $day_start_time;
                    $time             = $day_start_time;
                }

                $charging_timestamp = $one_day_interval ? $time + ( $one_day_interval / $r ) : 0;

                if ( $charging_timestamp >= sumo_get_subscription_timestamp() ) {
                    $timestamp = absint( $charging_timestamp );

                    //may be Last retry in Current Subscription status
                    if ( $payment_charging_days - 1 === $c && $charging_timestamp === $day_end_time ) {
                        $action_args[ 'next_eligible_status' ] = $next_eligible_status;

                        switch ( $next_eligible_status ) {
                            case 'Suspended':
                                $action_args[ 'payment_charging_days' ]       = sumosubs_get_suspend_days();
                                $action_args[ 'payment_retry_times_per_day' ] = sumosubs_get_payment_retry_times_per_day_in( 'Suspended' );

                                // BKWD COMPAT
                                if ( self::maybe_schedule_legacy( $subscription_id, $timestamp, $legacy_job_name, $action_args ) ) {
                                    $scheduled = true;
                                } else {
                                    $_action_args = array( 'args' => $action_args );

                                    if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_due_payment_retry', $_action_args, self::GROUP_NAME ) !== $timestamp ) {
                                        as_schedule_single_action( $timestamp, 'sumosubs_scheduled_subscription_due_payment_retry', $_action_args, self::GROUP_NAME );
                                    }

                                    $scheduled = true;
                                }
                                break;
                            case 'Cancelled':
                                // BKWD COMPAT
                                if ( self::maybe_schedule_legacy( $subscription_id, $timestamp, $legacy_job_name, $action_args ) ) {
                                    $scheduled = true;
                                } else {
                                    $_action_args = array( 'args' => $action_args );

                                    if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_due_payment_retry', $_action_args, self::GROUP_NAME ) !== $timestamp ) {
                                        as_schedule_single_action( $timestamp, 'sumosubs_scheduled_subscription_due_payment_retry', $_action_args, self::GROUP_NAME );
                                    }

                                    $scheduled = true;
                                }
                                break;
                        }
                    } else {
                        $action_args[ 'next_eligible_status' ] = $subscription_status;

                        // BKWD COMPAT
                        if ( self::maybe_schedule_legacy( $subscription_id, $timestamp, $legacy_job_name, $action_args ) ) {
                            $scheduled = true;
                        } else {
                            $_action_args = array( 'args' => $action_args );

                            if ( as_next_scheduled_action( 'sumosubs_scheduled_subscription_due_payment_retry', $_action_args, self::GROUP_NAME ) !== $timestamp ) {
                                as_schedule_single_action( $timestamp, 'sumosubs_scheduled_subscription_due_payment_retry', $_action_args, self::GROUP_NAME );
                            }

                            $scheduled = true;
                        }
                    }
                }
            }
        }

        return $scheduled;
    }

    /**
     * Schedule due payment for both automatic & manual payments for failed scenarios.
     *
     * @param int $subscription_id
     * @param string $due_date
     * @param int $payment_charging_days
     * @param int $retry_times_per_day
     * @return bool
     */
    public static function schedule_due_payment( $subscription_id, $due_date = '', $payment_charging_days = 0, $retry_times_per_day = 0 ) {
        $due_order_id         = get_post_meta( $subscription_id, 'sumo_get_renewal_id', true );
        $next_eligible_status = sumosubs_get_next_eligible_subscription_failed_status( $subscription_id );

        if ( empty( $due_date ) ) {
            $due_date = get_post_meta( $subscription_id, 'sumo_get_next_payment_date', true );
        }

        $scheduled = false;
        switch ( get_post_meta( $subscription_id, 'sumo_get_status', true ) ) {
            case 'Trial':
            case 'Active':
            case 'Pending':
            case 'Pending_Authorization':
                if ( $payment_charging_days > 0 ) {
                    $due_date = sumo_get_subscription_date( sumo_get_subscription_timestamp( $due_date ) + ( $payment_charging_days * 86400 ) );
                }

                switch ( $next_eligible_status ) {
                    case 'Overdue':
                        $scheduled = self::schedule_subscription_overdue( $subscription_id, $due_date, $due_order_id, sumosubs_get_overdue_days() );
                        break;
                    case 'Suspended':
                        $scheduled = self::schedule_subscription_suspend( $subscription_id, $due_date, $due_order_id, sumosubs_get_suspend_days() );
                        break;
                    case 'Cancelled':
                        $scheduled = self::schedule_subscription_cancel( $subscription_id, $due_date, $due_order_id );
                        break;
                }
                break;
            case 'Overdue':
                switch ( $next_eligible_status ) {
                    case 'Suspended':
                        if ( 'auto' === sumo_get_payment_type( $subscription_id ) ) {
                            $scheduled = self::schedule_due_payment_retry( $subscription_id, $due_order_id, $next_eligible_status, $payment_charging_days, $retry_times_per_day );
                        } else {
                            $scheduled = self::schedule_subscription_suspend( $subscription_id, '', $due_order_id, $payment_charging_days );
                        }
                        break;
                    case 'Cancelled':
                        if ( 'auto' === sumo_get_payment_type( $subscription_id ) ) {
                            $scheduled = self::schedule_due_payment_retry( $subscription_id, $due_order_id, $next_eligible_status, $payment_charging_days, $retry_times_per_day );
                        } else {
                            $scheduled = self::schedule_subscription_cancel( $subscription_id, '', $due_order_id, $payment_charging_days );
                        }
                        break;
                }
                break;
            case 'Suspended':
                switch ( $next_eligible_status ) {
                    case 'Cancelled':
                        if ( 'auto' === sumo_get_payment_type( $subscription_id ) ) {
                            $scheduled = self::schedule_due_payment_retry( $subscription_id, $due_order_id, $next_eligible_status, $payment_charging_days, $retry_times_per_day );
                        } else {
                            $scheduled = self::schedule_subscription_cancel( $subscription_id, '', $due_order_id, $payment_charging_days );
                        }
                        break;
                }
                break;
        }

        return $scheduled;
    }

    /**
     * Maybe unschedule all actions for the given subscription ID.
     *
     * @param int $subscription_id
     * @param string|array $hooks Specific hooks to unschedule. When left empty, unschedules all.
     */
    public static function unschedule_all_actions( $subscription_id, $hooks = array() ) {
        $scheduled_actions = as_get_scheduled_actions( array(
            'group'    => self::GROUP_NAME,
            'status'   => 'pending',
            'per_page' => -1,
                ) );

        if ( empty( $hooks ) ) {
            $hooks_to_unschedule = self::$action_hooks;
        } else if ( is_array( $hooks ) ) {
            $hooks_to_unschedule = $hooks;
        } else {
            $hooks_to_unschedule = array( $hooks );
        }

        foreach ( $scheduled_actions as $action ) {
            if ( in_array( $action->get_hook(), $hooks_to_unschedule ) ) {
                $args = $action->get_args();

                if ( isset( $args[ 'args' ][ 'subscription_id' ] ) && $subscription_id == $args[ 'args' ][ 'subscription_id' ] ) {
                    as_unschedule_all_actions( $action->get_hook(), $args, self::GROUP_NAME );
                }
            }
        }

        /**
         * Maybe unschedule the legacy if exists.
         */
        $legacy_scheduler = new SUMOSubs_Legacy_Scheduler( $subscription_id );
        $legacy_scheduler->unset_events( $legacy_scheduler->map_events_from_hooks( $hooks ) );

        if ( $legacy_scheduler->event_id > 0 && empty( $legacy_scheduler->cron_events ) ) {
            // Force delete the scheduled entry when no schedules are available, so that new scheduler will be used hereafter for this subscription.
            wp_delete_post( $legacy_scheduler->event_id, true );
        }
    }

    /**
     * Gets the scheduled status args.
     * 
     * @param int $subscription_id
     * @return array
     */
    public static function get_scheduled_status_args( $subscription_id ) {
        /**
         * Maybe gets legacy schedule if exists.
         */
        $legacy_scheduler = new SUMOSubs_Legacy_Scheduler( $subscription_id );

        if ( $legacy_scheduler->event_id > 0 && ! empty( $legacy_scheduler->cron_events ) ) {
            $args = $legacy_scheduler->get_scheduled_status_args();
        } else {
            $scheduled_actions = as_get_scheduled_actions( array(
                'group'    => self::GROUP_NAME,
                'status'   => 'pending',
                'per_page' => -1,
                    ) );

            $action_hooks = array(
                'sumosubs_scheduled_subscription_due_payment_automatic',
                'sumosubs_scheduled_subscription_due_payment_retry',
                'sumosubs_scheduled_subscription_overdue',
                'sumosubs_scheduled_subscription_suspend',
                'sumosubs_scheduled_subscription_cancel',
                'sumosubs_scheduled_subscription_expiration',
            );

            $args = array();
            foreach ( $scheduled_actions as $action ) {
                if ( in_array( $action->get_hook(), $action_hooks ) ) {
                    $args = $action->get_args();

                    if ( isset( $args[ 'args' ][ 'subscription_id' ] ) && $subscription_id == $args[ 'args' ][ 'subscription_id' ] ) {
                        $schedule            = $action->get_schedule();
                        $date                = $schedule ? $schedule->get_date() : null;
                        $args[ 'timestamp' ] = $date ? $date->getTimestamp() : 0;
                    }
                }
            }
        }

        $args = wp_parse_args( $args, array(
            'next_eligible_status' => '',
            'timestamp'            => 0
                ) );

        return $args;
    }
}
