<?php
namespace SlimSEOPro\ContentAnalysis;

use SlimSEOPro\Assets;
use SlimSEOPro\Common;

class Post {
	public function __construct() {
		add_action( 'init', [ $this, 'setup' ] );
	}

	public function setup() {
		if ( ! is_admin() ) {
			return;
		}

		if ( defined( 'SLIM_SEO_VER' ) ) {
			add_filter( 'slim_seo_metabox_tabs', [ $this, 'add_tab' ] );
			add_filter( 'slim_seo_metabox_panels', [ $this, 'add_panel' ] );
		} else {
			add_action( 'add_meta_boxes', [ $this, 'add_meta_box' ] );
		}

		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue' ] );
		add_action( 'admin_init', [ $this, 'setup_admin_columns' ] );
		add_action( 'save_post', [ $this, 'save' ] );
	}

	public function add_tab( array $tabs ): array {
		$screen = get_current_screen();

		if ( 'post' !== $screen->base ) {
			return $tabs;
		}

		$tabs['content-analysis'] = __( 'Writing assistant', 'slim-seo-pro' );

		return $tabs;
	}

	public function add_panel( array $panels ): array {
		$screen = get_current_screen();

		if ( 'post' !== $screen->base ) {
			return $panels;
		}

		ob_start();
		?>

		<div id="content-analysis" class="ss-tab-pane">
			<?php $this->render(); ?>
		</div>

		<?php
		$panels['content-analysis'] = ob_get_clean();
		return $panels;
	}

	public function admin_columns( array $columns ): array {
		$columns['content_analysis'] = esc_html__( 'Content', 'slim-seo-pro' );
		return $columns;
	}

	public function admin_column_render( $column, $post_id ) {
		if ( 'content_analysis' !== $column ) {
			return;
		}

		$data                  = get_post_meta( $post_id, 'slim_seo_pro', true ) ?: [];
		$content_analysis_data = $data['content_analysis'] ?? [];

		if ( ! isset( $content_analysis_data['good_keywords'] ) ) {
			return;
		}

		$ok = ! empty( $content_analysis_data['good_keywords'] ) && ! empty( $content_analysis_data['good_words_count'] );

		if ( isset( $content_analysis_data['good_media'] ) ) {
			$ok = $ok && ! empty( $content_analysis_data['good_media'] );
		}

		if ( $ok ) {
			echo '<span class="ssp-success" role="img" aria-label="' . esc_html__( 'Success', 'slim-seo-pro' ) . '"></span>';
		} else {
			echo '<span class="ssp-warning" role="img" aria-label="' . esc_html__( 'Warning', 'slim-seo-pro' ) . '"></span>';
		}
	}

	public function enqueue(): void {
		$screen = get_current_screen();

		if ( ! in_array( $screen->post_type, $this->get_types(), true ) ) {
			return;
		}

		wp_enqueue_style( 'slim-seo-pro-post', SLIM_SEO_PRO_URL . 'css/slim-seo-pro-post.css', [ 'wp-components' ], filemtime( SLIM_SEO_PRO_DIR . '/css/slim-seo-pro-post.css' ) );

		if ( 'post' !== $screen->base ) {
			return;
		}

		// Use components from Slim SEO: tabs, status icons.
		wp_enqueue_style( 'slim-seo-components', 'https://cdn.jsdelivr.net/gh/elightup/slim-seo@master/css/components.css', [], '1.0.0' );

		// If Slim SEO is active, create tabs for this feature, so use the tabs component from Slim SEO.
		if ( defined( 'SLIM_SEO_VER' ) ) {
			wp_enqueue_script( 'slim-seo-components', 'https://cdn.jsdelivr.net/gh/elightup/slim-seo@master/js/components.js', [], '1.0.0', true );
		}

		$data                  = get_post_meta( get_the_ID(), 'slim_seo_pro', true );
		$data                  = is_array( $data ) && ! empty( $data ) ? $data : [];
		$content_analysis_data = $data['content_analysis'] ?? [];

		Assets::enqueue_build_js( 'content-analysis', 'SSPro', [
			'homeURL'          => untrailingslashit( home_url() ),
			'rest'             => untrailingslashit( rest_url() ),
			'nonce'            => wp_create_nonce( 'wp_rest' ),
			'keywords'         => $content_analysis_data['keywords'] ?? '',
			'mainKeyword'      => $content_analysis_data['main_keyword'] ?? '',
			'SSLMActivated'    => defined( 'SLIM_SEO_LINK_MANAGER_VER' ),
			'SSSettingsPage'   => esc_url( admin_url( 'options-general.php?page=slim-seo' ) ),
			'siteLocale'       => get_locale(),
			'supportThumbnail' => post_type_supports( get_post_type(), 'thumbnail' ),
			'builderContent'   => $this->get_builder_content(),
		] );
	}

	public function render(): void {
		wp_nonce_field( 'save', 'ssp_content_analysis_nonce' );
		?>

		<div id="content-analysis-app"></div>

		<?php
	}

	public function add_meta_box() {
		$context    = apply_filters( 'slim_seo_pro_meta_box_context', 'normal' );
		$priority   = apply_filters( 'slim_seo_pro_meta_box_priority', 'low' );
		$post_types = $this->get_types();

		foreach ( $post_types as $post_type ) {
			add_meta_box( 'slim-seo-pro', __( 'Writing Assistant', 'slim-seo-pro' ), [ $this, 'render' ], $post_type, $context, $priority );
		}
	}

	public function setup_admin_columns(): void {
		$post_types = $this->get_types();

		foreach ( $post_types as $post_type ) {
			add_filter( "manage_{$post_type}_posts_columns", [ $this, 'admin_columns' ] );
			add_action( "manage_{$post_type}_posts_custom_column", [ $this, 'admin_column_render' ], 10, 2 );
		}
	}

	public function save( $post_id ): void {
		if ( ! check_ajax_referer( 'save', 'ssp_content_analysis_nonce', false ) || empty( $_POST ) ) {
			return;
		}

		$data = isset( $_POST['slim_seo_pro'] ) ? wp_unslash( $_POST['slim_seo_pro'] ) : []; // phpcs:ignore

		if ( empty( $data ) ) {
			delete_post_meta( $post_id, 'slim_seo_pro' );
		} else {
			update_post_meta( $post_id, 'slim_seo_pro', $data );
		}
	}

	public function get_types() {
		$post_types = Common::get_post_types();
		$post_types = apply_filters( 'slim_seo_pro_meta_box_post_types', $post_types );

		return $post_types;
	}

	public function get_builder_content( int $post_id = 0 ): string {
		$post_id = $post_id ?: get_the_ID();

		return apply_filters( 'slim_seo_pro_builder_content', '', $post_id );
	}
}
