<?php

/*
* Class: sppro_auto_update
* Inspired from: 
* http:// code. tutsplus. com/tutorials/a-guide-to-the-wordpress-http-api-automatic-plugin-updates--wp-25181
*/

if (!class_exists('SPPRO_AutoUpdate')) {
	class SPPRO_AutoUpdate
	{
		/**
		 * The plugins data
		 * @var string
		 */
		private $plugins;

		/**
		 * The update object from API
		 * @var string
		 */
		private $updates;
		/**
		 * The update request path for API
		 * @var string
		 */
		private $update_path;
		/**
		 * Plugin Slug (plugin_directory/plugin_file.php)
		 * @var string
		 */
		private $plugin_slug;
		/**
		 * The plugin's Purchase code
		 * @var string
		 */
		private $license_key;

		/**
		 * Initialize a new instance of the WordPress Auto-Update class
		 * @param string $current_version
		 * @param string $update_path
		 * @param string $plugin_slug
		 */
		public function __construct($license_key = '')
		{
            $this->plugins = array(
				array(
					'key'		=> $license_key,
					'slug'		=> SPPRO_PLUGIN_BASENAME,
					'name'		=> SPPRO_PLUGIN_NAME,
					'version'	=> SPPRO_VERSION,
					'title'     => SPPRO_PLUGIN_TITLE
				)
			);
			$this->wp = array(
				'wp_name'		=> get_bloginfo('name'),
				'wp_url'		=> home_url(),
				'wp_version'	=> get_bloginfo('version'),
				'wp_language'	=> get_bloginfo('language'),
				'wp_timezone'	=> get_option('timezone_string'),
			);

			// define the alternative API for updating checking
			add_filter( 'pre_set_site_transient_update_plugins', array( &$this, 'check_update' ) );
			// Define the alternative response for information checking
			add_filter( 'plugins_api', array( &$this, 'check_info' ), 10, 3 );

			add_action( 'admin_notices', array($this, 'sppro_notices_for_update') );

			add_action( 'admin_menu', array($this, 'sppro_add_updates_page'), 99 );
			add_action( 'wp_ajax_action_sppro_update_license', array($this, 'action_sppro_update_license') );

			// admin
			if( is_admin() ) {
				add_action('in_plugin_update_message-' . $this->plugins[0]['slug'], array($this, 'modify_plugin_update_message'), 10, 2 );
			}

		}

		public function sppro_add_updates_page() {
			global $sppro_hook;
			$sppro_hook[] = add_submenu_page( 'sp-pro', 'Updates', 'Enable Updates', 'manage_options', 'sp-pro-updates', array($this, 'sppro_updates_page'));
		}

		/**
		 * Add our self-hosted description to the filter
		 *
		 * @param boolean $false
		 * @param array $action
		 * @param object $arg
		 * @return bool|object
		 */
		public function check_info($result, $action = null, $args = null)
		{
			// vars
			$plugin = false;

			// only for 'plugin_information' action
			if( $action !== 'plugin_information' ) return $result;

			// find plugin via slug
			foreach( $this->plugins as $p ) {
				if( $args->slug == $p['slug'] ) $plugin = $p;
			}

			// bail early if plugin not found
			if( !$plugin ) return $result;

			// connect
			$response = $this->get_plugin_info();

			// bail early if no response
			if( !is_array($response) ) return $result;

			// remove tags (different context)
			if(isset($reponse[$args->slug]['tags'])) unset($response[$args->slug]['tags']);

			// convert to object
			$response = (object) $response[$args->slug];

			if(0) {
				// sections
				$sections = array(
					'description'		=> '',
					'installation'		=> '',
					'changelog'			=> '',
					'what_is_new'	=> ''
				);

				foreach( $sections as $k => $v ) {
					$sections[ $k ] = $response->sections[$k];
				}

				$response->sections = $sections;
			}

			// return
			return $response;
		}

		/**
		 * Return the remote version
		 *
		 * @return string $remote_version
		 */
		public function get_plugin_info()
		{
			// var
			$transient_name = 'sppro_update_transient';

			// delete transient (force-check is used to refresh)
			if( !empty($_GET['force-check']) ) {
				delete_transient($transient_name);
			}

			// try transient
			$transient = get_transient($transient_name);
			if( $transient !== false ) return $transient;

			$post = array(
				'plugins' => wp_json_encode($this->plugins),
			);

			$response = $this->request('/omakupd/v1/plugins/info', $post);

			// update transient
			set_transient($transient_name, $response, HOUR_IN_SECONDS );

			// return
			return $response;
		}

		/**
		 * Add our self-hosted autoupdate plugin to the filter transient
		 *
		 * @param $transient
		 * @return object $ transient
		 */
		public function check_update( $transient ) {
			// bail early if no response (error)
			if( !isset($transient->response) ) return $transient;

			// bail early if checked is empty
			if ( empty( $transient->last_checked ) ) return $transient;

			// fetch updates once (this filter is called multiple times during a single page load)
			if( !$this->updates ) {

				$post = array(
					'plugins'		=> wp_json_encode($this->plugins),
					'wp'			=> wp_json_encode($this->wp),
				);

				// connect
				$this->updates = $this->request('/omakupd/v1/plugins/check-update', $post);
			}

			// append
			if( is_array($this->updates) ) {
				foreach( $this->updates as $basename => $update ) {
					if(version_compare($this->plugins[0]['version'], $update['version']  , '<')) {
						$transient->response[ $basename ] = (object) $update;

						// no-license, invalid-license, duplicate-license
						if(isset($update['update_notice']) AND sizeof($update['update_notice'])) {
							$notices = $update['update_notice'];
							$sppro_notices = array();
							foreach($notices as $detail => $value) {
								$sppro_notices[$detail] = $value;
							}
							update_option('sppro_notices', $sppro_notices);
						}
						if(isset($update['in_plugin_update_message'])) {
							update_option('sppro_plugin_message', $update['in_plugin_update_message']);
						}
						if(isset($update['update_url']) AND !empty($update['update_url'])) {
							update_option('sppro_update_url', $update['update_url']);
						}
						if(isset($update['delete_update_url'])) {
							$this->delete_option('sppro_update_url');
						}
						if(isset($update['is_activated'])) {
							update_option('sppro_is_activated', $update['is_activated']);
						}
					}
					return $transient;
				}
			}
		}

		public function delete_option($key='sppro_notices') {
			delete_option($key);
		}

		public function show_notice($n) {

			if(!empty($n['message'])) {
				echo sprintf( '<div id="message" class="notice notice-'.$n['class'] . ' ' . $n['is-dismissible'].'" style="padding-top:5px;padding-bottom:5px;"><div class="sppro_notice_left" style="display:inline-block; vertical-align:middle;"><img style="float:left;margin-right:5px;max-height:75px;" title="'.SPPRO_PLUGIN_TITLE.'" src="'.SPPRO_PLUGIN_IMG_URL.'/slick-popup-logo.png"></div><div class="sppro_notice_right"><h3 style="margin-bottom:0;">%s</h3><p>%s</p></div></div>', SPPRO_PLUGIN_TITLE . ' - Easiest way to create popups', sprintf( $n['message'] ) );
				echo '<style>
						.sppro_notice_right {
							max-width: calc(100% - 120px);
							display: inline-block; 
							vertical-align: middle; 
							margin-left: 10px;
						}
					</style>';
			}

		}

		/**
		 * Notice Area Updater
		 * Since Version 2.0
		 *
		 * Echo the appropriate message after each action
		 */
		public function sppro_notices_for_update() {
			$sppro_notices = get_option('sppro_notices', array('is-shown'=>'', 'message'=>''));

			if(is_array($sppro_notices) && isset($sppro_notices['is-shown']) && !$sppro_notices['is-shown']) {
				$this->show_notice($sppro_notices);
				$sppro_notices['is-shown'] = true;

				// Update is-shown of the notice as the notice is displayed
				if(isset($sppro_notices['id']) && in_array($sppro_notices['id'], array('no-license', 'is-activated'))) {
					$sppro_notices['is-shown'] = false;
				}

				update_option('sppro_notices', $sppro_notices);

				return;
			}
			return;
		}


		public function modify_plugin_update_message( $plugin_data, $response ) {

			global $sp_opts;
			$message = get_option('sppro_plugin_message');

			// bail ealry if has key
			// Commented out for better control on this message from API
			// $license_key = get_option('sppro_license_key');
			// if( isset($sp_opts) AND isset($license_key) AND !empty($license_key) ) return;

			// bail ealry if no update_message
			if(!isset($message) OR empty($message)) return;

			echo'<br />' . sprintf( $message );
		}


		public function request( $query = '/omakupd/v1/plugins/info', $body = array() ) {

			// vars
			if(!$url = get_option('sppro_update_url') )
				$url = 'Overridden';

			// $url = 'http://localhost/clients/omak/pluginstesting';

			$url .= '/wp-json' . $query . '/';

            // post
			$raw_response = wp_remote_post( $url, array(
				'timeout'	=> 10,
				'body'		=> $body
			));

			// wp error
			if( is_wp_error($raw_response) ) {
				return $raw_response;

			// http error
			} elseif( wp_remote_retrieve_response_code($raw_response) != 200 ) {
				return new WP_Error( 'server_error', wp_remote_retrieve_response_message($raw_response) );
			}

			// decode response
			$json = json_decode( wp_remote_retrieve_body($raw_response), true );

			// allow non json value
			if( $json === null ) {
				return wp_remote_retrieve_body($raw_response);
			}

			// return
			return $json;
		}


		/**
		 * Returns the Response after Activating the License Key
		 *
		 * @return string $response
		 */
		public function activate_license($license_key) {

			$this->plugins[0]['key'] = $license_key;

			$post = array(
				'plugins' => wp_json_encode($this->plugins),
				'wp' => wp_json_encode($this->wp),
			);

			$response = $this->request('/omakupd/v1/plugins/activate', $post);

			// return
			return $response;
		}

		/**
		 * Returns the Response after Deactivating the License Key
		 *
		 * @return string $response
		 */
		public function deactivate_license($license_key) {

			$this->plugins[0]['key'] = $license_key;

			$post = array(
				'plugins' => wp_json_encode($this->plugins),
				'wp' => wp_json_encode($this->wp),
			);

            // return
			return $this->request('/omakupd/v1/plugins/deactivate', $post);
		}

        /**
		 * Updates Page
		 * Since Version 2.2.3
		 */
		public function sppro_updates_page() { ?>
			<style>
				input[type="text"]{
					width: 100%;
					display: block;
				}
				label {
					font-weight: bold;
				}
				.result-area {
					line-height: 1.5em;
					padding: 10px 15px;
				}
			</style>

			<?php
				$purchase_code = get_option('sppro_license_key', '');
				$is_activated = get_option('sppro_is_activated', false);
				$disabled = (!empty($purchase_code) and $is_activated) ? 'disabled' : '';
			?>
			<div class="notice-info settings-error notice is-dismissible"><!-- style="background: azure" -->
				<p class="font-weight-bold">For documentation and demos please visit <em><a target="_blank" href="https://www.slickpopup.com">www.slickpopup.com</a></em></p>
			</div>
			<div class="wrap">
				<div class="col-md-12 card">
					<span class="card-title text-center m-2 display-4"><?php echo esc_html__( "Updates", 'sp-pro-txt-domain' ); ?></span>
					<div class="card-body">
						<div class="row">
							<div class="card col-md-12">
								<span class="card-title text-left m-2 text-info font-weight-bold" style="font-size: 28px;"><?php echo esc_html__( "Purchase Code Information", 'sp-pro-txt-domain' ); ?></span>
								<form method="post" class="sppro-update-license mt-2" action="">
									<div class="input-group mb-3">
									    <div class="input-group-prepend">
									      <span class="input-group-text"><?php echo esc_html__( "Purchase Code", 'sp-pro-txt-domain' ); ?></span>
									    </div>
									    <input id="purchase_code" type="text" class="form-control" name="purchase_code" placeholder="<?php echo esc_html__( "Enter your Purchase Code", 'sp-pro-txt-domain' ); ?>" value="<?php echo $purchase_code; ?>" <?php echo $disabled ?> required>
									</div>
									<div class="input-group mt-2 mx-0 mb-1">
										<?php if(empty($purchase_code) OR !$is_activated) { ?>
											<input type="hidden" name="action" value="activate">
											<input type="submit" value="Activate License" class="btn btn-outline-primary btn-xs sp-update-license">
										<?php } else { ?>
											<input type="hidden" name="action" value="deactivate">
											<input type="submit" value="Deactivate License" class="btn btn-warning btn-xs sp-update-license">
										<?php } ?>
										<span class="sp-loader" style="margin-left:10px;visibility:hidden;"><img alt="Spinner Icon" src="<?php site_url('/wp-includes/images/spinner-2x.gif');?>" /></span>
									</div>
									<div class="result-area"></div>
								</form>
							</div>
						</div>
						<div class="row">
							<div class="card col-md-12">
								<span class="card-title text-left m-2 text-info font-weight-bold" style="font-size: 28px;"><?php esc_html__( "Update Information", 'sp-pro-txt-domain' ); ?></span>
								<div class="row">
									<div class="col-md-12">
										<p class="lead"><?php echo esc_html('In order to get the Automatic Updates of the Slick Popup Pro, all you need to do is to enter your Purchase code and then click on the ','sp-pro-txt-domain'); ?><strong class="text-info">"<?php echo esc_html('Activate License','sp-pro-txt-domain'); ?>"</strong><?php echo esc_html(' button. This Purchase code is for single use only.', 'sp-pro-txt-domain'); ?></p>
										<p class="lead">
											<strong><?php echo esc_html("If you don't have a ",'sp-pro-txt-domain'); ?><em class="text-info"><?php echo esc_html('Purchase Code','sp-pro-txt-domain'); ?></em><?php echo esc_html(', please see ','sp-pro-txt-domain'); ?><a class="text-info" target="_blank" href="https://codecanyon.net/item/slick-popup-pro/16115931?ref=OmAkSols"><?php echo esc_html('details & pricing','sp-pro-txt-domain'); ?></a><?php echo esc_html(' to buy one to enable automatic updates.','sp-pro-txt-domain'); ?></strong>
										</p>
									</div>
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>
		<?php }

		public function action_sppro_update_license() {
			//print_r( $_POST['fields'] );
			$ajaxy = array();
			$errors = array();

			if( !isset($_POST) OR !isset($_POST['fields']) OR empty($_POST['fields']) ) {
				$ajaxy['reason'] = 'Nothing sent to server, please retry.';
			}

			parse_str($_POST['fields'], $fields);
			//wp_send_json_error($fields); wp_die();

			// If Nothing is posted through AJAX
			if( !isset($fields['purchase_code']) OR empty($fields['purchase_code']) ) {
				$errors[] = 'Please enter the purchase code.';
			}

			if(sizeof($errors)) {
				$ajaxy['reason'] = implode('<br>', $errors);
				wp_send_json_error($ajaxy);
				wp_die();
			}

			if($fields['action']=='activate') {

				$response = $this->activate_license($fields['purchase_code']);

				update_option('sppro_is_activated', $response['is_activated']);

				$ajaxy['reason'] = $response['message'];
				$ajaxy['reload'] = $response['reload'];

				if($response['is_activated']) {
					update_option('sppro_license_key', $fields['purchase_code']);

					// update sppro_notice for user info
					$sppro_notices = array(
						'id' => 'is-activated',
						'class' => 'error',
					    'is-dismissible' => 'is-dismissible',
					    'is-shown' => false,
					    'message' => esc_html__( 'You have successfully activated your purchase code.', 'sp-pro-txt-domain' ).'<span style="display: block; margin: 0.5em 0.5em 0 0; clear: both; font-weight: bold;">'.sppro_get_help_links().'</span>',
					);
					update_option('sppro_notices', $sppro_notices);

					// send success result
					wp_send_json_success($ajaxy);
				}
				else
					wp_send_json_error($ajaxy);

				wp_die();
			}
			elseif($fields['action']=='deactivate') {
				$response = $this->deactivate_license($fields['purchase_code']);
                $ajaxy['reason'] = $response['message'];
				$ajaxy['reload'] = $response['reload'];

				if($response['is_deactivated']) {
					update_option('sppro_is_activated', $response['is_deactivated']);
					update_option('sppro_license_key', '');

					$sppro_notices = array(
						'id' => 'no-license',
						'class' => 'error',
					    'is-dismissible' => 'is-dismissible',
					    'is-shown' => false,
					    'message' => 'You have successfully deactivated the purchase code. Now, you can use this on any other website you want. Please note that one purchase code can only be used within one site at a time. If you want to use this on multiple websites then please buy another license, see <a href="https://codecanyon.net/item/slick-popup-pro/16115931?ref=OmAkSols">details & pricing</a>.',
					);
					update_option('sppro_notices', $sppro_notices);

					wp_send_json_success($ajaxy);
				}
				else {
					wp_send_json_error($ajaxy);
				}

				wp_die();
			}

			$ajaxy['reason'] = 'Something went wrong, please refresh this page and try again.';
			wp_send_json_error($ajaxy); wp_die();
		}
	}
}