<?php

namespace WPSecurityNinja\Plugin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Dashboard Widget Module
 *
 * Handles the Security Ninja dashboard widget functionality
 * Only loads on dashboard page for efficiency
 *
 * @author  Lars Koudal
 * @since   v001
 * @version v10  Sunday, May 11th,2025
 */
class Wf_Sn_Dashboard_Widget {

	/**
	 * Initialize the dashboard widget module
	 *
	 * @author  Lars Koudal
	 * @since   v00.1
	 * @version v10  Sunday, May 11th,2025 * @access  public static
	 * @return  void
	 */
	public static function init() {
    add_action( 'wp_dashboard_setup', array( __NAMESPACE__ . '\\Wf_Sn_Dashboard_Widget', 'add_dashboard_widgets' ) );

    add_action( 'admin_enqueue_scripts', array( __NAMESPACE__ . '\\Wf_Sn_Dashboard_Widget', 'admin_enqueue_scripts' ) );


	}

  public static function admin_enqueue_scripts( $hook ) {
    if ( 'wp-admin/update.php' === $GLOBALS['pagenow'] ) {
      return;
    }

    if ( 'index.php' === $hook ) {
      // Enqueue CSS
      wp_enqueue_style(
        'security-ninja-dashboard-css',
        WF_SN_PLUGIN_URL . 'modules/dashboard-widget/css/min/dashboard-widget-min.css',
        array(),
        filemtime( WF_SN_PLUGIN_DIR . 'modules/dashboard-widget/css/min/dashboard-widget-min.css' )
      );
      
    }

  }


	/**
		 * Add a widget to the dashboard.
		 *
		 * @author  Lars Koudal
		 * @since   v0.0.1
		 * @version v1.0.0  Wednesday, January 13th, 2021.
		 * @access  public static
		 * @return  void
		 */
		public static function add_dashboard_widgets() {
			if ( secnin_fs()->can_use_premium_code__premium_only() ) {
				// Checks if whitelabel is active or not...
				if ( class_exists( '\\WPSecurityNinja\\Plugin\\Wf_Sn_Wl' ) && method_exists( '\\WPSecurityNinja\\Plugin\\Wf_Sn_Wl', 'is_active' ) ) {
					if ( \WPSecurityNinja\Plugin\Wf_Sn_Wl::is_active() ) {
						return;
					}
				}
			}
			wp_add_dashboard_widget(
				'wpsn_dashboard_widget',
				'WP Security Ninja',
				// Is not whitelabelled, so nevermind
				array( __NAMESPACE__ . '\\Wf_Sn_Dashboard_Widget', 'wpsn_dashboard_widget_render' )
			);
		}


		/**
		 * Renders dashboard widget
		 *
		 * @author  Lars Koudal
		 * @since   v0.1
		 * @version v1.0.0  Wednesday, January 13221 * @access  public static
		 * @return  void
		 */
		public static function wpsn_dashboard_widget_render() {
			// Check if whitelabel is active
			$is_whitelabel = false;
			if ( secnin_fs()->can_use_premium_code__premium_only() ) {
				if ( class_exists( '\\WPSecurityNinja\\Plugin\\Wf_Sn_Wl' ) && method_exists( '\\WPSecurityNinja\\Plugin\\Wf_Sn_Wl', 'is_active' ) ) {
					$is_whitelabel = \WPSecurityNinja\Plugin\Wf_Sn_Wl::is_active();
				}
			}

			echo '<div class="secnin-dashboard-widget">';

			// Show icon only if not whitelabeled
			if ( ! $is_whitelabel && class_exists( __NAMESPACE__ . '\\Wf_Sn' ) ) {
				$icon_url = \WPSecurityNinja\Plugin\Utils::get_icon_svg( true, '000000' );
				echo '<img src="' . esc_url( $icon_url ) . '" class="secnin-widget-icon">';
			}

			// Firewall Protection (Primary Focus) - with caching
			if ( secnin_fs()->can_use_premium_code__premium_only() ) {
				$cache_key = 'secnin_dashboard_firewall_stats';
				$cached_stats = get_transient( $cache_key );
				
				if ( false === $cached_stats ) {
					global $wpdb;
					$table_name = $wpdb->prefix . 'wf_sn_el';
					
					// Check if table exists first
					$table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) );
					
					if ( $table_exists ) {
						// Get recent security events (last 24) with more attack types
						$recent_events = $wpdb->get_results(
							"SELECT action, COUNT(*) as count 
								FROM `{$table_name}` 
								WHERE `action` IN ('blocked_ip', 'blocked_ip_banned', 'banned_ip', 'blacklisted_IP', 'blocked_ip_suspicious_request', 'failed_login', 'login_denied_banned_ip', 'login_form_blocked_ip', 'blockadminlogin', 'login_error', 'blocked_ip_country_ban', 'firewall_ip_banned') 
								AND `module` = 'security_ninja' 
								AND `timestamp` >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
								GROUP BY action
								LIMIT 1000;"
						);

						$total_events = 0;
						foreach ( $recent_events as $event ) {
							$total_events += $event->count;
						}

						$blocked_count = get_option( 'wf_sn_cf_blocked_count' );					
						$cached_stats = array(
							'total_events' => $total_events,
							'blocked_count' => $blocked_count,
							'has_activity' => ( $total_events > 0 || $blocked_count > 0 ),
							'table_exists' => true
						);
					} else {
						$cached_stats = array(
							'total_events' => 0,
							'blocked_count' => 0,
							'has_activity' => false,
							'table_exists' => false
						);
					}
					
					// Cache for 5 minutes to reduce DB load
					set_transient( $cache_key, $cached_stats, 5 * MINUTE_IN_SECONDS );
				}
				
				// Show firewall status regardless of activity
				?>
				<div class="secnin-status-card secnin-status-card--firewall">
					<h3 class="secnin-card-header secnin-card-header--firewall">
						<span class="dashicons dashicons-shield" style="color: #28a745;"></span> 
						<?php esc_html_e( 'Firewall Protection', 'security-ninja' ); ?>
					</h3>
					<?php if ( $cached_stats['has_activity'] ) { ?>
						<div class="secnin-card-stats secnin-card-stats--firewall">
							<span><?php esc_html_e( 'Last 24h: ', 'security-ninja' ); ?> <strong><?php echo number_format_i18n( $cached_stats['total_events'] ); ?></strong></span>
							<?php if ( $cached_stats['blocked_count'] > 0 ) { ?>
								<span><?php esc_html_e( 'Total blocked: ', 'security-ninja' ); ?> <strong><?php echo number_format_i18n( $cached_stats['blocked_count'] ); ?></strong></span>
							<?php } ?>
						</div>
						<div class="secnin-card-link secnin-card-link--firewall">
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=wf-sn#sn_logger' ) ); ?>">
								<?php esc_html_e( 'View details', 'security-ninja' ); ?> →
							</a>
						</div>
					<?php } else { ?>
						<div class="secnin-card-stats secnin-card-stats--firewall">
							<span>✓ <?php esc_html_e( 'No threats detected', 'security-ninja' ); ?></span>
						</div>
						<div class="secnin-card-link secnin-card-link--firewall">
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=wf-sn#sn_logger' ) ); ?>">
								<?php esc_html_e( 'View logs', 'security-ninja' ); ?> →
							</a>
						</div>
					<?php } ?>
				</div>
				<?php
			}

			// Updates Available (Moved up under firewall)
			$updates_cache_key = 'secnin_dashboard_updates';
			$cached_updates = get_transient( $updates_cache_key );
			
			if ( false === $cached_updates ) {
				$plugin_updates = get_plugin_updates();
				$theme_updates = get_site_transient( 'update_themes' );
				$plugin_count = count( $plugin_updates );
				$theme_count = ! empty( $theme_updates->response ) ? count( $theme_updates->response ) : 0;
				$total_updates = $plugin_count + $theme_count;
				
				$cached_updates = array(
					'total_updates' => $total_updates,
					'plugin_count' => $plugin_count,
					'theme_count' => $theme_count
				);
				
				// Cache for 10 minutes
				set_transient( $updates_cache_key, $cached_updates, 10 * MINUTE_IN_SECONDS );
			}

			// Always show updates status
			?>
			<div class="secnin-status-card secnin-status-card--updates">
				<div class="secnin-card-content">
					<span class="secnin-card-header secnin-card-header--updates">
						<?php if ( $cached_updates['total_updates'] > 0 ) { ?>
							<span class="dashicons dashicons-update" style="color: #17a2b8;"></span> 
							<?php esc_html_e( 'Updates Available', 'security-ninja' ); ?>
						<?php } else { ?>
							<span class="dashicons dashicons-yes-alt" style="color: #28a745;"></span> 
							<?php esc_html_e( 'All Updates Applied', 'security-ninja' ); ?>
						<?php } ?>
					</span>
					<div class="secnin-score-display">
						<?php if ( $cached_updates['total_updates'] > 0 ) { ?>
							<span class="secnin-update-count"><?php echo number_format_i18n( $cached_updates['total_updates'] ); ?></span>
							<br>
							<a href="<?php echo esc_url( admin_url( 'update-core.php' ) ); ?>" class="secnin-card-link secnin-card-link--updates">
								<?php esc_html_e( 'Update', 'security-ninja' ); ?> →
							</a>
						<?php } else { ?>
							<span class="secnin-update-count" style="color: #28a745;">✓</span>
							<br>
							<a href="<?php echo esc_url( admin_url( 'update-core.php' ) ); ?>" class="secnin-card-link secnin-card-link--updates">
								<?php esc_html_e( 'Check Updates', 'security-ninja' ); ?> →
							</a>
						<?php } ?>
					</div>
				</div>
			</div>
			<?php

			// Security Score (Compact - matching firewall style)
			$test_scores = \WPSecurityNinja\Plugin\wf_sn::return_test_scores();
			?>
			<div class="secnin-status-card secnin-status-card--security">
				<div class="secnin-card-content">
					<span class="secnin-card-header secnin-card-header--security">
						<span class="dashicons dashicons-shield-alt" style="color: #6c757d;"></span> 
						<?php esc_html_e( 'Security Score', 'security-ninja' ); ?>
					</span>
					<div class="secnin-score-display">
						<?php if ( isset( $test_scores['score'] ) && '0' !== $test_scores['score'] ) { ?>
							<span class="secnin-score-value"><?php echo intval( $test_scores['score'] ); ?>%</span>
							<br>
							<div class="secnin-score-breakdown">
								<span class="secnin-score-good">✓ <?php echo intval( $test_scores['good'] ); ?></span>
								<span class="secnin-score-warning">⚠ <?php echo intval( $test_scores['warning'] ); ?></span>
								<span class="secnin-score-bad">✗ <?php echo intval( $test_scores['bad'] ); ?></span>
								<a href="<?php echo esc_url( admin_url( 'admin.php?page=wf-sn#sn_tests' ) ); ?>" class="secnin-card-link secnin-card-link--security">
								<?php esc_html_e( 'Run Tests', 'security-ninja' ); ?> →
							</a>
							</div>
						<?php } else { ?>
							<span class="secnin-score-value" style="color: #6c757d;">-</span>
							<br>
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=wf-sn' ) ); ?>" class="secnin-card-link secnin-card-link--security">
								<?php esc_html_e( 'Run Tests', 'security-ninja' ); ?> →
							</a>
						<?php } ?>
					</div>
				</div>
			</div>
			<?php

			// Vulnerabilities (Compact)
			$vulns = \WPSecurityNinja\Plugin\Wf_Sn_Vu::return_vulnerabilities();
			if ( $vulns ) {
				$total = \WPSecurityNinja\Plugin\Wf_Sn_Vu::return_vuln_count();
				?>
				<div class="secnin-status-card secnin-status-card--vulnerabilities">
					<div class="secnin-card-content">
						<span class="secnin-card-header secnin-card-header--vulnerabilities">
							<span class="dashicons dashicons-warning" style="color: #f39c12;"></span> 
							<?php echo esc_html( sprintf( _n( '%s Vulnerability Found', '%s Vulnerabilities Found', $total, 'security-ninja' ), number_format_i18n( $total ) ) ); ?>
						</span>
						<a href="<?php echo esc_url( admin_url( 'admin.php?page=wf-sn#sn_vuln' ) ); ?>" class="secnin-card-link secnin-card-link--vulnerabilities">
							<?php esc_html_e( 'View', 'security-ninja' ); ?> →
						</a>
					</div>
				</div>
				<?php
			}
			
			echo '</div>'; // Close secnin-dashboard-widget div
		}


} 