<?php

namespace Salesloo;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use \WP_Error;

/**
 * User classes
 */
class User
{
    private $userdata;

    public function __construct($user_id = false)
    {
        if ($user_id) {
            $this->userdata = get_userdata(intval($user_id));
        } else {
            $this->userdata = wp_get_current_user();
        }
    }

    /**
     * getter
     * @param  string $key
     * @return mixed
     */
    public function __get($key)
    {
        $data = get_object_vars($this->userdata);

        if (array_key_exists($key, $data))
            return $data[$key];

        return NULL;
    }

    /**
     * username_exists
     *
     * @param  string $username
     * @return mixed
     */
    public static function username_exists($username)
    {
        if (username_exists($username)) {
            return new WP_Error('failed', __('This username already exists, please choose another username', 'salesloo'));
        }

        return sanitize_text_field($username);
    }

    /**
     * check_is_email_exists
     *
     * @param  string $email
     * @return mixed
     */
    public static function email_exists($email)
    {
        $email = sanitize_text_field($email);
        if (!is_email($email)) {
            return new WP_Error('error', __('Invalid email format, Please enter a valid email address', 'salesloo'));
        }

        if (email_exists($email)) {
            return new WP_Error('error', sprintf(__('This email has been registered, please login with this email or use another email', 'salesloo'), $email));
        }

        return sanitize_email($email);
    }

    /**
     * register user
     *
     * @param  mixed $user
     * @return mixed
     */
    public static function register($user)
    {
        $required_fields = [
            'user_login' => 'Username',
            'first_name'  => 'First Name',
            'last_name' => 'Last Name',
            'user_email' => 'Email',
            'pass'  => 'Password'
        ];

        foreach ((array)$required_fields as $field => $label) {
            if (!isset($user[$field]) || isset($user[$field]) && !$user[$field]) {
                return new WP_Error('failed', sprintf(__('Field "%s" is required and can\'t be empty', 'salesloo'), $label));
            }
        }

        $username = sanitize_text_field($user['user_login']);
        $fname    = $user['first_name'];
        $lname    = $user['last_name'];
        $email    = sanitize_email($user['user_email']);
        $pass     = sanitize_text_field($user['pass']);

        $username = self::username_exists($username);

        if (is_wp_error($username)) {
            return $username;
        }

        $email = self::email_exists($email);
        if (is_wp_error($email)) {
            return $email;
        }

        $user_id = wp_create_user($username, $pass, $email);

        if (!is_wp_error($user_id)) {
            foreach ((array)$required_fields as $field => $label) {
                unset($user[$field]);
            }

            $userdata = [
                'ID' => $user_id
            ];
            $userdata['first_name'] = $fname;
            $userdata['last_name']  = $lname;

            if (isset($user['phone'])) {
                $userdata['phone'] = sanitize_text_field($user['phone']);
            }
            wp_update_user($userdata);

            foreach ((array)$user as $key => $value) {
                $value = is_array($value) ? $value : sanitize_text_field($value);
                update_user_meta($user_id, sanitize_text_field($key), $value);
            }

            $data = [
                'user_id' => $user_id,
                'passwd' => $pass
            ];

            salesloo_add_notification('register', $data);
            salesloo_add_event('register', $data);
        }

        return $user_id;
    }
}
