<?php

namespace Salesloo;

use Salesloo\Abstracts\Setting;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Settings
 */
class Settings
{

    private $current_tab;

    private $current_section;

    private $current_fields = '';

    /**
     * Register Menu
     * 
     * register system info menu
     */
    public function register_menu($submenu)
    {
        $submenu[] = [
            'page_title' => __('Salesloo Settings', 'salesloo'),
            'menu_title' => __('Settings', 'salesloo'),
            'capability' => 'manage_options',
            'slug'       => 'salesloo-settings',
            'callback'   => [$this, 'page'],
            'position'   => 80
        ];

        return $submenu;
    }


    /**
     * load_setting_pages
     *
     * @return void
     */
    public function load()
    {
        $settings_filename = [
            'general-user-area',
            'general-currency',
            'general-cronjob',
            'design-color-scheme',
            'affiliate-general',
            'affiliate-tracking',
            'notification-email',
            'tracking-pixelfb',
            'tracking-pixeltiktok',
            'checkout-general',
            'checkout-unique-number',
            'checkout-coupon',
            'invoice-general'
        ];

        foreach ($settings_filename as $filename) {
            require_once SALESLOO_PATH . 'includes/setting-pages/' . $filename . '.php';

            $class_name = str_replace('-', '_', $filename);

            $class_name = __NAMESPACE__ . '\Setting_Page\\' . $class_name;

            self::add_section_page(new $class_name, 1);
        }

        require_once SALESLOO_PATH . 'includes/setting-pages/payment-method.php';

        $payment_method_classes = \Salesloo\Plugin::instance()->payment_method->all();
        foreach ($payment_method_classes as $class) {
            self::add_section_page(new Setting_Page\Payment_Method($class), 1);
        }
    }

    /**
     * setting tabs
     *
     * @return string
     */
    private function tabs()
    {
        $tabs = '';

        $_tabs = [
            'general'        => __('General', 'salesloo'),
            'design'         => __('Design', 'salesloo'),
            'affiliate'      => __('Affiliate', 'salesloo'),
            'payment_method' => __('Payment Method', 'salesloo'),
            'notification'   => __('Notification', 'salesloo'),
            'tracking'       => __('Tracking', 'salesloo'),
            'checkout'       => __('Checkout', 'salesloo'),
            'invoice'        => __('Invoice', 'salesloo')
        ];

        $_tabs = apply_filters('salesloo/admin/settings/tabs', $_tabs);
        $this->current_tab = isset($_GET['tab']) ? trim($_GET['tab']) : 'general';

        foreach ($_tabs as $key => $name) :
            $is_active = $this->current_tab == $key ? ' nav-tab-active' : '';
            $url = add_query_arg('tab', $key);
            if (isset($_GET['section'])) {
                $url = remove_query_arg('section', $url);
            }
            $tabs .= '<a href="' . $url . '" class="nav-tab' . $is_active . '">' . $name . '</a>';
        endforeach;

        return $tabs;
    }

    /**
     * setting sections
     *
     * @return string
     */
    public function sections()
    {
        $sections = '';

        $_sections = apply_filters('salesloo/admin/settings/' . $this->current_tab . '/sections', array());
        $this->current_section = array_key_first($_sections);
        if (isset($_GET['section']) && array_key_exists($_GET['section'], $_sections)) {
            $this->current_section = trim($_GET['section']);
        }

        $section_html = [];
        foreach ((array)$_sections as $section_key => $val) {

            if (!isset($val['label']) || !isset($val['callback'])) continue;

            $is_section_active = '';
            if ($this->current_section == $section_key) {
                $this->current_fields = $val['callback'];
                $is_section_active = ' class="current"';
            }

            $section_html[] = sprintf(
                '<li><a href="%s"%s>%s</a></li>',
                add_query_arg('section', $section_key),
                $is_section_active,
                $val['label']
            );
        }

        $sections .= sprintf(
            '<div class="" style="position:relative;width: auto;margin-bottom: 10px;"><ul class="subsubsub">%s</ul></div><br class="clear">',
            implode(' | ', $section_html)
        );

        return $sections;
    }

    /**
     * Page
     * 
     * Output settings page
     */
    public function page()
    {

        $this->load();
        $tabs     = $this->tabs();
        $sections = $this->sections();

        echo '<div class="wrap">';
        echo '<h2>' . __('Settings', 'salesloo') . '</h2>';
        echo '<h2 class="nav-tab-wrapper wp-clearfix">' . $tabs . '</h2>';
        echo $sections;
        echo '<form action="" method="post" enctype="multipart/form-data" style="margin-top:30px">';
        if ($this->current_fields) {
            call_user_func($this->current_fields);
        }
        wp_nonce_field('salesloo', '__nonce');
        echo '</form>';
        echo '</div>';
    }

    /**
     * add section page
     *
     * @param  Abstracts\Setting_Page $setting_page
     * @return void
     */
    public static function add_section_page(Abstracts\Setting_Page $setting_page, $priority = 10)
    {
        add_filter('salesloo/admin/settings/' . $setting_page->tab . '/sections', function ($sections) use ($setting_page) {

            $sections[$setting_page->name] = [
                'label' => $setting_page->label,
                'callback' => [$setting_page, 'fields']
            ];

            return $sections;
        }, $priority);
    }

    /**
     * save_settings
     *
     * @return void
     */
    public function save_settings()
    {
        if (isset($_POST['save']) && isset($_POST['__nonce'])) :
            if (wp_verify_nonce($_POST['__nonce'], 'salesloo')) :

                unset($_POST['save']);
                unset($_POST['__nonce']);
                unset($_POST['_wp_http_referer']);

                foreach ($_POST as $key => $value) {
                    \salesloo_update_option($key, $value);
                }

                \flush_rewrite_rules();

                add_action('admin_notices', function () {
                    echo '<div id="message" class="updated notice notice-success"><p><strong>' . __('Your settings have been saved.', 'salesloo') . '</strong></p></div>';
                });
            endif;
        endif;
    }

    /**
     * Constructor
     */
    public function __construct()
    {
        add_filter('salesloo/admin/submenu', [$this, 'register_menu'], 1, 10);
        add_action('admin_init', [$this, 'save_settings']);
    }
}
