<?php

namespace Salesloo\Models;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Salesloo\Abstracts\Database;

/**
 * Product Model
 */
class Product extends Database
{
    protected $table = 'salesloo_product';

    protected $columns = [
        'ID'           => 'integer',
        'slug'         => 'string',
        'title'        => 'string',
        'thumbnail_id' => 'integer',
        'description'  => 'content',
        'duration'     => 'string',
        'affiliate'    => 'integer',
        'items'        => 'array',
        'status'       => 'string',
        'created_at'   => 'string',
        'updated_at'   => 'string'
    ];

    protected $attributes = [];

    protected $meta = [];

    protected $price = 0;

    /**
     * get thumbnail url
     */
    public function get_thumbnail_url()
    {
        return $this->thumbnail_id > 0 ? wp_get_attachment_url($this->thumbnail_id) : false;
    }

    /**
     * set price
     *
     * @param  int $price
     * @return void
     */
    public function set_price($price)
    {
        $this->price = salesloo_convert_price($price);

        return $this;
    }

    /**
     * get price
     *
     * @return string
     */
    public function get_price($type = 'regular')
    {
        $price = $this->price;

        if ('regular' != $type) {
            $price = $this->meta($type . '_price');
        }

        if (false === $price || empty($price)) {
            $price = $this->meta('regular_price');
        }

        return floatval($price);
    }

    /**
     * get fixed commission
     *
     * @param  float $price
     * @return float
     */
    public function get_commission($price = 0)
    {
        if (!$price) {
            $price  = $this->get_price();
        }

        $raw_commission = $this->get_raw_affiliate_commission();
        if ('percen' == $raw_commission['type']) {
            $commission = floatval($raw_commission['value']) * $price;
            $commission = $commission / 100;
        } else {
            $commission = $raw_commission['value'];
        };

        return floatval($commission);
    }


    /**
     * get raw commission
     *
     * @return array
     */
    public function get_raw_affiliate_commission()
    {
        $commission_array = [];
        $commission = $this->meta('affiliate_commission');

        if (strpos($commission, '%')) {
            $commission = str_replace('%', '', $commission);

            $commission_array = [
                'type' => 'percen',
                'value' => floatval($commission)
            ];
        } else {
            $commission_array = [
                'type' => 'fixed',
                'value' => floatval($commission)
            ];
        }

        return $commission_array;
    }

    /**
     * get meta
     */
    public function meta($meta_key = false)
    {
        if (empty($this->meta)) {
            $get_all_meta = Product_Meta::query('WHERE product_id = %d', $this->ID)->get();

            $all_meta = [];

            if ($get_all_meta) {
                foreach ($get_all_meta as $key => $value) {
                    if (isset($all_meta[$value->meta_key])) {

                        if (is_array($all_meta[$value->meta_key])) {
                            $all_meta[$value->meta_key][] = \maybe_unserialize($value->meta_value);
                        } else {
                            $temp = array(
                                $all_meta[$value->meta_key],
                                \maybe_unserialize($value->meta_value)
                            );

                            unset($all_meta[$value->meta_key]);

                            $all_meta[$value->meta_key] = $temp;
                        }
                    } else {
                        $all_meta[$value->meta_key] = \maybe_unserialize($value->meta_value);
                    }
                }
            }

            $this->meta = $all_meta;
        }

        if ($meta_key) {
            return isset($this->meta[$meta_key]) ? $this->meta[$meta_key] : false;
        }

        return $this->meta;
    }
}
