<?php


namespace ReviewX\Modules;

use ReviewX\Controllers\Admin\Core\ReviewxMetaBox;

class Gatekeeper
{
    private $review_status = [];

    /**
     * 
     *
     * @param [type] $productID
     * @param boolean $orderID
     * @return boolean
     */
    public static function hasPermit($productID, $orderID = false)
    {
        if (\is_user_logged_in()) {
            if (wc_customer_bought_product( '', get_current_user_id(), $productID )) {
                $review_status['success'] = (new static())->getLogForVerifiedUser($productID, $orderID, get_current_user_id());                
                $review_status['msg'] = __( 'Once reviewed item can not be re-reviewed without purchase it again.', 'reviewx' );
                return $review_status;
                // return (new static())->getLogForVerifiedUser($productID, $orderID);
            } else {             
                $review_status['success'] = (new static())->wcGuestPermission() && (new static())->getLogForNonVerifiedUser($productID);
                $review_status['msg'] = __( 'Can not submit (re)review without purchase.', 'reviewx' );
                return $review_status;
                // return (new static())->wcGuestPermission() && (new static())->getLogForNonVerifiedUser($productID);
            }
        } else {
            $review_status['success'] = (new static())->wcGuestPermission() && (new static())->getLogForGuest($productID);
            $review_status['msg'] = __( 'Once reviewed item can not be (re)reviewed without logged in.', 'reviewx' );
            return $review_status;
            // return (new static())->wcGuestPermission() && (new static())->getLogForGuest($productID);
        }
    }

    /**
     * Set log for user
     *
     * @param [type] $productID
     * @param boolean $orderID
     * @param boolean $reviewID
     * @return void
     */
    public static function setLogForUser($productID, $orderID = false, $reviewID = false)
    {
        if (\is_user_logged_in()) {
            if (wc_customer_bought_product( '', get_current_user_id(), $productID )) {
                (new static())->setLogForVerifiedUser($productID, $orderID, $reviewID);
            } else {
                (new static())->setLogForNonVerifiedUser($productID, $reviewID);
            }
        } else {
            (new static())->setLogForGuest($productID, $reviewID);
        }
    }

    /**
     * Check order status
     *
     * @param [type] $orderID
     * @return void
     */
    private function checkOrderStatus($orderID)
    {
        $order = wc_get_order($orderID);
        $orderStatus = wc_get_order_status_name( $order->get_status() );
        $settings = ReviewxMetaBox::get_metabox_settings();
        $mapStatus = [
            'pending_payment' => __('Pending Payment', 'reviewx'),
            'processing' => __('Processing', 'reviewx'),
            'on_hold' => __('On hold', 'reviewx'),
            'completed' => __('Completed', 'reviewx'),
            'cancelled' => __('Cancelled', 'reviewx'),
            'refunded' => __('Refunded', 'reviewx'),
            'failed' => __('Failed', 'reviewx')
        ];
        $orderStatuses = [];
        foreach ($mapStatus as $key => $label) {
            if ($settings->{$key} == 1) {
                $orderStatuses[] = $label;
            }
        }

        return in_array($orderStatus, $orderStatuses);
    }

    /**
     * Set log for verified user
     *
     * @param [type] $productID
     * @param [type] $orderID
     * @param [type] $reviewID
     * @return void
     */
    private function setLogForVerifiedUser($productID, $orderID, $reviewID)
    {
        if ($orderID) {
            $key = sprintf("RX_VERIFIED_%s_ORDER_%s_PRODUCT_%s", get_current_user_id(), $orderID, $productID);
            \update_option($key, 1);
            $this->reviewIssuedBy($reviewID, $key);
        } else {
            $orders = (new static())->getOrders('', get_current_user_id(), $productID);
            foreach ($orders as $order_id) {
                $key = sprintf("RX_VERIFIED_%s_ORDER_%s_PRODUCT_%s", get_current_user_id(), $order_id, $productID);
                if (!get_option($key)) {
                    \update_option($key, 1);
                    $this->reviewIssuedBy($reviewID, $key);
                    return;
                }
            }
        }
    }

    /**
     * Get log for verified user
     *
     * @param [type] $productID
     * @param [type] $orderID
     * @param [type] $userId
     * @return void
     */
    public function getLogForVerifiedUser($productID, $orderID, $userId)
    {
        if ($orderID) {
            $key = sprintf("RX_VERIFIED_%s_ORDER_%s_PRODUCT_%s", $userId, $orderID, $productID);
            return !get_option($key) && $this->checkOrderStatus($orderID);
        } else {
            $orders = (new static())->getOrders('', $userId, $productID);
            foreach ($orders as $order_id) {
                $key = sprintf("RX_VERIFIED_%s_ORDER_%s_PRODUCT_%s", $userId, $order_id, $productID);
                if (!get_option($key)) {
                    return true && $this->checkOrderStatus($order_id);
                }
            }
            return false;
        }
    }

    /**
     * set log for non verified user
     *
     * @param [type] $productID
     * @param [type] $reviewId
     * @return void
     */
    private function setLogForNonVerifiedUser($productID, $reviewId)
    {
        $key = sprintf("RX_NON_VERIFIED_%s_PRODUCT_%s", get_current_user_id(), $productID);
        \update_option($key, 1);
        $this->reviewIssuedBy($reviewId, $key);
    }

    /**
     * Get log for non verified user
     *
     * @param [type] $productID
     * @return void
     */
    private function getLogForNonVerifiedUser($productID)
    {
        $key = sprintf("RX_NON_VERIFIED_%s_PRODUCT_%s", get_current_user_id(), $productID);
        return (! \get_option($key));
    }

    /**
     * Set log for guest
     *
     * @param [type] $productID
     * @param [type] $reviewId
     * @return void
     */
    private function setLogForGuest($productID, $reviewId)
    {
        $ipAddress = str_replace('.', '_', (new static())->getIP());
        $key = sprintf("RX_GUEST_%s_PRODUCT_%s", $ipAddress, $productID);
        \update_option($key, 1);
        $this->reviewIssuedBy($reviewId, $key);
    }

    /**
     * Get log for guest
     *
     * @param [type] $productID
     * @return void
     */
    private function getLogForGuest($productID)
    {
        $ipAddress = str_replace('.', '_', (new static())->getIP());
        $key = sprintf("RX_GUEST_%s_PRODUCT_%s", $ipAddress, $productID);
        return (! \get_option($key));
    }

    /**
     * Get IP
     *
     * @return void
     */
    private function getIP()
    {
        $ipAddress = $_SERVER['REMOTE_ADDR'];
        if (array_key_exists('HTTP_X_FORWARDED_FOR', $_SERVER)) {
            $ipAddress = array_pop(explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']));
        }
        return $ipAddress;
    }

    /**
     * Get Orders
     *
     * @param [type] $customer_email
     * @param [type] $user_id
     * @param [type] $product_id
     * @return void
     */
    public function getOrders($customer_email, $user_id, $product_id)
    {
        
        $orderIds = [];
        $customer_orders 	= wc_get_orders( array(
            'meta_key' 		=> '_customer_user',
            'meta_value' 	=> $user_id,
            'numberposts' 	=> -1
        ) );

        foreach( $customer_orders as $order ) {
            $orderIds[] 	= method_exists( $order, 'get_id' ) ? $order->get_id() : $order->ID;
        }

        $orderItems = [];

        if (count($orderIds)) {
            $orderItems = wpFluent()->table('woocommerce_order_items')
                                ->whereIn('order_id', $orderIds)
                                ->get();
        }

        $orders = [];

        foreach ($orderItems as $orderItem) {
            $getProductID = wc_get_order_item_meta($orderItem->order_item_id, '_product_id', true);
            if ((int)$getProductID == $product_id) {
                $orders[] = $orderItem->order_id;
            }
        }

        return array_map('absint', $orders);
    }

    /**
     * Get WooCommerce Permission
     *
     * @return void
     */
    private function wcGuestPermission()
    {
        return \get_option( 'woocommerce_review_rating_verification_required' ) === 'no';
    }

    /**
     * Get review issue by
     *
     * @param [type] $reviewId
     * @param [type] $key
     * @return void
     */
    private function reviewIssuedBy($reviewId, $key)
    {
        \update_comment_meta($reviewId, 'review_issued_by', $key);
    }

    /**
     * Remove log
     *
     * @param [type] $reviewId
     * @return void
     */
    public static function removeLog($reviewId)
    {
        $getIssueKey = \get_comment_meta($reviewId, 'review_issued_by', true);
        \delete_option($getIssueKey);
    }

}