<?php

namespace ReviewX\Controllers\Storefront;

use ReviewX\Constants\Filters;
use ReviewX\Controllers\Admin\Core\ReviewxMetaBox;
use ReviewX\Controllers\Controller;

/**
 * Class GraphTemplateLoader
 * @package ReviewX\Controllers\Storefront
 */
class GraphTemplateLoader extends Controller
{
    const MAP_STYLE_VIEW = [
        'default' => 'style-default',
        'elementor' => [
            'graph_style_default'   => 'style-default',
            'graph_style_one'       => "style-one",
            'graph_style_two_free'  => "style-two-free",
        ],
        'reviewx' => [
            'graph_style_default'   => 'style-default',
            'graph_style_one'       => "style-one",
            'graph_style_two'       => "style-two",
            'graph_style_two_free'  => "style-two-free",
        ]
    ];

    public $settings;

    public $elementorController;

    public function __construct()
    {
        $this->setSettings(ReviewxMetaBox::get_metabox_settings());
        $this->setElementorController();
    }

    /**
     * @param mixed $settings
     */
    public function setSettings($settings)
    {
        $this->settings = $settings;
    }

    /**
     * @return mixed
     */
    public function getSettings()
    {
        return $this->settings;
    }

    /**
     * @param $elementorController
     */
    public function setElementorController($elementorController = null)
    {
        $this->elementorController = $elementorController ? : apply_filters(Filters::RX_LOAD_ELEMENTOR_STYLE_CONTROLLER, []);
    }

    /**
     * @return mixed
     */
    public function getElementorController()
    {
        return $this->elementorController;
    }

    /**
     * @param $property
     * @return |null
     */
    protected function getSettingsProperty($property)
    {
        if (isset($this->getSettings()->{$property})) {
            return $this->getSettings()->{$property};
        }
        return null;
    }

    /**
     * @return mixed|null
     */
    protected function elementorGraphStyle()
    {
        if (array_key_exists('rx_graph_type', $this->getElementorController())) {
            return array_get($this->getElementorController(), 'rx_graph_type', null);
        }
        return null;
    }

    /**
     * @return mixed
     */
    protected function getDefaultStyle()
    {
        return array_get(self::MAP_STYLE_VIEW, 'default', 'style-default');
    }

    /**
     * @param $key
     * @return mixed
     */
    protected function getStyleView($key)
    {
        return array_get(self::MAP_STYLE_VIEW, $key, []);
    }

    /**
     * @return mixed
     */
    protected function getStyle()
    {
        $default = $this->getDefaultStyle();

        if ($this->elementorGraphStyle()) {
            return  array_get(
                $this->getStyleView('elementor'),
                $this->elementorGraphStyle(),
                $default
            );
        }

        if ($this->getSettingsProperty('graph_style')) {
            return  array_get(
                $this->getStyleView('reviewx'),
                $this->getSettingsProperty('graph_style'),
                $default
            );
        }

        return $default;
    }

    /**
     * @return string
     */
    protected function getStylePath()
    {
        return "storefront/review-summery/" . $this->getStyle() . ".php";
    }

    /**
     * Load View
     */
    public function loadView()
    {
        $review_criteria = $this->getSettingsProperty('review_criteria');
        $arg = array(
            'post_type' 			=> 'product',
            'post_id' 				=> get_the_ID(),
            'type'  				=> 'review',
        );

        $comment_lists 				= get_comments($arg);
        $get_all_recommended_query	= reviewx_product_recommendation_count_meta(get_the_ID());
        $criteria_arr 				= array();
        $criteria_count 			= array();
        $total_review_count 		= array();
        $rx_total_rating_sum        = 0;

        foreach( $comment_lists as $comment_list ) {

            $get_criteria 			= get_comment_meta($comment_list->comment_ID, 'reviewx_rating', true );
            $criteria_query_array 	= $get_criteria;
            $criteria_query_obj 	= (object)$get_criteria;
            $get_rating_val 			= get_comment_meta( $comment_list->comment_ID, 'rating', true );
            if ( !empty( $get_rating_val ) ) {
                $rx_total_rating_add =  $get_rating_val;
                $rx_total_rating_count = array_push( $total_review_count, $rx_total_rating_add );
                $rx_total_rating_sum += $get_rating_val;
            }
            
            if( is_array($review_criteria) ) {
                foreach( $review_criteria as $key => $single_criteria ) {
                    $hasKey = array_key_exists($key, $criteria_query_array);
                    if ($hasKey) {
                        if (isset($criteria_arr[$key])) {
                            $criteria_arr[$key] += $criteria_query_array[$key];
                        } else {
                            $criteria_arr[$key] = $criteria_query_array[$key];
                        }
                    }
                    if($hasKey) {
                        if (isset($criteria_count[$key])) {
                            $criteria_count[$key] += 1;
                        } else {
                            $criteria_count[$key] = 1;
                        }
                    }
                }
            }
        }
       
        $rx_count_total_rating_avg = round( $rx_total_rating_sum / $rx_total_rating_count, 2);
        $cri = apply_filters(Filters::REVIEWX_ADD_CRITERIA, $this->getSettingsProperty('review_criteria'));
        $allow_recommendation = get_option( '_rx_option_allow_recommendation' );

        return view_v1($this->getStylePath(), compact(
            'cri',
            'rx_count_total_rating_avg',
            'rx_total_rating_count',
            'allow_recommendation',
            'criteria_arr',
            'criteria_count'
        ));
    }
}