<?php

namespace ReviewX\Controllers\Admin\Email;

use ReviewX\Controllers\Admin\Core\ReviewxMetaBox;
use ReviewX\Controllers\Controller;
use ReviewX\Constants\ReminderEmail as ConstantsReminderEmail;
use ReviewX\Modules\Gatekeeper;

class EmailSettings extends Controller
{
    /**
    * The plugin option
    *
    * @since    1.0.0
    * @access   public
    * @var string option.
    */
    public static $prefix = 'rx_option_';

    /**
     * Send test email
     *
     * @return void
     */
    public function sendTestEmail()
    {
        try {
            $email = sanitize_email( $_POST['test_email'] );
            $headers = array('Content-Type: text/html; charset=UTF-8');
            $subject = __('Your Feedback Means a Lot to Us! | [SHOP_NAME] | ReviewX', 'reviewx' );

            wp_mail(
                $email,
                $subject,
                $this->sendTestEmailTemplate(),
                $headers
            );

            $this->response([
                'message' => __( 'Successfully email sent to your mail!', 'reviewx' )
            ]);
        } catch (\Exception $exception) {
             $this->response([
                'message' =>  __( 'Internal error to sent email! Please Try Again.', 'reviewx' )
            ], 400);
        }
    }

    /**
     * Email template for send test email
     *
     * @return void
     */
    protected function sendTestEmailTemplate()
    {
        return '<table class="body" style="border-collapse: collapse; border-spacing: 0; vertical-align: top; mso-table-lspace: 0pt; mso-table-rspace: 0pt; -ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; height: 100% !important; width: 100% !important; min-width: 100%; -moz-box-sizing: border-box; -webkit-box-sizing: border-box; box-sizing: border-box; -webkit-font-smoothing: antialiased !important; -moz-osx-font-smoothing: grayscale !important; background-color: #f1f1f1; color: #444; font-family: Helvetica,Arial,sans-serif; font-weight: normal; padding: 0; margin: 0; text-align: left; font-size: 14px; mso-line-height-rule: exactly; line-height: 140%;" border="0" width="100%" cellspacing="0" cellpadding="0"><tbody><tr style="padding: 0; vertical-align: top; text-align: left;"><td class="body-inner wp-mail-smtp" style="word-wrap: break-word; -webkit-hyphens: auto; -moz-hyphens: auto; hyphens: auto; border-collapse: collapse !important; vertical-align: top; mso-table-lspace: 0pt; mso-table-rspace: 0pt; -ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; color: #444; font-family: Helvetica,Arial,sans-serif; font-weight: normal; padding: 0; margin: 0; font-size: 14px; mso-line-height-rule: exactly; line-height: 140%; text-align: center;" align="center" valign="top"><table class="container" style="border-collapse: collapse; border-spacing: 0; padding: 0; vertical-align: top; mso-table-lspace: 0pt; mso-table-rspace: 0pt; -ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; width: 600px; margin: 0 auto 30px auto; text-align: inherit;" border="0" cellspacing="0" cellpadding="0"><tbody><tr style="padding: 0; vertical-align: top; text-align: left;"><td class="content" style="word-wrap: break-word; -webkit-hyphens: auto; -moz-hyphens: auto; hyphens: auto; border-collapse: collapse !important; vertical-align: top; mso-table-lspace: 0pt; mso-table-rspace: 0pt; -ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; color: #444; font-family: Helvetica,Arial,sans-serif; font-weight: normal; margin: 0; text-align: left; font-size: 14px; mso-line-height-rule: exactly; line-height: 140%; background-color: #ffffff; padding: 60px 75px 45px 75px; border-right: 1px solid #ddd; border-bottom: 1px solid #ddd; border-left: 1px solid #ddd; border-top: 3px solid #809eb0;" align="left" valign="top"><div class="success"><p class="text-large" style="-ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; color: #444; font-family: Helvetica,Arial,sans-serif; font-weight: normal; padding: 0; text-align: left; mso-line-height-rule: exactly; line-height: 140%; margin: 0 0 15px 0; font-size: 14px;">Hey [CUSTOMER_NAME],</p><p class="text-large" style="-ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; color: #444; font-family: Helvetica,Arial,sans-serif; font-weight: normal; padding: 0; text-align: left; mso-line-height-rule: exactly; line-height: 140%; margin: 0 0 15px 0; font-size: 14px;">Thank you for purchasing items from the [SHOP_NAME]. We love to know your experiences with the product(s) that you recently purchased.</p><p class="text-large" style="-ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; color: #444; font-family: Helvetica,Arial,sans-serif; font-weight: normal; padding: 0; text-align: left; mso-line-height-rule: exactly; line-height: 140%; margin: 0 0 15px 0; font-size: 14px;">You can browse a list of orders from your account page and can submit your feedback based on multiple criteria that we specially designed for you. To browse your orders: [MY_ORDERS_PAGE]</p><p class="text-large" style="-ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; color: #444; font-family: Helvetica,Arial,sans-serif; font-weight: normal; padding: 0; text-align: left; mso-line-height-rule: exactly; line-height: 140%; margin: 0 0 15px 0; font-size: 14px;">[ORDER_DATE] you placed the order [ORDER_ID]</p><p class="text-large" style="-ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; color: #444; font-family: Helvetica,Arial,sans-serif; font-weight: normal; padding: 0; text-align: left; mso-line-height-rule: exactly; line-height: 140%; margin: 0 0 15px 0; font-size: 14px;">[ORDER_ITEMS]</p><p class="text-large" style="-ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; color: #444; font-family: Helvetica,Arial,sans-serif; font-weight: normal; padding: 0; text-align: left; mso-line-height-rule: exactly; line-height: 140%; margin: 0 0 15px 0; font-size: 14px;">Your feedback means a lot to us! Thanks for being a loyal [SHOP_NAME] customer.</p><p class="text-large" style="-ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; color: #444; font-family: Helvetica,Arial,sans-serif; font-weight: normal; padding: 0; text-align: left; mso-line-height-rule: exactly; line-height: 140%; margin: 0 0 15px 0; font-size: 14px;">Regards,</p><p class="text-large" style="-ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; color: #444; font-family: Helvetica,Arial,sans-serif; font-weight: normal; padding: 0; text-align: left; mso-line-height-rule: exactly; line-height: 140%; margin: 0 0 15px 0; font-size: 14px;">Team [SHOP_NAME]</p></div><h6>If you want to unsubscribe this email please go to this [UNSUBSCRIBE_LINK]</h6></td></tr></tbody></table></td></tr></tbody></table>';
    }

    /**
     * Check is processable or not
     *
     * @param [type] $orderId
     * @return boolean
     */
    public function isProcessAble($orderId)
    {
        return $this->checkOrderEmailConsent($orderId);
    }

    /**
     * Check order email consent
     *
     * @param [type] $orderId
     * @return void
     */
    protected function checkOrderEmailConsent($orderId)
    {
        $consent = true;
        if ( \ReviewX_Helper::is_pro() ) {
            $consentOption = get_option('_rx_option_consent_email_subscription', false);
            if ($consentOption) {
                $value = get_post_meta($orderId, 'consent_email_subscription', true);
                $consentValue = is_numeric($value) ? $value : null;
                if (is_null($consentValue)) {
                    $consent = true;
                } else {
                    $consent = $consentValue ? true : false;
                }
            }
        }
        return $consent;
    }

    /**
     * Add Consent Field in checkout
     *
     * @param [type] $fields
     * @return void
     */
    public function addConsentFieldInCheckout($fields)
    {
        $fields['order']['consent_email_subscription'] = [
            'type' => 'checkbox',
            'label' => __('I want to subscribe email', 'reviewx'),
        ];
        return $fields;
    }

    /**
     * Update Consent Field meta
     *
     * @param [type] $orderId
     * @return void
     */
    public function updateConsentFieldMeta($orderId)
    {
        $field = isset( $_POST['consent_email_subscription'] ) ? $_POST['consent_email_subscription'] : 0;
        update_post_meta( $orderId, 'consent_email_subscription', ( filter_var($field, FILTER_SANITIZE_NUMBER_INT) ) );
    }

    /**
     * Process Bulk Email
     *
     * @param [type] $unreviewdOrderWiseProduct
     * @return void
     */
    public function processBulkEmail(&$unreviewdOrderWiseProduct)
    {
        $orderItems = wpFluent()->table('woocommerce_order_items')
                    ->join('posts', function ($query) {
                        $query->on('posts.ID', '=', 'woocommerce_order_items.order_id');
                    })
                    ->get();

        foreach ($orderItems as $order) {
            try {
                $getProductId = wc_get_order_item_meta($order->order_item_id, '_product_id', true);
                $userId = get_post_meta($order->ID, '_customer_user', true);
                $isPossibleToSendEmail = (new Gatekeeper())->getLogForVerifiedUser($getProductId, $order->ID, $userId);
                
                if ($isPossibleToSendEmail) {
                    if (isset($unreviewdOrderWiseProduct[$order->ID])) {
                        $unreviewdOrderWiseProduct[$order->ID][] = $getProductId;
                    } else {
                        $unreviewdOrderWiseProduct[$order->ID] = [$getProductId];
                    }
                }
            } catch (\Exception $e) {
                throw new \Error($e->getMessage());
            }
        }

        $defaultArgs = ReviewxMetaBox::get_quick_setup_args();
        $defaultSubject = $defaultArgs['tabs']['email_tab']['sections']['image']['fields']['email_subject']['default'];
        $defaultEditor = $defaultArgs['tabs']['email_tab']['sections']['image']['fields']['email_editor']['default'];

        $email_subject = get_option('_'.self::$prefix.'email_subject', $defaultSubject) ? : $defaultSubject;
        $email_editor = get_option( '_'.self::$prefix.'email_editor', $defaultEditor) ? : $defaultEditor;

        foreach ($unreviewdOrderWiseProduct as $orderId => $products) {
            $emailConsent = $this->isProcessAble($orderId);

            if ($emailConsent) {

                $reminderEmail = (new ReminderEmail())->getEmailLog($orderId);

                if (! is_null($reminderEmail)) {
                    
                    $isProcessable = (new ReminderEmail())->isProcessAble($reminderEmail);

                    if ($isProcessable) {
                        $email = unserialize($reminderEmail->processed_email);
                    
                        wp_mail(
                            $reminderEmail->customer_email,
                            $email['subject'],
                            $email['text'],
                            array($email['headers'])
                        );

                        $totalDelivery = intval($reminderEmail->total_delivery) + 1;
                        (new ReminderEmail())->updateLog($reminderEmail->id, [
                            'total_delivery' => $totalDelivery,
                            'scheduled_at' => date('Y-m-d H:i:s')
                        ]);
                    }

                } else {
                    
                    $editorParser = (new Editor($orderId, $products, $email_editor));
                    $subjectParser = (new Editor($orderId, $products, apply_filters("rx_email_subject", $email_subject)));
                    $headers = 'Content-Type: text/html; charset=UTF-8';
                    $text = $editorParser->prepareEmail(apply_filters('rx_email_editor_keys', []));
                    $subject = $subjectParser->prepareEmail([]);

                    $email = [
                        'user_id' => $userId,
                        'headers' => $headers,
                        'subject' => $subject,
                        'text' => $text,
                    ];
                    
                    $wcOrder = new \WC_Order($orderId);
                    $userEmail = $wcOrder->get_billing_email();
                    $wcOrder->get_customer_id();

                    wp_mail($userEmail, $subject, $text, array($headers));

                    (new ReminderEmail())->setEmailLog(
                        $orderId,
                        $userEmail,
                        $products,
                        serialize($email),
                        date('Y-m-d H:i:s'),
                        ConstantsReminderEmail::STATUS_DELIVERED,
                        1
                    );
                }  
                
            }
        }
    }

    public function getAutoReviewReminderConsent()
    {
        return get_option('_rx_option_auto_review_reminder', false);
    }

    /**
     * Process Single Email
     *
     * @param [type] $orderId
     * @return void
     */
    public function processSingleEmail($orderId)
    {
        if (! $this->getAutoReviewReminderConsent()) {
            return;
        }

        $products = [];
        
        $orderItems = wpFluent()->table('woocommerce_order_items')
                    ->where('order_id', $orderId)
                    ->get();

        foreach ($orderItems as $order) {
            $getProductId = wc_get_order_item_meta($order->order_item_id, '_product_id', true);
            $products[] = $getProductId;
        }

        $emailConsent = $this->isProcessAble($orderId);

        $email_subject = get_option('_'.self::$prefix.'email_subject') ? : '';
        $email_editor = get_option( '_'.self::$prefix.'email_editor') ? : '';

        if ($emailConsent) {

            $reminderEmail = (new ReminderEmail())->getEmailLog($orderId);

            $daysToWait = get_option('_rx_option_get_review_email', 1);
                
            $cronAfter = (( 24 * 60 * 60 * intval($daysToWait)));
            $scheduleAt = time() + $cronAfter;

            if (is_null($reminderEmail)) {
                
                $editorParser = (new Editor($orderId, $products, $email_editor));
                $subjectParser = (new Editor($orderId, $products, apply_filters("rx_email_subject", $email_subject)));
                $headers = 'Content-Type: text/html; charset=UTF-8';
                $text = $editorParser->prepareEmail(apply_filters('rx_email_editor_keys', []));
                $subject = $subjectParser->prepareEmail([]);

                $email = [
                    'headers' => $headers,
                    'subject' => $subject,
                    'text' => $text,
                ];
                
                $wcOrder = new \WC_Order($orderId);

                $userEmail = $wcOrder->get_billing_email();

                (new ReminderEmail())->setEmailLog(
                    $orderId,
                    $userEmail,
                    $products,
                    serialize($email),
                    date('Y-m-d H:i:s', $scheduleAt),
                    ConstantsReminderEmail::STATUS_SCHEDULED,
                    0
                );

            }

            wp_schedule_single_event($scheduleAt, 'rx_reminder_email_dispatch_scheduled', [
                'order_id' => $orderId
            ]);
        }

    }

    /**
     * Send Schedule Email
     *
     * @param [type] $orderId
     * @return void
     */
    public function sendScheduleEmail($orderId)
    {
        $reminderEmail = (new ReminderEmail())->getEmailLog($orderId);

        $email = unserialize($reminderEmail->processed_email);
            
        wp_mail(
            $reminderEmail->customer_email,
            $email['subject'],
            $email['text'],
            array($email['headers'])
        );

        $totalDelivery = intval($reminderEmail->total_delivery) + 1;

        (new ReminderEmail())->updateLog($reminderEmail->id, [
            'total_delivery' => $totalDelivery,
            'status' => ConstantsReminderEmail::STATUS_DELIVERED
        ]);
    }

}